/*
	SuperCollider real time audio synthesis system
    Copyright (c) 2002 James McCartney. All rights reserved.
	http://www.audiosynth.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <Cocoa/Cocoa.h>


static int linepos, lineno, charno, yylen;
static int textpos, textlen, totallen;
static unichar *text;
static unichar yytext[256];

static bool streq(char *cstr, unichar *ustr)
{
    while (true) {
        char c = *cstr++;
        unichar u = *ustr++;
        if (u != c) return false;
        else if (u == 0) return true;
    }
    return false;
}

static unichar input() 
{
	unichar c;
	if (textpos >= textlen) {
		c = 0;
	} else {
		c = text[textpos++];
		charno++;
	}
	if (c == '\n' || c == '\r') {
		lineno++;
		linepos = textpos;
		charno = 0;
	}
	if (c != 0) yytext[yylen++] = c;
	//if (gDebugLexer) post("input '%c' %d\n",c,c);
	return c;
}

static unichar input0() 
{
	unichar c;
	if (textpos >= textlen) {
		c = 0;
		textpos++; // so unput will work properly
	} else {
		c = text[textpos++];
		charno++;
	}
	if (c == '\n' || c == '\r') {
		lineno++;
		linepos = textpos;
		charno = 0;
	}
	//if (gDebugLexer) post("input0 '%c' %d\n",c,c);
	return c;
}

static void unput(unichar c) 
{
	if (textpos>0) textpos--;
	if (c) {
		if (yylen) --yylen;
		if (charno) --charno;
		if (c == '\n' || c == '\r') {
			--lineno;
		}
	}
}

unichar* startColorizer(NSTextView* textView);
unichar* startColorizer(NSTextView* textView) 
{
    totallen = [[textView textStorage] length];

    NSRange range = [textView selectedRange];
    if (range.length == 0) range = NSMakeRange(0, totallen);

    if (![textView shouldChangeTextInRange: range replacementString: nil]) return 0;
    
    textpos = range.location;
    textlen = textpos + range.length;
    
    text = (unichar*)malloc((totallen+1) * sizeof(unichar));
    [[[textView textStorage] string] getCharacters: text];
    text[totallen] = 0;
    
    linepos = 0;
    lineno = 1;
    charno = 0;

    yylen = 0;

    return text;
}

void SyntaxColorize(NSTextView* textView)
{
	unichar c;
	long startrun = 0, endrun;
        NSRange range;
	int startline, clevel, prevc;
        NSColor* black = [NSColor colorWithCalibratedRed: 0. green: 0. blue: 0. alpha: 1.];
        NSColor* darkRed = [NSColor colorWithCalibratedRed: 0.75 green: 0. blue: 0. alpha: 1.];
        NSColor* darkBlue = [NSColor colorWithCalibratedRed: 0. green: 0. blue: 0.75 alpha: 1.];
        NSColor* darkGreen = [NSColor colorWithCalibratedRed: 0. green: 0.45 blue: 0. alpha: 1.];
        NSColor* darkGray = [NSColor colorWithCalibratedRed: 0.375 green: 0.375 blue: 0.375 alpha: 1.];
	
	if (!startColorizer(textView)) return;
	
	// reset to black
	        
        range = NSMakeRange(textpos, textlen - textpos);
        [textView setTextColor: black range: range];
	
	bool bracket = false;

	for (;;) {
		yylen = 0;
		c = input();
		if (c == 0 || textpos >= textlen) goto leave;
		else if (c==' ' || c=='\t' || c=='\v' || c=='\f' || c=='\n' || c=='\r') {
			continue;
		}
		
		if (c == '|' && bracket) goto arglist;
		if (c == '{') { bracket = true; continue; }
		bracket = false;
		
		if (c >= 'A' && c <= 'Z' || c >= 'a' && c <= 'z' || c == '_') goto ident;
		else if (c == '\\') goto symbol1;
		else if (c == '\'') goto symbol3;
		else if (c == '"') goto string1;
		else if (c == '/') {
			c = input0();
			if (c == '/') goto comment1;
			else if (c == '*') goto comment2;
			else {
				unput(c);
				continue; 
			}
		}
		else if (c == '$') { 	
			c = input0();
			if (c == '\\') {
				c = input0();
			}
			continue; 
		} else {
			continue; 
		}
	ident:
		c = input();
		
		if (c >= 'A' && c <= 'Z' || c >= 'a' && c <= 'z' 
			|| c == '_' || c >= '0' && c <= '9') goto ident;
		else if (c == ':') {
			yytext[yylen] = 0;
			//r = processkeywordbinop(yytext) ; 
			continue;
		} else {
			unput(c);
			yytext[yylen] = 0;
			endrun = textpos;
			startrun = textpos - yylen;
                        range = NSMakeRange(startrun, endrun - startrun);
			if (yytext[0] == '_') {
                                [textView setTextColor: darkBlue range: range];
                        }
			else if (yytext[0] >= 'A' && yytext[0] <= 'Z') {
                                [textView setTextColor: darkBlue range: range];
                        }
			else if (  streq("var",yytext)
					|| streq("arg",yytext)
					|| streq("classvar",yytext) 
					|| streq("const",yytext)
			) {
                                range = NSMakeRange(startrun, endrun - startrun);
                                [textView setTextColor: darkBlue range: range];
				}
			else if (  streq("this",yytext)
					|| streq("super",yytext)
					|| streq("thisProcess",yytext)
					|| streq("thisThread",yytext) 
					|| streq("thisMethod",yytext)
					|| streq("thisFunctionDef",yytext)
					|| streq("thisFunction",yytext)
			) {
                                [textView setTextColor: darkBlue range: range];
                        }
			else if (  streq("nil",yytext)
					|| streq("false",yytext)
					|| streq("true",yytext) 
					|| streq("inf",yytext)
			) {
                                [textView setTextColor: darkBlue range: range];
                        }
			continue;
		}
	symbol1:
		c = input();
		
		if (c >= 'A' && c <= 'Z' || c >= 'a' && c <= 'z' 
			|| c == '_') {
				startrun = textpos - 2;
				goto symbol2;
		} else {
			unput(c);
			yytext[yylen] = 0;
			continue;
		}
	symbol2:
		c = input();
		
		if (c >= 'A' && c <= 'Z' || c >= 'a' && c <= 'z' 
			|| c == '_' || c >= '0' && c <= '9') goto symbol2;
		else {
			unput(c);
			endrun = textpos;
                        range = NSMakeRange(startrun, endrun - startrun);
                        [textView setTextColor: darkGreen range: range];
			continue;
		}
	symbol3 : {	
			int startline;
                        unichar endchar;
			startrun = textpos - 1;
			startline = lineno;
			endchar = '\'';
		
			do {
				c = input0();
				if (c == '\\') {
					c = input0();
				}
			} while (c != endchar && c != 0);
			if (c == 0 || textpos >= textlen) {
				//postbuf("Open ended symbol ... started on line %d in file '%s'\n", 
				//	startline, curfilename);
				goto error2;
			} else {
				endrun = textpos;
                                range = NSMakeRange(startrun, endrun - startrun);
                                [textView setTextColor: darkGreen range: range];
				continue;
			}
		}

	string1 : {	
			int startline;
                        unichar endchar;
                        
			startrun = textpos - 1;
			startline = lineno;
			endchar = '\"';
		
			do  {
				c = input0();
				if (c == '\\') {
					c = input0();
				}
			} while (c != endchar && c != 0);
			if (c == 0 || textpos >= textlen) {
				//postbuf("Open ended string ... started on line %d in file '%s'\n", 
				//	startline, curfilename);
				goto error2;
			} else {
				endrun = textpos;
                                range = NSMakeRange(startrun, endrun - startrun);
                                [textView setTextColor: darkGray range: range];
				continue;
			}
		}
	arglist:
		startrun = textpos-1;
		do {
			c = input0(); 
		} while (c != '|' && c != 0);
		endrun = textpos;
                range = NSMakeRange(startrun, endrun - startrun);
                [textView setTextColor: darkBlue range: range];
		continue;
	comment1:	/* comment -- to end of line */
		startrun = textpos-2;
		do {
			c = input0(); 
		} while (c != '\n' && c != '\r' && c != 0);
		endrun = textpos;
                range = NSMakeRange(startrun, endrun - startrun - 1);
                [textView setTextColor: darkRed range: range];
		continue;

	comment2 : 
		startrun = textpos-2;
		startline = lineno;
		prevc = 0;
		clevel = 1;
		do {
			c = input0();
			if (c == '/' && prevc == '*') {
				if (--clevel <= 0) break;
			} else if (c == '*' && prevc == '/') clevel++;
			prevc = c;
		} while (c != 0);
		if (textpos > textlen) {
			goto error2;
		} else {
			endrun = textpos;
                        range = NSMakeRange(startrun, endrun - startrun);
                        [textView setTextColor: darkRed range: range];
			continue;
		}
	}
	
error2:
	
leave:
    free(text);
}
