'''
 ====================================================================
 Copyright (c) 2003-2005 Barry A Scott.  All rights reserved.

 This software is licensed as described in the file LICENSE.txt,
 which you should have received as part of this distribution.

 ====================================================================

    wb_subversion_tree_handler.py

'''
import os
import pysvn
import wx
import wb_tree_panel
import wb_ids
import wb_subversion_utils
import wb_subversion_project_info
import wb_subversion_utils
import wb_subversion_info_dialog
import wb_subversion_history
import wb_subversion_properties_dialog

class SubversionProject(wb_tree_panel.TreeProjectItem):
    def __init__( self, app, project_info ):
        wb_tree_panel.TreeProjectItem.__init__( self )
        self.project_info = project_info
        self.app = app

    def updateStatus( self ):
        self.project_info.updateStatus()

    def getProjectInfo( self ):
        return self.project_info

    def mayExpand( self ):
        dir_status = self.project_info.getDirStatus()
        if dir_status is None:
            # no status available - make a guess
            if not os.path.exists( self.project_info.wc_path ):
                # nothing there 
                return False
            else:
                # some dir assume can expand
                return True

        for file in self.project_info.getTreeFilesStatus():
            if( (file.entry is not None and file.entry.kind == pysvn.node_kind.dir)
            or (file.entry is None and os.path.isdir( file.path )) ):
                return True

        return False

    def getExpansion( self ):
        project_info_list = []

        for file in self.project_info.getTreeFilesStatus():

            if( (file.entry is None and os.path.isdir( file.path ))
            or (file.entry is not None and file.entry.kind == pysvn.node_kind.dir) ):
                pi = wb_subversion_project_info.ProjectInfo( self.app, self.project_info )
                name = os.path.basename( file.path )
                if file.entry is None:
                    # fake up the url - maybe a failed checkout/update
                    url = '%s/%s' % (self.project_info.url, name )
                else:
                    url = file.entry.url

                pi.init( name, url=url, wc_path=file.path,
                    client_fg=self.project_info.client_fg,
                    client_bg=self.project_info.client_bg )
                project_info_list.append( pi )

        return project_info_list

    def getTreeNodeColour( self ):
        dir_status = self.project_info.getDirStatus()
        if dir_status is None:
            # no status available - make a guess
            if not os.path.exists( self.project_info.wc_path ):
                # nothing there
                return wx.RED
            elif not os.path.exists( os.path.join( self.project_info.wc_path, '.svn' ) ):
                # not versioned
                return wx.GREEN
            else:
                # versioned and present
                return wx.BLACK

        elif not os.path.exists( dir_status.path ):
            # nothing there
            return wx.RED
        elif dir_status.text_status in [pysvn.wc_status_kind.unversioned]:
            # unversioned
            return wx.GREEN

        # versioned and present
        return wx.BLACK
    
    def getState( self ):
        state = wb_tree_panel.TreeState()

        dir_status = self.project_info.getDirStatus()


        state.is_project_parent = self.is_project_parent

        if dir_status is None:
            state.modified = False
            state.new_versioned = False
            state.versioned = True
            state.unversioned = False
            state.need_checkin = False
            state.conflict = False
            state.file_exists = False
        else:
            state.modified = True
            state.new_versioned = True
            state.versioned = True
            state.unversioned = True
            state.need_checkin = True
            state.conflict = True
            state.file_exists = True

            filename = dir_status.path
            if not os.path.exists( filename ):
                state.file_exists = False

            text_status = dir_status.text_status
            if text_status in [pysvn.wc_status_kind.unversioned]:
                state.versioned = False
            else:
                state.unversioned = False

            state.new_versioned = state.new_versioned and text_status in [pysvn.wc_status_kind.added]

            prop_status = dir_status.prop_status
            state.modified = (text_status in [pysvn.wc_status_kind.modified,
                                pysvn.wc_status_kind.conflicted]
                        or
                        prop_status in [pysvn.wc_status_kind.modified,
                                pysvn.wc_status_kind.conflicted])
            state.need_checkin = (text_status in [pysvn.wc_status_kind.added,
                                    pysvn.wc_status_kind.deleted,
                                    pysvn.wc_status_kind.modified]
                            or
                            prop_status in [pysvn.wc_status_kind.added,
                                    pysvn.wc_status_kind.deleted,
                                    pysvn.wc_status_kind.modified])
            state.conflict = text_status in [pysvn.wc_status_kind.conflicted]

        return state

    def getContextMenu( self, state ):
        menu_item =[('', wb_ids.id_Command_Shell, '&Command Shell')
            ,('', wb_ids.id_File_Browser, '&File Browser')
            ,('-', 0, 0 )
            ,('', wb_ids.id_SP_History, 'History...' )
            ,('', wb_ids.id_SP_Info, 'Information...' )
            ,('', wb_ids.id_SP_Properties, 'Properties...' )
            ,('-', 0, 0 )]
        if not state.file_exists and state.versioned and self.isProjectParent():
            menu_item += [('', wb_ids.id_SP_Checkout, 'Checkout' )]
        else:
            menu_item += [('', wb_ids.id_SP_Update, 'Update' )]
        menu_item += [('-', 0, '' )
            ,('', wb_ids.id_SP_Checkin, 'Checkin...' )
            ,('-', 0, 0 )
            ,('', wb_ids.id_SP_Mkdir, 'Make directory...' )
            ,('', wb_ids.id_SP_Add, 'Add...' )
            ,('', wb_ids.id_SP_Rename, 'Rename...' )
            ,('-', 0, 0 )
            ,('', wb_ids.id_SP_Delete, 'Delete' )
            ,('', wb_ids.id_SP_Revert, 'Revert' )
            ,('-', 0, 0 )
            ,('', wb_ids.id_SP_Cleanup, 'Clean up' )
            ]

        return wb_subversion_utils.populateMenu( wx.Menu(), menu_item )

    def Cmd_Dir_Add( self ):
        name = os.path.basename( self.project_info.wc_path )
        force = self.app.addFile( 'Add Directory', name, False )
        if force is None:
            return

        try:
            self.project_info.client_fg.add( self.project_info.wc_path, recurse=True, force=force )
        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        self.app.refreshFrame()

    def Cmd_Dir_Checkout( self ):
        self.app.setAction( 'Checkout %s...' % self.project_info.url )

        yield self.app.backgroundProcess
        try:
            # lose any trailing / on the URL it stops checkout working
            url = self.project_info.url
            if url[-1] == '/':
                url = url[:-1]
            self.project_info.client_bg.checkout( url, self.project_info.wc_path, recurse=True )
        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        yield self.app.foregroundProcess

        self.app.setAction( 'Ready' )
        self.app.refreshFrame()

    def Cmd_Dir_Cleanup( self ):
        self.app.setAction( 'Clean up %s...' % self.project_info.wc_path )

        yield self.app.backgroundProcess
        try:
            self.project_info.client_bg.cleanup( self.project_info.wc_path )
        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        yield self.app.foregroundProcess

        self.app.setAction( 'Ready' )
        self.app.refreshFrame()

    def Cmd_Dir_Checkin( self ):
        # get a status output to show the user

        all_files = self.project_info.client_fg.status( self.project_info.wc_path, recurse=True, get_all=False, ignore=True, update=False )
        status_output = self._cmd_status_output( all_files )
        if len(status_output) == 0:
            wx.MessageBox( "There are no changes to checkin",
                "Warning", style=wx.OK|wx.ICON_EXCLAMATION )
            return

        message = self.app.getLogMessage( 'Check in', status_output )

        if not message:
            return

        self.app.setAction( 'Check in %s...' % self.project_info.wc_path )
        self.app.setProgress( 'Sent %(count)d of %(total)d', len( status_output ) )

        yield self.app.backgroundProcess

        ok = False
        try:
            rev = self.project_info.client_bg.checkin( self.project_info.wc_path, message )
            ok = True
        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        yield self.app.foregroundProcess

        if ok:
            if rev:
                self.app.log.info( 'Checkin created revision %d' % rev.number )
            else:
                self.app.log.warning( 'No changes to checkin ' )

        self.app.refreshFrame()
        self.app.setAction( 'Ready' )
        self.app.clearProgress()

    def _cmd_status_output( self, all_files ):
        output_lines = []
        all_files.sort( wb_subversion_utils.by_path )
        for file in all_files:
            state = wb_subversion_utils._status_format( file )

            if( wb_subversion_utils.wc_status_checkin_map[ file.text_status ]
            or  wb_subversion_utils.wc_status_checkin_map[ file.prop_status ] ):
                output_lines.append( (state, file.path) )

        return output_lines

    def Cmd_Dir_Delete( self ):
        if self.app.confirmAction( 'Delete', [('', self.project_info.wc_path)] ):
            try:
                self.project_info.client_fg.remove( self.project_info.wc_path )
            except pysvn.ClientError, e:
                self.app.log_client_error( e )
            self.app.refreshFrame()

    def Cmd_Dir_History( self ):
        filename = self.project_info.wc_path
        self.app.setAction( 'History %s...' % filename )

        yield self.app.backgroundProcess

        ok = False
        history_entries = []
        try:
                log_entries = self.project_info.client_bg.log( filename, strict_node_history=True )
                for log in log_entries:
                    history_entries.append( (log['revision'].number,
                        log['author'],
                        log['date'],
                        '',
                        log['message'] ) )
                info = self.project_info.client_bg.info( filename )

                tags_url = self.project_info.getTagsUrl( info.url )
                if tags_url:
                    for ls_info in self.project_info.client_bg.ls( tags_url ):
                        history_entries.append( (ls_info['created_rev'].number,
                            ls_info['last_author'],
                            ls_info['time'],
                            'Tag ' + ls_info['name'].split('/')[-1],
                            '' ) )


                history_entries.sort()

                ok = True
        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        yield self.app.foregroundProcess

        if ok:
            h_frame = wb_subversion_history.HistoryDirFrame( self.app, filename, info.url, history_entries )
            h_frame.Show( True )

        self.app.setAction( 'Ready' )

    def Cmd_Dir_Info( self ):
        try:
            entry = self.project_info.client_fg.info( self.project_info.wc_path )

            dialog = wb_subversion_info_dialog.InfoDialog( self.app,
                    self.app.frame.tree_panel.tree_ctrl,
                    self.project_info.wc_path,
                    entry )
            dialog.ShowModal()

        except pysvn.ClientError, e:
            self.app.log_client_error( e )


    def Cmd_Dir_Mkdir( self ):
        rc, name = self.app.getFilename( 'Make directory', 'Name:' )
        if not rc:
            return

        try:
            new_path = os.path.join( self.project_info.wc_path, name )
            # pass an empty message
            self.project_info.client_fg.mkdir( new_path,  '' )

        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        # open up the current tree node to show the created item
        self.app.expandSelectedTreeNode()

    def Cmd_Dir_Properties( self ):
        client_fg = self.project_info.client_fg

        filename = self.project_info.wc_path

        try:
            prop_list = client_fg.proplist( filename,
                    revision=pysvn.Revision( pysvn.opt_revision_kind.working ) )
            if len(prop_list) == 0:
                prop_dict = {}
            else:
                _, prop_dict = prop_list[0]
            dialog = wb_subversion_properties_dialog.DirPropertiesDialog( self.app,
                    self.app.frame.tree_panel.tree_ctrl,
                    filename,
                    prop_dict )
            if dialog.ShowModal() == wx.OK:
                for present, name, value in dialog.getModifiedProperties():
                    if not present:
                        # delete name
                        client_fg.propdel( name, filename )
                    else:
                        # add/update name value
                        client_fg.propset( name, value, filename )

        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        self.app.refreshFrame()

    def Cmd_Dir_Rename( self ):
        old_filename = self.project_info.wc_path
        old_name = os.path.basename( old_filename )

        new_name, force = self.app.renameFile( "Rename Directory", old_name, False )

        if new_name is None:
            return

        if new_name != old_name:
            new_full_filename = os.path.join( os.path.dirname( old_filename ), new_name )
            dir_status = self.project_info.getDirStatus()
            print 'Rename',old_filename, new_full_filename
            if dir_status is None or dir_status.text_status not in [pysvn.wc_status_kind.unversioned]:
                try:
                    self.project_info.client_fg.move( old_filename, new_full_filename, force=force )
                except pysvn.ClientError, e:
                    self.app.log_client_error( e )
            else:
                try:
                    os.rename( old_filename, new_full_filename )
                except (OSError,IOError), e:
                    self.app.log.error( str(e) )

            self.app.selectTreeNodeInParent( new_name )
        self.app.refreshFrame()

    def Cmd_Dir_Revert( self ):
        # get a status output to show the user
        all_files = self.project_info.client_fg.status( self.project_info.wc_path, recurse=True, get_all=False, ignore=True, update=False )
        status_output = self._cmd_status_output( all_files )
        if len(status_output) == 0:
            wx.MessageBox( "There are no changes to revert",
                "Warning", style=wx.OK|wx.ICON_EXCLAMATION )
            return


        if self.app.confirmAction( 'Revert', status_output ):
            try:
                self.project_info.client_fg.revert( self.project_info.wc_path, recurse=True )
            except pysvn.ClientError, e:
                self.app.log_client_error( e )
            self.app.refreshFrame()

    def Cmd_Dir_Update( self ):
        self.app.setAction( 'Update %s...' % self.project_info.wc_path )
        self.app.setProgress( 'Updated %(count)d', 0 )

        yield self.app.backgroundProcess

        ok = False
        try:
            rev = self.project_info.client_bg.update( self.project_info.wc_path, recurse=True )
            ok = True
        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        yield self.app.foregroundProcess

        if ok:
            if rev.number > 0:
                count = self.app.getProgressValue( 'count' )
                if count == 0:
                    self.app.log.info( 'Updated to revision %d, no new updates' % rev.number )
                elif count == 1:
                    self.app.log.info( 'Updated to revision %d, 1 new update' % rev.number )
                else:
                    self.app.log.info( 'Updated to revision %d, %d new updates' % (rev.number, count) )
            else:
                self.app.log.warning( 'Already up to date' )

        self.app.clearProgress()
        self.app.setAction( 'Ready' )
        self.app.refreshFrame()

    def Cmd_Dir_Copy( self, all_filenames ):
        try:
            for src_filename in all_filenames:
                self.project_info.client_fg.copy( src_filename, self.project_info.wc_path )
                self.app.log.info( 'Copied %s to %s' % (src_filename, self.project_info.wc_path) )
        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        self.app.refreshFrame()

    def Cmd_Dir_Move( self, all_filenames ):
        try:
            for src_filename in all_filenames:
                self.project_info.client_fg.move( src_filename, self.project_info.wc_path )
                self.app.log.info( 'Moved %s to %s' % (src_filename, self.project_info.wc_path) )
        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        self.app.refreshFrame()
