#include <stdlib.h>
#include <string.h>
#ifndef WIN32
#include "config.h"
#endif

#ifdef ENABLE_NLS
#include <libintl.h>
#endif

#define         _ISOC9X_SOURCE  1
#define         _ISOC99_SOURCE  1
#define         __USE_ISOC99    1
#define         __USE_ISOC9X    1

#include <math.h>

#include "ladspa.h"

#ifdef WIN32
#define _WINDOWS_DLL_EXPORT_ __declspec(dllexport)
int bIsFirstTime = 1; 
void _init(); // forward declaration
#else
#define _WINDOWS_DLL_EXPORT_ 
#endif

#line 10 "vocoder_1911.xml"

#include "util/biquad.h"

#define BANDS 8

const static float bfreqs[BANDS] = {
  50.0f, 100.0f, 200.0f, 400.0f, 800.0f, 1600.0f, 3200.0f, 6400.0f
};

#define VOCODER_ANAL                   0
#define VOCODER_SYNTH                  1
#define VOCODER_OUT                    2

static LADSPA_Descriptor *vocoderDescriptor = NULL;

typedef struct {
	LADSPA_Data *anal;
	LADSPA_Data *synth;
	LADSPA_Data *out;
	float *      amp;
	biquad *     filter;
	float        fs;
	float *      sig;
	LADSPA_Data run_adding_gain;
} Vocoder;

_WINDOWS_DLL_EXPORT_
const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {

#ifdef WIN32
	if (bIsFirstTime) {
		_init();
		bIsFirstTime = 0;
	}
#endif
	switch (index) {
	case 0:
		return vocoderDescriptor;
	default:
		return NULL;
	}
}

static void activateVocoder(LADSPA_Handle instance) {
	Vocoder *plugin_data = (Vocoder *)instance;
	float *amp = plugin_data->amp;
	biquad *filter = plugin_data->filter;
	float fs = plugin_data->fs;
	float *sig = plugin_data->sig;
#line 39 "vocoder_1911.xml"
	unsigned int i;

	memset(amp, 0, sizeof(float) * BANDS);
	memset(sig, 0, sizeof(float) * BANDS);
	for (i=0; i<BANDS * 2; i++) {
	  biquad_init(filter+i);
	  bp_set_params(filter+i, bfreqs[i % BANDS], 1.0f, fs);
	}
	plugin_data->amp = amp;
	plugin_data->filter = filter;
	plugin_data->fs = fs;
	plugin_data->sig = sig;

}

static void cleanupVocoder(LADSPA_Handle instance) {
#line 50 "vocoder_1911.xml"
	Vocoder *plugin_data = (Vocoder *)instance;
	free(plugin_data->filter);
	free(instance);
}

static void connectPortVocoder(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Vocoder *plugin;

	plugin = (Vocoder *)instance;
	switch (port) {
	case VOCODER_ANAL:
		plugin->anal = data;
		break;
	case VOCODER_SYNTH:
		plugin->synth = data;
		break;
	case VOCODER_OUT:
		plugin->out = data;
		break;
	}
}

static LADSPA_Handle instantiateVocoder(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Vocoder *plugin_data = (Vocoder *)malloc(sizeof(Vocoder));
	float *amp = NULL;
	biquad *filter = NULL;
	float fs;
	float *sig = NULL;

#line 26 "vocoder_1911.xml"
	unsigned int i;

	fs = (float)s_rate;
	filter = malloc(sizeof(biquad) * BANDS * 2);
	amp = malloc(sizeof(float) * BANDS * 2);
	sig = malloc(sizeof(float) * BANDS);

	for (i=0; i<BANDS * 2; i++) {
	  biquad_init(filter+i);
	}

	plugin_data->amp = amp;
	plugin_data->filter = filter;
	plugin_data->fs = fs;
	plugin_data->sig = sig;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

static void runVocoder(LADSPA_Handle instance, unsigned long sample_count) {
	Vocoder *plugin_data = (Vocoder *)instance;

	/* Analysis (array of floats of length sample_count) */
	const LADSPA_Data * const anal = plugin_data->anal;

	/* Synthesis (array of floats of length sample_count) */
	const LADSPA_Data * const synth = plugin_data->synth;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data * const out = plugin_data->out;
	float * amp = plugin_data->amp;
	biquad * filter = plugin_data->filter;
	float fs = plugin_data->fs;
	float * sig = plugin_data->sig;

#line 54 "vocoder_1911.xml"
	unsigned long pos, i;

	for (pos = 0; pos < sample_count; pos++) {
	  float comb = 0.0f; /* combined result of synth filters */

	  for (i=0; i<BANDS; i++) {
	    amp[i] = fabs(biquad_run(filter+i, anal[pos]));
	    sig[i] = biquad_run(filter+BANDS+i, synth[pos]);
	    amp[BANDS+i] = fabs(sig[i]) + 0.01f;
	    //comb += sig[i] * (amp[i] / amp[BANDS+i]);
	    comb += sig[i];
	  }
	  buffer_write(out[pos], comb);
	}
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

static void setRunAddingGainVocoder(LADSPA_Handle instance, LADSPA_Data gain) {
	((Vocoder *)instance)->run_adding_gain = gain;
}

static void runAddingVocoder(LADSPA_Handle instance, unsigned long sample_count) {
	Vocoder *plugin_data = (Vocoder *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Analysis (array of floats of length sample_count) */
	const LADSPA_Data * const anal = plugin_data->anal;

	/* Synthesis (array of floats of length sample_count) */
	const LADSPA_Data * const synth = plugin_data->synth;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data * const out = plugin_data->out;
	float * amp = plugin_data->amp;
	biquad * filter = plugin_data->filter;
	float fs = plugin_data->fs;
	float * sig = plugin_data->sig;

#line 54 "vocoder_1911.xml"
	unsigned long pos, i;

	for (pos = 0; pos < sample_count; pos++) {
	  float comb = 0.0f; /* combined result of synth filters */

	  for (i=0; i<BANDS; i++) {
	    amp[i] = fabs(biquad_run(filter+i, anal[pos]));
	    sig[i] = biquad_run(filter+BANDS+i, synth[pos]);
	    amp[BANDS+i] = fabs(sig[i]) + 0.01f;
	    //comb += sig[i] * (amp[i] / amp[BANDS+i]);
	    comb += sig[i];
	  }
	  buffer_write(out[pos], comb);
	}
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

#ifdef ENABLE_NLS
#define D_(s) dgettext(PACKAGE, s)
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
#else
#define D_(s) (s)
#endif


	vocoderDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (vocoderDescriptor) {
		vocoderDescriptor->UniqueID = 1911;
		vocoderDescriptor->Label = "vocoder";
		vocoderDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		vocoderDescriptor->Name =
		 D_("Vocoder");
		vocoderDescriptor->Maker =
		 "Steve Harris <steve@plugin.org.uk>";
		vocoderDescriptor->Copyright =
		 "GPL";
		vocoderDescriptor->PortCount = 3;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(3,
		 sizeof(LADSPA_PortDescriptor));
		vocoderDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(3,
		 sizeof(LADSPA_PortRangeHint));
		vocoderDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(3, sizeof(char*));
		vocoderDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Analysis */
		port_descriptors[VOCODER_ANAL] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[VOCODER_ANAL] =
		 D_("Analysis");
		port_range_hints[VOCODER_ANAL].HintDescriptor = 0;

		/* Parameters for Synthesis */
		port_descriptors[VOCODER_SYNTH] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[VOCODER_SYNTH] =
		 D_("Synthesis");
		port_range_hints[VOCODER_SYNTH].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[VOCODER_OUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[VOCODER_OUT] =
		 D_("Output");
		port_range_hints[VOCODER_OUT].HintDescriptor = 0;

		vocoderDescriptor->activate = activateVocoder;
		vocoderDescriptor->cleanup = cleanupVocoder;
		vocoderDescriptor->connect_port = connectPortVocoder;
		vocoderDescriptor->deactivate = NULL;
		vocoderDescriptor->instantiate = instantiateVocoder;
		vocoderDescriptor->run = runVocoder;
		vocoderDescriptor->run_adding = runAddingVocoder;
		vocoderDescriptor->set_run_adding_gain = setRunAddingGainVocoder;
	}
}

void _fini() {
	if (vocoderDescriptor) {
		free((LADSPA_PortDescriptor *)vocoderDescriptor->PortDescriptors);
		free((char **)vocoderDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)vocoderDescriptor->PortRangeHints);
		free(vocoderDescriptor);
	}

}
