/*
   SwingWT
   Copyright(c)2003-2005, R. Rawson-Tetley
 
   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.
 
   Contact me by electronic mail: bobintetley@users.sourceforge.net
 
   $Log: Window.java,v $
   Revision 1.56  2005/01/05 09:22:25  bobintetley
   Updated copyright year on source

   Revision 1.55  2005/01/05 08:37:11  bobintetley
   Many compatibility fixes from David Barron

   Revision 1.54  2004/11/02 11:26:05  bobintetley
   Fix to default size of JTextArea/JTextField
 
   Revision 1.53  2004/11/02 11:07:30  bobintetley
   Fixed a number of small compatibility bugs
 
   Revision 1.52  2004/11/01 12:16:45  bobintetley
   Fixed JFileChooser drive box bug, Dialog size/display bug and a bug in the
   classpath file for Eclipse editing
 
   Revision 1.51  2004/10/30 20:11:54  bobintetley
   Code cleanup
 
   Revision 1.50  2004/10/29 10:24:04  bobintetley
   Separate EditorPane implementations to decouple platform specifics,
   stubbing of HTMLEditorKit
 
   Revision 1.49  2004/10/08 15:09:27  bobintetley
   Definitely think I got repaintFix this time - needs to subtly go a few
   ms after window creation
 
   Revision 1.48  2004/10/08 14:54:29  bobintetley
   Definitely nailled that resizing shell bug under GTK2 this time. Replaced
      peer resizing in repaintFix() with an SWT layout
 
   Revision 1.47  2004/10/07 10:09:14  bobintetley
   Bug fix to GTK2 to prevent Windows intermittently opening with a 1x1 pixel size
 
   Revision 1.46  2004/09/24 12:38:59  bobintetley
   MacOSX specific fixes for window position and use of platform toolbuttons
 
   Revision 1.45  2004/07/16 00:29:29  dannaab
   Bug fixes: Fix init order in JTable to ensure registration of all event listeners; reset isClosed in Window (to false) on peer creation, allowing reuse of the swing window objects (a common practice)
 
   Revision 1.44  2004/06/10 07:57:57  dannaab
   Added AWTSwingWrapper to map AWT components to the corresponding Swing version.  Implemted using it: TextComponent, TextArea, TextField; used similar technique for Menu components
 
   Revision 1.43  2004/06/08 09:24:21  dannaab
   Rename Component.getPeer() -> getSWTPeer().  added ComponentPeer and stubbed out support classes.
 
   Revision 1.42  2004/06/07 14:01:27  bobintetley
   JTable fixes for Win32, and ability to turn off SWT VIRTUAL tables. Off by
   default due to Win32 bugs in 3.0M8
 
   Revision 1.41  2004/06/04 13:57:29  bobintetley
   Fix to Window descendants to position themselves correctly
 
   Revision 1.40  2004/05/27 03:04:50  dannaab
   Add a couple methods from Component.java to Window.java so the Shell peer is used instead of the generic composite peer
 
   Revision 1.39  2004/05/26 13:46:23  bobintetley
   Fixes to new Windowing code to stop non-modal dialogs failling and
      missing cached items
 
   Revision 1.38  2004/05/26 04:46:06  dannaab
   Changed how Window/Dialog/Frame are initialized.
   Now, the SWT peer's instantiation is delayed until setVisible(true).  This
   has the benefit of allowing Swing's setter methods that can only be set via
   the SWT Shell peer's construction (ie, setResizable(), setModel())
 
   Revision 1.37  2004/05/05 23:28:40  laurentmartelli
   setTitle: checks peer is not disposed
 
   Revision 1.36  2004/05/05 13:24:30  bobintetley
   Bugfixes and Laurent's patch for binary compatibility on Container.add()
 
   Revision 1.35  2004/05/05 12:43:19  bobintetley
   Patches/new files from Laurent Martell
 
   Revision 1.34  2004/05/04 16:30:01  bobintetley
   Fix to check peer in repaintFix()
 
   Revision 1.33  2004/05/04 14:59:55  bobintetley
   Fix to ensure getBounds() works correctly for Window descendants
 
   Revision 1.32  2004/04/30 13:20:40  bobintetley
   Fix to unrealised peer preferred sizes, forwarding window events to
   content panes and fix for mouse drag events.
 
   Revision 1.31  2004/04/19 14:50:54  bobintetley
   Window.isActive() support
 
   Revision 1.30  2004/04/19 12:49:33  bobintetley
   JTaskTray implementation (and demo), along with Frame repaint fix
 
   Revision 1.29  2004/04/18 20:07:09  bobintetley
   GTK2 repainting bug fixed
 
   Revision 1.28  2004/04/16 10:19:06  dannaab
   Misc bug fixes, InputMap implementation, preliminary undo support
 
   Revision 1.27  2004/03/22 15:10:21  bobintetley
   JRootPane and JLayeredPane implementation
 
   Revision 1.26  2004/03/03 09:13:12  bobintetley
   JList threading fixed and top level error handling
 
   Revision 1.25  2004/03/01 12:25:46  bobintetley
   Better HTML conversion, custom JFileChooser support, JLabel, Window and
   Image fixes to improve compatibility
 
   Revision 1.24  2004/02/24 09:51:34  bobintetley
   Better contentPane support
 
   Revision 1.23  2004/02/02 15:20:45  bobintetley
   setContentPane() support and pack now includes frame decorations
 
   Revision 1.22  2004/02/02 14:40:22  bobintetley
   Non-modal dialogs now work correctly
 
   Revision 1.21  2004/02/02 14:17:45  bobintetley
   Window pack() now works correctly
 
   Revision 1.20  2004/01/26 08:10:59  bobintetley
   Many bugfixes and addition of SwingSet
 
   Revision 1.19  2004/01/23 08:04:24  bobintetley
   JComboBox fixes and better Action implementation
 
   Revision 1.18  2004/01/20 07:38:05  bobintetley
   Bug fixes and compatibility methods
 
   Revision 1.17  2004/01/16 09:35:46  bobintetley
   Full event dispatch thread support!
 
   Revision 1.16  2004/01/06 10:29:21  bobintetley
   Dispatch thread retardation code
 
   Revision 1.15  2004/01/05 16:52:35  bobintetley
   Minor class name fixes
 
   Revision 1.14  2004/01/05 09:18:05  bobintetley
   Merged Daniel's changes
 
   Revision 1.13  2004/01/05 02:50:39  djspiewak
   Added JToolBar peer functionality and commenced AWT layout manager support
 
   Revision 1.12  2003/12/16 15:47:44  bobintetley
   Thread safety added to common methods
 
   Revision 1.11  2003/12/16 14:51:16  bobintetley
   Fixed hang when a window close event closes itself again
 
   Revision 1.10  2003/12/16 09:19:02  bobintetley
   Various small fixes to match Swing more closely
 
   Revision 1.9  2003/12/15 18:29:56  bobintetley
   Changed setParent() method to setSwingWTParent() to avoid conflicts with applications
 
   Revision 1.8  2003/12/15 15:54:25  bobintetley
   Additional core methods
 
   Revision 1.7  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers
 
 */

package swingwt.awt;

import org.eclipse.swt.widgets.*;
import org.eclipse.swt.*;
import org.eclipse.swt.events.*;

import swingwtx.swing.*;
import java.util.*;

public class Window extends Container {
    
    protected Shell peer = null;
    protected Display display = null;
    protected Vector windowListeners = new Vector();
    protected boolean isClosed = false;
    protected boolean isIcon = false;
    
    protected JRootPane rootPane = null;
    
    protected String title;
    protected Image iconImage;
    protected JMenuBar pMenu;
    protected MenuBar pMenuBar;
    protected boolean delayPack = false;
    
    /** Number of pixels to allow for frame decoration during pack() */
    protected final static int FRAME_DECORATION_HEIGHT = 34;
    /** Number of pixels to allow for frame decoration during pack() */
    protected final static int FRAME_DECORATION_WIDTH = 8;
    
    private Object retval = null;
    
    /** Does nothing */
    GraphicsConfiguration graphicsConfig = null;
    
    public Window(Frame parent) {
        this.parent = parent;
        init();
    }
    public Window(Window parent) {
        this.parent = parent;
        init();
    }
    public Window(Window parent, GraphicsConfiguration gc) {
        this.parent = parent;
        init();
    }
    
    private void init() {
        graphicsConfig = (parent == null ? null : parent.getGraphicsConfiguration());
        rootPane = new JRootPane(this);
        rootPane.parent = this;
    }
    
    // Default frame type for direct Window instances
    protected int getSWTFrameType() {
        return SWT.NO_TRIM;
    }
    
    protected void create() {
        create(getSWTFrameType());
    }
    
    /**
     * Actually does the work of creating the Window peer
     * All the descendants of Window call this with an appropriate SWT type
     * as part of their constructor.
     *
     * A bit cheaty, but we use the Swing JRootPane stuff here as well
     * so it only needs doing once and subclasses can use it if they want it.
     */
    protected void create(final int SWTFrameType) {
        SwingWTUtils.incrementWindowReferences();
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                createImpl(SWTFrameType);
            }
        });
    }
    
    /**
     * The actual create implementation. This is just so we don't
     * have to muck about inside anonymous inner classes.
     *
     * This routine creates the peer and rootpane ready for use.
     *
     * @param SWTFrameType The SWT constant to use when creating the Shell
     */
    private void createImpl(int SWTFrameType) {
        
        display = SwingWTUtils.getDisplay();
        
        if (parent == null)
            peer = new Shell(display, SWTFrameType);
        else
            peer = new Shell( (Shell)parent.getSWTPeer(), SWTFrameType);
        
        composite = peer;
        peer.setLayout(new swingwt.awt.swtcustomlayout.SWTBorderLayout());
        peer.setLayoutData(swingwt.awt.swtcustomlayout.SWTBorderLayout.CENTER);
        registerWindowEvents();
        
        setCachedProperties();
        
        try { rootPane.setSwingWTParent(this); } catch (Exception e) { e.printStackTrace(); }
        
        isClosed = false;
    }
    
    public void setCachedProperties() {
        if (!SwingWTUtils.isSWTControlAvailable(peer)) return;
        if (title != null) peer.setText(title);
        if (iconImage != null) peer.setImage(iconImage.image);
        if (pMenu != null) setJMenuBar(pMenu);
        if (pLocation != null) setLocation(pLocation);
        if (pSize != null) setSize(pSize);
        if (delayPack) pack();
    }
    
    public void setLocation(final Point p) {
        pLocation = p;
        if (SwingWTUtils.isMacOSX() && (p.y == 0))
            p.y = SwingWTUtils.MACOSX_MENUBAR_BOTTOM;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer))
                    peer.setBounds(p.x, p.y, peer.getSize().x, peer.getSize().y);
            }
        });
    }
    
    /** FIXME: Bad algorithm here, but can't find appropriate routine. Display.map() looks
     *  interesting, but it is undocumented.
     */
    public Point getLocationOnScreen() {
        return getLocation();
        /*
        final Point p = new Point(0, 0);
        if (!SwingWTUtils.isSWTControlAvailable(peer)) return p;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                p.x = peer.getLocation().x + getLocation().x;
                p.y = peer.getLocation().y + getLocation().y;
            }
        });
        return p;
         */
    }
    
    /** Overridden to point to getContentPane() rather than throwing an error */
    public Component add(swingwt.awt.Component c) { rootPane.getContentPane().add(c); return c; }
    /** Overridden to point to getContentPane() rather than throwing an error */
    public void add(swingwt.awt.Component c, Object layoutModifier) { rootPane.getContentPane().add(c, layoutModifier); }
    /** Overridden to point to getContentPane() rather than throwing an error */
    public void remove(swingwt.awt.Component c) { rootPane.getContentPane().remove(c); }
    /** Overridden to point to getContentPane() rather than throwing an error */
    public LayoutManager getLayout() {
        return rootPane.getContentPane().getLayout();
    }
    /** Overridden to add mouse listeners to the content pane instead */
    public void addMouseListener(swingwt.awt.event.MouseListener l) {
        rootPane.getContentPane().addMouseListener(l);
    }
    /** Overridden to remove mouse listeners from the content pane instead */
    public void removeMouseListener(swingwt.awt.event.MouseListener l) {
        rootPane.getContentPane().removeMouseListener(l);
    }
    /** Overridden to point to getContentPane() rather than throwing an error */
    public void setLayout(LayoutManager l) {
        rootPane.getContentPane().setLayout(l);
    }
    /** Overridden to point to getContentPane() rather than throwing an error */
    public void setLayout(LayoutManager2 l) {
        rootPane.getContentPane().setLayout(l);
    }
    
    public void pack() {
        final Window me = this;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer)) {
                    Dimension d = rootPane.getContentPane().getPreferredSize();
                    d.width += FRAME_DECORATION_WIDTH;
                    d.height += FRAME_DECORATION_HEIGHT;
                    peer.setSize( d.width, d.height );
                    pSize = d;
                }
                else {
                    delayPack = true;
                }
            }
        });
    }
    
    
    public void setLocation(int x, int y) {
        if (SwingWTUtils.isMacOSX() && (y == 0))
            y = SwingWTUtils.MACOSX_MENUBAR_BOTTOM;
        pLocation = new Point(x,y);
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    peer.setLocation(pLocation.x, pLocation.y);
                }
            });
        }
    }
    
    public void setLocationRelativeTo(Component c) {
        // Center the Window
        Dimension paneSize = getSize();
        Dimension screenSize = getToolkit().getScreenSize();
        setLocation((screenSize.width - paneSize.width) / 2,
        (screenSize.height - paneSize.height) / 2);
    }
    
    public Dimension getSize() {
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    retval = new Dimension(peer.getBounds().width, peer.getBounds().height);
                }
            });
        }
        else {
            retval = pSize;
        }
        
        if (retval == null) {
            retval = new Dimension(0,0);
        }
        
        return (Dimension) retval;
    }
    
    public Rectangle getBounds() {
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    retval = new Rectangle(peer.getLocation().x, peer.getLocation().y, peer.getSize().x, peer.getSize().y);
                }
            });
        }
        else {
            if (pLocation == null) {
                retval = new Rectangle(0, 0, 0, 0);
            }
            else {
                retval = new Rectangle(pLocation.x, pLocation.y, pSize.width, pSize.height);
            }
        }
        return (Rectangle) retval;
    }

    public void setSize(final int width, final int height) {
        setSize(new Dimension(width, height));
    }
    public void setSize(Dimension d) {
        pSize = d;
        // If for some dumb reason setSize is called after pack() in user space, reset the delayed pack()
        delayPack = false;
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    peer.setSize(pSize.width, pSize.height);
                }
            });
        }
    }
    public void setBounds(int x, int y, final int width, final int height) {
        if (SwingWTUtils.isMacOSX() && (y == 0))
            y = SwingWTUtils.MACOSX_MENUBAR_BOTTOM;
        pLocation = new Point(x, y);
        pSize = new Dimension(width, height);
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    peer.setLocation(pLocation.x, pLocation.y);
                    peer.setSize(width, height);
                }
            });
        }
    }
    
    public void setBounds(final Rectangle r) {
        if (SwingWTUtils.isMacOSX() && (r.y == 0))
            r.y = SwingWTUtils.MACOSX_MENUBAR_BOTTOM;
        pLocation = new Point(r.x, r.y);
        pSize = new Dimension(r.width, r.height);
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    peer.setLocation(r.x, r.y);
                    peer.setSize(r.width, r.height);
                }
            });
        }
    }
    
    public boolean isActive() {
        final boolean[] ret = new boolean[1];
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                ret[0] = (SwingWTUtils.getDisplay().getActiveShell() == peer);
            }
        });
        return ret[0];
    }
    
    public void show() { invalidate(); setVisible(true); }
    public void hide() { setVisible(false); }
    public void addWindowListener(swingwt.awt.event.WindowListener l) { windowListeners.add(l); }
    public void removeWindowListener(swingwt.awt.event.WindowListener l) { windowListeners.remove(l); }
    
    public void setVisible(final boolean b) {
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (b) {
                    if (!SwingWTUtils.isSWTControlAvailable(peer))
                        create();
                    
                    peer.open();
                    processWindowEvent(swingwt.awt.event.WindowEvent.WINDOW_OPENED);
                    repaintFix();
                }
                else {
                    // If the Window is already closing because the user hit the x
                    // button, don't do anything
                    if (isClosed) return;
                    if (SwingWTUtils.isSWTControlAvailable(peer))
                        peer.close();
                }
            }
        });
    }
    
    public String getTitle() {
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    retval = peer.getText();
                }
            });
        }
        else retval = title;
        
        return retval.toString();
    }
    
    public void setTitle(String s) {
        this.title = s;
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    peer.setText(title);
                }
            });
        }
    }
    
    public Image getIconImage() {
        final Image[] ret = new Image[1];
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    ret[0] = new Image();
                    ret[0].image = peer.getImage();
                }
            });
        }
        else ret[0] = iconImage;
        
        return ret[0];
    }
    
    public void setIconImage(final Image icon) {
        this.iconImage = icon;
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    peer.setImage(icon.image);
                }
            });
        }
    }
    
    public void dispose() {
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                peer.dispose();
            }
        });
        super.dispose();
    }
    
    public void setJMenuBar(JMenuBar menu) {
        this.pMenu = menu;
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            rootPane.setJMenuBar(menu);
        }
    }
    
    public JMenuBar getJMenuBar() {
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            retval = rootPane.getJMenuBar();
        }
        else {
            retval = pMenu;
        }
        return (JMenuBar) retval;
    }
    
    public void setMenuBar(MenuBar menu) {
        this.pMenuBar = menu;
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            rootPane.setMenuBar(menu);
        }
    }
    
    public MenuBar getMenuBar() {
        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            retval = rootPane.getMenuBar();
        }
        else {
            retval = pMenuBar;
        }
        return (MenuBar) retval;
    }
    
    public void setDefaultButton(JButton button) {
        rootPane.setDefaultButton(button);
    }
    
    public JButton getDefaultButton() {
        return rootPane.getDefaultButton();
    }
    
    public void registerWindowEvents() {
        
        peer.addShellListener(new ShellListener() {
            public void shellActivated(ShellEvent e) {
                processWindowEvent(swingwt.awt.event.WindowEvent.WINDOW_ACTIVATED);
            }
            public void shellClosed(ShellEvent e) {
                isClosed = true;
                processWindowEvent(swingwt.awt.event.WindowEvent.WINDOW_CLOSING);
                processWindowEvent(swingwt.awt.event.WindowEvent.WINDOW_CLOSED);
                SwingWTUtils.decrementWindowReferences();
            }
            public void shellDeactivated(ShellEvent e) {
                processWindowEvent(swingwt.awt.event.WindowEvent.WINDOW_DEACTIVATED);
            }
            public void shellDeiconified(ShellEvent e) {
                isIcon = false;
                processWindowEvent(swingwt.awt.event.WindowEvent.WINDOW_DEICONIFIED);
            }
            public void shellIconified(ShellEvent e) {
                isIcon = true;
                processWindowEvent(swingwt.awt.event.WindowEvent.WINDOW_ICONIFIED);
            }
        });
    }
    
    protected void processWindowEvent(int id) {
        swingwt.awt.event.WindowEvent we = new swingwt.awt.event.WindowEvent(this, id);
        Iterator i = windowListeners.iterator();
        while (i.hasNext()) {
            swingwt.awt.event.WindowListener l = (swingwt.awt.event.WindowListener) i.next();
            switch(id) {
                case (swingwt.awt.event.WindowEvent.WINDOW_ACTIVATED): l.windowActivated(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_CLOSED): l.windowClosed(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_CLOSING): l.windowClosing(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_DEACTIVATED): l.windowDeactivated(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_DEICONIFIED): l.windowDeiconified(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_ICONIFIED): l.windowIconified(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_OPENED): l.windowOpened(we); break;
            }
        }
    }
    
    protected void processWindowEvent(swingwt.awt.event.WindowEvent e) {
        swingwt.awt.event.WindowEvent we = e;
        Iterator i = windowListeners.iterator();
        while (i.hasNext()) {
            swingwt.awt.event.WindowListener l = (swingwt.awt.event.WindowListener) i.next();
            switch(e.getID()) {
                case (swingwt.awt.event.WindowEvent.WINDOW_ACTIVATED): l.windowActivated(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_CLOSED): l.windowClosed(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_CLOSING): l.windowClosing(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_DEACTIVATED): l.windowDeactivated(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_DEICONIFIED): l.windowDeiconified(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_ICONIFIED): l.windowIconified(we); break;
                case (swingwt.awt.event.WindowEvent.WINDOW_OPENED): l.windowOpened(we); break;
            }
        }
    }
    
    protected void dispatchEvents() {
        while (!peer.isDisposed()) {
            try {
                if (!display.readAndDispatch()) {
                    // Send this thread to sleep to allow other
                    // processes a go :)
                    display.sleep();
                }
                // If thread retardation is on, sleep
                if (SwingWTUtils.isRetardDispatchThread()) {
                    try {
                        Thread.sleep(SwingWTUtils.getRetardationInterval());
                    }
                    catch (InterruptedException e) {}
                }
            }
            // By catching all exceptions and errors here, we
            // can prevent the app going down for small event
            // related problems. User code can still catch where
            // necessary.
            catch (Error e) {
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
    }
    
    /** Sends the window to the front */
    public void toFront() {
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer))
                    peer.forceActive();
            }
        });
    }
    
    /** Sends the window to the back */
    public void toBack() {
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer)) {
                    // Find another shell to make active
                    Shell[] s = SwingWTUtils.getDisplay().getShells();
                    for (int i = 0; i < s.length; i++) {
                        if (!s[i].equals(peer))
                            s[i].forceActive();
                    }
                }
            }
        });
    }
    
    /**
     * Under GTK2 and MacOSX, SWT has some trouble issuing repaints,
     * the only way around it seems to be to tell SWT to relayout
     * the top-level container again after creation and child
     * components have been added.
     */
    protected void repaintFix() {
        SwingUtilities.invokeIn(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer)) {
                    peer.layout(true);
                }
                SwingUtilities.invokeIn(new Runnable() {
                    public void run() {
                        if (SwingWTUtils.isSWTControlAvailable(peer)) {
                            
                            // This is an interesting GTK bug - sometimes the size is
                            // reported as < 40, the peer doesn't paint or layout
                            // and everything goes pear-shaped. If we detect this, and
                            // reapply the Shell's size, all becomes well again
                            // in GTK land. This bug was a BITCH to find and fix.
                            if (peer.getSize().x < 40 && peer.getSize().y < 40) {
                                peer.setSize(pSize.width, pSize.height);
                            }
                            
                            // This code is no longer needed - the painting only fails
                            // when the size drops below the 40 threshold. This is one
                            // weird SWT/GTK bug.
                            
                            //else
                            //	peer.setSize(peer.getSize().x + 1, peer.getSize().y);
                        }
                    }
                }, 75);
            }
        }, 10);
    }
    
    public Control getSWTPeer() { return peer; }
    public void setSwingWTParent(Composite parent) throws Exception { throw new Exception("Window can't have parent."); }
    
}

