// Copyright (C) 2009-2009 Martin Sandve Alnæs
// Licensed under the GNU LGPL Version 3.0.
//
// First added:  2009-04-08
// Last changed: 2009-04-08
//
// This demo program solves the harmonic map problem in 2D.

#include <cmath>
#include <dolfin.h>
#include "generated_code/HarmonicMap.h"

using namespace dolfin;
using namespace HarmonicMap;

class Value: public Function
{
public:
  Value(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    values[0] = 1.0;
  }
};

class Solution: public Function
{
public:
  Solution(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0]-0.5;
    double dy = x[1]-0.5;
    double r = sqrt(dx*dx + dy*dy);
    if(r == 0)
    {
        values[0] = 1.0;
        values[1] = 1.0;
        values[2] = 1.0;
        warning("Singularity.");
    }
    values[0] = 1.0*dx/r;
    values[1] = 1.0*dy/r;
    values[2] = 1.0;
  }
};

class BoundaryValue: public Function
{
public:
  BoundaryValue(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0]-0.5;
    double dy = x[1]-0.5;
    double r = sqrt(dx*dx + dy*dy);
    if(r == 0)
    {
        values[0] = 1.0;
        values[1] = 1.0;
        warning("Singularity.");
    }
    values[0] = dx/r;
    values[1] = dy/r;
  }
};

class DirichletBoundary: public SubDomain
{
public:
  bool inside(const double* x, bool on_boundary) const
  {
    return on_boundary;
  }
};

int main(int argc, char ** argv)
{
    int n = 11;
    UnitSquare mesh(n, n);

    BilinearForm::TrialSpace V(mesh);
    //CoefficientSpace_x X(mesh);
    //CoefficientSpace_y Y(mesh);
    SubSpace X(V, 0);
    SubSpace Y(V, 1);

    Function u(V);
    Solution usol(V);
    usol.interpolate(u.vector(), V);

    BilinearForm a(V, V);
    LinearForm L(V);
    a.u = u;
    L.u = u;

    DirichletBoundary boundary;
    BoundaryValue ubc(X);
    DirichletBC bc(X, ubc, boundary);

    VariationalProblem problem(a, L, bc, true);
    problem.solve(u);

    Function x = u[0];
    Function y = u[1];

    File xfile("x.pvd");
    xfile << x;
    File yfile("y.pvd");
    yfile << y;

    //plot(x);
    //plot(y);
}

