/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009, 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "mousedevicesmonitor.h"
#include <KDebug>
#include <QtDBus/QDBusInterface>
#include <QtDBus/QDBusReply>
#include <QtCore/QStringList>


using namespace synaptiks;


static const char HAL_SERVICE[] = "org.freedesktop.Hal";
static const char HAL_MANAGER_PATH[] = "/org/freedesktop/Hal/Manager";
static const char HAL_MANAGER_IFACE[] = "org.freedesktop.Hal.Manager";
static const char HAL_DEVICE_IFACE[] = "org.freedesktop.Hal.Device";


#define DEVICE_INTERFACE(device, udi) \
    QDBusInterface device(HAL_SERVICE, udi, HAL_DEVICE_IFACE, \
                          QDBusConnection::systemBus());      \


namespace synaptiks {
    class MouseDevicesMonitorPrivate {
    public:
        Q_DECLARE_PUBLIC(MouseDevicesMonitor)

        MouseDevicesMonitorPrivate(MouseDevicesMonitor *qq);

        virtual ~MouseDevicesMonitorPrivate() {};

        void _k_deviceAdded(const QString &udi);
        void _k_deviceRemoved(const QString &udi);
        QStringList pluggedMouseDevices() const;
        bool queryCapability(const QString &udi,
                             const QString &capability) const;
        QString getProperty(const QString &udi,
                            const QString &property) const;

        QDBusInterface *hal_manager;
        QSet<QString> mouseDevices;
        MouseDevicesMonitor *q_ptr;
    };
}


MouseDevicesMonitorPrivate::MouseDevicesMonitorPrivate(
    MouseDevicesMonitor *qq): q_ptr(qq) {
    Q_Q(MouseDevicesMonitor);
    this->hal_manager = new QDBusInterface(
        HAL_SERVICE, HAL_MANAGER_PATH, HAL_MANAGER_IFACE, QDBusConnection::systemBus(), q);
    q->connect(this->hal_manager, SIGNAL(DeviceAdded(const QString&)),
               SLOT(_k_deviceAdded(const QString&)));
    q->connect(this->hal_manager, SIGNAL(DeviceRemoved(const QString&)),
               SLOT(_k_deviceRemoved(const QString&)));
    this->mouseDevices = QSet<QString>::fromList(
        this->pluggedMouseDevices());
}

void MouseDevicesMonitorPrivate::_k_deviceAdded(const QString& udi) {
    if (this->queryCapability(udi, "input.mouse")) {
        Q_Q(MouseDevicesMonitor);
        this->mouseDevices.insert(udi);
        emit q->mousePlugged(udi);
    }
}

void MouseDevicesMonitorPrivate::_k_deviceRemoved(const QString& udi) {
    if (this->mouseDevices.contains(udi)) {
        Q_Q(MouseDevicesMonitor);
        this->mouseDevices.remove(udi);
        emit q->mouseUnplugged(udi);
    }
}

QStringList MouseDevicesMonitorPrivate::pluggedMouseDevices() const {
    kDebug() << "enumerating plugged devices";
    QDBusReply<QStringList> devices = this->hal_manager->call(
        "FindDeviceByCapability", "input.mouse");
    if (!devices.isValid()) {
        kError() << "dbus error:" << devices.error().name()
                 << ":" << devices.error().message();
        return QStringList();
    } else
        return devices.value();
}

bool MouseDevicesMonitorPrivate::queryCapability(
    const QString &udi, const QString &capability) const {
    DEVICE_INTERFACE(device, udi);
    QDBusReply<bool> hasCapability = device.call("QueryCapability",
                                                 capability);
    if (!hasCapability.isValid()) {
        kError() << "dbus error:" << hasCapability.error().name()
                 << ":" << hasCapability.error().message();
        return false;
    } else {
        return hasCapability.value();
    }
}

QString MouseDevicesMonitorPrivate::getProperty(
    const QString &udi, const QString &property) const {
    DEVICE_INTERFACE(device, udi);
    QDBusReply<QString> value = device.call("GetProperty", property);
    if (!value.isValid()) {
        kError() << "dbus error:" << value.error().name()
                 << ":" << value.error().message();
        return QString();
    } else {
        return value.value();
    }
}

MouseDevicesMonitor::MouseDevicesMonitor(QObject *parent):
    QObject(parent), d_ptr(new MouseDevicesMonitorPrivate(this)) {}

MouseDevicesMonitor::~MouseDevicesMonitor() {
    delete this->d_ptr;
}

QStringList MouseDevicesMonitor::pluggedMouseDevices() const {
    Q_D(const MouseDevicesMonitor);
    return d->pluggedMouseDevices();
}

QString MouseDevicesMonitor::productName(const QString &id) const {
    Q_D(const MouseDevicesMonitor);
    return d->getProperty(id, "info.product");
}

bool MouseDevicesMonitor::isTouchpad(const QString &id) const {
    Q_D(const MouseDevicesMonitor);
    return d->queryCapability(id, "input.touchpad");
}

#include "moc_mousedevicesmonitor.cpp"
