/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009, 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include "touchpadmanager.h"
#include "touchpad.h"
#include "mousedevicesmonitor.h"
#include "keyboardmonitor.h"
#include <QtCore/QPointer>
#include <QtCore/QHash>
#include <QtCore/QSet>
#include <KLocalizedString>
#include <KDebug>


namespace synaptiks {

    class TouchpadManagerPrivate {

    public:
        Q_DECLARE_PUBLIC(TouchpadManager)

        TouchpadManagerPrivate(Touchpad *touchpad, MouseDevicesMonitor *md,
                               TouchpadManager *qq):
            q_ptr(qq), touchpad(touchpad), mouseDevices(md),
            interactivelyEnabled(false), monitorMouseDevices(false) {
            Q_Q(TouchpadManager);
            this->touchpad->setParent(qq);

            this->mouseDevices->setParent(qq);
            this->mouseDevices->setObjectName("mouseDevices");
            q->connect(this->mouseDevices,
                       SIGNAL(mousePlugged(const QString &)),
                       SLOT(_k_registerMouseDevice(const QString&)));
            q->connect(this->mouseDevices,
                       SIGNAL(mouseUnplugged(const QString &)),
                       SLOT(_k_unregisterMouseDevice(const QString&)));
        }

        bool shallIgnoreMouse(const QString &mouse) {
            return (this->mouseDevices->isTouchpad(mouse) ||
                    this->ignoredMouseDevices.contains(mouse) ||
                    // don't register the same mouse twice
                    this->pluggedMouses.contains(mouse));
        }

        void _k_registerMouseDevice(const QString &udi) {
            Q_Q(TouchpadManager);
            if (!this->shallIgnoreMouse(udi)) {
                QString name = this->mouseDevices->productName(udi);
                this->pluggedMouses.insert(udi, name);
                kDebug() << name << "plugged";
                if (this->monitorMouseDevices) {
                    q->setTouchpadOn(false, "mouse", name);
                }
            }
        }

        void _k_unregisterMouseDevice(const QString &udi) {
            Q_Q(TouchpadManager);
            if (!this->pluggedMouses.contains(udi)) {
                return;
            }
            QString name = this->pluggedMouses.take(udi);
            kDebug() << name << "unplugged";
            if (this->monitorMouseDevices &&
                this->pluggedMouses.isEmpty()) {
                q->setTouchpadOn(true, "mouse", name);
            }
        }

        void _k_keyboardActivity() {
            Q_Q(TouchpadManager);
            if (this->keyboardMonitor->isKeyboardActive()) {
                kDebug() << "User started typing";
                q->setTouchpadOn(false, "keyboard");
            } else {
                kDebug() << "User stopped typing";
                q->setTouchpadOn(true, "keyboard");
            }
        }

        TouchpadManager *q_ptr;

        Touchpad *touchpad;
        MouseDevicesMonitor *mouseDevices;
        QPointer<KeyboardMonitor> keyboardMonitor;
        QHash<QString, QString> pluggedMouses;
        QSet<QString> ignoredMouseDevices;
        QSet<QString> reasonsForSwitchOff;
        bool interactivelyEnabled;
        bool monitorMouseDevices;
    };
}


using namespace synaptiks;


TouchpadManager::TouchpadManager(Touchpad *touchpad,
                                 MouseDevicesMonitor *mouseDevicesMonitor,
                                 QObject *parent):
    QObject(parent), d_ptr(new TouchpadManagerPrivate(
                               touchpad, mouseDevicesMonitor, this)) {
}

TouchpadManager::~TouchpadManager() {
}

void TouchpadManager::setTouchpadOn(
    bool on, const QString &reason, const QVariant &closure) {
    Q_D(TouchpadManager);
    if (reason == "interactive") {
        d->interactivelyEnabled = on;
    }
    if (on) {
        d->reasonsForSwitchOff.remove(reason);
        if (!d->interactivelyEnabled &&
            !d->reasonsForSwitchOff.isEmpty()) {
            kDebug() << "not enabling touchpad automatically";
            return;
        }
    } else {
        d->reasonsForSwitchOff.insert(reason);
    }
    try {
        if (d->touchpad->isOn() != on) {
            d->touchpad->setOn(on);
            emit this->touchpadSwitched(on, reason, closure);
            if (closure.isNull())
                emit this->touchpadSwitched(on, reason, QDBusVariant(""));
            else {
                emit this->touchpadSwitched(on, reason,
                                            QDBusVariant(closure));
            }
        }
    } catch (const QXDeviceError &error) {
        if (this->calledFromDBus()) {
            this->sendErrorReply("org.kde.TouchpadError",
                                 error.toString());
        }
        // emit error signals
        emit this->touchpadError(error.toString());
    }
}

void TouchpadManager::setTouchpadOn(
    bool on, const QString &reason, const QDBusVariant &closure) {
    this->setTouchpadOn(on, reason, closure.variant());
}

bool TouchpadManager::isTouchpadOn() {
    try {
        return this->touchpad()->isOn();
    } catch (const QXDeviceError &error) {
        if (this->calledFromDBus()) {
            this->sendErrorReply("org.kde.TouchpadError",
                                 error.toString());
        }
        emit this->touchpadError(error.toString());
        return false;
    }
}

bool TouchpadManager::monitorMouseDevices() const {
    Q_D(const TouchpadManager);
    return d->monitorMouseDevices;
}

void TouchpadManager::setMonitorMouseDevices(bool enabled) {
    Q_D(TouchpadManager);
    if (enabled == d->monitorMouseDevices)
        return;

    d->monitorMouseDevices = enabled;
    if (!d->pluggedMouses.isEmpty()) {
        QString name = *d->pluggedMouses.begin();
        // if mouse device monitoring is disabled, then enable the touchpad
        // again.  if mouse device monitoring is enabled, then enable the
        // touchpad.
        this->setTouchpadOn(!enabled, "mouse", name);
    }
}

bool TouchpadManager::monitorKeyboard() const {
    Q_D(const TouchpadManager);
    return !d->keyboardMonitor.isNull();
}

void TouchpadManager::setMonitorKeyboard(bool enabled) {
    Q_D(TouchpadManager);
    if (enabled) {
        if (d->keyboardMonitor.isNull()) {
            d->keyboardMonitor = new KeyboardMonitor(this);
            this->connect(d->keyboardMonitor, SIGNAL(typingStarted()),
                          SLOT(_k_keyboardActivity()));
            this->connect(d->keyboardMonitor, SIGNAL(typingStopped()),
                          SLOT(_k_keyboardActivity()));
        }
        d->keyboardMonitor->start();
    } else  {
        delete d->keyboardMonitor;
    }
}

Touchpad *TouchpadManager::touchpad() const {
    Q_D(const TouchpadManager);
    return d->touchpad;
}

KeyboardMonitor *TouchpadManager::keyboardMonitor() const {
    Q_D(const TouchpadManager);
    return d->keyboardMonitor;
}

MouseDevicesMonitor *TouchpadManager::mouseDevicesMonitor() const {
    Q_D(const TouchpadManager);
    return d->mouseDevices;
}

QStringList TouchpadManager::ignoredMouseDevices() const {
    Q_D(const TouchpadManager);
    return d->ignoredMouseDevices.toList();
}

void TouchpadManager::setIgnoredMouseDevices(const QStringList &devices) {
    Q_D(TouchpadManager);
    d->ignoredMouseDevices = QSet<QString>::fromList(devices);
    if (d->mouseDevices) {
        foreach (const QString &udi,
                 d->mouseDevices->pluggedMouseDevices()) {
            d->_k_registerMouseDevice(udi);
        }
        foreach (const QString &udi, d->ignoredMouseDevices) {
            d->_k_unregisterMouseDevice(udi);
        }
    }
}

#include "moc_touchpadmanager.cpp"
