/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009, 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include "mousedevicesmodel.h"
#include <KLocalizedString>
#include <KDebug>
#include <QtDBus/QDBusInterface>
#include <QtDBus/QDBusReply>
#include <QtCore/QStringList>
#include <QtCore/QSet>


using namespace synaptiks;


namespace synaptiks {

    class MouseDevicesModelPrivate {
    public:
        Q_DECLARE_PUBLIC(MouseDevicesModel)

        MouseDevicesModelPrivate(MouseDevicesModel *qq);

        virtual ~MouseDevicesModelPrivate() {};

        void resetDeviceIndex();

        void _k_mousePlugged(const QString &udi);
        void _k_mouseUnplugged(const QString &udi);

        /**
         * Check, whether udi may be added to the device index.
         *
         * A device may be added, if ignoreTouchpads is @c false or the
         * device is not a touchpad.
         *
         * @return @c true, if the device may be added, @c false otherwise
         */
        bool mayAddDevice(const QString &udi) {
            if (this->touchpadsIgnored) {
                QDBusReply<bool> isTouchpad =
                    this->devices->call("isTouchpad", udi);
                return !(isTouchpad.isValid() && isTouchpad.value());
            }
            return true;
        };

        MouseDevicesModel *q_ptr;
        QDBusInterface *devices;
        QStringList deviceIndex;
        QSet<QString> checkedDevices;
        bool touchpadsIgnored;
    };
}


MouseDevicesModelPrivate::MouseDevicesModelPrivate(MouseDevicesModel *qq):
    q_ptr(qq), touchpadsIgnored(false) {
    Q_Q(MouseDevicesModel);
    this->devices = new QDBusInterface(
        "org.kde.synaptiks", "/MouseDevicesMonitor",
        "org.kde.MouseDevicesMonitor", QDBusConnection::sessionBus(), q);
    if (this->devices->isValid()) {
        this->resetDeviceIndex();
        q->connect(this->devices, SIGNAL(mousePlugged(const QString&)),
               SLOT(_k_mousePlugged(const QString&)));
        q->connect(this->devices, SIGNAL(mouseUnplugged(const QString&)),
               SLOT(_k_mouseUnplugged(const QString&)));
    }
}

void MouseDevicesModelPrivate::_k_mousePlugged(const QString &udi) {
    Q_Q(MouseDevicesModel);
    if (this->mayAddDevice(udi)) {
        int position = q->rowCount();
        q->beginInsertRows(QModelIndex(), position, position);
        this->deviceIndex.append(udi);
        q->endInsertRows();
    }
}

void MouseDevicesModelPrivate::_k_mouseUnplugged(const QString &udi) {
    Q_Q(MouseDevicesModel);
    int position = this->deviceIndex.indexOf(udi);
    if (position >= 0) {
        q->beginRemoveRows(QModelIndex(), position, position);
        this->deviceIndex.removeAt(position);
        q->endRemoveRows();
    }
}

void MouseDevicesModelPrivate::resetDeviceIndex() {
    this->deviceIndex.clear();
    QDBusReply<QStringList> pluggedMouses =
        this->devices->call("pluggedMouseDevices");
    if (!pluggedMouses.isValid())
        return;
    foreach (const QString &udi, pluggedMouses.value()) {
        if (this->mayAddDevice(udi)) {
            this->deviceIndex.append(udi);
        }
    }
}


MouseDevicesModel::MouseDevicesModel(QObject *parent):
    QAbstractListModel(parent),
    d_ptr(new MouseDevicesModelPrivate(this)) {
}

MouseDevicesModel::~MouseDevicesModel() {
    delete this->d_ptr;
}

int MouseDevicesModel::rowCount(const QModelIndex&) const {
    Q_D(const MouseDevicesModel);
    return d->deviceIndex.count();
}

Qt::ItemFlags MouseDevicesModel::flags(const QModelIndex &index) const {
    Q_UNUSED(index);
    return Qt::ItemIsEnabled | Qt::ItemIsUserCheckable;
}

QVariant MouseDevicesModel::data(const QModelIndex &index, int role) const {
    Q_D(const MouseDevicesModel);
    if (index.isValid()) {
        QString udi = d->deviceIndex.at(index.row());
        QDBusReply<QString> productName =
            d->devices->call("productName", udi);
        switch (role) {
        case Qt::DisplayRole:
            if (productName.isValid())
                return productName.value();
            else
                return QString();
        case Qt::ToolTipRole:
            return udi;
        case Qt::CheckStateRole:
            if (d->checkedDevices.contains(udi)) {
                return Qt::Checked;
            } else {
                return Qt::Unchecked;
            }
        default:
            return QVariant();
        }
    } else {
        return QVariant();
    }
}

bool MouseDevicesModel::setData(const QModelIndex &index,
                                const QVariant &value, int role) {
    Q_D(MouseDevicesModel);
    bool changed = false;
    if (index.isValid() && role == Qt::CheckStateRole) {
        QString udi = d->deviceIndex.at(index.row());
        int state = value.toInt();
        if (state == Qt::Checked) {
            kDebug() << "checking device" << udi;
            d->checkedDevices.insert(udi);
            changed = true;
        } else if (state == Qt::Unchecked) {
            kDebug() << "unchecking device" << udi;
            d->checkedDevices.remove(udi);
            changed = true;
        }
    }
    if (changed) {
        emit this->dataChanged(index, index);
        emit this->checkedDevicesChanged(this->checkedDevices());
    }
    return changed;
}

bool MouseDevicesModel::touchpadsIgnored() const {
    Q_D(const MouseDevicesModel);
    return d->touchpadsIgnored;
}

void MouseDevicesModel::setTouchpadsIgnored(bool ignore) {
    Q_D(MouseDevicesModel);
    if (this->touchpadsIgnored() != ignore) {
        d->touchpadsIgnored = ignore;
        d->resetDeviceIndex();
    }
}

QStringList MouseDevicesModel::checkedDevices() const {
    Q_D(const MouseDevicesModel);
    return d->checkedDevices.toList();
}

void MouseDevicesModel::setCheckedDevices(const QStringList &devices) {
    Q_D(MouseDevicesModel);
    foreach (const QString &device, devices) {
        kDebug() << "set device" << device << "to checked";
        d->checkedDevices.insert(device);
        // notify views
        int row = d->deviceIndex.indexOf(device);
        if (row >= 0) {
            QModelIndex index = this->index(row, 0);
            emit this->dataChanged(index, index);
        }
    }
    emit this->checkedDevicesChanged(this->checkedDevices());
}

#include "moc_mousedevicesmodel.cpp"
