# Python Parser
# Copyright(c) 2000 by Stephen Davies
#
# This file is under the GPL.
"""Main module for the Python parser.
Parsing python is achieved by using the code in the Python distribution that
is an example for parsing python by using the built-in parser. This parser
returns a parse tree which we can traverse and translate into Synopsis' AST.
The exparse module contains the enhanced example code (it has many more
features than the simple example did), and this module translates the
resulting intermediate AST objects into Synopsis.Core.AST objects.
"""

import parser, sys, os, string, getopt
from Synopsis.Core import AST, Type
import exparse

verbose = 0

def addDeclaration(decl):
    """Adds the given declaration to the current top scope and to the
    SourceFile for this file."""
    global scopes, sourcefile
    scopes[-1].declarations().append(decl)
    sourcefile.declarations().append(decl)

def push(scope):
    """Pushes the given scope onto the top of the stack"""
    scopes.append(scope)

def pop():
    """Pops the scope stack by one level"""
    del scopes[-1]

def scopeName(name):
    """Scopes the given name. If the given name is a list then it is returned
    verbatim, else it is concatenated with the (scoped) name of the current
    scope"""
    if type(name) == type([]): return tuple(name)
    return tuple(list(scopes[-1].name()) + [name])

def process_ModuleInfo(mi):
    """Processes a ModuleInfo object. The comments are extracted, and any
    functions and comments recursively processed."""
    comm = AST.Comment(mi.get_docstring(), sourcefile, -1)
    scopes[-1].comments().append(comm)
    for name in mi.get_function_names():
	process_FunctionInfo(mi.get_function_info(name))
    for name in mi.get_class_names():
	process_ClassInfo(mi.get_class_info(name))

def add_params(func, fi):
    """Adds the parameters of 'fi' to the AST.Function 'func'."""
    for name, value in map(None, fi.get_params(), fi.get_param_defaults()):
	func.parameters().append(AST.Parameter('', returnType,'',name,value))

def process_FunctionInfo(fi):
    """Process a FunctionInfo object. An AST.Function object is created and
    inserted into the current scope."""
    name = scopeName(fi.get_name())
    func = AST.Function(sourcefile, -1, 'Python', 'function', '', returnType, name, name[-1])
    comm = AST.Comment(fi.get_docstring(), '', -1)
    func.comments().append(comm)
    add_params(func, fi)
    addDeclaration(func)

def process_MethodInfo(fi):
    """Process a MethodInfo object. An AST.Operation object is created and
    inserted into the current scope."""
    name = scopeName(fi.get_name())
    func = AST.Operation(sourcefile,-1, 'Python', 'operation', '', returnType, name, name[-1])
    comm = AST.Comment(fi.get_docstring(), '', -1)
    func.comments().append(comm)
    add_params(func, fi)
    addDeclaration(func)


def process_ClassInfo(ci):
    """Process a ClassInfo object. An AST.Class object is created and
    inserted into the current scope. The inheritance of the class is also
    parsed, and nested classes and methods recursively processed."""
    name = scopeName(ci.get_name())
    clas = AST.Class(sourcefile,-1, 'Python', 'class', name)
    comm = AST.Comment(ci.get_docstring(), '', -1)
    clas.comments().append(comm)
    # Figure out bases:
    for base in ci.get_base_names():
	try:
	    t = types[scopeName(base)]
	except KeyError:
	    #print "Adding Unknown type for",scopeName(base)
	    t = Type.Unknown("Python", scopeName(base))
	clas.parents().append(AST.Inheritance('', t, ''))
    # Add the new class
    addDeclaration(clas)
    types[clas.name()] = Type.Declared("Python", clas.name(), clas)
    push(clas)
    for name in ci.get_class_names():
	process_ClassInfo(ci.get_class_info(name))
    for name in ci.get_method_names():
	process_MethodInfo(ci.get_method_info(name))
    pop()
    
def usage():
    """Prints a usage message"""
    print \
"""
  -b <basename>                        Specify basename"""

def __parseArgs(args, config_obj):
    """Parses the command line arguments and the config object"""
    global basename, verbose
    # Set defaults
    basename = ''

    # Parse config object
    if hasattr(config_obj, 'verbose') and config_obj.verbose:
	verbose = 1
    if hasattr(config_obj, 'basename') and config_obj.basename:
	basename = config_obj.basename

    # Parse command line arguments
    try:
        opts,remainder = getopt.getopt(args, "b:v")
    except getopt.error, e:
        sys.stderr.write("Error in arguments: " + str(e) + "\n")
        sys.exit(1)

    for opt in opts:
        o,a = opt
        if o == "-b": basename = a
        elif o == "-v": verbose = 1

def get_synopsis(file):
    """Returns the docstring from the top of an open file"""
    mi = exparse.get_docs(file)
    return mi.get_docstring()

def parse(file, extra_files, parser_args, config_obj):
    """Entry point for the Python parser"""
    __parseArgs(parser_args, config_obj)
    global scopes, filename, types, basename, returnType, sourcefile

    ast = AST.AST()
    filename = file
    types = ast.types()

    # Split the filename into a scoped tuple name
    name = file
    if file[:len(basename)] == basename:
	name = filename = file[len(basename):]
    name = string.split(os.path.splitext(name)[0], os.sep)
    exparse.packagepath = string.join(string.split(file, os.sep)[:-1], os.sep)
    exparse.packagename = name[:-1]
    #print "package path, name:",exparse.packagepath, exparse.packagename
    sourcefile = AST.SourceFile(filename, file, 'Python')
    sourcefile.set_is_main(1)

    # Create the enclosing module scopes
    scopes = [AST.Scope(sourcefile, -1, 'Python', 'file scope', name)]
    for depth in range(len(name) - 1):
	module = AST.Module(sourcefile, -1, 'Python', 'package', name[:depth+1])
	addDeclaration(module)
	types[module.name()] = Type.Declared("Python", module.name(), module)
	push(module)
    # Add final name as Module -- unless module is __init__
    if name[-1] != '__init__':
	module = AST.Module(sourcefile, -1, 'Python', 'module', name)
	addDeclaration(module)
	types[module.name()] = Type.Declared("Python", module.name(), module)
	push(module)
    
    # Parse the file
    mi = exparse.get_docs(file)

    # Create return type for Python functions:
    returnType = Type.Base('Python',('',))

    # Process the parsed file
    process_ModuleInfo(mi)

    # Store the declarations
    ast.files()[filename] = sourcefile
    ast.declarations().extend(scopes[0].declarations())

    # Return new AST object
    return ast

