/*
 *	tardy - a tar post-processor
 *	Copyright (C) 1998, 1999 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate cpios
 */

#include <ac/string.h>

#include <mem.h>
#include <tar/output/cpio.h>


tar_output_cpio::~tar_output_cpio()
{
	delete fp;
}


tar_output_cpio::tar_output_cpio(file_output *arg_fp, int arg_padding)
	: fp(arg_fp), padding(arg_padding), pos(0)
{
}


void
tar_output_cpio::write_data(const void *buffer, int nbytes)
{
	fp->write(buffer, nbytes);
	pos += nbytes;
}


void
tar_output_cpio::write_data_padding()
{
	if (padding <= 0)
		return;
	int n = pos % padding;
	if (!n)
		return;

	/*
	 * This serves all class instances, even thogh the chances
	 * of more than one are unlikely.  Each instance could
	 * have a different padding multiple.
	 */
	static char *dummy;
	static int dummy_size;
	if (dummy_size < padding)
	{
		dummy_size = padding;
		dummy = (char *)mem_change_size(dummy, dummy_size);
		memset(dummy, 0, dummy_size);
	}
	write_data(dummy, padding - n);
}


void
tar_output_cpio::write_archive_end()
{
	tar_header h;
	h.name = "TRAILER!!!";
	h.inode_number = 0;
	write_header(h);
	write_header_padding();
	write_data_padding();
}


const char *
tar_output_cpio::filename()
	const
{
	return fp->filename();
}


tar_output_cpio::tar_output_cpio()
{
	fatal("bug %s %d", __FILE__, __LINE__);
}


tar_output_cpio::tar_output_cpio(const tar_output_cpio &)
{
	fatal("bug %s %d", __FILE__, __LINE__);
}


tar_output_cpio &
tar_output_cpio::operator = (const tar_output_cpio &)
{
	fatal("bug %s %d", __FILE__, __LINE__);
	return *this;
}


int
tar_output_cpio::calculate_mode(const tar_header &h)
{
	/*
	 * Don't use the native UNIX defintions, because
	 * (a) this may not be UNIX, and
	 * (b) the vendor may have gratuitously changed things.
	 */
	int hibits;
	hibits = 0100000;
	switch (h.type)
	{
	case tar_header::type_socket:
		hibits = 0140000;
		break;

	case tar_header::type_link_symbolic:
		hibits = 0120000;
		break;

	case tar_header::type_normal:
	case tar_header::type_normal_contiguous:
	case tar_header::type_link_hard:
		hibits = 0100000;
		break;

	case tar_header::type_device_block:
		hibits = 0060000;
		break;

	case tar_header::type_directory:
		hibits = 0040000;
		break;

	case tar_header::type_device_character:
		hibits = 0020000;
		break;

	case tar_header::type_fifo:
		hibits = 0010000;
		break;
	}
	return (h.mode | hibits);
}
