/*
 *	tardy - a tar post-processor
 *	Copyright (C) 1998, 1999, 2004 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: interface definition for common/file/input.cc
 */

#ifndef COMMON_FILE_INPUT_H
#define COMMON_FILE_INPUT_H

#include <main.h>

/**
  * The file_input class represents an abstract base class for input
  * data sources.  It shall be derived from before it can be used.
  */
class file_input
{
public:
    /**
      * The destructor.
      */
    virtual ~file_input();

    /**
      * The fatal method is used to report fatal errors occurring when
      * operating on this output stream.  The program name and the file
      * name are prepended.  It exits with an exit status of one.  It
      * does not return.
      *
      * \param fmt
      *     The message format to print.  See printf(3) for a
      *     description of the format and its arguments.
      */
    void fatal(const char *fmt, ...) const ATTR_PRINTF(2, 3);

    /**
      * The nfatal method is used to report fatal errors occurring when
      * operating on this output stream.  The program name and the
      * file name is prepended.  The strerror translation of errno is
      * appended.  It exits with an exit status of one.  It does not
      * return.
      *
      * \param fmt
      *     The message format to print.  See printf(3) for a
      *     description of the format and its arguments.
      */
    void nfatal(const char *fmt, ...) const ATTR_PRINTF(2, 3);

    /**
      * The read method is used to read data from the file.  It only
      * returns if there is no error.
      *
      * \param data
      *     The location to write the data read from the file.
      * \param nbytes
      *     The maximum number of bytes read into the buffer from the file.
      * \returns
      *     The number of bytes read into the buffer, or 0 for end of file.
      */
    virtual int read(void *data, int nbytes) const = 0;

    /**
      * The filename method is used to determine the name of the file.
      */
    virtual const char *filename() const = 0;

protected:
    /**
      * The default constructor.
      * Only derived classes may use this constructor.
      */
    file_input();

private:
    /**
      * The copy constructor.  Do not use.
      */
    file_input(const file_input &);

    /**
      * The assignment operator.  Do not use.
      */
    file_input &operator = (const file_input &);
};

#endif /* COMMON_FILE_INPUT_H */
