//
//	tardy - a tar post-processor
//	Copyright (C) 1998, 1999, 2003, 2004 Peter Miller;
//	All rights reserved.
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
//
// MANIFEST: interface definition for common/file/output.cc
//

#ifndef COMMON_FILE_OUTPUT_H
#define COMMON_FILE_OUTPUT_H

#include <main.h>

/**
  * The file_output class is used to represent an abstract form of binary
  * file output.  Shall be derived from before it can be used.
  */
class file_output
{
public:
    /**
      * The destructor.
      */
    virtual ~file_output();

    /**
      * The fatal method is used to report fatal errors occurring when
      * operating on this output stream.  The program name and the file
      * name are prepended.  It exits with an exit status of one.  It
      * does not return.
      *
      * \param fmt
      *     The message format to print.  See printf(3) for a
      *     description of the format and its arguments.
      */
    void fatal(const char *fmt, ...) const ATTR_PRINTF(2, 3);

    /**
      * The fatal method is used to report fatal errors occurring when
      * operating on this ooutput stream. errno will be appended to the
      * message.  The program name and the file name is prepended.  The
      * strerror translation of errno is appended.  It exits with an
      * exit status of one.  It does not return.
      *
      * \param fmt
      *     The message format to print.  See printf(3) for a
      *     description of the format and its arguments.
      */
    void nfatal(const char *fmt, ...) const ATTR_PRINTF(2, 3);

    /**
      * The filename method is used to obtain the name of the file
      * being written.
      */
    virtual const char *filename() const = 0;

    /**
      * The write method is used to write data to the output stream.
      *
      * \param data
      *     This argument points to a data structure to be written.
      *     Usually an array of bytes.
      * \param nbytes
      *     This argument indicates the number fo bytes to be written.
      */
    virtual void write(const void *data, int nbytes) = 0;

protected:
    /**
      * The default constructor.
      * May onlly be used by derived classes.
      */
    file_output();

private:
    /**
      * The copy constructor.  Do not use.
      */
    file_output(const file_output &);

    /**
      * The assignment operator.  Do not use.
      */
    file_output &operator = (const file_output &);
};

#endif // COMMON_FILE_OUTPUT_H
