/*
 *	tclcrypt.c -- tcl wrapper for libc crypt and random functionality.
 *
 * This software and its doumentation may be freely distributed under
 * the terms of the GNU General Public License(GPL), Version 2
 * See http://www.fsf.org/licenses/gpl.txt for details.
 *
 * Copyright by Jan Kandziora <tkkasse@users.sf.net>
 *
 */
#define _XOPEN_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <tcl.h>
#include <unistd.h>
#include "tclcrypt.h"

#define VERSION "1.0"


/*
 *	Constants: 
 */
char saltlick[]="abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789./";


/*
 *	Utility functions:
 */
unsigned int limited_rand(unsigned int lower_limit, unsigned int upper_limit)
{
	unsigned long long int min=lower_limit;
	unsigned long long int max=upper_limit;

	return (min+((max-min)*rand())/RAND_MAX) & 0xffffffff;
}


/*
 *	User functions:
 */
static int tclcrypt_random(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	unsigned int            count;
	unsigned long long int  max=0;
	unsigned long long int  min=0;
 	int                     mode;
	char                   *p; 
	unsigned int            r;
	int                     read_bytes;
	char                    salt[9];
	unsigned int            seed;
	Tcl_Channel             seed_channel;


	/*
	 *	Check the subcommand.	
	 */
	if (argc<2)
	{
		Tcl_AppendResult(interp, "wrong # args: should be \"",argv[0]," option ?arg ...?\"",(char*)NULL);
		return TCL_ERROR;
	}
	if (strcmp(argv[1],"seed")==0)
	{
		/*
		 *	The "seed" subcommand. Check the args.
		 */
		switch (argc)
		{
			case 3:
				/*
				 *	Try to read the seed from the command line.
				 */
				if (sscanf(argv[2],"%d",&seed)!=1)
				{
					/*
					 *	Failed. Try to read the seed from a channel.
					 */
					seed_channel=Tcl_GetChannel(interp,argv[2],&mode);
					if (seed_channel==(Tcl_Channel)NULL)
					{
						return TCL_ERROR;
					}
					if ((mode & TCL_READABLE)==0)
					{
						Tcl_AppendResult(interp, "channel \"",argv[2],"\" wasn't opened for reading",(char*)NULL);
						return TCL_ERROR;
					}

					/*
					 *	Read seed from a channel.
					 */
					p=(char*)&seed;
					count=sizeof(seed);
					while (count>0)
					{
						if ((read_bytes=Tcl_Read(seed_channel,p,count))<0)
						{
							Tcl_AppendResult(interp,argv[0],": ",Tcl_GetChannelName(seed_channel),Tcl_PosixError(interp),(char*)NULL);
							return TCL_ERROR;
						}
						p+=read_bytes;
						count-=read_bytes;
					}
				}
				break;
			default:
				Tcl_AppendResult(interp, "wrong # args: should be \"",argv[0]," seed value\"",(char*)NULL);
				return TCL_ERROR;
		}

		/*
		 *	Set the random seed.
		 */
		srand(seed);
		return TCL_OK;
	} else
	if (strcmp(argv[1],"get")==0)
	{
		/*
		 *	The "get" subcommand. Check the args.
		 */
		switch (argc)
		{
			case 2:
				min=0;
				max=RAND_MAX;
				break;
			case 4:
				if (sscanf(argv[2],"%d",&min)!=1)
				{
					Tcl_AppendResult(interp,"parameter min must be an integer",(char*)NULL);
					return TCL_ERROR;
				}
				if (sscanf(argv[3],"%d",&max)!=1)
				{
					Tcl_AppendResult(interp,"parameter max must be an integer",(char*)NULL);
					return TCL_ERROR;
				}
				if (min>=max)
				{
					Tcl_AppendResult(interp,"parameter min must be smaller than max",(char*)NULL);
					return TCL_ERROR;
				}
				if (max>RAND_MAX)
				{
					Tcl_AppendResult(interp,"parameter max must be within 0 and RAND_MAX",(char*)NULL);
					return TCL_ERROR;
				}
				break;
			default:
				Tcl_AppendResult(interp, "wrong # args: should be \"",argv[0]," get ?min max?\"",(char*)NULL);
				return TCL_ERROR;
		}

		/*
		 *	Return the random value.
		 */
		sprintf(interp->result,"%d",limited_rand(min,max));
		return TCL_OK;
	} else
	if (strcmp(argv[1],"salt")==0)
	{
		/*
		 *	The "salt" subcommand.	
		 */
		if (argc!=3)
		{
			Tcl_AppendResult(interp, "wrong # args: should be \"",argv[0]," salt des|md5\"",(char*)NULL);
			return TCL_ERROR;
		}
		if (strcmp(argv[2],"des")==0)
		{
			/*
			 *	Make a DES salt.
			 */
			r=limited_rand(0,4096);
			salt[0]=saltlick[ r     & 0x3f];
			salt[1]=saltlick[(r>>6) & 0x3f];
			salt[2]='\0';

			/*
			 *	Return the salt.
			 */
			strcpy(interp->result,salt);
			return TCL_OK;
		} else
		if (strcmp(argv[2],"md5")==0)
		{
			/*
			 *	Make a MD5 salt.
			 */
			r=limited_rand(0,64*64*64*64);
			salt[0]=saltlick[ r      & 0x3f];
			salt[1]=saltlick[(r>>6)  & 0x3f];
			salt[2]=saltlick[(r>>12) & 0x3f];
			salt[3]=saltlick[(r>>18) & 0x3f];
			r=limited_rand(0,64*64*64*64);
			salt[4]=saltlick[ r      & 0x3f];
			salt[5]=saltlick[(r>>6)  & 0x3f];
			salt[6]=saltlick[(r>>12) & 0x3f];
			salt[7]=saltlick[(r>>18) & 0x3f];
			salt[8]='\0';

			/*
			 *	Return the salt.
			 */
			sprintf(interp->result,"$1$%s$",salt);
			return TCL_OK;
		} else
		{
			Tcl_AppendResult(interp, "bad option \"",argv[1],"\": must be des or md5",(char*)NULL);
			return TCL_ERROR;
		}
	} else
	{
		Tcl_AppendResult(interp, "bad subcommand \"",argv[1],"\": must be seed, get or salt",(char*)NULL);
		return TCL_ERROR;
	}
}



static int tclcrypt_salt(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
}

static int tclcrypt_password(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	/*
	 *	Check the args.
	 */
	if (argc!=3)
	{
		Tcl_AppendResult(interp,"wrong # args: should be \"",argv[0]," password salt|encryptedPassword\"",(char*)NULL);
		return TCL_ERROR;
	}

	/*
	 *	Crypt the string and return it.
	 */
	strcpy(interp->result,crypt(argv[1],argv[2]));
	return TCL_OK;
}


/*
 *	Initialization of the tclcrypt package:
 */
int Tclcrypt_Init(Tcl_Interp *interp)
{
	/*
	 *	Check Tcl version.
	 */
	if (Tcl_InitStubs(interp, "8.0",0) == NULL)
	{
		return TCL_ERROR;
	}

	if (Tcl_PkgRequire(interp, "Tcl",TCL_VERSION,0) == NULL)
	{
		if (TCL_VERSION[0] == '7')
		{
			if (Tcl_PkgRequire(interp,"Tcl","8.0",0) == NULL)
			{
				return TCL_ERROR;
			}
		}
	}

	/*
	 *	Provide a package.
	 */
	if (Tcl_PkgProvide(interp,"crypt",VERSION) != TCL_OK)
	{
		return TCL_ERROR;
	}

	/*
	 *	Install commands.
	 */
	Tcl_CreateCommand(interp,"crypt::random",(Tcl_CmdProc*)tclcrypt_random,(ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp,"crypt::password",(Tcl_CmdProc*)tclcrypt_password,(ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);

	return TCL_OK;
}

