/***************************************************************************
                          tea_text_document.c  -  description
                             -------------------
    begin                : Mon Dec 1 2003
    copyright            : (C) 2003, 2004, 2005 by Peter 'Roxton' Semiletov
    email                : roxton@ua.fm
 ***************************************************************************/

/*
Copyright (C) 1998 Olivier Sessink and Chris Mazuc
Copyright (C) 1999-2003 Olivier Sessink
(c)Oskar Swida <swida@aragorn.pb.bialystok.pl>
Copyright (C) 1999-2001  David A Knight
Copyright (C)2004 Tarot Osuji <tarot@sdf.lonestar.org>
*/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <stdio.h>
#include <strings.h>
#include <unistd.h>

#include "tea_defs.h"
#include "tea_text_document.h"
#include "callbacks.h"
#include "interface.h"
#include "tea_config.h"
#include "tea_funx.h"
#include "tea_hl.h"
#include "tea_gtk_utils.h"


////////////////////
/*
from Leafpad::indent.c
(C)2004 Tarot Osuji <tarot@sdf.lonestar.org>
*/
static gchar *compute_indentation(GtkTextBuffer *buffer, gint line) // from gedit
{
	GtkTextIter start_iter, end_iter;
	gunichar ch;
	
	gtk_text_buffer_get_iter_at_line(buffer, &start_iter, line);
	end_iter = start_iter;
	ch = gtk_text_iter_get_char(&end_iter);
	while (g_unichar_isspace(ch) && ch != '\n') {
		if (!gtk_text_iter_forward_char(&end_iter))
			break;
		ch = gtk_text_iter_get_char(&end_iter);
	}
	if (gtk_text_iter_equal(&start_iter, &end_iter))
		return NULL;
	
	return gtk_text_iter_get_text(&start_iter, &end_iter);
}

void indent_real(GtkWidget *text_view)
{
	GtkTextIter iter;
	gchar *ind, *str;
	
	GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text_view));
	
	gtk_text_buffer_get_iter_at_mark(buffer, &iter, gtk_text_buffer_get_insert(buffer));
	ind = compute_indentation(buffer, gtk_text_iter_get_line(&iter));
	str = g_strconcat("\n", ind, NULL);
        gtk_text_buffer_insert(buffer, &iter, str, -1);
	g_free(str);
	g_free(ind);
	
	gtk_text_view_scroll_mark_onscreen(
		GTK_TEXT_VIEW(text_view),
		gtk_text_buffer_get_insert(buffer));
}
///////////////////////////////


//from Bluefish::document.c
/**
 * doc_toggle_highlighting_cb:
 * @doc: a #Tdocument*
 * @unindent: #gboolean
 *
 * Indent the selected block in current document.
 * Set unindent to TRUE to unindent.
 *
 * Return value: void
 **/
void doc_indent_selection (t_note_page *doc, gboolean unindent)
 {
	GtkTextIter itstart,itend;
	if (gtk_text_buffer_get_selection_bounds(doc->text_buffer,&itstart,&itend)) {
		GtkTextMark *end;
/*	
		/* we have a selection, now we loop trough the characters, and for every newline
		we add or remove a tab, we set the end with a mark */
		end = gtk_text_buffer_create_mark(doc->text_buffer,NULL,&itend,TRUE);
		if (gtk_text_iter_get_line_offset(&itstart)>0) {
			gtk_text_iter_set_line_index(&itstart,0);
		}	
		while(gtk_text_iter_compare(&itstart,&itend) < 0) {
			GtkTextMark *cur;
/*			if (firstrun && !gtk_text_iter_starts_line(&itstart)) {
				gtk_text_iter_forward_line(&itstart);
			}
			firstrun = FALSE;*/
			cur = gtk_text_buffer_create_mark(doc->text_buffer,NULL,&itstart,TRUE);
			if (unindent) {
				/* when unindenting we try to set itend to the end of the indenting step
				which might be a tab or 'tabsize' spaces, then we delete that part */
				gboolean cont=TRUE;
				gchar *buf = NULL;
				gunichar cchar = gtk_text_iter_get_char(&itstart);
				if (cchar == 9) { /* 9 is ascii for tab */
					itend = itstart;
					cont = gtk_text_iter_forward_char(&itend);
					buf = g_strdup("\t");
				} else if (cchar == 32) { /* 32 is ascii for space */
					gchar *tmpstr;
					gint i=0;
					itend = itstart;
					gtk_text_iter_forward_chars(&itend,confile.tab_size);
					tmpstr = gtk_text_buffer_get_text(doc->text_buffer,&itstart,&itend,FALSE);
					//DEBUG_MSG("tab_width=%d, strlen(tmpstr)=%d, tmpstr='%s'\n",main_v->props.editor_tab_width,strlen(tmpstr),tmpstr);
					while (cont && tmpstr[i] != '\0') {
						cont = (tmpstr[i] == ' ');
						i++;
					}
					if (cont) {
						buf = tmpstr;
					} else {
						g_free(tmpstr);
					}
				} else {
					cont = FALSE;
				}
				if (cont) {
					gint offsetstart, offsetend;				
					offsetstart = gtk_text_iter_get_offset(&itstart);
					offsetend = gtk_text_iter_get_offset(&itend);
					gtk_text_buffer_delete(doc->text_buffer,&itstart,&itend);
				}
#ifdef DEBUG
				else {
					//DEBUG_MSG("doc_indent_selection, NOT continue!!\n");
				}
#endif
			} else { /* indent */
				gint offsetstart = gtk_text_iter_get_offset(&itstart);
				gchar *indentstring;
				gint indentlen;
				if (confile.ins_spaces_on_tab_press) {
                                          indentstring = g_strnfill (confile.tab_size, ' '); 
                  
					//indentstring = bf_str_repeat(" ", confile.tab_size);
					indentlen = confile.tab_size;
				} else {
					indentstring = g_strdup("\t");
					indentlen=1;
				}
				gtk_text_buffer_insert(doc->text_buffer,&itstart,indentstring,indentlen);
				//doc_unre_add(doc, indentstring, offsetstart, offsetstart+indentlen, UndoInsert);
				g_free(indentstring);
			}
			gtk_text_buffer_get_iter_at_mark(doc->text_buffer,&itstart,cur);
			gtk_text_buffer_get_iter_at_mark(doc->text_buffer,&itend,end);
			gtk_text_buffer_delete_mark(doc->text_buffer,cur);
			gtk_text_iter_forward_line(&itstart);
		}
		gtk_text_buffer_delete_mark(doc->text_buffer,end);
	} else {
		/* there is no selection, work on the current line */
		GtkTextIter iter;
		gtk_text_buffer_get_iter_at_mark(doc->text_buffer,&iter,gtk_text_buffer_get_insert(doc->text_buffer));
		gtk_text_iter_set_line_offset(&iter,0);
		if (unindent) {
			gint deletelen = 0;
			gchar *tmpstr, *tmp2str;
			GtkTextIter itend = iter;
			gtk_text_iter_forward_chars(&itend,confile.tab_size);
			tmpstr = gtk_text_buffer_get_text(doc->text_buffer,&iter,&itend,FALSE);
                        tmp2str = g_strnfill  (confile.tab_size, ' '); 
			if (tmpstr[0] == '\t') {
				deletelen = 1;
			} else if (tmpstr && strncmp(tmpstr,tmp2str,confile.tab_size)==0) {
				deletelen = confile.tab_size;
			}
			g_free(tmpstr);
			g_free(tmp2str);
			if (deletelen) {
				itend = iter;
				gtk_text_iter_forward_chars(&itend,deletelen);
				gtk_text_buffer_delete(doc->text_buffer,&iter,&itend);
			}
		} else { /* indent */
			gchar *indentstring;
			gint indentlen;
			if (confile.ins_spaces_on_tab_press) {
                                indentstring = g_strnfill  (confile.tab_size, ' '); 
				indentlen = confile.tab_size;
			} else {
				indentstring = g_strdup("\t");
				indentlen=1;
			}
			gtk_text_buffer_insert(doc->text_buffer,&iter,indentstring,indentlen);
			g_free(indentstring);
		}
	}
}
//////////////////////


//from Bluefish:: gtk_easy.c, modified by roxton

/**
 * widget_get_string_size:
 * @widget: #GtkWidget* to put the string on
 * @string: #ghcar* with the string
 *
 * This function will calculate the width in pixels from the
 * string passed to it in string, using the font from widget
 *
 * Return value: #gint pixels
 */
gint widget_get_string_size (GtkWidget *w, gchar *s) 
{
  gint r = -1;

  PangoLayout *l = gtk_widget_create_pango_layout (w, s);
  if (l)
     {
      pango_layout_get_pixel_size (l, &r, NULL);
      g_object_unref (G_OBJECT (l));
     }

  return r;
}


//from Bluefish:: document.c
/**
 * This function is taken from gtksourceview
 * Copyright (C) 2001
 * Mikael Hermansson <tyan@linux.se>
 * Chris Phelps <chicane@reninet.com>
 */
static gint textview_calculate_real_tab_width(GtkWidget *textview, gint tab_size) {
	gchar *tab_string;
	gint counter = 0;
	gint tab_width = 0;

	if (tab_size <= 0)
		return 0;

	tab_string = g_malloc (tab_size + 1);
	while (counter < tab_size) {
		tab_string[counter] = ' ';
		counter++;
	}
	tab_string[tab_size] = '\0';
	tab_width =  widget_get_string_size(textview, tab_string);
	g_free(tab_string);
/*	if (tab_width < 0) tab_width = 0;*/
	return tab_width;
}

/**
 * doc_set_tabsize:
 * @doc: a #Tdocument
 * @tabsize: a #gint with the tab size
 *
 * this function will set the textview from doc to use the tabsize
 * described by tabsize
 *
 * Return value: void
 **/
void doc_set_tabsize(t_note_page *doc, gint tabsize) {
	PangoTabArray *tab_array;
	gint pixels = textview_calculate_real_tab_width(GTK_WIDGET(doc->text_view), tabsize);
	//DEBUG_MSG("doc_set_tabsize, tabsize=%d, pixels=%d\n", tabsize, pixels);
	tab_array = pango_tab_array_new (1, TRUE);
	pango_tab_array_set_tab (tab_array, 0, PANGO_TAB_LEFT, pixels);
	gtk_text_view_set_tabs (GTK_TEXT_VIEW (doc->text_view), tab_array);
	pango_tab_array_free(tab_array);
}


 //from Bluefish :: document.c
/* contributed by Oskar Swida <swida@aragorn.pb.bialystok.pl>, with help from the gedit source */
static gboolean doc_textview_expose_event_lcb(GtkWidget * widget, GdkEventExpose * event, gpointer data) {
	GtkTextView *view = (GtkTextView*)widget;
	GdkRectangle rect;
	GdkWindow *win;
	GtkTextIter l_start,l_end,it;
	gint l_top1,l_top2;
	PangoLayout *l;
	gchar *pomstr;
	gint numlines,w,i;

	win = gtk_text_view_get_window(view,GTK_TEXT_WINDOW_LEFT);
	if (win!=event->window) return FALSE;

	gtk_text_view_get_visible_rect(view,&rect);
	gtk_text_view_get_line_at_y(view,&l_start,rect.y,&l_top1);
	gtk_text_view_get_line_at_y(view,&l_end,rect.y+rect.height,&l_top2);
	l = gtk_widget_create_pango_layout(widget,"");

	numlines = gtk_text_buffer_get_line_count(gtk_text_view_get_buffer(view));
	pomstr = g_strdup_printf("%d",MAX(99,numlines));
	pango_layout_set_text(l,pomstr,-1);
	g_free(pomstr);
	pango_layout_get_pixel_size(l,&w,NULL);
	gtk_text_view_set_border_window_size(view,GTK_TEXT_WINDOW_LEFT,w+4);
	it = l_start;
	for(i=gtk_text_iter_get_line(&l_start);i<=gtk_text_iter_get_line(&l_end);i++) {
		gtk_text_iter_set_line(&it,i);
		gtk_text_view_get_line_yrange(view,&it,&w,NULL);
		gtk_text_view_buffer_to_window_coords(view,GTK_TEXT_WINDOW_LEFT,0,w,NULL,&w);

		pomstr = g_strdup_printf("%d",i+1);
		pango_layout_set_text(l,pomstr,-1);

		gtk_paint_layout(widget->style,win,GTK_WIDGET_STATE(widget),FALSE,NULL,widget,NULL,2,w,l);
      g_free(pomstr);
	}
	g_object_unref(G_OBJECT(l));
	return TRUE;
}

/**
 * document_set_line_numbers:
 * @doc: a #Tdocument*
 * @value: a #gboolean
 *
 * Show or hide linenumbers (at the left of the main GtkTextView).
 *
 * Return value: void
 **/
void document_set_line_numbers(t_note_page* doc, gboolean value) {
	if (value) {
		gtk_text_view_set_left_margin(GTK_TEXT_VIEW(doc->text_view),2);
		gtk_text_view_set_border_window_size(GTK_TEXT_VIEW(doc->text_view),GTK_TEXT_WINDOW_LEFT,20);
		g_signal_connect(G_OBJECT(doc->text_view),"expose-event",G_CALLBACK(doc_textview_expose_event_lcb),doc);
	} else {
		gtk_text_view_set_left_margin(GTK_TEXT_VIEW(doc->text_view),0);
		gtk_text_view_set_border_window_size(GTK_TEXT_VIEW(doc->text_view),GTK_TEXT_WINDOW_LEFT,0);
	}
  doc->linenums = value;

}
//////////
static gchar* get_full_fname (const gchar *fname, const gchar *linkname)
{
  gchar *dir = g_path_get_dirname (fname);
  gchar *filename = create_full_path (linkname, dir);
  g_free (dir);
  return filename;
}


static void msg_wrong_encoding (gchar *filename)
{
  log_to_memo (_("Sorry, but I cannot determine the charset of %s. Please try again and select it manually."), filename, LM_ERROR);
}


void do_backup (gchar *file_name, gboolean do_check)
{
  if (do_check)
     if (! confile.do_backup)
        return;

  if (! g_file_test (file_name, G_FILE_TEST_EXISTS))
     return;

  gchar *bak = g_strconcat (file_name, ".bak", NULL);
  
  if (copy_file (file_name, bak))
     log_to_memo (_("%s is created"), bak, LM_NORMAL);
  else
      log_to_memo (_("Cannot do backup for %s"), file_name, LM_ERROR);
                       
  g_free (bak);
}


gboolean text_doc_save (t_note_page *doc, gchar *a_filename)
{
  if (! doc || ! a_filename)
     return FALSE;

  if (doc->readonly)
     {
      log_to_memo (_("This file is read-only"), NULL, LM_ERROR);
      return FALSE;
     }

  gboolean result = FALSE;
  
  if (g_file_test (a_filename, G_FILE_TEST_EXISTS))
  if (access (a_filename, W_OK) != 0)
     {
      log_to_memo (_("%s is not writable for you!"), a_filename, LM_ERROR);
      return FALSE;
     }

  if (g_file_test (a_filename, G_FILE_TEST_IS_DIR))
     {
      log_to_memo (_("And how I can save this text file as a DIRECTORY?!"), NULL, LM_ERROR);
      return FALSE;
     }

  if (g_utf8_collate (doc->encoding, "UTF-8") == 0)
     {
      do_backup (a_filename, TRUE);
      result = doc_save_buffer_to_file (doc->text_buffer, a_filename);
     }
  else
      {
       if (g_utf8_collate (doc->encoding, CURR_LOCALE) == 0)
         {
          do_backup (a_filename, TRUE);
          result = doc_save_buffer_to_file_l (doc->text_buffer, a_filename);
         }
       else
           {
            do_backup (a_filename, TRUE);
            result = doc_save_buffer_to_file_iconv (doc->text_buffer, a_filename, doc->encoding);
           }
      }
  
  if (g_utf8_collate (a_filename, confile.tea_rc) == 0)
     {
      confile_reload ();
      doc_update_all ();
      log_to_memo (_("config reloaded"), NULL, LM_NORMAL);
     }
  else
  if (g_utf8_collate (a_filename, confile.tea_hotkeys) == 0)
     {
      reload_hotkeys ();
      log_to_memo (_("hotkeys reloaded"), NULL, LM_NORMAL);
     }
  else
  if (g_utf8_collate (a_filename, confile.user_menu_file) == 0)
     {
      reload_usermenu ();
      log_to_memo (_("user menu reloaded"), NULL, LM_NORMAL);
     }
  else
  if (g_utf8_collate (a_filename, confile.tea_kwas_bookmarks) == 0)
     {
      glist_strings_free (gl_tea_kwas_bookmarks); 
      if (g_file_test (confile.tea_kwas_bookmarks, G_FILE_TEST_EXISTS))
          gl_tea_kwas_bookmarks = load_file_to_glist (confile.tea_kwas_bookmarks);

      log_to_memo (_("kwas bookmarks reloaded"), NULL, LM_NORMAL);
     }
  else
  if (g_utf8_collate (a_filename, confile.tea_autoreplace) == 0)
      reload_autoreplace ();
  else
  if (g_utf8_collate (a_filename, confile.bmx_file) == 0)
     {
      bmx_reload ();
      log_to_memo (_("%s reloaded"), confile.bmx_file, LM_NORMAL);
     }
  else
  if (g_utf8_collate (a_filename, confile.autosave_file) == 0)
     {
      reload_autosave;
      log_to_memo (_("%s reloaded"), confile.autosave_file, LM_NORMAL);
     }

  gchar *hm = get_hl_name (doc->file_name_utf8);
  doc->hl_mode = ch_str (doc->hl_mode, hm);
  g_free (hm);   
  
  gtk_text_buffer_set_modified (doc->text_buffer, FALSE);  
  
  return result;
}


gchar* doc_get_sel (t_note_page *doc) 
{
  GtkTextIter start;
  GtkTextIter end;
  if (gtk_text_buffer_get_selection_bounds (doc->text_buffer, &start, &end))
     return (gtk_text_buffer_get_text (doc->text_buffer, &start, &end, FALSE));
  else
      return NULL;
}


gboolean doc_is_sel (GtkTextBuffer *text_buffer)
{
  return gtk_text_buffer_get_selection_bounds (text_buffer, NULL, NULL);
}


gchar* doc_get_buf (GtkTextBuffer *text_buffer)
{
  GtkTextIter itstart, itend;

  gtk_text_buffer_get_iter_at_offset (text_buffer, &itstart, 0);
  gtk_text_buffer_get_iter_at_offset(text_buffer, &itend, gtk_text_buffer_get_char_count (text_buffer));

  return gtk_text_buffer_get_text (text_buffer, &itstart, &itend, FALSE);
}


void doc_select_line (t_note_page *doc, gint line)
{
  GtkTextIter itstart, itend;
  gtk_text_buffer_get_iter_at_line (doc->text_buffer, &itstart, line - 1);
  itend = itstart;
  if (gtk_text_iter_forward_to_line_end (&itend))
     {
      gtk_text_buffer_place_cursor (doc->text_buffer, &itstart);
      GtkTextMark *m = gtk_text_buffer_get_mark (doc->text_buffer, "insert");
      if (m)
        {
         gtk_text_view_scroll_to_mark (doc->text_view, m, 0.0, TRUE, 0.0, 0.0);
         gtk_text_buffer_move_mark_by_name (doc->text_buffer, "selection_bound", &itend);
        }
     } 
}


gboolean doc_save_buffer_to_file (GtkTextBuffer *text_buffer, gchar *filename)
{
  gchar *t = doc_get_buf (text_buffer);
  if (! t)
     return FALSE;
  gboolean r = save_string_to_file (filename, t);
  g_free (t);
  return r;
}


gboolean doc_save_buffer_to_file_l (GtkTextBuffer *text_buffer, gchar *filename)
{
  gsize bytes_read;
  gsize bytes_written;
  gchar *buf = doc_get_buf (text_buffer);

  if (! buf)
     return FALSE;

  gchar *lbuf = g_locale_from_utf8 (buf, -1, &bytes_read, &bytes_written, NULL);
  gboolean result = save_string_to_file (filename, lbuf);

  g_free (buf);
  g_free (lbuf);
                        
  return result;
}


gboolean doc_save_buffer_to_file_iconv (GtkTextBuffer *text_buffer, gchar *filename, gchar *enc)
{
  gchar *t;
  gboolean result = FALSE;
  gsize bytes_read;
  gsize bytes_written;

  gchar *buf = doc_get_buf (text_buffer);
  if (! buf)
     return FALSE;

  t = g_convert (buf, -1, enc, "UTF-8", &bytes_read, &bytes_written, NULL);

  if (! t)
     {
      g_free (buf);
      return FALSE;
     }

  result = save_string_to_file (filename, t);

  g_free (buf);
  g_free (t);

  return result;
}


static const char *xpm_data[] = 
                                {
                                 "6 6 2 1",
                                 "  c None",
                                 ". c #000000000000",
                                 ".    .",
                                 " .  . ",
                                 "  ..  ",
                                 "  ..  ",
                                 " .  . ",
                                 ".    ."
                                };


t_note_page* page_create_new (void)
{
  t_note_page *page = (t_note_page *) g_malloc (sizeof (t_note_page));

  page->toggle_images_visibility = FALSE;
  page->position = 1;
  page->readonly = FALSE;
  page->current_path = NULL;
  page->last_searched_text = NULL;
  page->hl_mode = g_strdup (HL_NONE);
  page->autosave = FALSE;

  page->scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (page->scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_show (page->scrolledwindow);

  page->text_view = gtk_text_view_new ();
  //gtk_widget_set_name (page->text_view, "MyTextView");

  page->encoding = g_strdup ("UTF-8");
  page->linenums = FALSE;

  gtk_widget_show (page->text_view);
 
  if (confile.word_wrap == 1)
     gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (page->text_view), GTK_WRAP_WORD);
  else
      gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (page->text_view), GTK_WRAP_NONE);

  gtk_container_set_border_width (GTK_CONTAINER (page->text_view), 1);

  gtk_container_add (GTK_CONTAINER (page->scrolledwindow), page->text_view);

  page->tab_label = gtk_label_new (NULL);
  GTK_WIDGET_UNSET_FLAGS (page->tab_label, GTK_CAN_FOCUS);

  GtkWidget *hbox;
  hbox = gtk_hbox_new (FALSE, 0);

  page->but = gtk_button_new ();
  page->dclicked_y = 0;

  GtkWidget *w_pixmap;
  GdkPixmap *pixmap;
  GdkBitmap *mask;
  GtkStyle *style;

  style = gtk_widget_get_style (tea_main_window);
  pixmap = gdk_pixmap_create_from_xpm_d (tea_main_window->window,  &mask,
                                         &style->bg[GTK_STATE_NORMAL],
                                         (gchar **) xpm_data);

  w_pixmap = gtk_image_new_from_pixmap (pixmap, mask);
  gtk_widget_show (w_pixmap);
  gtk_container_add (GTK_CONTAINER (page->but), w_pixmap);

  gtk_box_pack_start (GTK_BOX(hbox), page->tab_label, TRUE, FALSE, UI_PACKFACTOR);
  gtk_box_pack_start (GTK_BOX(hbox), page->but, FALSE, FALSE, UI_PACKFACTOR);
  gtk_widget_show (hbox);
  gtk_widget_show (page->but);
  gtk_widget_show (page->tab_label);
 
  g_signal_connect ((gpointer) page->but, "clicked",
                    G_CALLBACK (on_button_close),
                    page);
  
  page->b_saved = FALSE;
  page->file_name_local = g_strdup ("noname");
  page->file_name_utf8 = g_strdup ("noname");

  gtk_text_view_set_buffer (page->text_view, gtk_text_buffer_new (main_tags_table));
  page->text_buffer = gtk_text_view_get_buffer (page->text_view);

  g_signal_connect (page->text_view, "key-press-event",
                    G_CALLBACK (on_editor_keypress), page);

  page->um = UNDOMGR (undomgr_new ());
  undomgr_attach (page->um, page->text_view);

  dox = g_list_append (dox, page);
  gtk_notebook_append_page_menu (GTK_NOTEBOOK(notebook1), page->scrolledwindow, hbox, NULL);

  if (confile.show_line_nums == 1)
     document_set_line_numbers (page, TRUE);

  set_dnd_accept (page->text_view);

  return page;
}


gpointer glist_find_by_index (GList *list, int i)
{
  return g_list_nth_data (list, i);
}


t_note_page* get_page (void)
{
  cur_text_doc = NULL;

  gint i = gtk_notebook_get_current_page (notebook1);
  if (i == -1)
    return NULL;
  else
      cur_text_doc = (t_note_page *) g_list_nth_data (dox, i);

  return cur_text_doc;
}


t_note_page* get_page_by_index (int i)
{
  return (t_note_page *) g_list_nth_data (dox, i);
}


void page_del_by_index (int i)
{
  t_note_page *page;

  if (i != -1)
     {
      page = glist_find_by_index (dox, i);
      dox = g_list_remove (dox, page);
      page_free (page);
      gtk_notebook_remove_page (notebook1, i);
      update_recent_list_menu (FALSE);
     }
}


t_note_page* doc_clear_new (void) 
{
  t_note_page *doc = page_create_new ();

  g_free (doc->file_name_local);
  g_free (doc->file_name_utf8);
 
  doc->file_name_local = get_noname_name ();
  doc->file_name_utf8 = g_strdup (doc->file_name_local);

  gtk_label_set_label (doc->tab_label, doc->file_name_utf8);
  doc_apply_settings (doc);
  gtk_notebook_set_current_page (notebook1, g_list_length (dox) - 1);
  return doc;
}


t_note_page* doc_open_file (gchar *a_filename)
{
  if (! a_filename)
     return NULL;
  
  gsize length;
  GError *error = NULL;
  gchar *buf;
  gchar *newlabel;
  gsize bytes_read;
  gsize bytes_written;
  gchar *enc;
  gchar *t;
  gchar *filename;
  
  t_note_page* page = NULL;
  gchar *utf8_name = get_8_filename (a_filename);
  if (! utf8_name)
     utf8_name = g_strdup ("errorname");   

  if (! g_file_test (a_filename, G_FILE_TEST_EXISTS))
     {
      log_to_memo (_("%s is not exists!"), utf8_name, LM_ERROR);
      g_free (utf8_name); 
      return NULL;       
     }
    
  if (access (a_filename, R_OK) != 0)
     {
      log_to_memo (_("You have NO access to %s"), utf8_name, LM_ERROR);
      g_free (utf8_name);
      return NULL;
     }

  filename = g_strdup (a_filename);
  
  if (g_file_test (a_filename, G_FILE_TEST_IS_SYMLINK))
     {
      g_free (filename);
      filename = (gchar *) g_malloc (2048);
      readlink (a_filename, filename, 2048);
  
      g_free (utf8_name);    
      utf8_name = get_8_filename (filename);

      if (access (filename, R_OK) != 0)
        {
         log_to_memo (_("You have NO access to %s"), utf8_name, LM_ERROR);
         g_free (filename);
         g_free (utf8_name);
         return NULL;
        }
     }
  /*
   if (is_rtf (utf8_name))
      {
       buf = text_load_rtf (filename);
       page = doc_ins_to_new (buf); 
       g_free (buf);

       g_free (page->file_name_local);
       g_free (page->file_name_utf8);

       page->file_name_utf8 = g_strdup (utf8_name);
       page->file_name_local = g_strdup (filename); 

       page->b_saved = TRUE;
       page->readonly = TRUE;
       newlabel = g_path_get_basename (page->file_name_local);
       gtk_label_set_label (page->tab_label, newlabel);
                
       doc_apply_settings (page);
       editor_set_pos (page, 0);
       gtk_text_buffer_set_modified (page->text_buffer, FALSE);

       log_to_memo (_("%s is opened"), page->file_name_utf8, LM_NORMAL);
       g_free (filename);
       g_free (utf8_name);
       g_free (newlabel);
       tabs_reload ();
     
       return page;
      }   
*/
   if (is_format_readonly (utf8_name))
      {
       if (check_ext (filename, ".sxw"))
          buf = read_xml_text (filename, "content.xml", "text:p");
       else
          if (check_ext (filename, ".abw"))
              buf = read_abw_text (filename, filename, "p");
       else
          if (check_ext (filename, ".kwd"))
             buf = read_xml_text (filename, "maindoc.xml", "TEXT");
       else
           if (is_rtf (utf8_name))
              buf = text_load_rtf (filename);
       else
            if (is_ext (utf8_name, ".gz", ".zip", ".bz2", NULL))   
                buf = text_load_gzip (filename);
       
     if (! g_utf8_validate (buf, -1, NULL)) 
        {
         log_to_memo (_("Sorry, but %s data is not UTF-8 :("), utf8_name, LM_ERROR);
         g_free (utf8_name);
         return NULL;
        }
     

       page = doc_ins_to_new (buf); 

       g_free (buf);

       g_free (page->file_name_local);
       g_free (page->file_name_utf8);
 

       page->file_name_utf8 = g_strdup (utf8_name);
       page->file_name_local = g_strdup (filename); 
  
       page->b_saved = TRUE;
       page->readonly = TRUE;
       newlabel = g_path_get_basename (page->file_name_local);
       gtk_label_set_label (page->tab_label, newlabel);
                
       doc_apply_settings (page);
       editor_set_pos (page, 0);
       gtk_text_buffer_set_modified (page->text_buffer, FALSE);

       log_to_memo (_("%s is opened"), page->file_name_utf8, LM_NORMAL);
       g_free (filename);
       g_free (utf8_name);
       g_free (newlabel);
       tabs_reload ();
     
       return page;
      }
/*
  if (is_ext (utf8_name, ".gz", ".zip", ".bz2", NULL))   
     {
      page = doc_clear_new ();

      if (page->encoding)
         g_free (page->encoding);

      if (cur_settings.selected_enc)
         page->encoding = g_strdup (cur_settings.selected_enc);
      else
          page->encoding = g_strdup (confile.default_charset);

      buf = text_load_gzip (filename);

      if (g_utf8_validate (buf, -1, NULL)) 
         gtk_text_buffer_set_text (page->text_buffer, buf, -1);
      
      g_free (page->file_name_utf8);
      g_free (page->file_name_local);

      page->file_name_utf8 = g_strdup (utf8_name);
      page->file_name_local = g_strdup (filename);

      page->b_saved = TRUE;
      page->readonly = TRUE;
      newlabel = g_path_get_basename (page->file_name_utf8);
      gtk_label_set_label (page->tab_label, newlabel);
                
      doc_apply_settings (page);
      editor_set_pos (page, 0);

      gchar *hm = get_hl_name (page->file_name_utf8);
      page->hl_mode = ch_str (page->hl_mode, hm);
      g_free (hm);   

      log_to_memo (_("%s is opened"), page->file_name_utf8, LM_NORMAL);

      if (confile.scan_for_links_on_doc_open == 1)
         scan_links();

      if (confile.do_hl_on_fileopen)
        apply_hl (page);

      g_free (filename);
      g_free (newlabel);
      g_free (buf);
      g_free (utf8_name);

      gtk_text_buffer_set_modified (page->text_buffer, FALSE);
      tabs_reload ();

      return page;
     } 
*/
  if (! g_file_get_contents (filename, &buf, &length, &error))
     {
      g_free (filename);
      g_free (utf8_name);
      return NULL;  
     }
  
  error = NULL;
  page = doc_clear_new ();
  
  if (page->encoding)
     g_free (page->encoding);

  if (cur_settings.selected_enc)
      page->encoding = g_strdup (cur_settings.selected_enc);
  else
      page->encoding = g_strdup (confile.default_charset);
  
  if (g_utf8_collate (page->encoding, CHARSET_JAPANESE) == 0)
     {
      enc = detect_charset_ja (buf);
      page->encoding = ch_str (page->encoding, enc);
     }

  if (g_utf8_collate (page->encoding, CHARSET_AUTODETECT) == 0)
     {
      enc = enc_guess (buf);
      if (enc)
         page->encoding = ch_str (page->encoding, enc);
      else
          {
           g_free (buf);
           page_del_by_index (gtk_notebook_get_current_page ((GtkNotebook *) notebook1));
           msg_wrong_encoding (filename); 
           g_free (utf8_name);  
           g_free (filename);
           return NULL; 
          }
         }
           
  if (g_utf8_collate (page->encoding, "UTF-8") == 0)
     {
      if (! g_utf8_validate (buf, -1, NULL))
         {
          g_free (buf);
          page_del_by_index (gtk_notebook_get_current_page ((GtkNotebook *) notebook1));
          msg_wrong_encoding (filename); 
          g_free (utf8_name);
          g_free (filename);
          return NULL; 
         }

       gtk_text_buffer_set_text (page->text_buffer, buf, length);
      }
  else
     {
      if (g_utf8_collate (page->encoding, CURR_LOCALE) == 0)
         {
          t = g_locale_to_utf8 (buf, length, &bytes_read, &bytes_written, NULL);

          if (! t)
            {
             g_free (buf);
             page_del_by_index (gtk_notebook_get_current_page ((GtkNotebook *) notebook1));
             msg_wrong_encoding (filename); 
             g_free (filename);
             g_free (utf8_name);
             return NULL; 
            }

          if (g_utf8_validate (t, bytes_written, NULL)) 
             gtk_text_buffer_set_text (page->text_buffer, t, bytes_written);
       
          g_free (t);
         }
     else //all other charsets
        { 
         t = g_convert (buf, length, "UTF-8", page->encoding, &bytes_read, &bytes_written, NULL);

         if (! t)
            {
             g_free (buf);
             page_del_by_index (gtk_notebook_get_current_page ((GtkNotebook *) notebook1));
             msg_wrong_encoding (filename); 
             g_free (filename);
             return NULL; 
            }

         if (g_utf8_validate (t, bytes_written, NULL)) 
            gtk_text_buffer_set_text (page->text_buffer, t, bytes_written);

         g_free (t);
        }
     }

  g_free (page->file_name_local);
  g_free (page->file_name_utf8);

  page->file_name_utf8 = g_strdup (utf8_name);
  page->file_name_local = g_strdup (filename);

  if (str_in_glist (gl_autosave, page->file_name_utf8))
     page->autosave = TRUE;

  page->b_saved = TRUE;
  newlabel = g_path_get_basename (page->file_name_utf8);
  gtk_label_set_label (page->tab_label, newlabel);
                
  doc_apply_settings (page);
  editor_set_pos (page, 0);

  gchar *hm = get_hl_name (page->file_name_utf8);
  page->hl_mode = ch_str (page->hl_mode, hm);
  g_free (hm);   

  log_to_memo (_("%s is opened"), page->file_name_utf8, LM_NORMAL);

  if (confile.scan_for_links_on_doc_open == 1)
      scan_links();

  if (confile.do_hl_on_fileopen)
     apply_hl (page);

  g_free (filename);
  g_free (utf8_name);
  g_free (newlabel);
  g_free (buf);

  gtk_text_buffer_set_modified (page->text_buffer, FALSE);
  tabs_reload ();

  return page;
}


void doc_apply_settings (t_note_page *doc)
{
  if (! doc) return;
  
  PangoFontDescription *font_desc = pango_font_description_from_string (confile.editor_font);
  gtk_widget_modify_font (doc->text_view, font_desc);
  pango_font_description_free (font_desc);

  g_object_set (G_OBJECT (tag_comment), "foreground", confile.tag_comment, "font", confile.tag_comment_font, NULL);
  g_object_set (G_OBJECT (tag_identifier), "foreground", confile.tag_identifier, "font", confile.tag_identifier_font, NULL);
  g_object_set (G_OBJECT (tag_digit), "foreground", confile.tag_digit, "font", confile.tag_digit_font, NULL);
  g_object_set (G_OBJECT (tag_string), "foreground", confile.tag_string, "font", confile.tag_string_font, NULL);
  g_object_set (G_OBJECT (tag_html_tag), "foreground", confile.tag_html_tag, "font", confile.tag_html_tag_font, NULL);
  g_object_set (G_OBJECT (tag_preprocessor), "foreground", confile.tag_preprocessor, "font", confile.tag_preprocessor_font, NULL);
  g_object_set (G_OBJECT (tag_type), "foreground", confile.tag_type, "font", confile.tag_type_font, NULL);
   
  GdkColor background;
  GdkColor text;
  GdkColor selection;
  GdkColor sel_text;

  gdk_color_parse (confile.text_foreground, &text);
  gdk_color_parse (confile.text_background, &background);
  gdk_color_parse (confile.text_sel_background, &selection);
  gdk_color_parse (confile.text_sel_foreground, &sel_text);
  
  widget_set_colors (doc->text_view, &background, &text, &selection, &sel_text);  
  doc_set_tabsize (doc, confile.tab_size);
}


void widget_apply_colors (GtkWidget *w)
{
  if (! w) return;
  
  GdkColor background;
  GdkColor text;
  GdkColor selection;
  GdkColor sel_text;

  gdk_color_parse (confile.text_foreground, &text);
  gdk_color_parse (confile.text_background, &background);
  gdk_color_parse (confile.text_sel_background, &selection);
  gdk_color_parse (confile.text_sel_foreground, &sel_text);
  
  widget_set_colors (w, &background, &text, &selection, &sel_text);  
}

           
void page_free (t_note_page *page)
{
  if (page->text_view)
     undomgr_reset (page->um);

  gchar *f = page->file_name_local;
   
  if (f)
    {
     if (page->autosave)
        text_doc_save (page, f);
     else 
        {
         if (strcmp (f, confile.crapbook_file) != 0)
            if (page->b_saved && confile.prompt_on_not_saved && gtk_text_buffer_get_modified (page->text_buffer)) 
               if (dlg_question (_("Question"), _("%s is modified but not saved. Save it?"), page->file_name_utf8))
                  text_doc_save (page, f);
        }
      
     if (strcmp (f, confile.crapbook_file) == 0)
        doc_save_buffer_to_file (page->text_buffer, confile.crapbook_file); 
     else 
         if (g_file_test (f, G_FILE_TEST_EXISTS))
            add_recent_internal (page);
    }
  
  g_free (page->last_searched_text);
  g_free (page->encoding);
  g_free (page->hl_mode);
  g_free (page->file_name_local);
  g_free (page->file_name_utf8);
  g_free (page->current_path);
  g_free (page);

  tabs_reload ();
}


void doc_insert_at_cursor (t_note_page* doc, gchar *text)
{
  if (! doc || ! text)
     return;

  gtk_text_buffer_begin_user_action (doc->text_buffer);
  gtk_text_buffer_insert_at_cursor (doc->text_buffer, text, -1);
  gtk_text_buffer_end_user_action (doc->text_buffer);
}

      
void doc_update_cb (gpointer data, gpointer user_data)
{
  if (data)
     doc_apply_settings ((t_note_page*) data);
}


void doc_update_all (void)
{
  g_list_foreach (dox, doc_update_cb, NULL);
  set_lm_colors ();
  if (confile.do_show_main_toolbar)
     gtk_widget_show (tb_main_toolbar);
  else
      gtk_widget_hide (tb_main_toolbar);
}


gint editor_get_pos (t_note_page* doc)
{
  if (! doc->text_buffer && ! doc->text_view)
     return 0;

  GtkTextMark *mark;
  GtkTextIter it;
  mark = gtk_text_buffer_get_mark (doc->text_buffer, "insert");
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &it, mark);
  return gtk_text_iter_get_offset (&it);
}


void tv_logmemo_set_pos (guint pos )
{
  GtkTextIter it;
  gtk_text_buffer_get_iter_at_offset (log_memo_textbuffer, &it, pos);
  gtk_text_buffer_place_cursor (log_memo_textbuffer, &it);
  gtk_text_view_scroll_to_iter (tv_logmemo, &it, 0.0, FALSE, 0.0, 0.0);
}


gboolean doc_reload_text (t_note_page *doc, gchar *filename, gchar *enc)
{
  gsize *length;
  GError *error = NULL;
  gchar *buf;
  gchar *t;
  gsize bytes_read;
  gsize bytes_written;

  if (! g_file_get_contents (filename, &buf, &length, &error))
     return FALSE;
  else
  if (g_utf8_collate (enc, CURR_LOCALE) == 0)
     {
      gchar *t = g_locale_to_utf8 (buf, length, &bytes_read, &bytes_written, NULL);

      if (! t)
         {
          g_free (buf);
          return FALSE;
         }

      gtk_text_buffer_set_text (doc->text_buffer, t, bytes_written);

      g_free (buf);
      g_free (t);

      return TRUE;
     }
  
  else
      if (g_utf8_collate (enc, "UTF-8") == 0)
        {
         if (! g_utf8_validate (buf, -1, NULL))
            {
             g_free (buf);
             return FALSE;
            }

         gtk_text_buffer_set_text (doc->text_buffer, buf, length);

         g_free (buf);
         return TRUE;
        }
       else
           {//not UTF-8
            error = NULL;
            t = g_convert (buf, length, "UTF-8", enc, &bytes_read, &bytes_written, NULL);
            if (! t)
               {
                g_free (buf);
                return FALSE;
               }

            gtk_text_buffer_set_text (doc->text_buffer, t, bytes_written);

            g_free (buf);
            g_free (t);
            
            return TRUE;
      }
}


gint get_n_page_by_filename (gchar *filename)
{
  gint i = 0;
  t_note_page *t;

  GList *tl = g_list_first (dox);

  while (tl)
        {
         t = (t_note_page*) tl->data;
         if (strcmp (filename, t->file_name_utf8) == 0)
            return i;
         i++;
       
         tl = g_list_next (tl);
        }

  return -1;
}


static gint log_to_memo_counter = 0;

void log_to_memo (gchar *m1, gchar* m2, gint mode)
{
  if (! log_memo_textbuffer || ! m1)
     return;

  if ((++log_to_memo_counter) == confile.logmemo_lines_max)
     {
      log_to_memo_counter = 0;
      GtkTextIter itstart, itend;
      gtk_text_buffer_get_bounds (log_memo_textbuffer, &itstart, &itend);   
      gtk_text_buffer_delete (log_memo_textbuffer, &itstart, &itend);
     }

  if (confile.msg_counter == G_MAXINT)
     confile.msg_counter = 0;

  gchar *prefix = g_strdup_printf  ("(%d) ", ++confile.msg_counter);
  gchar *st;

  if (! m2)
     {
      st = g_strconcat (prefix, m1, "\n", NULL);

      GtkTextIter it;
      gtk_text_buffer_get_iter_at_offset (log_memo_textbuffer, &it, 0);
      gtk_text_buffer_place_cursor (log_memo_textbuffer, &it);

      if (mode == LM_NORMAL)
          gtk_text_buffer_insert_with_tags_by_name (log_memo_textbuffer, &it, st, -1, "lm_normal", NULL);
      else
          if (mode == LM_ERROR)
             gtk_text_buffer_insert_with_tags_by_name (log_memo_textbuffer, &it, st, -1, "lm_error", NULL);
      else
          if (mode == LM_ADVICE)
             gtk_text_buffer_insert_with_tags_by_name (log_memo_textbuffer, &it, st, -1, "lm_advice", NULL);
      else
          if (mode == LM_GREET)
             gtk_text_buffer_insert_with_tags_by_name (log_memo_textbuffer, &it, st, -1, "lm_greet", NULL);

      g_free (st);
      g_free (prefix);
      tv_logmemo_set_pos (0);
      return;
     }
 
  gulong len = strlen (m1) + strlen (m2) + 2024;
  gchar *message = g_malloc (len);

  g_snprintf (message, len, m1, m2);
  gchar *s = g_strconcat (prefix, message, "\n", NULL);
  
  gtk_text_buffer_insert_at_cursor (log_memo_textbuffer, s, -1);

  g_free (message);
  g_free (s);
  g_free (prefix);
  tv_logmemo_set_pos (0);
  
  return;
}


gboolean find_space (gunichar ch, gpointer user_data)
{
  if (g_unichar_isspace (ch) || (ch == '\0')) 
     return TRUE;
  else
      return FALSE; 
}


gboolean find_quote (gunichar ch, gpointer user_data)
{
  if ((ch == '"') || (ch == '\'')) 
     return TRUE;
  else
      return FALSE; 
}


//n.p. Nirvana - I Hate Myself And I Want To Die
gchar* doc_get_current_word (t_note_page *doc, GtkTextIter *itstart, GtkTextIter *itend) 
{
  GtkTextIter ittemp;
  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);

  gunichar c;
  
  *itstart = ittemp;
  *itend = ittemp;

  if (! gtk_text_iter_forward_find_char (itend, find_space, NULL, NULL))
      gtk_text_buffer_get_end_iter (doc->text_buffer, itend);

  if (! gtk_text_iter_backward_find_char (itstart, find_space, NULL, NULL))
     gtk_text_buffer_get_start_iter (doc->text_buffer, itstart); 
    else     
       gtk_text_iter_forward_char (itstart);  
         
  return (gtk_text_buffer_get_text (doc->text_buffer, itstart, itend, FALSE));
}


void doc_set_new_text (t_note_page *doc, GtkTextIter *itstart, GtkTextIter *itend, gchar *text) 
{
  gtk_text_buffer_begin_user_action (doc->text_buffer);

  gtk_text_buffer_delete (doc->text_buffer, itstart, itend);   
  gtk_text_buffer_insert (doc->text_buffer, itstart, text, -1);

  gtk_text_buffer_end_user_action (doc->text_buffer);
}


void doc_rep_sel (t_note_page *page, const gchar *newstring)
{
  if (! newstring)
     return; 

  GtkTextIter start;
  GtkTextIter end;
  if (gtk_text_buffer_get_selection_bounds (page->text_buffer, &start, &end))
     doc_set_new_text (page, &start, &end, newstring);  
}


gchar* get_c_url (t_note_page *doc) 
{
  GtkTextIter ittemp;
  GtkTextIter itstart;
  GtkTextIter itend;
  gchar *filename;

  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);

  itstart = ittemp;
  itend = ittemp;

  if (! gtk_text_iter_forward_find_char (&itend, find_quote, NULL, NULL))
      gtk_text_buffer_get_end_iter (doc->text_buffer, &itend);

  if (! gtk_text_iter_backward_find_char (&itstart, find_quote, NULL, NULL))
     gtk_text_buffer_get_start_iter (doc->text_buffer, &itstart); 
    else     
       gtk_text_iter_forward_char (&itstart);  

  gchar *result = gtk_text_buffer_get_text (doc->text_buffer, &itstart, &itend, FALSE);

  if (g_utf8_get_char (result) == '#')
     {
      doc_goto_local_label (result);
      g_free (result);
      return NULL;
     }

  gchar *dir = g_path_get_dirname (doc->file_name_local);

  if (g_utf8_get_char (result) == G_DIR_SEPARATOR)  
     filename = create_full_path (g_utf8_find_next_char (result, NULL), dir);
  else
      filename = create_full_path (result, dir);

  g_free (dir);
  g_free (result);
  gchar *result2 = strdup (filename);
  g_free (filename);

  return result2;
}


gboolean doc_search_f (t_note_page *doc, const gchar *text) 
{
  if (! text)
      return FALSE; 

  GtkTextIter ittemp;
  GtkTextIter end;
  GtkTextIter match_start;
  GtkTextIter match_end;
  gboolean result = FALSE;

  gtk_text_buffer_get_bounds (doc->text_buffer, &ittemp, &end);

  if (gtk_text_iter_forward_search (&ittemp, text, GTK_TEXT_SEARCH_TEXT_ONLY, &match_start, &match_end, &end))
     {
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "insert", &match_end);
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "selection_bound", &match_start);
      gtk_text_view_scroll_to_iter (doc->text_view, &match_start, 0.0, TRUE, 0.0, 0.0 );
      g_free (doc->last_searched_text);
      doc->last_searched_text = g_strdup (text); 
      result = TRUE;
     }

   return result;
}


gboolean doc_search_f_next (t_note_page *doc) 
{
  gboolean result = FALSE;
  if (! doc->last_searched_text)
     return FALSE;

  GtkTextIter ittemp;
  GtkTextIter end;
  GtkTextIter match_start;
  GtkTextIter match_end;
  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);

  gtk_text_buffer_get_end_iter (doc->text_buffer, &end); 

  if (gtk_text_iter_forward_search (&ittemp, doc->last_searched_text, GTK_TEXT_SEARCH_TEXT_ONLY, &match_start, &match_end, &end))
     {
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "insert", &match_end);
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "selection_bound", &match_start);
      gtk_text_view_scroll_to_iter (doc->text_view, &match_start, 0.0, TRUE, 0.0, 0.0 );
      result = TRUE;
     }

  return result;
}


void doc_replace_all (t_note_page *doc, const gchar *text, const gchar *new_text) 
{
  if (! text)
      return;
 
  if (! new_text)
      return;

  if (doc_search_f (doc, text))
     {
      doc_rep_sel (doc, new_text);
      while (doc_search_f_next (doc))
             doc_rep_sel (doc, new_text);
     }  
}


gint find_index_by_page (t_note_page *page)
{
 gint i = -1;

 GList *p = g_list_first (dox);

 while (p)
       {
        i++;  
        if (page == p->data)
           return i;

        p = g_list_next (p);  
       }

 return i;
}


gchar* get_c_url_pure (t_note_page *doc) 
{
  GtkTextIter ittemp;
  GtkTextIter itstart;
  GtkTextIter itend;

  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);

  itstart = ittemp;
  itend = ittemp;

  if (! gtk_text_iter_forward_find_char (&itend, find_quote, NULL, NULL))
      gtk_text_buffer_get_end_iter (doc->text_buffer, &itend);

  if (! gtk_text_iter_backward_find_char (&itstart, find_quote, NULL, NULL))
     gtk_text_buffer_get_start_iter (doc->text_buffer, &itstart); 
    else     
       gtk_text_iter_forward_char (&itstart);  
  
  gchar *result = gtk_text_buffer_get_text (doc->text_buffer, &itstart, &itend, FALSE);

  return result;
}


t_note_page* doc_ins_to_new (const gchar *newstring)
{
  if (! newstring) 
     return NULL;

  t_note_page *page = doc_clear_new ();
  gtk_window_set_title (GTK_WINDOW (tea_main_window), page->file_name_utf8);
  gtk_widget_grab_focus (page->text_view);
  doc_insert_at_cursor (page, newstring);
  tabs_reload ();
  return page;
}


t_note_page* get_page_text (void)
{
  cur_text_doc = NULL;

  gint i = gtk_notebook_get_current_page (notebook1);
  if (i == -1)
    return NULL;
  else
      cur_text_doc = (t_note_page *) g_list_nth_data (dox, i);

  return cur_text_doc;
}


void set_title (t_note_page *t)
{
  if (! t)
     return;

  if (! t->file_name_utf8)
     return;      

  if (! g_utf8_validate (t->file_name_utf8, -1, NULL))
      return;

  gchar *nc = NULL;

  if (confile.main_wnd_show_full_path)
      nc = g_strdup (t->file_name_utf8);
  else 
      nc = g_path_get_basename (t->file_name_utf8);

  gtk_window_set_title (GTK_WINDOW (tea_main_window), nc);
  statusbar_msg (t->encoding);
  set_last_dir (t);

  g_free (nc);
}
  

t_note_page* open_file_std (gchar *f)
{
  cur_text_doc = doc_open_file (f);
  tabs_reload ();
  set_title (cur_text_doc);
  return cur_text_doc;
}


void doc_move_to_pos_bw (t_note_page* doc, gint pos)
{
  if (! doc->text_buffer && ! doc->text_view)
     return 0;

  GtkTextMark *mark;
  GtkTextIter it;
  mark = gtk_text_buffer_get_mark (doc->text_buffer, "insert");
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &it, mark);
  
  if (gtk_text_iter_backward_chars (&it, pos))
      gtk_text_buffer_place_cursor (doc->text_buffer, &it);
  }


//n.p. Albinoni - Adagio G-moll
void doc_move_to_pos_bw_quote (t_note_page* doc)
{
  if (! doc->text_buffer && ! doc->text_view)
     return 0;

  GtkTextMark *mark;
  GtkTextIter it;
  mark = gtk_text_buffer_get_mark (doc->text_buffer, "insert");
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &it, mark);
  
  if (gtk_text_iter_backward_find_char (&it, find_quote, NULL, NULL))
      gtk_text_buffer_place_cursor (doc->text_buffer, &it);
}


void editor_set_pos (t_note_page *doc, guint pos)
{
  GtkTextIter it;
  GtkTextMark* m; 

  gtk_text_buffer_get_iter_at_offset (doc->text_buffer, &it, pos);
  gtk_text_buffer_place_cursor (doc->text_buffer, &it);
  m = gtk_text_buffer_get_mark (doc->text_buffer, "insert");
  if (m)
     gtk_text_view_scroll_to_mark (doc->text_view, m, 0.0, TRUE, 0.0, 0.0);
}
 

void editor_goto_selection (t_note_page *doc)
{
  GtkTextIter start;
  GtkTextIter end;

  if (gtk_text_buffer_get_selection_bounds (doc->text_buffer, &start, &end))
      gtk_text_view_scroll_to_iter (doc->text_view, &start, 0.0, TRUE, 0.0, 0.0); 
}


gboolean doc_search_f_ncase (t_note_page *doc, const gchar *text) 
{
  if (! text)
     return FALSE;

  gboolean result = FALSE;

  gchar *f = g_utf8_strdown (text, -1);
 
  GtkTextIter start;
  GtkTextIter end;
  GtkTextIter ittemp;
  GtkTextIter b_end;

  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);

  gtk_text_iter_forward_chars (&ittemp, g_utf8_strlen (f, -1));
  
  gint int_char_offset = gtk_text_iter_get_offset (&ittemp);

  gtk_text_buffer_get_end_iter (doc->text_buffer, &b_end);

  gchar *t = gtk_text_buffer_get_text (doc->text_buffer, &ittemp, &b_end, FALSE);
  gchar *x = g_utf8_strdown (t, -1);
  gchar *found = strstr (x, f);
 
  if (found)
     {
      glong i = g_utf8_pointer_to_offset (x, found) + int_char_offset;
      gtk_text_buffer_get_iter_at_offset (doc->text_buffer, &start, i);
      gtk_text_buffer_get_iter_at_offset (doc->text_buffer, &end,
                                          i + g_utf8_strlen (f, -1));

      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "insert", &start);
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "selection_bound", &end);
      g_free (doc->last_searched_text);
      doc->last_searched_text = g_strdup (text); 
      result = TRUE;
     }     

  g_free (t);
  g_free (x);
  g_free (f);

  return result;
}


gboolean doc_search_f_next_ncase (t_note_page *doc) 
{
  if (! doc->last_searched_text)
      return FALSE;

  GtkTextIter start;
  GtkTextIter end;

  gchar *f = g_utf8_strdown (doc->last_searched_text, -1);

  gboolean result = FALSE;
  GtkTextIter ittemp;
  GtkTextIter b_start;
  GtkTextIter b_end;
  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);

  gtk_text_iter_forward_chars (&ittemp, g_utf8_strlen (f, -1));
  gint int_char_offset = gtk_text_iter_get_offset (&ittemp);
  gtk_text_buffer_get_end_iter (doc->text_buffer, &b_end);

  gchar *t = gtk_text_buffer_get_text (doc->text_buffer, &ittemp, &b_end, FALSE);
  gchar *x = g_utf8_strdown (t, -1);
  gchar *found = strstr (x, f);

  if (found)
     {
      glong i = g_utf8_pointer_to_offset (x, found) + int_char_offset;
  
      gtk_text_buffer_get_iter_at_offset (doc->text_buffer, &start, i);
      gtk_text_buffer_get_iter_at_offset (doc->text_buffer, &end, i + g_utf8_strlen (f, -1));
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "insert", &start);
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "selection_bound", &end);

      result = TRUE;
     }     

  g_free (t);
  g_free (x);
  g_free (f);
  return result;
}


gchar* upcase_each_first_letter_ofw (t_note_page *doc)
{
  gunichar u;
  
  GtkTextIter start;
  GtkTextIter end;

  if (! gtk_text_buffer_get_selection_bounds (doc->text_buffer, &start, &end))
      return NULL; 
 
  GString* gs = g_string_sized_new (1024);

  do
    {
     if (gtk_text_iter_equal (&start, &end))
        break;

     u = gtk_text_iter_get_char (&start);

     if ( gtk_text_iter_starts_word (&start))        
        u = g_unichar_toupper (u); 
     
     gs = g_string_append_unichar (gs, u);        
    }  
   while ( gtk_text_iter_forward_char (&start) );

  gchar *s = gs->str;
  g_string_free (gs, FALSE);

  return s;
}


void doc_move_cursor_forw (t_note_page *doc, gint i) 
{
  GtkTextIter ittemp;
  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);

  if (gtk_text_iter_forward_chars (&ittemp, i))
     {
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "insert", &ittemp);
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "selection_bound", &ittemp);
     } 
}


void doc_move_cursor_backw (t_note_page *doc, gint i) 
{
  GtkTextIter ittemp;
  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);

  if (gtk_text_iter_backward_chars (&ittemp, i))
     {
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "insert", &ittemp);
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "selection_bound", &ittemp);
     }
}


static gboolean predicate_find_tag_end (gunichar ch, gpointer user_data)
{
  if (ch == '>')
     return TRUE;
  else 
      return FALSE;
}


void doc_move_cursor_backw_middle_tags (t_note_page *doc) 
{
  GtkTextIter ittemp;
  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);

  if (gtk_text_iter_backward_find_char (&ittemp, predicate_find_tag_end, NULL, NULL))
  if (gtk_text_iter_backward_find_char (&ittemp, predicate_find_tag_end, NULL, NULL))
  if (gtk_text_iter_forward_char (&ittemp))
     {
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "insert", &ittemp);
      gtk_text_buffer_move_mark_by_name (doc->text_buffer, "selection_bound", &ittemp);
     }
}


void doc_header_source_switch (t_note_page *doc)
{
  gchar *f = get_c_header_fname (cur_text_doc->file_name_utf8);
  if (f)
     {
      handle_file (f, 0, FALSE);
      g_free (f);
      return;
     }

  f = get_c_source_fname (cur_text_doc->file_name_utf8);

  if (f)
     {
      handle_file (f, 0, FALSE);
      g_free (f);
      return;
     }
}

void doc_toggle_images_visibility (t_note_page *p)
{
  GtkTextIter it_start;
  GtkTextIter it_end;

  if (! p->toggle_images_visibility)
     {
      p->toggle_images_visibility = TRUE;
      gchar *f;
      gchar *t;
      GdkPixbuf *pb;

      gtk_text_buffer_get_start_iter (p->text_buffer, &it_start); 
      gtk_text_buffer_get_end_iter (p->text_buffer, &it_end); 

      while (gtk_text_iter_forward_search (&it_start, "src=\"", GTK_TEXT_SEARCH_TEXT_ONLY, &it_start, &it_end, NULL))
           {
            it_start = it_end;
            if (gtk_text_iter_forward_find_char (&it_end, find_quote, NULL, NULL))   
               {
                f = gtk_text_buffer_get_text (p->text_buffer, &it_start, &it_end, FALSE);
                t = get_full_fname (p->file_name_utf8, f);    
                if (t)
                if (is_image (t))
                   {
                    pb = gdk_pixbuf_new_from_file_at_size (t, confile.thumb_width, confile.thumb_height, NULL);
                    gtk_text_buffer_insert_pixbuf (p->text_buffer, &it_end, pb);
                    gtk_text_buffer_set_modified (p->text_buffer, FALSE);  
                   }

                g_free (f);
                g_free (t);
               }

            it_start = it_end;
          }

      while (gtk_text_iter_forward_search (&it_start, "SRC=\"", GTK_TEXT_SEARCH_TEXT_ONLY, &it_start, &it_end, NULL))
           {
            it_start = it_end;
            if (gtk_text_iter_forward_find_char (&it_end, find_quote, NULL, NULL))   
               {
                f = gtk_text_buffer_get_text (p->text_buffer, &it_start, &it_end, FALSE);
                t = get_full_fname (p->file_name_utf8, f);    
                if (t)
                if (is_image (t))
                   {
                    pb = gdk_pixbuf_new_from_file_at_size (t, confile.thumb_width, confile.thumb_height, NULL);
                    gtk_text_buffer_insert_pixbuf (p->text_buffer, &it_end, pb);
                    gtk_text_buffer_set_modified (p->text_buffer, FALSE);  
                   }

                 g_free (f);
                 g_free (t);  
                }
           it_start = it_end;
          }
       }
  else
     {
      p->toggle_images_visibility = FALSE;
      gtk_text_buffer_get_start_iter (p->text_buffer, &it_start);
      do  
         {
          if (gtk_text_iter_get_pixbuf (&it_start))
             {
              it_end = it_start;
              if (gtk_text_iter_forward_char (&it_end))
                 gtk_text_buffer_delete (p->text_buffer, &it_start, &it_end);
              gtk_text_buffer_set_modified (p->text_buffer, FALSE);  
      
             }    
          }
      while (gtk_text_iter_forward_char (&it_start));
     } 
}


void doc_close_all (void)
{
  gint c = gtk_notebook_get_n_pages (notebook1);
  if (c == -1)
     return;

  c--;
  gint i;

  t_note_page *page;

   for (i = 0; i <= c; ++i)
      {
       page = glist_find_by_index (dox, 0);
       dox = g_list_remove (dox, page);
       page_free (page);
       gtk_notebook_remove_page (notebook1, 0);
      }

  update_recent_list_menu (FALSE);
}


GList* add_session_item_composed (GList *list, t_note_page *doc)
{
  if (! doc) return list;

  GList *l = list;
 
  l = g_list_append (list, g_strdup_printf ("%s,%s,%d",
                     doc->file_name_utf8, doc->encoding, 
                     editor_get_pos (doc)));
  return l;
}


//FIXME?
GList* add_recent_item_composed (GList *list, t_note_page *doc)
{
  if (! doc) return list;

  GList *l = list;

  l = g_list_prepend (list, g_strdup_printf ("%s,%s,%d",
                       doc->file_name_utf8, doc->encoding, 
                       editor_get_pos (doc)));


  return l;
}


void current_tag_close (t_note_page *doc, gint mode)
{
  gboolean found = FALSE;

  GtkTextIter ittemp;
  GtkTextIter itstart;
  GtkTextIter itend;
  gunichar u;
  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);

  itstart = ittemp;
  itend = ittemp;

  while (gtk_text_iter_backward_char (&itstart))
       {
        u = gtk_text_iter_get_char (&itstart);
        if (u == '/')
           {
            found = FALSE;
            break;
           }
        
        if (u == '<')
           {
            found = TRUE;
            break;
           }
       }   

     if (! found)
        return; 

     if (! gtk_text_iter_forward_char (&itstart))
        return; 
 
     itend = itstart; 
     while (gtk_text_iter_forward_char (&itend))
           {  
            u = gtk_text_iter_get_char (&itend);
            if (! g_unichar_isalpha (u) && ! g_unichar_isdigit (u) && u != '_')
               break;
           }
  
  gchar *tag = gtk_text_buffer_get_text (doc->text_buffer, &itstart, &itend, FALSE);

  if (def_mm == MM_HTML)
  if (str_in (tag, "br", "hr", "img", NULL))
     {
      gtk_text_buffer_insert (doc->text_buffer, &ittemp, ">", -1);  
      g_free (tag);
      return;
     }

  gchar *complete; 
  if (mode == 0)
      complete = g_strconcat ("></", tag, ">", NULL);
  else    
      complete = g_strconcat ("</", tag, ">", NULL);

  gtk_text_buffer_insert (doc->text_buffer, &ittemp, complete, -1);
   

  g_free (tag);
  g_free (complete);
  doc_move_cursor_backw_middle_tags (doc);  

}


void doc_goto_local_label (const gchar *l)
{
  if (! l)
     return;

  gchar *t = g_utf8_find_next_char (l, NULL);
  gchar *s = g_strconcat ("<a name=\"", t, NULL);

  GtkTextIter iter;
  GtkTextIter match_start;

  GtkTextMark *m = gtk_text_buffer_get_insert (cur_text_doc->text_buffer); 
  gtk_text_buffer_get_iter_at_mark (cur_text_doc->text_buffer, &iter, m);

  if (gtk_text_iter_forward_search  (&iter, s, GTK_TEXT_SEARCH_TEXT_ONLY, &match_start, NULL, NULL))
     {
      gtk_text_buffer_place_cursor (cur_text_doc->text_buffer, &match_start );
      gtk_text_view_scroll_to_iter (cur_text_doc->text_view, &match_start, 0.0, TRUE, 0.0, 0.0 );
      gtk_text_view_place_cursor_onscreen (GTK_TEXT_VIEW (cur_text_doc->text_view));
     }

  g_free (s);
}

/*
gchar* doc_get_word_at_left (t_note_page *doc, GtkTextIter *itstart, GtkTextIter *itend) 
{
  GtkTextIter ittemp;
  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);
  
  *itstart = ittemp;
  *itend = ittemp;
  
  if (! gtk_text_iter_backward_find_char (itstart, find_space, NULL, NULL))
     gtk_text_buffer_get_start_iter (doc->text_buffer, itstart); 
  else     
      gtk_text_iter_forward_char (itstart);  
         
  return (gtk_text_buffer_get_text (doc->text_buffer, itstart, itend, FALSE));
}
*/

gchar* doc_get_word_at_left (t_note_page *doc, GtkTextIter *itstart, GtkTextIter *itend) 
{
  gunichar u;
  
  GtkTextIter ittemp;
  
  GtkTextMark *mark = gtk_text_buffer_get_insert (doc->text_buffer);
  gtk_text_buffer_get_iter_at_mark (doc->text_buffer, &ittemp, mark);
  
  *itstart = ittemp;
  *itend = ittemp;
  
  if (! gtk_text_iter_backward_find_char (itstart, find_space, NULL, NULL))
     gtk_text_buffer_get_start_iter (doc->text_buffer, itstart); 
  else     
      gtk_text_iter_forward_char (itstart);  

  *itend = *itstart; 
  while (gtk_text_iter_forward_char (itend))
           {  
            u = gtk_text_iter_get_char (itend);
            if (! g_unichar_isalpha (u) && ! g_unichar_isdigit (u) && u != '_')
               break;
           }
         
  return (gtk_text_buffer_get_text (doc->text_buffer, itstart, itend, FALSE));
}


void set_last_dir (t_note_page *doc)
{
  if (g_file_test (doc->file_name_local, G_FILE_TEST_EXISTS))
     last_dir = ch_str (last_dir, g_path_get_dirname (doc->file_name_local));
}