/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-channel.c:
 * GObject wrapper for D-Bus method calls to org.freedesktop.Telepathy.Channel
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-channel.h"
#include "ti-signals-marshal.h"

#include <dbus/dbus-glib.h>

struct _TIChannelClass {
    GObjectClass parent;
    
    guint closed_id; // "closed" signal id
    guint received_id; // "received" signal id
    guint sent_id; // "sent" signal id
    guint members_changed_id; // "members-changed" signal id
    guint group_flags_changed_id; // "group-flags-changed" signal id
};

G_DEFINE_TYPE (TIChannel, ti_channel, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIChannelPrivate {
    gboolean disposed;

    DBusGProxy* dbus_proxy; // Proxy for main channel interface;
    DBusGProxy* dbus_proxy_type_text; // Proxy for Type.Text interface;
    DBusGProxy* dbus_proxy_group; // Proxy for Interface.Group interface;
};
typedef struct _TIChannelPrivate TIChannelPrivate;

#define TI_CHANNEL_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_CHANNEL, TIChannelPrivate))

// Function prototypes.
static void _on_closed (gpointer proxy, TIChannel* channel);
static void _on_received (gpointer proxy,
                          guint id, guint timestamp, guint sender, guint type, guint flags, const gchar* text,
                          TIChannel* channel);
static void _on_sent (gpointer proxy,
                      guint timestamp, guint type, const gchar* text,
                      TIChannel* channel);
static void _on_members_changed (gpointer proxy,
                                 const gchar* message,
                                 GArray* added, GArray* removed,
                                 GArray* local_pending, GArray* remote_pending,
                                 guint actor, guint reason,
                                 TIChannel* channel);
static void _on_group_flags_changed (gpointer proxy, guint added, guint removed, TIChannel* channel);

/**
 * Drop all references to other objects.
 */
static void
ti_channel_dispose (GObject *object)
{
    TIChannel* channel = TI_CHANNEL (object);
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE (channel);

    if (priv->disposed) {
        return;
    } else {
        priv->disposed = TRUE;
    }

    if (priv->dbus_proxy != NULL) {
        g_object_unref (priv->dbus_proxy);
        priv->dbus_proxy = NULL;
    }

    if (priv->dbus_proxy_type_text != NULL) {
        g_object_unref (priv->dbus_proxy_type_text);
        priv->dbus_proxy_type_text = NULL;
    }

    if (priv->dbus_proxy_group != NULL) {
        g_object_unref (priv->dbus_proxy_group);
        priv->dbus_proxy_group = NULL;
    }

    G_OBJECT_CLASS (ti_channel_parent_class)->dispose (object);
}

/**
 * Finalizes the object, marking the memory as ready for reuse
 */
static void
ti_channel_finalize (GObject *object)
{
	//TIChannel* channel = TI_CHANNEL (object);
    //TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE (channel);

	G_OBJECT_CLASS (ti_channel_parent_class)->finalize (object);
}

/**
 * Class initialization.
 */
static void
ti_channel_class_init (TIChannelClass *ti_channel_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_channel_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_channel_dispose;
    gobject_class->finalize = ti_channel_finalize;

	/* Add private */
	g_type_class_add_private (ti_channel_class, sizeof (TIChannelPrivate));
	
	/* Signals */
	
    ti_channel_class->closed_id =
        g_signal_new ("closed",
                      G_OBJECT_CLASS_TYPE (ti_channel_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE,
                      0);

    ti_channel_class->received_id =
        g_signal_new ("received",
                      G_OBJECT_CLASS_TYPE (ti_channel_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      ti_marshal_VOID__UINT_UINT_UINT_UINT_UINT_STRING,
                      G_TYPE_NONE,
                      6, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING);

    ti_channel_class->sent_id =
        g_signal_new ("sent",
                      G_OBJECT_CLASS_TYPE (ti_channel_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      ti_marshal_VOID__UINT_UINT_STRING,
                      G_TYPE_NONE,
                      3, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING);

    ti_channel_class->members_changed_id =
        g_signal_new ("members-changed",
                      G_OBJECT_CLASS_TYPE (ti_channel_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      ti_marshal_VOID__STRING_BOXED_BOXED_BOXED_BOXED_UINT_UINT,
                      G_TYPE_NONE,
                      7, G_TYPE_STRING, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, G_TYPE_UINT, G_TYPE_UINT);

    ti_channel_class->group_flags_changed_id =
        g_signal_new ("group-flags-changed",
                      G_OBJECT_CLASS_TYPE (ti_channel_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      ti_marshal_VOID__UINT_UINT,
                      G_TYPE_NONE,
                      2, G_TYPE_UINT, G_TYPE_UINT);
}

/**
 * Instance initialization.
 */
static void
ti_channel_init (TIChannel *ti_channel)
{
    TIChannelPrivate *priv = TI_CHANNEL_GET_PRIVATE (ti_channel);

    priv->disposed = FALSE;

    priv->dbus_proxy = NULL;
    priv->dbus_proxy_type_text = NULL;
    priv->dbus_proxy_group = NULL;
}

/**
 * Creates a new channel
 */
TIChannel*
ti_channel_new (const gchar* service_name, const gchar* obj_path)
{
    TIChannel* channel;
    TIChannelPrivate* priv;
    DBusGConnection* dbus_conn;
    GError* error;

    error = NULL;
    dbus_conn = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (dbus_conn == NULL)
    {
        g_printerr ("Failed to open channel to bus: %s\n", error->message);
        g_error_free (error);
        return NULL;
    }

	channel = g_object_new (TI_TYPE_CHANNEL, NULL);
    priv = TI_CHANNEL_GET_PRIVATE(channel);

    priv->dbus_proxy = dbus_g_proxy_new_for_name (dbus_conn, service_name, obj_path,
                                                  "org.freedesktop.Telepathy.Channel");

    priv->dbus_proxy_type_text =
        dbus_g_proxy_new_for_name (dbus_conn, service_name, obj_path,
                                   "org.freedesktop.Telepathy.Channel.Type.Text");

    priv->dbus_proxy_group =
        dbus_g_proxy_new_for_name (dbus_conn, service_name, obj_path,
                                   "org.freedesktop.Telepathy.Channel.Interface.Group");
                                                  
    /* Tell DBus what the type signature of the signal callback is; this
     * allows us to sanity-check incoming messages before invoking the
     * callback.  You need to do this once for each proxy you create,
     * not every time you want to connect to the signal.
     */

    // Main interface signals
    dbus_g_proxy_add_signal (priv->dbus_proxy, "Closed",
                             G_TYPE_INVALID);

    dbus_g_proxy_add_signal (priv->dbus_proxy_type_text, "Received",
                             G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);

    // Type.Text interface signals 
    dbus_g_proxy_add_signal (priv->dbus_proxy_type_text, "Sent",
                             G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);

    // Interface.Group interface signals
    dbus_g_proxy_add_signal (priv->dbus_proxy_group, "MembersChanged",
                             G_TYPE_STRING, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID);

    dbus_g_proxy_add_signal (priv->dbus_proxy_group, "GroupFlagsChanged",
                             G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID);

    /* Actually connect to the signal.  Note you can call
     * dbus_g_proxy_connect_signal multiple times for one invocation of
     * dbus_g_proxy_add_signal.
     */

    // Main interface signals
    dbus_g_proxy_connect_signal (priv->dbus_proxy, "Closed", G_CALLBACK (_on_closed),
                                 channel, NULL);
     
    // Type.Text interface signals    
    dbus_g_proxy_connect_signal (priv->dbus_proxy_type_text, "Received", G_CALLBACK (_on_received),
                                 channel, NULL);
                                 
    dbus_g_proxy_connect_signal (priv->dbus_proxy_type_text, "Sent", G_CALLBACK (_on_sent),
                                 channel, NULL);

    // Interface.Group interface signals
    dbus_g_proxy_connect_signal (priv->dbus_proxy_group, "MembersChanged", G_CALLBACK (_on_members_changed),
                                 channel, NULL);

    dbus_g_proxy_connect_signal (priv->dbus_proxy_group, "GroupFlagsChanged", G_CALLBACK (_on_group_flags_changed),
                                 channel, NULL);

    return channel;
}

/**
 * Get Interfaces
 */
gchar**
ti_channel_get_interfaces (TIChannel* channel)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);
    GError* error;
    gchar** interfaces;

    error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "GetInterfaces", &error,
                            G_TYPE_INVALID,
                            G_TYPE_STRV, &interfaces, G_TYPE_INVALID)) {
        if (error != NULL) {
            g_printerr ("Error: %s\n", error->message);
            g_error_free (error);
        }
        return NULL;
    }

    return interfaces;
}

/**
 * Get Channel Type
 */
gchar*
ti_channel_get_channel_type (TIChannel* channel, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);
    gchar* type;

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "GetChannelType", error,
                            G_TYPE_INVALID,
                            G_TYPE_STRING, &type, G_TYPE_INVALID)) {
        if (*error != NULL) {
            g_printerr ("Error: %s\n", (*error)->message);
        }
        return NULL;
    }

    return type;
}

/**
 * Get Handle
 */
void
ti_channel_get_handle (TIChannel* channel, guint* handle_type, guint* handle_number, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "GetHandle", error,
                            G_TYPE_INVALID,
                            G_TYPE_UINT, handle_type, G_TYPE_UINT, handle_number, G_TYPE_INVALID)) {
        if (*error != NULL) {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * Close
 */
void ti_channel_close (TIChannel* channel, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "Close", error,
                            G_TYPE_INVALID,
                            G_TYPE_INVALID)) {
        if (*error != NULL) {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * AddMembers method from Group interface
 */
void
ti_channel_igroup_add_members (TIChannel* channel, GArray* members, const gchar* message, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);

    g_assert (members != NULL);
    g_assert (message != NULL);
    g_assert (error != NULL);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_group, "AddMembers", error,
                            DBUS_TYPE_G_UINT_ARRAY, members, G_TYPE_STRING, message, G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * GetGroupFlags method from Group interface
 */
guint
ti_channel_igroup_get_group_flags (TIChannel* channel, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);
    guint flags;

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_group, "GetGroupFlags", error,
                            G_TYPE_INVALID,
                            G_TYPE_UINT, &flags, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return flags;
}

/**
 * RemoveMembers method from Group interface
 */
void
ti_channel_igroup_remove_members (TIChannel* channel, GArray* members, const gchar* message, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_group, "RemoveMembers", error,
                            DBUS_TYPE_G_UINT_ARRAY, members, G_TYPE_STRING, message, G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * GetMembers method from Group interface
 */
GArray*
ti_channel_igroup_get_members (TIChannel* channel, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);
    GArray* handles = NULL;

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_group, "GetMembers", error,
                            G_TYPE_INVALID,
                            DBUS_TYPE_G_UINT_ARRAY, &handles, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return handles;
}

/**
 * GetLocalPendingMembers method from Group interface
 */
GArray*
ti_channel_igroup_get_local_pending_members (TIChannel* channel, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);
    GArray* handles = NULL;

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_group, "GetLocalPendingMembers", error,
                            G_TYPE_INVALID,
                            DBUS_TYPE_G_UINT_ARRAY, &handles, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return handles;
}

/**
 * GetRemotePendingMembers method from Group interface
 */
GArray*
ti_channel_igroup_get_remote_pending_members (TIChannel* channel, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);
    GArray* handles = NULL;

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_group, "GetRemotePendingMembers", error,
                            G_TYPE_INVALID,
                            DBUS_TYPE_G_UINT_ARRAY, &handles, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return handles;
}

/**
 * Send method from Type.Text interface
 */
void
ti_channel_typetext_send (TIChannel* channel, guint type, const gchar* text, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_type_text, "Send", error,
                            G_TYPE_UINT, type, G_TYPE_STRING, text, G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * On Closed
 */
static void
_on_closed (gpointer proxy, TIChannel* channel)
{
    TIChannelClass* klass = TI_CHANNEL_GET_CLASS (channel);
    
    g_debug ("ti-channel - _on_closed()");
    
    g_signal_emit (channel, klass->closed_id, 0);
}

/**
 * On Received
 */
static void
_on_received (gpointer proxy,
              guint id, guint timestamp, guint sender, guint type, guint flags, const gchar* text,
              TIChannel* channel)
{
    TIChannelClass* klass = TI_CHANNEL_GET_CLASS (channel);

    g_debug ("ti-channel - _on_received()");

    g_signal_emit (channel, klass->received_id, 0, id, timestamp, sender, type, flags, text);
}

/**
 * On Sent
 */
static void
_on_sent (gpointer proxy,
          guint timestamp, guint type, const gchar* text,
          TIChannel* channel)
{
    TIChannelClass* klass = TI_CHANNEL_GET_CLASS (channel);

    g_debug ("ti-channel - _on_sent()");

    g_signal_emit (channel, klass->sent_id, 0, timestamp, type, text);
}

/**
 * ListPendingMessages method from Type.Text interface
 */
void
ti_channel_typetext_list_pending_messages (TIChannel* channel, gboolean clear, GPtrArray** messages, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE (channel);
    static GType message_gtype = G_TYPE_INVALID;

    if (message_gtype == G_TYPE_INVALID)
    {
        message_gtype = dbus_g_type_get_struct ("GValueArray",
                                                G_TYPE_UINT,
                                                G_TYPE_UINT,
                                                G_TYPE_UINT,
                                                G_TYPE_UINT,
                                                G_TYPE_UINT,
                                                G_TYPE_STRING,
                                                G_TYPE_INVALID);
    }

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_type_text, "ListPendingMessages", error,
                            G_TYPE_BOOLEAN, clear, G_TYPE_INVALID,
                            dbus_g_type_get_collection ("GPtrArray", message_gtype), messages, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * AcknowledgePendingMessages method from Type.Text interface
 */
void
ti_channel_typetext_acknowledge_pending_messages (TIChannel* channel, GArray* ids, GError** error)
{
    TIChannelPrivate* priv = TI_CHANNEL_GET_PRIVATE(channel);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_type_text, "AcknowledgePendingMessages", error,
                            DBUS_TYPE_G_UINT_ARRAY, ids, G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * On Members Changed
 */
static void
_on_members_changed (gpointer proxy,
                     const gchar* message,
                     GArray* added, GArray* removed,
                     GArray* local_pending, GArray* remote_pending,
                     guint actor, guint reason,
                     TIChannel* channel)
{
    TIChannelClass* klass = TI_CHANNEL_GET_CLASS (channel);

    g_debug ("ti-channel - _on_members_changed()");

    g_signal_emit (channel, klass->members_changed_id, 0, message, added, removed, local_pending, remote_pending, actor, reason);
}

/**
 * On Group Flags Changed
 */
static void
_on_group_flags_changed (gpointer proxy, guint added, guint removed, TIChannel* channel)
{
    TIChannelClass* klass = TI_CHANNEL_GET_CLASS (channel);

    g_debug ("ti-channel - _on_group_flags_changed()");

    g_signal_emit (channel, klass->group_flags_changed_id, 0, added, removed);
}
