/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-connection.h:
 * GObject wrapper for D-Bus method calls to org.freedesktop.Telepathy.Connection
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-connection.h"
#include "ti-constants.h"
#include "ti-signals-marshal.h"

#include <dbus/dbus-glib.h>

struct _TIConnectionClass {
    GObjectClass parent;

    // "aliases-changed" signal id
    // org.freedesktop.Telepathy.Connection.Interface.Aliasing - AliasesChanged signal
    guint aliases_changed_id;

    guint status_changed_id;
    
    // "presence-update" signal id
    // org.freedesktop.Telepathy.Connection.Interface.Presence - PresenceUpdate signal
    guint presence_update_id;

    // GType for parameter "presence" on [...].Interface.Presence.PresenceUpdate(presence) signal
    GType g_type_presence_update;
};

G_DEFINE_TYPE (TIConnection, ti_connection, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIConnectionPrivate {
    DBusGProxy* dbus_proxy; // Proxy for main interface
    DBusGProxy* dbus_proxy_aliasing; // Proxy for Aliasing interface
    DBusGProxy* dbus_proxy_presence; // Proxy for Presence interface
    
    gchar* service_name;
};
typedef struct _TIConnectionPrivate TIConnectionPrivate;

#define TI_CONNECTION_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_CONNECTION, TIConnectionPrivate))

// Function prototypes.
static void _on_aliases_changed (gpointer proxy, GPtrArray* aliases, TIConnection* connection);
static void _on_status_changed (gpointer proxy, guint status, guint reason, TIConnection* connection);
static void _on_presence_update (gpointer proxy, GHashTable* presence, TIConnection* connection);

/**
 * Drop all references to other objects.
 */
static void
ti_connection_dispose (GObject *object)
{
    TIConnection* connection = TI_CONNECTION (object);
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE (connection);

    if (priv->dbus_proxy != NULL)
    {
        g_object_unref (priv->dbus_proxy);
        priv->dbus_proxy = NULL;
    }

    if (priv->dbus_proxy_aliasing != NULL)
    {
        g_object_unref (priv->dbus_proxy_aliasing);
        priv->dbus_proxy_aliasing = NULL;
    }

    if (priv->dbus_proxy_presence != NULL)
    {
        g_object_unref (priv->dbus_proxy_presence);
        priv->dbus_proxy_presence = NULL;
    }

    G_OBJECT_CLASS (ti_connection_parent_class)->dispose (object);
}

/**
 * Finalizes the object, marking the memory as ready for reuse
 */
static void
ti_connection_finalize (GObject *object)
{
	TIConnection* connection = TI_CONNECTION (object);
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE (connection);

    g_free (priv->service_name);

	G_OBJECT_CLASS (ti_connection_parent_class)->finalize (object);
}

/**
 * Class initialization.
 */
static void
ti_connection_class_init (TIConnectionClass *ti_connection_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_connection_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_connection_dispose;
    gobject_class->finalize = ti_connection_finalize;

    // A monster GType - a{u(ua{sa{sv}})}
    ti_connection_class->g_type_presence_update =
        (dbus_g_type_get_map ("GHashTable", G_TYPE_UINT, (dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, (dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, (dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE)))), G_TYPE_INVALID))));

	/* Add private */
	g_type_class_add_private (ti_connection_class, sizeof (TIConnectionPrivate));

    ti_connection_class->aliases_changed_id =
        g_signal_new ("aliases-changed",
                      G_OBJECT_CLASS_TYPE (ti_connection_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOXED,
                      G_TYPE_NONE,
                      1, dbus_g_type_get_collection ("GPtrArray", dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_STRING)) );

    ti_connection_class->status_changed_id =
        g_signal_new ("status-changed",
                      G_OBJECT_CLASS_TYPE (ti_connection_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      ti_marshal_VOID__UINT_UINT,
                      G_TYPE_NONE,
                      2, G_TYPE_UINT, G_TYPE_UINT);

    ti_connection_class->presence_update_id =
        g_signal_new ("presence-update",
                      G_OBJECT_CLASS_TYPE (ti_connection_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOXED,
                      G_TYPE_NONE,
                      1, ti_connection_class->g_type_presence_update);
}

/**
 * Instance initialization.
 */
static void
ti_connection_init (TIConnection *ti_connection)
{
    TIConnectionPrivate *priv = TI_CONNECTION_GET_PRIVATE (ti_connection);

    priv->dbus_proxy = NULL;
    priv->dbus_proxy_aliasing = NULL;
    priv->dbus_proxy_presence = NULL;
}

/**
 * Creates a new connection
 */
TIConnection*
ti_connection_new (const gchar* service_name, const gchar* obj_path)
{
    TIConnection* connection;
    TIConnectionPrivate* priv;
    TIConnectionClass* klass;
    DBusGConnection* dbus_conn;
    GError* error;

    error = NULL;
    dbus_conn = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (dbus_conn == NULL)
    {
        g_printerr ("Failed to open connection to bus: %s\n", error->message);
        g_error_free (error);
        return NULL;
    }

	connection = g_object_new (TI_TYPE_CONNECTION, NULL);
    priv = TI_CONNECTION_GET_PRIVATE (connection);
    klass = TI_CONNECTION_GET_CLASS (connection);

    priv->dbus_proxy = dbus_g_proxy_new_for_name (dbus_conn, service_name, obj_path,
                                                  "org.freedesktop.Telepathy.Connection");

    priv->dbus_proxy_aliasing = dbus_g_proxy_new_for_name (dbus_conn, service_name, obj_path,
                                                           "org.freedesktop.Telepathy.Connection.Interface.Aliasing");

    priv->dbus_proxy_presence = dbus_g_proxy_new_for_name (dbus_conn, service_name, obj_path,
                                                           "org.freedesktop.Telepathy.Connection.Interface.Presence");
    priv->service_name = g_strdup (service_name);                                                  

    /* Tell DBus what the type signature of the signal callback is; this
     * allows us to sanity-check incoming messages before invoking the
     * callback.  You need to do this once for each proxy you create,
     * not every time you want to connect to the signal.
     */

    dbus_g_proxy_add_signal (priv->dbus_proxy, "StatusChanged", G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID);

    dbus_g_proxy_add_signal (priv->dbus_proxy_presence, "PresenceUpdate", klass->g_type_presence_update, G_TYPE_INVALID);

    dbus_g_proxy_add_signal (priv->dbus_proxy_aliasing, "AliasesChanged",
                             dbus_g_type_get_collection ("GPtrArray", dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_STRING)), G_TYPE_INVALID);

    /* Actually connect to the signal.  Note you can call
     * dbus_g_proxy_connect_signal multiple times for one invocation of
     * dbus_g_proxy_add_signal.
     */

    dbus_g_proxy_connect_signal (priv->dbus_proxy, "StatusChanged", G_CALLBACK (_on_status_changed),
                                 connection, NULL);

    dbus_g_proxy_connect_signal (priv->dbus_proxy_presence, "PresenceUpdate", G_CALLBACK (_on_presence_update),
                                 connection, NULL);

    dbus_g_proxy_connect_signal (priv->dbus_proxy_aliasing, "AliasesChanged", G_CALLBACK (_on_aliases_changed),
                                 connection, NULL);

    return connection;
}

/**
 * Get Service Name
 */
const gchar*
ti_connection_get_service_name (TIConnection* connection)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);

    return priv->service_name;
}

/**
 * Get Interfaces
 */
gchar**
ti_connection_get_interfaces (TIConnection* connection)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    GError* error;
    gchar** interfaces;

    error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "GetInterfaces", &error,
                            G_TYPE_INVALID,
                            G_TYPE_STRV, &interfaces, G_TYPE_INVALID)) {
        if (error != NULL) {
            g_printerr ("Error: %s\n", error->message);
            g_error_free (error);
        }
        return NULL;
    }

    return interfaces;
}

/**
 * Get Status
 */
guint
ti_connection_get_status (TIConnection* connection)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    GError* error;
    guint status = 0;

    error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "GetStatus", &error,
                            G_TYPE_INVALID,
                            G_TYPE_UINT, &status, G_TYPE_INVALID)) {
        if (error != NULL) {
            g_printerr ("Error: %s\n", error->message);
            g_error_free (error);
        }
    }

    return status;
}

/**
 * Get Protocol
 */
gchar*
ti_connection_get_protocol (TIConnection* connection)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    GError* error;
    gchar* protocol;

    error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "GetProtocol", &error,
                            G_TYPE_INVALID,
                            G_TYPE_STRING, &protocol, G_TYPE_INVALID)) {
        if (error != NULL) {
            g_printerr ("Error: %s\n", error->message);
            g_error_free (error);
        }
        return NULL;
    }

    return protocol;
}

/**
 * Connect
 */
gboolean
ti_connection_connect (TIConnection* connection)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    GError* error;

    error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "Connect", &error,
                            G_TYPE_INVALID,
                            G_TYPE_INVALID)) {
        if (error != NULL) {
            g_printerr ("Error: %s\n", error->message);
            g_error_free (error);
        }
        return FALSE;
    }

    return TRUE;
}

/**
 * Disconnect
 */
gboolean
ti_connection_disconnect (TIConnection* connection)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    GError* error;

    error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "Disconnect", &error,
                            G_TYPE_INVALID,
                            G_TYPE_INVALID)) {
        if (error != NULL) {
            g_printerr ("Error: %s\n", error->message);
            g_error_free (error);
        }
        return FALSE;
    }

    return TRUE;
}

/**
 * On Aliases Changed
 */
static void
_on_aliases_changed (gpointer proxy, GPtrArray* aliases, TIConnection* connection)
{
    TIConnectionClass* klass = TI_CONNECTION_GET_CLASS (connection);

    g_signal_emit (connection, klass->aliases_changed_id, 0, aliases);
}

/**
 * On Status Changed
 */
static void
_on_status_changed (gpointer proxy, guint status, guint reason, TIConnection* connection)
{
    TIConnectionClass* klass = TI_CONNECTION_GET_CLASS (connection);

    g_signal_emit (connection, klass->status_changed_id, 0, status, reason);
}

/**
 * On Presence Update
 */
static void
_on_presence_update (gpointer proxy, GHashTable* presence, TIConnection* connection)
{
    TIConnectionClass* klass = TI_CONNECTION_GET_CLASS (connection);

    g_signal_emit (connection, klass->presence_update_id, 0, presence);
}

/**
 * Hold Single Handle
 */
void
ti_connection_hold_single_handle (TIConnection* connection, guint handle_type, guint handle_number, GError** error)
{
    GArray* handles = NULL;

    handles = g_array_new (FALSE, FALSE, sizeof (guint));
    g_array_append_val (handles, handle_number);

    ti_connection_hold_handles (connection, handle_type, handles, error);

    // Clean up
    g_array_free (handles, TRUE);
}

/**
 * Hold Handles
 */
void
ti_connection_hold_handles (TIConnection* connection, guint handle_type, GArray* handles, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "HoldHandles", error,
                            G_TYPE_UINT, handle_type, DBUS_TYPE_G_UINT_ARRAY, handles, G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * Inspect Single Handle
 */
gchar*
ti_connection_inspect_single_handle (TIConnection* connection, guint handle_type, guint handle_number, GError** error)
{
    GArray* handles = NULL;
    gchar** handle_names = NULL;
    gchar* name = NULL;

    handles = g_array_new (FALSE, FALSE, sizeof (guint));
    g_array_append_val (handles, handle_number);

    handle_names = ti_connection_inspect_handles (connection, handle_type, handles, error);

    name = g_strdup (handle_names[0]);

    // Clean up
    g_array_free (handles, TRUE);
    g_strfreev (handle_names);

    return name;
}

/**
 * Inspect Handles
 */
gchar**
ti_connection_inspect_handles (TIConnection* connection, guint handle_type, GArray* handles, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    gchar** handle_names = NULL;

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "InspectHandles", error,
                            G_TYPE_UINT, handle_type, DBUS_TYPE_G_UINT_ARRAY, handles, G_TYPE_INVALID,
                            G_TYPE_STRV, &handle_names, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return handle_names;
}

/**
 * Release Handles
 */
void
ti_connection_release_handles (TIConnection* connection, guint handle_type, GArray* handles, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "ReleaseHandles", error,
                            G_TYPE_UINT, handle_type, DBUS_TYPE_G_UINT_ARRAY, handles, G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * Request Handles
 */
GArray*
ti_connection_request_handles (TIConnection* connection, guint type, gchar** names, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    GArray* handles = NULL;

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "RequestHandles", error,
                            G_TYPE_UINT, type, G_TYPE_STRV, names, G_TYPE_INVALID,
                            DBUS_TYPE_G_UINT_ARRAY, &handles, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return handles;
}

/**
 * Request Channel
 */
TIChannel*
ti_connection_request_channel (TIConnection* connection,
                               gchar* type, guint handle_type, guint handle, gboolean suppress_handler,
                               GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    gchar* obj_path = NULL;
    TIChannel* channel = NULL;

    *error = NULL;

    if (!dbus_g_proxy_call (priv->dbus_proxy, "RequestChannel", error,
                            G_TYPE_STRING, type, G_TYPE_UINT, handle_type, G_TYPE_UINT, handle, G_TYPE_BOOLEAN, suppress_handler, G_TYPE_INVALID,
                            DBUS_TYPE_G_OBJECT_PATH, &obj_path, G_TYPE_INVALID))
    {
        if (error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
    else
    {
        channel = ti_channel_new (priv->service_name, obj_path);
    }
    
    g_free (obj_path);
    
    return channel;
}

/**
 * List Channels
 */
GPtrArray*
ti_connection_list_channels (TIConnection* connection, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    static GType channel_gtype = G_TYPE_INVALID;
    GPtrArray* channels = NULL;

    if (channel_gtype == G_TYPE_INVALID)
    {
        channel_gtype = dbus_g_type_get_struct ("GValueArray",
                                                DBUS_TYPE_G_OBJECT_PATH,
                                                G_TYPE_STRING,
                                                G_TYPE_UINT,
                                                G_TYPE_UINT,
                                                G_TYPE_INVALID);
    }

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "ListChannels", error,
                            G_TYPE_INVALID,
                            dbus_g_type_get_collection ("GPtrArray", channel_gtype), &channels, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return channels;
}

/**
 * AddStatus method from Presence interface
 */
void
ti_connection_ipresence_add_status (TIConnection* connection, const gchar* status, GHashTable* opt_params, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE (connection);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_presence, "AddStatus", error,
                            G_TYPE_STRING, status,
                            dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE), opt_params,
                            G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * ClearStatus method from Presence interface
 */
void
ti_connection_ipresence_clear_status (TIConnection* connection, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE (connection);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_presence, "ClearStatus", error,
                            G_TYPE_INVALID, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * GetStatuses method from Presence interface
 */
GHashTable* ti_connection_ipresence_get_statuses (TIConnection* connection, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    static GType statuses_gtype = G_TYPE_INVALID;
    static GType status_info_gtype = G_TYPE_INVALID;
    GHashTable* statuses = NULL;

    if (status_info_gtype == G_TYPE_INVALID)
    {
        status_info_gtype = dbus_g_type_get_struct ("GValueArray",
                                                    G_TYPE_UINT,
                                                    G_TYPE_BOOLEAN,
                                                    G_TYPE_BOOLEAN,
                                                    DBUS_TYPE_G_STRING_STRING_HASHTABLE,
                                                    G_TYPE_INVALID);
    }

    if (statuses_gtype == G_TYPE_INVALID)
    {
        statuses_gtype = dbus_g_type_get_map ("GHashTable",
                                              G_TYPE_STRING,
                                              status_info_gtype);
    }

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_presence, "GetStatuses", error,
                            G_TYPE_INVALID,
                            statuses_gtype, &statuses, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return statuses;
}

/**
 * RemoveStatus method from Presence interface
 */
void
ti_connection_ipresence_remove_status (TIConnection* connection, const gchar* status, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE (connection);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_presence, "RemoveStatus", error,
                            G_TYPE_STRING, status, G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * RequestPresence method from Presence interface
 */
void
ti_connection_ipresence_request_presence (TIConnection* connection, GArray* handles, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_presence, "RequestPresence", error,
                            DBUS_TYPE_G_UINT_ARRAY, handles, G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * RequestAliases method from Aliasing interface
 */
gchar**
ti_connection_ialiasing_request_aliases (TIConnection* connection, GArray* handles, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    gchar** aliases;

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_aliasing, "RequestAliases", error,
                            DBUS_TYPE_G_UINT_ARRAY, handles, G_TYPE_INVALID,
                            G_TYPE_STRV, &aliases, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return aliases;
}

/**
 * SetAliases method from Aliasing interface
 */
void
ti_connection_ialiasing_set_aliases (TIConnection* connection, GHashTable* aliases, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_aliasing, "SetAliases", error,
                            dbus_g_type_get_map ("GHashTable", G_TYPE_UINT, G_TYPE_STRING), aliases, G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}

/**
 * GetAliasFlags method from Aliasing interface
 */
guint
ti_connection_ialiasing_get_alias_flags (TIConnection* connection, GError** error)
{
    TIConnectionPrivate* priv = TI_CONNECTION_GET_PRIVATE(connection);
    guint flags = 0;

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy_aliasing, "GetAliasFlags", error,
                            G_TYPE_INVALID,
                            G_TYPE_UINT, &flags, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return flags;
}
