/***************************************************************************
    copyright            : (C) 2001-2005 by Robby Stephenson
    email                : robby@periapsis.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License as  *
 *   published by the Free Software Foundation;                            *
 *                                                                         *
 ***************************************************************************/

#include "collection.h"
#include "field.h"
#include "entry.h"
#include "tellico_debug.h"

#include <klocale.h>
#include <kglobal.h> // for KMAX

#include <qregexp.h>

using Tellico::Data::Collection;

const QString Collection::s_emptyGroupTitle = i18n("(Empty)");
const QString Collection::s_peopleGroupName = QString::fromLatin1("_people");

Collection::Collection(const QString& title_, const QString& entryName_, const QString& entryTitle_)
    : QObject(), KShared(), m_title(title_), m_entryName(entryName_), m_entryTitle(entryTitle_), m_entryIdDict(101) {
  m_entryGroupDicts.setAutoDelete(true);

  m_id = getID();
  m_iconName = entryName_ + 's';
}

Collection::~Collection() {
}

bool Collection::addFields(FieldVec list_) {
  bool success = true;
  for(FieldVec::Iterator it = list_.begin(); it != list_.end(); ++it) {
    success &= addField(it);
  }
  return success;
}

bool Collection::addField(Field* field_) {
  if(!field_) {
    return false;
  }

  // fieldByName() returns 0 if there's no field by that name
  // this essentially checks for duplicates
  if(fieldByName(field_->name())) {
    myDebug() << "Collection::addField() - replacing " << field_->name() << endl;
    removeField(fieldByName(field_->name()), true);
  }

//  kdDebug() << "Collection::addField() - adding " << field_->name() << endl;
  m_fields.append(field_);
  if(field_->formatFlag() == Field::FormatName) {
    m_peopleFields.append(field_); // list of people attributes
    if(m_peopleFields.count() > 1) {
      // the second time that a person field is added, add a "pseudo-group" for people
      if(m_entryGroupDicts.find(s_peopleGroupName) == 0) {
        m_entryGroupDicts.insert(s_peopleGroupName, new EntryGroupDict());
        m_entryGroups.prepend(s_peopleGroupName);
      }
    }
  }
  m_fieldNameDict.insert(field_->name(), field_);
  m_fieldTitleDict.insert(field_->title(), field_);
  m_fieldNames << field_->name();
  m_fieldTitles << field_->title();
  if(field_->type() == Field::Image) {
    m_imageFields.append(field_);
  }

  if(!field_->category().isEmpty() && m_fieldCategories.findIndex(field_->category()) == -1) {
    m_fieldCategories << field_->category();
  }

  if(field_->flags() & Field::AllowGrouped) {
    // m_entryGroups autoDeletes each QDict when the Collection d'tor is called
    EntryGroupDict* dict = new EntryGroupDict();
    // don't autoDelete, since the group is deleted when it becomes empty
    m_entryGroupDicts.insert(field_->name(), dict);
    // cache the possible groups of entries
    m_entryGroups << field_->name();
  }

  for(EntryVecIt it = m_entries.begin(); it != m_entries.end(); ++it) {
    populateDicts(it);
  }

  if(m_defaultGroupField.isEmpty() && field_->flags() & Field::AllowGrouped) {
    m_defaultGroupField = field_->name();
  }

  return true;
}

bool Collection::mergeField(Field* newField_) {
  if(!newField_) {
    return false;
  }

  Field* currField = fieldByName(newField_->name());
  if(!currField) {
    // does not exist in current collection, add it
    return addField(newField_->clone());
  }

  // the original field type is kept
  if(currField->type() != newField_->type()) {
    // make special exception for music collections and the track field, ok
    // to convert Table to Table2
    if(type() == Album && currField->type() == Field::Table && newField_->type() == Field::Table2) {
//      currField->setType(Field::Table2);
      newField_->setType(Data::Field::Table);
      currField->setProperty(QString::fromLatin1("columns"), QChar('2'));
    } else {
      kdDebug() << "Collection::mergeField() - skipping, field type mismatch for " << currField->title() << endl;
      return false;
    }
  }

  // if field is a Choice, then make sure all values are there
  if(currField->type() == Field::Choice && currField->allowed() != newField_->allowed()) {
    QStringList allowed = currField->allowed();
    const QStringList& newAllowed = newField_->allowed();
    for(QStringList::ConstIterator it = newAllowed.begin(); it != newAllowed.end(); ++it) {
      if(allowed.findIndex(*it) == -1) {
        allowed.append(*it);
      }
    }
    currField->setAllowed(allowed);
  }

  // don't change original format flags
  // don't change original category
  // add new description if current is empty
  if(currField->description().isEmpty()) {
    currField->setDescription(newField_->description());
  }

  // if new field has additional extended properties, add those
  for(StringMap::ConstIterator it = newField_->propertyList().begin(); it != newField_->propertyList().end(); ++it) {
    if(currField->property(it.key()).isEmpty()) {
      currField->setProperty(it.key(), it.data());
    }
  }

  // combine flags
  currField->setFlags(currField->flags() | newField_->flags());
  return true;
}

// be really careful with these field pointers, try not to call to many other functions
// which may depend on the field list
bool Collection::modifyField(Field* newField_) {
  if(!newField_) {
    return false;
  }
//  kdDebug() << "Collection::modifyField() - " << newField_->name() << endl;

// the field name never changes
  const QString& fieldName = newField_->name();
  Field* oldField = fieldByName(fieldName);
  if(!oldField) {
    kdDebug() << "Collection::modifyField() - no field named " << fieldName << endl;
    return false;
  }

  // update name dict
  m_fieldNameDict.replace(fieldName, newField_);

  // update titles
  const QString& oldTitle = oldField->title();
  const QString& newTitle = newField_->title();
  if(oldTitle == newTitle) {
    m_fieldTitleDict.replace(newTitle, newField_);
  } else {
    m_fieldTitleDict.remove(oldTitle);
    m_fieldTitles.remove(oldTitle);
    m_fieldTitleDict.insert(newTitle, newField_);
    m_fieldTitles.append(newTitle);
  }

  // now replace the field pointer in the list
  FieldVec::Iterator it = m_fields.find(oldField);
  if(it != m_fields.end()) {
    m_fields.insert(it, newField_);
    m_fields.remove(oldField);
  } else {
    kdDebug() << "Collection::modifyField() - no index found!" << endl;
    return false;
  }

  // update category list.
  if(oldField->category() != newField_->category()) {
    m_fieldCategories.clear();
    for(FieldVec::Iterator it = m_fields.begin(); it != m_fields.end(); ++it) {
      // add category if it's not in the list yet
      if(!it->category().isEmpty() && !m_fieldCategories.contains(it->category())) {
        m_fieldCategories += it->category();
      }
    }
  }

  // keep track of if the entry groups will need to be reset
  bool resetGroups = false;

  // if format is different, go ahead and invalidate all formatted entry values
  if(oldField->formatFlag() != newField_->formatFlag()) {
    // invalidate cached format strings of all entry attributes of this name
    for(EntryVecIt it = m_entries.begin(); it != m_entries.end(); ++it) {
      it->invalidateFormattedFieldValue(fieldName);
    }
    resetGroups = true;
  }

  // check to see if the people "pseudo-group" needs to be updated
  // only if only one of the two is a name
  bool wasPeople = oldField->formatFlag() == Field::FormatName;
  bool isPeople = newField_->formatFlag() == Field::FormatName;
  if(wasPeople) {
    m_peopleFields.remove(oldField);
    if(!isPeople) {
      resetGroups = true;
    }
  }
  if(isPeople) {
    // if there's more than one people field and no people dict exists yet, add it
    if(m_peopleFields.count() > 1 && !m_entryGroupDicts.find(s_peopleGroupName)) {
      m_entryGroupDicts.insert(s_peopleGroupName, new EntryGroupDict());
      // put it at the top of the list
      m_entryGroups.prepend(s_peopleGroupName);
    }
    m_peopleFields.append(newField_);
    if(!wasPeople) {
      resetGroups = true;
    }
  }

  bool wasGrouped = oldField->flags() & Field::AllowGrouped;
  bool isGrouped = newField_->flags() & Field::AllowGrouped;
  if(wasGrouped) {
    if(!isGrouped) {
      // in order to keep list in the same order, don't remove unless new field is not groupable
      m_entryGroups.remove(fieldName);
      m_entryGroupDicts.remove(fieldName);
      resetGroups = true;
    } else {
      // don't do this, it wipes out the old groups!
//      m_entryGroupDicts.replace(fieldName, new EntryGroupDict());
    }
  } else if(isGrouped) {
    m_entryGroupDicts.insert(fieldName, new EntryGroupDict());
    // cache the possible groups of entries
    m_entryGroups << fieldName;
    resetGroups = true;
  }

  if(oldField->type() == Field::Image) {
    m_imageFields.remove(oldField);
  }
  if(newField_->type() == Field::Image) {
    m_imageFields.append(newField_);
  }

  if(resetGroups) {
    invalidateGroups();
  }

  // now to update all entries if the field is a dependent and the description changed
  if(newField_->type() == Field::Dependent && oldField->description() != newField_->description()) {
    emit signalRefreshField(newField_);
  }

  return true;
}

bool Collection::removeField(const QString& name_, bool force_) {
  return removeField(fieldByName(name_), force_);
}

// force allows me to force the deleting of the title field if I need to
bool Collection::removeField(Field* field_, bool force_/*=false*/) {
  if(!field_ || !m_fields.contains(field_)) {
    return false;
  }
//  kdDebug() << "Collection::deleteField() - name = " << field_->name() << endl;

  // can't delete the title field
  if((field_->flags() & Field::NoDelete) && !force_) {
    return false;
  }

  bool success = true;
  if(field_->formatFlag() == Field::FormatName) {
    success &= m_peopleFields.remove(field_);
  }
  if(field_->type() == Field::Image) {
    success &= m_imageFields.remove(field_);
  }
  success &= m_fieldNameDict.remove(field_->name());
  success &= m_fieldTitleDict.remove(field_->title());
  success &= m_fieldNames.remove(field_->name());
  success &= m_fieldTitles.remove(field_->title());

  if(fieldsByCategory(field_->category()).count() == 1) {
    success &= m_fieldCategories.remove(field_->category());
  }

  for(EntryVecIt it = m_entries.begin(); it != m_entries.end(); ++it) {
    // setting the fields to an empty string removes the value from the entry's list
    it->setField(field_->name(), QString::null);
  }

  if(field_->flags() & Field::AllowGrouped) {
    success &= m_entryGroupDicts.remove(field_->name());
    success &= m_entryGroups.remove(field_->name());
    if(field_->name() == m_defaultGroupField) {
      setDefaultGroupField(m_entryGroups[0]);
    }
  }

  success &= m_fields.remove(field_);

//  delete field_; // removeFieldCommand will delete the field
  return success;
}

void Collection::reorderFields(const FieldVec& list_) {
// assume the lists have the same pointers!
  m_fields = list_;

  // also reset category list, since the order may have changed
  m_fieldCategories.clear();
  for(FieldVec::Iterator it = m_fields.begin(); it != m_fields.end(); ++it) {
    if(!it->category().isEmpty() && !m_fieldCategories.contains(it->category())) {
      m_fieldCategories << it->category();
    }
  }
}

void Collection::addEntry(Entry* entry_) {
  if(!entry_) {
    return;
  }

#ifndef NDEBUG
  if(entry_->collection() != this) {
    // should the addEntry() call be done in the Entry constructor?
    kdWarning() << "Collection::addEntry() - entry is not being added to its parent collection" << endl;
  }
#endif

  m_entries.append(entry_);
//  kdDebug() << "Collection::addEntry() - added entry (" << entry_->title() << ")" << endl;

  populateDicts(entry_);
  int id = KMAX(1, entry_->id()); // id is -1 for entries as created, but not added
  while(m_entryIdDict[id]) {
    ++id;
  }
  m_entryIdDict.insert(id, entry_);
  entry_->setId(id);
}

void Collection::removeEntryFromDicts(Entry* entry_) {
  // need a copy of the vector since it gets changed
  PtrVector<EntryGroup> groups = entry_->groups();
  for(PtrVector<EntryGroup>::Iterator group = groups.begin(); group != groups.end(); ++group) {
    if(entry_->removeFromGroup(group.ptr())) {
      emit signalGroupModified(this, group.ptr());
    }
    if(group->isEmpty()) {
      EntryGroupDict* dict = m_entryGroupDicts.find(group->fieldName());
      if(!dict) {
        continue;
      }
      dict->remove(group->groupName());
      delete group.ptr();
    }
  }
}

// this function gets called whenever a entry is modified. Its purpose is to keep the
// groupDicts current. It first removes the entry from every group to which it belongs,
// then it repopulates the dicts with the entry's fields
void Collection::updateDicts(Entry* entry_) {
//  myDebug() << "Collection::updateDicts()" << endl;
  if(!entry_) {
    return;
  }

  removeEntryFromDicts(entry_);
  populateDicts(entry_);
}

bool Collection::removeEntry(Entry* entry_) {
  if(!entry_) {
    return false;
  }

//  kdDebug() << "Collection::deleteEntry() - deleted entry - " << entry_->title() << endl;
  removeEntryFromDicts(entry_);
  bool success = m_entryIdDict.remove(entry_->id());

  success &= m_entries.remove(entry_);;

  return success;
}

Tellico::Data::FieldVec Collection::fieldsByCategory(const QString& cat_) {
#ifndef NDEBUG
  if(m_fieldCategories.findIndex(cat_) == -1) {
    kdDebug() << "Collection::fieldsByCategory() - '" << cat_ << "' is not in category list" << endl;
  }
#endif
  if(cat_.isEmpty()) {
    kdDebug() << "Collection::fieldsByCategory() - empty category!" << endl;
    return FieldVec();
  }

  FieldVec list;
  for(FieldVec::Iterator it = m_fields.begin(); it != m_fields.end(); ++it) {
    if(it->category() == cat_) {
      list.append(it);
    }
  }
  return list;
}

const QString& Collection::fieldNameByTitle(const QString& title_) const {
  if(title_.isEmpty()) {
    return QString::null;
  }
  Field* f = fieldByTitle(title_);
  if(!f) {
    kdWarning() << "Collection::fieldNameByTitle() - no field titled " << title_ << endl;
    return QString::null;
  }
  return f->name();
}

const QString& Collection::fieldTitleByName(const QString& name_) const {
  if(name_.isEmpty()) {
    return QString::null;
  }
  Field* f = fieldByName(name_);
  if(!f) {
    kdWarning() << "Collection::fieldTitleByName() - no field named " << name_ << endl;
    return QString::null;
  }
  return f->title();
}

QStringList Collection::valuesByFieldName(const QString& name_) const {
  if(name_.isEmpty()) {
    return QStringList();
  }
  bool multiple = (fieldByName(name_)->flags() & Field::AllowMultiple);

  QStringList strlist;
  for(EntryVec::ConstIterator it = m_entries.begin(); it != m_entries.end(); ++it) {
    if(multiple) {
      strlist += it->fields(name_, false);
    } else {
      strlist += it->field(name_);
    }
  } // end entry loop
  strlist.sort();

  QStringList::Iterator it = strlist.begin();
  while(it != strlist.end()) {
    const QString& s = *it;
    ++it;
    while(it != strlist.end() && s == *it) {
      it = strlist.remove(it);
    }
  }
  return strlist;
}

Tellico::Data::Field* const Collection::fieldByName(const QString& name_) const {
  return m_fieldNameDict.isEmpty() ? 0 : m_fieldNameDict.find(name_);
}

Tellico::Data::Field* const Collection::fieldByTitle(const QString& title_) const {
  return m_fieldTitleDict.isEmpty() ? 0 : m_fieldTitleDict.find(title_);
}

bool Collection::isAllowed(const QString& key_, const QString& value_) const {
  // empty string is always allowed
  if(value_.isEmpty()) {
    return true;
  }

  // find the field with a name of 'key_'
  Field* field = fieldByName(key_);

  // if the type is not multiple choice or if value_ is allowed, return true
  if(field && (field->type() != Field::Choice || field->allowed().findIndex(value_) > -1)) {
    return true;
  }

  return false;
}

Tellico::Data::EntryGroupDict* const Collection::entryGroupDictByName(const QString& name_) const {
  return m_entryGroupDicts.isEmpty() ? 0 : m_entryGroupDicts.find(name_);
}

void Collection::populateDicts(Entry* entry_) {
  if(m_entryGroupDicts.isEmpty()) {
    return;
  }

  // iterate over all the possible groupDicts
  // for each dict, get the value of that field for the entry
  // if multiple values are allowed, split the value and then insert the
  // entry pointer into the dict for each value
  QDictIterator<EntryGroupDict> dictIt(m_entryGroupDicts);
  for( ; dictIt.current(); ++dictIt) {
    EntryGroupDict* dict = dictIt.current();
    // the field name might be the people group name
    QString fieldName = dictIt.currentKey();
    bool isBool = fieldByName(fieldName) && fieldByName(fieldName)->type() == Field::Bool;

    QStringList groups = entryGroupNamesByField(entry_, fieldName);
    for(QStringList::ConstIterator groupIt = groups.begin(); groupIt != groups.end(); ++groupIt) {
      // find the group for this group name
      // bool fields used the field title
      EntryGroup* group;
      if(isBool && *groupIt != s_emptyGroupTitle) {
        group = dict->find(fieldTitleByName(fieldName));
      } else {
        group = dict->find(*groupIt);
      }
      // if the group doesn't exist, create it
      if(!group) {
        // if it's a bool, rather than showing "true", use field title instead of "true"
        // as long as it's not the empty group name
        if(isBool && *groupIt != s_emptyGroupTitle) {
          QString t = fieldTitleByName(fieldName);
          group = new EntryGroup(t, fieldName);
          dict->insert(t, group);
        } else {
          group = new EntryGroup(*groupIt, fieldName);
          dict->insert(*groupIt, group);
        }
      }
      if(entry_->addToGroup(group)) {
        emit signalGroupModified(this, group);
      }
    } // end group loop
//    kdDebug() << "Collection::populateDicts - end of group loop" << endl;
  } // end dict loop
//  kdDebug() << "Collection::populateDicts - end of full loop" << endl;
}

// return a string list for all the groups that the entry belongs to
// for a given field. Normally, this would just be splitting the entry's value
// for the field, but if the field name is the people pseudo-group, then it gets
// a bit more complicated
QStringList Collection::entryGroupNamesByField(Entry* entry_, const QString& fieldName_) {
  if(fieldName_ != s_peopleGroupName) {
    return entry_->groupNamesByFieldName(fieldName_);
  }

  QStringList groups;
  for(FieldVec::Iterator it = m_peopleFields.begin(); it != m_peopleFields.end(); ++it) {
    groups += entry_->groupNamesByFieldName(it->name());
  }

  // just want unique values
  // so if there's more than one people field, need to remove any duplicates
  if(m_peopleFields.count() > 1) {
    groups.sort();
    unsigned i = 1;
    while(i < groups.count()) {
      if(groups[i] == groups[i-1]) {
        groups.remove(groups.at(i));
      } else {
        ++i;
      }
    }
  }

  // don't want the empty group
  // effectively, this means that if the people group is used, entries with no
  // people are not shown in the group view
  groups.remove(s_emptyGroupTitle);
  return groups;
}

void Collection::invalidateGroups() {
  QDictIterator<EntryGroupDict> dictIt(m_entryGroupDicts);
  for( ; dictIt.current(); ++dictIt) {
    dictIt.current()->clear();
  }

  for(EntryVecIt it = m_entries.begin(); it != m_entries.end(); ++it) {
    it->invalidateFormattedFieldValue();
    // populateDicts() will make signals that the group view is connected to, block those
    blockSignals(true);
    populateDicts(it);
    blockSignals(false);
  }
}

Tellico::Data::Entry* Collection::entryById(int id_) {
  return m_entryIdDict[id_];
}

void Collection::addBorrower(Data::Borrower* borrower_) {
  if(!borrower_) {
    return;
  }
  m_borrowers.append(borrower_);
}

void Collection::addFilter(Filter* filter_) {
  if(!filter_) {
    return;
  }

  m_filters.append(filter_);
}

bool Collection::removeFilter(Filter* filter_) {
  if(!filter_) {
    return false;
  }

  // TODO: test for success
  m_filters.remove(filter_);
  return true;
}

// static
int Collection::getID() {
  static int id = 0;
  return ++id;
}

#include "collection.moc"
