/*
    terminatorX - realtime audio scratching software
    Copyright (C) 1999-2003  Alexander Knig
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
 
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
 
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
    File: tX_engine.c
 
    Description: Contains the code that does the real "Scratching
    		 business": XInput, DGA, Mouse and Keyboardgrabbing
		 etc.

    02 Jun 1999: Implemented high-priority/rt-FIFO-Scheduling use for
                 engine-thread.
		 
    04 Jun 1999: Changed warp-feature behaviour: still connected to
                 mouse-speed (should be changed to maybe) but now
		 depends on sample size -> you can warp through all
		 samples with the same mouse-distance.
		 
    12 Aug 2002: Complete rewrite - tX_engine is now a class and the thread
	is created on startup and kept alive until termination
*/    

#include <config.h>
#include "tX_types.h"
#include "tX_engine.h"
#include "tX_audiodevice.h"
#include "tX_mouse.h"
#include "tX_vtt.h"
#include <pthread.h>
#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>
#include "tX_mastergui.h"
#include "tX_global.h"
#include "tX_tape.h"
#include "tX_widget.h"
#include <config.h>
#include "tX_sequencer.h"
#include <errno.h>
#include <sched.h>
#include "tX_capabilities.h"

#include <sys/time.h>
#include <sys/resource.h>

tX_engine *tX_engine::engine=NULL;

tX_engine *tX_engine::get_instance() {
	if (!engine) {
		engine=new tX_engine();
	}
	
	return engine;
}

void tX_engine::set_grab_request() {
	grab_request=true;
}

int16_t* tX_engine::render_cycle() {
	/* Checking whether to grab or not  */
	if (grab_request!=grab_active) {
		if (grab_request) {
			/* Activating grab... */
			int result=mouse->grab(); 
			if (result!=0) {
				tX_error("tX_engine::render_cycle(): failed to grab mouse - error %i", result);
				grab_active=false;
				/* Reseting grab_request, too - doesn't help keeping it, does it ? ;) */
				grab_request=false;
				mouse->ungrab();
				grab_off();
			} else {
				grab_active=true;
			}
		} else {
			/* Deactivating grab... */
			mouse->ungrab();
			grab_active=false;
			grab_off(); // for the mastergui this is...
		}
	}

	/* Handling input events... */
	if (grab_active) {
		if (mouse->check_event()) {
			/* If we're here the user pressed ESC */
			grab_request=false;
		}
	}

	/* Forward the sequencer... */
	sequencer.step();

	/* Render the next block... */
	int16_t *data=vtt_class::render_all_turntables();
	
	/* Record the audio if necessary... */
	if (is_recording()) tape->eat(data);
	
	return  data;
}

void tX_engine::loop() {
	while (!thread_terminate) {
		/* Waiting for the trigger */
		pthread_mutex_lock(&start);
#ifdef USE_SCHEDULER
		pid_t pid=getpid();
		struct sched_param parm;
			
		if (globals.use_realtime && (globals.audiodevice_type!=JACK)) {
			sched_getparam(pid, &parm);
			parm.sched_priority=sched_get_priority_max(SCHED_FIFO);
						
			if (pthread_setschedparam(pthread_self(), SCHED_FIFO, &parm)) {
				tX_error("loop(): failed to set realtime priority.");
			} else {
				tX_debug("loop(): set SCHED_FIFO.");
			}
		} else {
			sched_getparam(pid, &parm);
			parm.sched_priority=sched_get_priority_max(SCHED_OTHER);
						
			if (pthread_setschedparam(pthread_self(), SCHED_OTHER, &parm)) {
				tX_error("loop(): failed to set non-realtime priority.");
			} else {
				tX_debug("loop(): set SCHED_OTHER.");
			}			
		}
#endif		
		loop_is_active=true;
		pthread_mutex_unlock(&start);

		if (!stop_flag) device->start(); // Hand flow control over to the device
		
		/* Stopping engine... */
		loop_is_active=false;
	}
}

void *engine_thread_entry(void *engine_void) {
	tX_engine *engine=(tX_engine*) engine_void;
	int result;
	
	/* Dropping root privileges for the engine thread - if running suid. */
	
	if ((!geteuid()) && (getuid() != geteuid())) {
#ifdef USE_CAPABILITIES
		tX_error("engine_thread_entry(): using capabilities but still suid root?");
		tX_error("engine_thread_entry(): Please report this.");
		exit(-1);
#endif
		
#ifndef ALLOW_SUID_ROOT
		tX_error("This binary doesn't support running suid-root.");
		tX_error("Reconfigure with --enable-capabilities or --enable-suidroot if you really want that.");
		exit(-1);
#endif		
		tX_debug("engine_thread_entry() - Running suid root - dropping privileges.");
		
		result=setuid(getuid());
		
		if (result!=0) {
			tX_error("engine_thread_entry() - Failed to drop root privileges.");
			exit(2);
		}
	}

#ifdef USE_SCHEDULER
	pid_t pid=getpid();
	struct sched_param parm;

#ifdef USE_CAPABILITIES
	if (have_nice_capability()) {
		if (globals.use_realtime) {
			sched_getparam(pid, &parm);
			parm.sched_priority=sched_get_priority_max(SCHED_FIFO);
						
			if (pthread_setschedparam(pthread_self(), SCHED_FIFO, &parm)) {
				tX_error("engine_thread_entry(): failed to set realtime priority.");
			} else {
				tX_debug("engine_thread_entry(): set SCHED_FIFO via capabilities.");
			}
		}
	} else {
		tX_warning("engine_thread_entry(): can't set SCHED_FIFO -> lacking capabilities.");
	}
#endif //USE_CAPABILITIES
	int policy=0;
	
	pthread_getschedparam(pthread_self(), &policy, &parm);
	if (policy!=SCHED_FIFO) {
		tX_warning("engine_thread_entry() - engine has no realtime priority scheduling.");
	}
#endif //USE_SCHEDULER
		
#ifdef USE_JACK
	/* Create the client now, so the user has something to connect to. */
	tX_jack_client::get_instance();
#endif	

#ifdef USE_SCHEDULER
	tX_debug("engine_thread_entry() engine thread is p: %i, t: %i and has policy %i.", getpid(), (int) pthread_self(), sched_getscheduler(getpid()));
#endif
	
	engine->loop();
	
	tX_debug("engine_thread_entry() engine thread terminating.");
	
	pthread_exit(NULL);
}

tX_engine :: tX_engine() {
	int result;
	
	pthread_mutex_init(&start, NULL);
	pthread_mutex_lock(&start);
	thread_terminate=false;
	
	/* Creating the actual engine thread.. */
#ifdef USE_SCHEDULER	
	if (!geteuid() /* && globals.use_realtime */) {

#ifndef ALLOW_SUID_ROOT
		tX_error("This binary doesn't support running suid-root.");
		tX_error("Reconfigure with --enable-capabilities or --enable-suidroot if you really want that.");
		exit(-1);
#endif		
		
		pthread_attr_t pattr;
		struct sched_param sparm;
		
		tX_debug("tX_engine() - setting SCHED_FIFO.");
		
		pthread_attr_init(&pattr);
		pthread_attr_setdetachstate(&pattr, PTHREAD_CREATE_JOINABLE);
		pthread_attr_setschedpolicy(&pattr, SCHED_FIFO);
	
		sched_getparam(getpid(), &sparm);
		sparm.sched_priority=sched_get_priority_max(SCHED_FIFO);
	
		pthread_attr_setschedparam(&pattr, &sparm);
		pthread_attr_setinheritsched(&pattr, PTHREAD_EXPLICIT_SCHED);
		pthread_attr_setscope(&pattr, PTHREAD_SCOPE_SYSTEM);
		
		result=pthread_create(&thread, &pattr, engine_thread_entry, (void *) this);
	} else {
#endif // USE_SCHEDULER
		//tX_debug("tX_engine() - can't set SCHED_FIFO euid: %i.", geteuid());
		
		result=pthread_create(&thread, NULL, engine_thread_entry, (void *) this);
#ifdef USE_SCHEDULER		
	}
#endif
	
	if (result!=0) {
		tX_error("tX_engine() - Failed to create engine thread. Errno is %i.", errno);
		exit(1);
	}
	
	/* Dropping root privileges for the main thread - if running suid. */
	
	if ((!geteuid()) && (getuid() != geteuid())) {
		tX_debug("tX_engine() - Running suid root - dropping privileges.");
		
		result=setuid(getuid());
		
		if (result!=0) {
			tX_error("tX_engine() - Failed to drop root privileges.");
			exit(2);
		}
	}
	
	mouse=new tx_mouse();
#ifdef USE_ALSA_MIDI_IN	
	midi=new tX_midiin();
#endif	
	tape=new tx_tapedeck();

	device=NULL;
	recording=false;
	recording_request=false;
	loop_is_active=false;
	grab_request=false;
	grab_active=false;
}

void tX_engine :: set_recording_request (bool recording) {
	this->recording_request=recording;
}

tX_engine_error tX_engine :: run() {
	list <vtt_class *> :: iterator vtt;
	
	if (loop_is_active) return ERROR_BUSY;
	
	switch (globals.audiodevice_type) {
#ifdef USE_OSS	
		case OSS:
			device=new tX_audiodevice_oss(); 
		break;
#endif			

#ifdef USE_ALSA			
		case ALSA:
			device=new tX_audiodevice_alsa(); 
		break;
#endif

#ifdef USE_JACK
		case JACK:
			device=new tX_audiodevice_jack();
		break;
#endif
		
		default:
			device=NULL; return ERROR_AUDIO;
	}
	
	if (device->open()) {
		if (device->get_is_open()) device->close();
		delete device;
		device=NULL;		
		return ERROR_AUDIO;
	}	

	vtt_class::set_sample_rate(device->get_sample_rate());
	
	if (recording_request) {
		if (tape->start_record(globals.record_filename, vtt_class::get_mix_buffer_size(), device->get_sample_rate())) {
			device->close();
			delete device;
			device=NULL;			
			return ERROR_TAPE;			
		} else {
			recording=true;
		}
	}
	
	for (vtt=vtt_class::main_list.begin(); vtt!=vtt_class::main_list.end(); vtt++) {
		(*vtt)->sync_countdown=0;
		if ((*vtt)->autotrigger) (*vtt)->trigger();
	}

	sequencer.forward_to_start_timestamp(1);	
	stop_flag=false;
	/* Trigger the engine thread... */
	pthread_mutex_unlock(&start);
	
	return NO_ERROR;
}

void tX_engine :: stop() {
	list <vtt_class *> :: iterator vtt;
	
	if (!loop_is_active) {
		tX_error("tX_engine::stop() - but loop's not running?");
	}
	
	pthread_mutex_lock(&start);
	stop_flag=true;
	
	tX_debug("tX_engine::stop() - waiting for loop to stop.");
	
	while (loop_is_active) {
		/* Due to gtk+ signal handling this can cause a deadlock
		   on the seqpars' update list. So we need to handle events
		   while waiting...			
		*/
		while (gtk_events_pending()) gtk_main_iteration();
		usleep(50);
	}
	
	tX_debug("tX_engine::stop() - loop has stopped.");

	if (device->get_is_open()) device->close();
	delete device;
	device=NULL;
	
	for (vtt=vtt_class::main_list.begin(); vtt!=vtt_class::main_list.end(); vtt++) {
		(*vtt)->stop();
		(*vtt)->ec_clear_buffer();
	}
	
	if (is_recording()) tape->stop_record();
	recording=false;
}

tX_engine :: ~tX_engine() {
	void *dummy;
		
	thread_terminate=true;
	stop_flag=true;
	pthread_mutex_unlock(&start);
	tX_debug("~tX_engine() - Waiting for engine thread to terminate.");
	pthread_join(thread, &dummy);	
	
	delete mouse;
#ifdef USE_ALSA_MIDI_IN		
	delete midi;
#endif	
	delete tape;	
}
