/*

    File: analyse.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <stdio.h>
#include <string.h>
#include "types.h"
#include "common.h"
#include "fnctdsk.h" /* get_LBA_part */
#include "analyse.h"
#include "bfs.h"
#include "swap.h"
#include "bsd.h"
#include "fat.h"
#include "ntfs.h"
#include "ext2.h"
#include "netware.h"
#include "intrface.h"
#include "rfs.h"
#include "lvm.h"
#include "md.h"
#include "jfs_superblock.h"
#include "jfs.h"
int search_NTFS_backup(t_param_disk *disk_car,t_diskext *partition, const int debug, const int dump_ind)
{
  unsigned char buffer[SECTOR_SIZE];
  if(disk_car->read(disk_car,1, &buffer, partition->lba)!=0)
    return -1;
  /* NTFS recovery using backup sector */
  if(recover_NTFS(disk_car,(struct ntfs_boot_sector*)&buffer,partition,debug,dump_ind,1)==0)
  {
    strncpy(partition->info,"NTFS found using backup sector!",sizeof(partition->info));
    return 1;
  }
  return 0;
}

int search_FAT_backup(t_param_disk *disk_car,t_diskext *partition, const int debug, const int dump_ind)
{
  unsigned char buffer[3*SECTOR_SIZE];
  if(disk_car->read(disk_car,3, &buffer, partition->lba)!=0)
    return -1;
  /* FAT32 recovery using backup sector */
  if(recover_FAT32(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind,1)==0)
  {
    strncpy(partition->info,"FAT found using backup sector!",sizeof(partition->info));
    return 1;
  }
  return 0;
}

int search_type_0(t_param_disk *disk_car,t_diskext *partition, const int debug, const int dump_ind)
{
  unsigned char buffer[8*SECTOR_SIZE];
  if(disk_car->read(disk_car,8, &buffer, partition->lba)!=0)
    return -1;
  if(recover_Linux_SWAP(disk_car,(const union swap_header *)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_LVM(disk_car,(const pv_disk_t*)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_FAT12(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_FAT16(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_FAT32(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind,0)==0) return 1;
  if(recover_HPFS(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_OS2MB(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_NTFS(disk_car,(const struct ntfs_boot_sector*)&buffer,partition,debug,dump_ind,0)==0) return 1;
  if(recover_netware((const struct disk_netware *)&buffer,partition)==0) return 1;
  return 0;
}

int search_type_1(t_param_disk *disk_car,t_diskext *partition,const int debug, const int dump_ind)
{
  unsigned char buffer[8*SECTOR_SIZE];
  if(debug>1)
  {
    ecrit_rapport("search_type_1 lba=%lu\n",partition->lba);
  }
  if(disk_car->read(disk_car,8, &buffer, partition->lba)!=0)
    return -1;
  if(recover_BSD(disk_car,(const struct disklabel *)&buffer[SECTOR_SIZE],partition,debug,dump_ind)==0) return 1;
  if(recover_BeFS(disk_car,(const struct disk_super_block *)&buffer[SECTOR_SIZE],partition,debug,dump_ind)==0) return 1;
  return 0;
}

int search_type_2(t_param_disk *disk_car,t_diskext *partition,const int debug, const int dump_ind)
{
  unsigned char buffer[8*SECTOR_SIZE];
  if(debug>1)
  {
    ecrit_rapport("search_type_2 lba=%ld\n",partition->lba);
  }
  if(partition->lba+8<disk_car->size)
  {
    if(disk_car->read(disk_car,2, &buffer[0x400], partition->lba+2)==0)
    {
      if(recover_EXT2(disk_car,(const struct ext2_super_block*)&buffer[0x400],partition,debug,dump_ind)==0)
      {
	return 1;
      }
    }
  }
  return 0;
}

int search_type_64(t_param_disk *disk_car,t_diskext *partition,const int debug, const int dump_ind)
{
  if(partition->lba+64<disk_car->size)
  {
	unsigned char buffer[8*SECTOR_SIZE];
	if(debug>1)
	{
	  ecrit_rapport("search_type_64 lba=%ld\n",partition->lba);
	}
	/* Test JFS */
	if(disk_car->read(disk_car,8, &buffer, partition->lba+64)!=0) /* 32k offset */
	  return -1;
	if(recover_JFS(disk_car,(const struct jfs_superblock*)&buffer,partition,debug,dump_ind)==0) return 1;
  }
  return 0;
}

int search_type_128(t_param_disk *disk_car,t_diskext *partition,const int debug, const int dump_ind)
{
  if(partition->lba+128<disk_car->size)
  {
	unsigned char buffer[8*SECTOR_SIZE];
	if(debug>1)
	{
	  ecrit_rapport("search_type_128 lba=%ld\n",partition->lba);
	}
	/* Test ReiserFS */
	if(disk_car->read(disk_car,8, &buffer, partition->lba+128)!=0) /* 64k offset */
	  return -1;
	if(recover_rfs(disk_car,(const struct reiserfs_super_block*)&buffer,partition,debug,dump_ind)==0) return 1;
  }
  return 0;
}

int check_part(t_param_disk *disk_car,const int debug,t_diskext *partition)  
{
  int ret=0;
  switch(partition->part_type)
  {
    case P_12FAT:
    case P_16FAT:
    case P_16FATBD:
    case P_32FAT:
    case P_32FAT_LBA:
    case P_16FATBD_LBA:
    case P_12FATH:
    case P_16FATH:
    case P_16FATBDH:
    case P_32FATH:
    case P_32FAT_LBAH:
    case P_16FATBD_LBAH:
      ret=check_FAT(disk_car,partition,debug);
      break;
    case P_LINUX:
      ret=check_JFS(disk_car,partition,debug);
      if(ret!=0)
      {
	ret=check_rfs(disk_car,partition,debug);
      }
      if(ret!=0)
      {
	ret=check_EXT2(disk_car,partition,debug);
      }
      if(ret!=0)
      { aff_buffer(BUFFER_ADD,"No EXT2, JFS or Reiser marker\n"); }
      break;
    case P_NTFS:
    case P_NTFSH:
      ret=check_NTFS(disk_car,partition,debug,0);
      if(ret!=0)
      { aff_buffer(BUFFER_ADD,"Invalid NTFS boot\n"); }
      break;
    case P_LVM:
      ret=check_LVM(disk_car,partition,debug);
      break;
    case P_BEOS:
      ret=check_BeFS(disk_car,partition,debug);
      break;
    case P_FREEBSD:
      ret=check_BSD(disk_car,partition,debug,BSD_MAXPARTITIONS);
      break;
    case P_OPENBSD:
      ret=check_BSD(disk_car,partition,debug,OPENBSD_MAXPARTITIONS);
      break;
    case P_NETBSD:
      ret=check_BSD(disk_car,partition,debug,BSD_MAXPARTITIONS);
      break;
    case P_LINSWAP:
      ret=check_Linux_SWAP(disk_car,partition,debug);
      break;
    case P_RAID:
      ret=check_MD(disk_car,partition,debug);
      if(ret!=0)
      { aff_buffer(BUFFER_ADD,"Invalid RAID superblock\n"); }
      break;
    case P_EXTENDED:
    case P_EXTENDX:
    case P_LINUXEXTENDX:
    case P_NETWARE:
      break;
    default:
      if(debug>0)
	ecrit_rapport("check_part %u type %02X: no test\n",partition->order,partition->part_type);
      break;
  }
  if(ret!=0)
  {
    ecrit_rapport("check_part %02X\n", partition->part_type);
    aff_part_buffer(AFF_PART_ORDER,disk_car,partition);
  }
  return ret;
}

