/*

    File: ufs.c

    Copyright (C) 2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 59
    Temple Place - Suite 330, Boston MA 02111-1307, USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "ufs.h"
#include "fnctdsk.h"
#include "intrf.h"

#define UFS_SECTOR_SIZE 512
#define UFS_SBLOCK 8192
#define UFS_MAGIC 0x00011954
#define UFS_CIGAM 0x54190100 /* byteswapped MAGIC */


/* HP specific MAGIC values */

#define UFS_MAGIC_LFN   0x00095014 /* fs supports filenames > 14 chars */
#define UFS_CIGAM_LFN   0x14500900 /* srahc 41 < semanelif stroppus sf */

#define UFS_MAGIC_SEC   0x00612195 /* B1 security fs */
#define UFS_CIGAM_SEC   0x95216100

#define UFS_MAGIC_FEA   0x00195612 /* fs_featurebits supported */
#define UFS_CIGAM_FEA   0x12561900

#define UFS_MAGIC_4GB   0x05231994 /* fs > 4 GB && fs_featurebits */
#define UFS_CIGAM_4GB   0x94192305


static int set_ufs_info(const t_param_disk *disk_car, const struct ufs_super_block_first *usb1,t_partition *partition, const int debug, const int dump_ind);
static int test_ufs(const t_param_disk *disk_car, const struct ufs_super_block_third *usb3,t_partition *partition,const int debug, const int dump_ind);

int check_ufs(t_param_disk *disk_car,t_partition *partition,const int debug)
{
  unsigned char ubh[2048];
  const struct ufs_super_block_first*usb1=(const struct ufs_super_block_first*)ubh;
  const struct ufs_super_block_third *usb3=(const struct ufs_super_block_third *)(ubh+UFS_SECTOR_SIZE*2);
  if(disk_car->read(disk_car,sizeof(ubh), &ubh, partition->part_offset+UFS_SBLOCK)!=0)
  { return 1; }
  if(test_ufs(disk_car,usb3,partition,debug,0)!=0)
    return 1;
  set_ufs_info(disk_car,usb1,partition,debug,0);
  return 0;
}

static int test_ufs(const t_param_disk *disk_car, const struct ufs_super_block_third *usb3,t_partition *partition,const int debug, const int dump_ind)
{
  if ((usb3->fs_magic==UFS_MAGIC) ||(usb3->fs_magic==UFS_CIGAM))
  {
    partition->upart_type = UP_UFS;
  }
  else
    return 1;
  if(debug!=0)
    ecrit_rapport("\nUFS Marker at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
  return 0;
}

int recover_ufs(t_param_disk *disk_car, const unsigned char *ubh, t_partition *partition,const int debug, const int dump_ind)
{
  const struct ufs_super_block_first*usb1=(const struct ufs_super_block_first*)ubh;
  const struct ufs_super_block_third *usb3=(const struct ufs_super_block_third *)(ubh+UFS_SECTOR_SIZE*2);
  if(test_ufs(disk_car,usb3,partition,debug,dump_ind)!=0)
    return 1;
  if(debug!=0 || dump_ind!=0)
  {
    ecrit_rapport("\nrecover_ufs\n");
    if(dump_ind!=0)
    {
      dump(stdscr,usb1,sizeof(*usb1));
    }
  }
  switch(le32(usb3->fs_magic))
  {
    case UFS_MAGIC:
      partition->part_size = (uint64_t)le32(usb1->fs_size)*le32(usb1->fs_fsize);
      break;
    default:
      partition->part_size = (uint64_t)be32(usb1->fs_size)*be32(usb1->fs_fsize);
      break;
  }
  set_ufs_info(disk_car,usb1,partition,debug,dump_ind);
  partition->part_type = (unsigned char)0x02;
  return 0;
}

static int set_ufs_info(const t_param_disk *disk_car,const struct ufs_super_block_first *usb1,t_partition *partition, const int debug, const int dump_ind)
{
  partition->info[0]='\0';
  set_part_name(partition,usb1->fs_fsmnt,sizeof(partition->name));
  return 0;
}
