/*

    File: file_zip.c

    Copyright (C) 1998-2007 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 51
    Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <stdio.h>
#include <string.h>
#include "types.h"
#include "filegen.h"

static void register_header_check_zip(file_stat_t *file_stat);
static int header_check_zip(const unsigned char *buffer, const unsigned int buffer_size, const unsigned int safe_header_only, const file_recovery_t *file_recovery, file_recovery_t *file_recovery_new);
static void file_check_zip(file_recovery_t *file_recovery);
static unsigned int pos_in_mem(const unsigned char *haystack, const unsigned int haystack_size, const unsigned char *needle, const unsigned int needle_size);

const file_hint_t file_hint_zip= {
  .extension="zip",
  .description="zip archive",
  .min_header_distance=0,
  .max_filesize=PHOTOREC_MAX_FILE_SIZE,
  .recover=1,
  .header_check=&header_check_zip,
  .register_header_check=&register_header_check_zip
};

static const unsigned char zip_header[4]  = { 'P', 'K', 0x03, 0x04};
static const unsigned char zip_header2[8]  = { 'P', 'K', '0', '0', 'P', 'K', 0x03, 0x04}; /* WinZIPv8-compressed files. */

static void register_header_check_zip(file_stat_t *file_stat)
{
  register_header_check(0, zip_header,sizeof(zip_header), &header_check_zip, file_stat);
  register_header_check(0, zip_header2,sizeof(zip_header2), &header_check_zip, file_stat);
}

static int header_check_zip(const unsigned char *buffer, const unsigned int buffer_size, const unsigned int safe_header_only, const file_recovery_t *file_recovery, file_recovery_t *file_recovery_new)
{
  unsigned char tmp_buffer[2000];
  char *ptr;
  if(memcmp(buffer,zip_header,sizeof(zip_header))==0)
  {
    reset_file_recovery(file_recovery_new);
    file_recovery_new->min_filesize=21;
    file_recovery_new->file_check=&file_check_zip;
    if(memcmp(&buffer[30],"mimetypeapplication/vnd.sun.xml.",32)==0)
     {
       if(memcmp(&buffer[62],"calc",4)==0)
       {
	 file_recovery_new->extension="sxc";
       }
       else if(memcmp(&buffer[62],"draw",4)==0)
       {
	 file_recovery_new->extension="sxd";
       }
       else if(memcmp(&buffer[62],"impress",7)==0)
       {
	 file_recovery_new->extension="sxi";
       }
       else if(memcmp(&buffer[62],"writer",6)==0)
       {
	 file_recovery_new->extension="sxw";
       }
       else
       { /* default to writer */
	 file_recovery_new->extension="sxw";
       }
     }
     else if(memcmp(&buffer[30],"mimetypeapplication/vnd.oasis.opendocument.graphics",51)==0)
       file_recovery_new->extension="odg";
     else if(memcmp(&buffer[30],"mimetypeapplication/vnd.oasis.opendocument.presentation",55)==0)
       file_recovery_new->extension="odp";
     else if(memcmp(&buffer[30],"mimetypeapplication/vnd.oasis.opendocument.spreadsheet",54)==0)
       file_recovery_new->extension="ods";
     else if(memcmp(&buffer[30],"mimetypeapplication/vnd.oasis.opendocument.text",47)==0)
       file_recovery_new->extension="odt";
     else if(memcmp(&buffer[30],"[Content_Types].xml",19)==0)
     {
       if(pos_in_mem(&buffer[0],2000,"word",4)!=0)
	 file_recovery_new->extension="docx";
       else if(pos_in_mem(&buffer[0],2000,"xl",2)!=0)
	 file_recovery_new->extension="xlsx";
       else if(pos_in_mem(&buffer[0],2000,"ppt",3)!=0)
	 file_recovery_new->extension="pptx";
       else
	 file_recovery_new->extension="docx";
     }
    else
      file_recovery_new->extension=file_hint_zip.extension;
    return 1;
  }
  else if(memcmp(buffer,zip_header2,sizeof(zip_header2))==0)
  {
    reset_file_recovery(file_recovery_new);
    file_recovery_new->file_check=&file_check_zip;
    file_recovery_new->extension=file_hint_zip.extension;
    return 1;
  }
  return 0;
}

static void file_check_zip(file_recovery_t *file_recovery)
{
  const unsigned char zip_footer[4]= {0x4b, 0x05, 0x06, 0x00};
  file_search_footer(file_recovery, zip_footer,sizeof(zip_footer));
  if(file_recovery->file_size>0)
    file_recovery->file_size+=17;
}

static unsigned int pos_in_mem(const unsigned char *haystack, const unsigned int haystack_size, const unsigned char *needle, const unsigned int needle_size)
{
  unsigned int i;
  for(i=0;i<haystack_size;i++)
    if(memcmp(&haystack[i],needle,needle_size)==0)
      return (i+needle_size);
  return 0;
}

