/* -*- mode: c; c-file-style: "gnu" -*-
 * conffile.l -- configuration file lexer for Thy
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
/* arch-tag: bc84d942-9e17-4aa3-b02b-e9375a19e5c2 */
%{
#define YYSTYPE char *

#include "system.h"

#include <string.h>

#include "compat/compat.h"
#include "bh-libs/list.h"

#include "config.h"
#include "cfg_grammar.tab.h"

#ifndef yylval
extern YYSTYPE yylval;
#endif

extern bhl_list_t *conffile_list;
const char *current_file;
int yyparse (void);

#define MAX_STR_CONST 1024
#define qspush(c) \
	if (string_buf_ptr < string_buf+MAX_STR_CONST) \
		*string_buf_ptr++ = c; \
	else \
		yyerror ("quoted string overflow");

char string_buf[MAX_STR_CONST];
char *string_buf_ptr;

int yyerror (char const *msg);

#if malloc == rpl_malloc
#undef malloc
#undef realloc
#endif
%}

WHITE	[ \t\n\r]*
BOOL_OFF	(off|false)
BOOL_ON		(on|true)

%option yylineno case-insensitive nomain noyywrap

%x QSTRING
%%

{WHITE}		{/* eat white space */}
#.*		{/* eat comments */}

\<		{return '<';}
\>		{return '>';}
\/		{return '/';}

=		{return '=';}

{BOOL_OFF}	{return T_BOOL_OFF;}
{BOOL_ON}	{return T_BOOL_ON;}

KeepAliveTimeout	{ return T_KEEPALIVETIMEOUT; }
MaxKeepAliveRequests	{ return T_MAXKEEPALIVEREQUESTS; }

DocumentRoot	{ return T_DOCUMENTROOT; }
Listen		{ return T_LISTEN; }
DefaultType	{ return T_DEFAULT_TYPE; }
Timeout		{ return T_TIMEOUT; }
UserDir		{ return T_USERDIR; }
UserID		{ return T_USERID; }
UID		{ return T_UID; }
PIDFile		{ return T_PIDFILE; }
IPv4		{ return T_IPV4; }
IPv6		{ return T_IPV6; }
\<Map		{ return T_MAP; }
\<\/Map		{ return T_UNMAP; }
\<VirtualHost	{ return T_VIRTUALHOST; }
\<\/VirtualHost	{ return T_UNVIRTUALHOST; }
Indexes		{ return T_INDEXES; }
CGIExts		{ return T_CGIEXTS; }
ErrorDoc	{ return T_ERRORDOC; }
CGIDirs		{ return T_CGIDIRS; }
AddEnv		{ return T_ADDENV; }
AddHandler	{ return T_ADDHANDLER; }
AddMethod	{ return T_ADDMETHOD; }
AddHeader	{ return T_ADDHEADER; }
AddType		{ return T_ADDTYPE; }
Alias		{ return T_ALIAS; }
ScriptAlias	{ return T_SCRIPTALIAS; }

GZip		{ return T_GZIP; }
Type		{ return T_TYPE; }
Level		{ return T_LEVEL; }
GZipType	{ return T_GZIP_TYPE; }
GZipLevel	{ return T_GZIP_LEVEL; }

Limit		{ return T_LIMIT; }
CGIs		{ return T_CGIS; }
Header		{ return T_HEADER; }
POST_Buffer	{ return T_BUFFER; }
Buffer		{ return T_BUFFER; }
MMap		{ return T_MMAP; }
LimitMMap	{ return T_LIMIT_MMAP; }
LimitCGIs	{ return T_LIMIT_CGIS; }
LimitHeader	{ return T_LIMIT_HEADER; }
LimitBuffer	{ return T_LIMIT_POST_BUFFER; }
LimitMaxClients	{ return T_LIMIT_MAXCLIENTS; }
MaxClients	{ return T_MAXCLIENTS; }

Auth		{ return T_AUTH; }
Path		{ return T_PATH; }
Args		{ return T_ARGS; }
File		{ return T_FILE; }
AuthFile	{ return T_AUTH_FILE; }
AuthPath	{ return T_AUTH_PATH; }
AuthArgs	{ return T_AUTH_ARGS; }
AuthUID		{ return T_AUTH_UID; }

ETag		{ return T_ETAG; }
DirETag		{ return T_DIRETAG; }

Worker		{ return T_WORKER; }
WorkerPath	{ return T_WORKER_PATH; }
WorkerArgs	{ return T_WORKER_ARGS; }
WorkerUID	{ return T_WORKER_UID; }

SSL		{ return T_SSL; }
CA		{ return T_CA; }
Cert		{ return T_CERT; }
KeyRing		{ return T_KEYRING; }
Key		{ return T_KEY; }
TrustDB		{ return T_TRUSTDB; }
Verify		{ return T_VERIFY; }
SRPPasswd	{ return T_SRPPASSWD; }
SRPConf		{ return T_SRPCONF; }
SSLSRPPasswd	{ return T_SSL_SRPPASSWD; }
SSLSRPConf	{ return T_SSL_SRPCONF; }
SSLCA		{ return T_SSL_CA; }
SSLCert		{ return T_SSL_CERT; }
SSLKeyRing	{ return T_SSL_KEYRING; }
SSLKey		{ return T_SSL_KEY; }
SSLTrustDB	{ return T_SSL_TRUSTDB; }
SSLType		{ return T_SSL_TYPE; }
SSLVerify	{ return T_SSL_VERIFY; }

CheapVHost	{ return T_CHEAP_VHOST; }
CGI		{ return T_CGI; }
UserCGI		{ return T_USERCGI; }
CaseMIME	{ return T_CASEMIME; }
FollowAll	{ return T_FOLLOWALL; }
Server		{ return T_SERVER; }
Vary		{ return T_VARY; }
Chroot		{ return T_CHROOT; }
Expect		{ return T_EXPECT; }
DirIndex	{ return T_DIRINDEX; }
HardLimit	{ return T_HARDLIMIT; }
Stats		{ return T_STATS; }
LazyCGI		{ return T_LAZYCGI; }

Cache		{ return T_CACHE; }

max-age		{ return T_CACHE_MAX_AGE; }
no-cache	{ return T_CACHE_NO_CACHE; }
no-store	{ return T_CACHE_NO_STORE; }
no-transform	{ return T_CACHE_NO_TRANSFORM; }
must-revalidate { return T_CACHE_MUST_REVALIDATE; }
expiry-base	{ return T_CACHE_EXPIRY_BASE; }

[[:digit:]]+	{
			yylval = (char *)strdup (yytext);
			return T_INT;
		}

\'[^\']*\'	{
			yytext[strlen(yytext)-1] = '\0';
			yylval = (char *)strdup(++yytext);
			return T_STRING;
		}

\"		{
  			string_buf_ptr = string_buf;
			BEGIN (QSTRING);
		}

<QSTRING>{
\"		{
			/* saw closing quote - all done */
			BEGIN (INITIAL);
			*string_buf_ptr = '\0';
			/* return string constant token type and value to parser */
			yylval = string_buf;
			return T_STRING;
		}

\n		{
			/* error - unterminated string constant */
			/* generate error message */
			yyerror ("unterminated string constant");
		}

\\[0-7]{1,3}	{
			/* octal escape sequence */
			int result;

			sscanf (yytext + 1, "%o", &result);

			if (result > 0xff)
				yyerror ("octal constant out of bounds");
			qspush(result);
		}

\\[0-9]+	{
			/* generate error - bad escape sequence; something
			 * like '\48' or '\0777777'
			 */
			yyerror ("bad escape sequence");
		}

\\n		qspush ('\n');
\\t		qspush ('\t');
\\r		qspush ('\r');
\\b		qspush ('\b');
\\f		qspush ('\f');

\\(.|\n)	*string_buf_ptr++ = yytext[1];

[^\\\n\"]+	{
			char *yptr = yytext;
			while (*yptr) qspush (*yptr++);
		}
} /* End of QSTRING string state. */

.		{ return ERROR; }
%%
int
thy_config_file_parse (const char *fn)
{
  FILE *conf;
  int err;

  /* return if user has not defined a config file */
  if (!fn)
    return 0;

  /* try and open it */
  conf = fopen (fn, "r");

  /* if we succeeded, try and parse it */
  if (conf)
    {
      yylineno = 1;
      yyin = conf;
      current_file = fn;

      bhl_list_free (conffile_list);
      conffile_list = bhl_list_init (1, NULL);
      bhl_list_append_string (conffile_list, "thy");

      err = yyparse ();
      fclose (conf);

      if (err)
	return 1;
    }
  else
    return 2;

  return 0;
}
