
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

#include <debug/memory.h>
#include <ber/ber.h>
#include <abz/typedefs.h>

#include <tinysnmp/tinysnmp.h>
#include <tinysnmp/manager/udp.h>
#include <tinysnmp/manager/snmp.h>

/* initial size of table (this can be anything bigger than or equal to 1) */
#define SIZE 10

typedef struct
{
   size_t n;
   snmp_table_t *table;
   uint32_t **oid;
} priv_table_t;

static int encode_pdu (ber_t *e,const snmp_pdu_t *pdu)
{
   int i;
   uint32_t offset;

   for (i = pdu->n - 1; i >= 0; i--)
	 {
		offset = e->offset;
		if (ber_encode_null (e) < 0 ||													/* value NULL							*/
			ber_encode_oid (e,pdu->oid[i]) < 0 ||										/* name OBJECT IDENTIFIER				*/
			ber_encode_sequence (e,e->offset - offset) < 0)								/* VarBind ::= SEQUENCE					*/
		  return (-1);
	 }

   if (ber_encode_sequence (e,e->offset) < 0 ||											/* VarBindList ::= SEQUENCE OF			*/
	   ber_encode_integer (e,0) < 0 ||													/* ErrorIndex INTEGER					*/
	   ber_encode_integer (e,0) < 0 ||													/* ErrorStatus INTEGER { noError(0) }	*/
	   ber_encode_integer (e,pdu->RequestID) < 0)										/* Request-ID INTEGER					*/
	 return (-1);

   if (pdu->type == BER_GetRequest && ber_encode_get_request (e) < 0)					/* PDU { GetRequest-PDU(0) }			*/
	 return (-1);
   else if (pdu->type == BER_GetNextRequest && ber_encode_get_next_request (e) < 0)		/* PDU { GetNextRequest-PDU(0) }		*/
	 return (-1);

   if (ber_encode_octet_string (e,&pdu->community) < 0 ||								/* community OCTET STRING				*/
	   ber_encode_integer (e,pdu->version) < 0 ||										/* version INTEGER { version-1(0) }		*/
	   ber_encode_sequence (e,e->offset) < 0)											/* Message ::= SEQUENCE					*/
	 return (-1);

   return (0);
}

static int octet_string_compare (const octet_string_t *a,const octet_string_t *b)
{
   if (a->len < b->len)
	 return (-1);

   if (a->len > b->len)
	 return (1);

   return (memcmp (a->buf,b->buf,a->len));
}

static int decode_response_prefix (ber_t *d,const snmp_pdu_t *pdu)
{
   int32_t version,RequestID,ErrorStatus,ErrorIndex;
   octet_string_t community;
   int result;

   if ((result = ber_decode_sequence (d)) < 0 ||										/* Message ::= SEQUENCE					*/
	   (result = ber_decode_integer (&version,d)) < 0 ||								/* version INTEGER { version-1(0) }		*/
	   (result = ber_decode_octet_string (&community,d)) < 0 ||							/* community OCTET STRING				*/
	   (result = ber_decode_get_response (d)) < 0 ||									/* PDU { GetResponse-PDU(0) }			*/
	   (result = ber_decode_integer (&RequestID,d)) < 0 ||								/* Request-ID INTEGER					*/
	   (result = ber_decode_integer (&ErrorStatus,d)) < 0 ||							/* ErrorStatus INTEGER { noError(0) }	*/
	   (result = ber_decode_integer (&ErrorIndex,d)) < 0 ||								/* ErrorIndex INTEGER					*/
	   (result = ber_decode_sequence (d)) < 0)											/* VarBindList ::= SEQUENCE OF			*/
	 {
		if (community.len) mem_free (community.buf);
		return (result);
	 }

   if (version != pdu->version || octet_string_compare (&community,&pdu->community) ||
	   RequestID != pdu->RequestID/* || ErrorStatus || ErrorIndex*/)
	 {
		if (community.len) mem_free (community.buf);
		return (-EINVAL);
	 }

   if (community.len) mem_free (community.buf);

   return (0);
}

static int decode_oid_sequence (snmp_next_value_t *seq,ber_t *d)
{
   int result;

   if ((result = ber_decode_sequence (d)) < 0 ||										/* VarBind ::= SEQUENCE					*/
	   (result = ber_decode_oid (&seq->oid,d)) < 0)										/* name OBJECT IDENTIFIER				*/
	 return (result);

   if (d->offset >= d->size)
	 {
		mem_free (seq->oid);
		return (-EINVAL);
	 }

   switch (d->buf[d->offset])															/* value ObjectSyntax					*/
	 {
	  case BER_INTEGER:
		seq->value.type = BER_INTEGER;
		result = ber_decode_integer (&seq->value.data.INTEGER,d);
		break;
	  case BER_Counter32:
		seq->value.type = BER_Counter32;
		result = ber_decode_counter32 (&seq->value.data.Counter32,d);
		break;
	  case BER_Gauge32:
		seq->value.type = BER_Gauge32;
		result = ber_decode_gauge32 (&seq->value.data.Gauge32,d);
		break;
	  case BER_TimeTicks:
		seq->value.type = BER_TimeTicks;
		result = ber_decode_timeticks (&seq->value.data.TimeTicks,d);
		break;
	  case BER_Counter64:
		seq->value.type = BER_Counter64;
		result = ber_decode_counter64 (&seq->value.data.Counter64,d);
		break;
	  case BER_OID:
		seq->value.type = BER_OID;
		result = ber_decode_oid (&seq->value.data.OID,d);
		break;
	  case BER_OCTET_STRING:
		seq->value.type = BER_OCTET_STRING;
		result = ber_decode_octet_string (&seq->value.data.OCTET_STRING,d);
		break;
	  case BER_NULL:
		seq->value.type = BER_NULL;
		result = ber_decode_null (d);
		break;
	  case BER_IpAddress:
		seq->value.type = BER_IpAddress;
		result = ber_decode_ipaddress (&seq->value.data.IpAddress,d);
		break;
	  default:
		result = -ENOSYS;
	 }

   if (result < 0)
	 {
		mem_free (seq->oid);
		return (result);
	 }

   return (0);
}

static void free_data (uint8_t type,snmp_data_t *data)
{
   if (type == BER_OID)
	 mem_free (data->OID);
   else if (type == BER_OCTET_STRING && data->OCTET_STRING.len)
	 mem_free (data->OCTET_STRING.buf);
}

/*
 * Free memory allocated by snmp_get() routine.
 */
void snmp_free_values (snmp_value_t **values,size_t n)
{
   size_t i;

   for (i = 0; i < n; i++)
	 free_data ((*values)[i].type,&(*values)[i].data);

   mem_free (*values);
   *values = NULL;
}

/*
 * Free memory allocated by snmp_get_next() routine.
 */
void snmp_free_next_values (snmp_next_value_t **values,size_t n)
{
   size_t i;

   for (i = 0; i < n; i++)
	 {
		mem_free ((*values)[i].oid);
		free_data ((*values)[i].value.type,&(*values)[i].value.data);
	 }

   mem_free (*values);
   *values = NULL;
}

/*
 * Free memory allocated by snmp_get_tables() routine.
 */
void snmp_free_tables (snmp_table_t **table,size_t n)
{
   size_t i;
   uint32_t j;

   for (i = 0; i < n; i++)
	 {
		for (j = 0; j < (*table)[i].n; j++)
		  {
			 free_data ((*table)[i].type,(*table)[i].data + j);
			 mem_free ((*table)[i].oid[j]);
		  }

		mem_free ((*table)[i].data);
		mem_free ((*table)[i].oid);
	 }

   mem_free (*table);
   *table = NULL;
}

static snmp_value_t *decode_get_response1 (ber_t *d,const snmp_pdu_t *pdu)
{
   int i,result;
   snmp_value_t *value;
   snmp_next_value_t tmp;

   if ((result = decode_response_prefix (d,pdu)) < 0)
	 {
		errno = -result;
		return (NULL);
	 }

   if ((value = (snmp_value_t *) mem_alloc (sizeof (snmp_value_t) * pdu->n)) == NULL)
	 {
		errno = ENOMEM;
		return (NULL);
	 }

   for (i = 0; i < pdu->n; i++)
	 {
		if ((result = decode_oid_sequence (&tmp,d)) < 0)
		  {
			 snmp_free_values (&value,i);
			 errno = -result;
			 return (NULL);
		  }

		if (memcmp (tmp.oid,pdu->oid[i],(tmp.oid[0] + 1) * sizeof (uint32_t)))
		  {
			 snmp_free_values (&value,i);
			 mem_free (tmp.oid);
			 errno = EINVAL;
			 return (NULL);
		  }

		mem_free (tmp.oid);

		memcpy (&value[i],&tmp.value,sizeof (snmp_value_t));
	 }

   if (d->offset != d->size)
	 {
		snmp_free_values (&value,pdu->n);
		errno = EINVAL;
		return (NULL);
	 }

   return (value);
}

static snmp_next_value_t *decode_get_response2 (ber_t *d,const snmp_pdu_t *pdu)
{
   int i,result;
   snmp_next_value_t *next;

   if ((result = decode_response_prefix (d,pdu)) < 0)
	 {
		errno = -result;
		return (NULL);
	 }

   if (!pdu->n)
	 {
		errno = EINVAL;
		return (NULL);
	 }

   if ((next = (snmp_next_value_t *) mem_alloc (sizeof (snmp_next_value_t) * pdu->n)) == NULL)
	 {
		errno = ENOMEM;
		return (NULL);
	 }

   if (d->offset != d->size)
	 {
		for (i = 0; i < pdu->n; i++)
		  if ((result = decode_oid_sequence (next + i,d)) < 0)
			{
			   snmp_free_next_values (&next,i);
			   errno = -result;
			   return (NULL);
			}

		if (d->offset != d->size)
		  {
			 snmp_free_next_values (&next,i);
			 errno = EINVAL;
			 return (NULL);
		  }
	 }

   return (next);
}

static int32_t make_request_id ()
{
   struct timeval tv;
   uint32_t id;

   if (gettimeofday (&tv,NULL) < 0) return (0x13a48f75);

   id = (uint32_t) tv.tv_sec | (uint32_t) tv.tv_usec;

   return ((int32_t) id);
}

static void *getpdu (udp_t *udp,const octet_string_t *community,uint32_t *const *oid,int n,uint8_t type)
{
   ber_t ber;
   snmp_pdu_t pdu;
   void *values = NULL;
   uint16_t tmp;
   uint32_t zeroDotZero[2] = { 1, 0 };
   uint32_t *zeroOid;
   int saved;

   if (udp == NULL || community == NULL || (type != BER_GetRequest && type != BER_GetNextRequest))
	 {
		errno = EINVAL;
		return (NULL);
	 }

   /* initialize the BER data structure */
   ber.size = UDP_DATAGRAM_SIZE;
   ber.offset = 0;
   if ((ber.buf = (uint8_t *) mem_alloc (ber.size)) == NULL)
	 {
		errno = ENOMEM;
		return (NULL);
	 }

   /* initialize the PDU data structure */
   pdu.version = SNMP_VERSION_1;
   pdu.community.len = community->len;
   pdu.community.buf = community->buf;
   pdu.RequestID = make_request_id ();
   if (n)
	 {
		pdu.oid = (uint32_t **) oid;
		pdu.n = n;
	 }
   else
	 {
		zeroOid = zeroDotZero;
		pdu.oid = (uint32_t **) &zeroOid;
		pdu.n = 1;
	 }

   pdu.type = type;

   /* encode get-request or get-next-request PDU, depending on type */
   if (encode_pdu (&ber,&pdu) < 0)
	 {
		mem_free (ber.buf);
		errno = ERANGE;
		return (NULL);
	 }

   /* establish a connection to the SNMP server */
   if (udp_connect (udp) < 0)
	 {
		saved = errno;
		mem_free (ber.buf);
		errno = saved;
		return (NULL);
	 }

   /* send our get-request/get-next-request packet to the SNMP server */
   if (udp_send (udp,ber.buf + ber.size - ber.offset,ber.offset) < 0)
	 {
		udp_io_error:
		saved = errno;
		udp_disconnect (udp);
		mem_free (ber.buf);
		errno = saved;
		return (NULL);
	 }

   /* receive the get-response packet from agent */
   ber.offset = 0;
   tmp = UDP_DATAGRAM_SIZE;
   if (udp_receive (udp,ber.buf,&tmp) < 0) goto udp_io_error;
   ber.size = tmp;

   /* close the connection */
   udp_disconnect (udp);

   /* decode the get-response packet */
   if (type == BER_GetRequest)
	 values = decode_get_response1 (&ber,&pdu);
   else if (type == BER_GetNextRequest)
	 values = decode_get_response2 (&ber,&pdu);

   mem_free (ber.buf);

   return (values);
}

/*
 * Create a handle for use with functions below.
 *        agent           snmp agent info
 *        addr            address/port of agent
 *        community       community string used for authentication
 *        timeout         timeout in milliseconds for any operations. if timeout
 *                        is 0, the timeout is disabled (i.e. we don't use
 *                        non-blocking I/O).
 *        retries         number of retries before get routines fail. if retries
 *                        is 0, only one attempt is made to communicate with the
 *                        agent.
 *
 * Returns 0 if successful, or -1 if some error occurred. Check
 * errno to see what error occurred.
 */
int snmp_open (snmp_agent_t *agent,const struct sockaddr_in *addr,const char *community,uint32_t timeout,uint16_t retries)
{
   agent->retries = retries;
   agent->community.len = strlen (community);

   if ((agent->community.buf = (uint8_t *) mem_alloc (agent->community.len)) == NULL)
	 return (-1);

   memcpy (agent->community.buf,community,agent->community.len);

   udp_open (&agent->udp,addr,timeout);

   return (0);
}

/*
 * Destroy handle created by snmp_open().
 */
void snmp_close (snmp_agent_t *agent)
{
   mem_free (agent->community.buf);
   udp_close (&agent->udp);
}

/*
 * Retrieve values from the agent.
 *
 *        agent           snmp agent info
 *        oid             list of ObjectID's to retrieve
 *        n               number of ObjectID's in list
 *
 * Return the values if successful, or NULL if some error occurred. Check
 * errno to see what error occurred.
 */
snmp_value_t *snmp_get (snmp_agent_t *agent,uint32_t *const *oid,size_t n)
{
   uint16_t i;
   snmp_value_t *value;

   for (i = 0; i <= agent->retries; i++)
	 if ((value = getpdu (&agent->udp,&agent->community,oid,n,BER_GetRequest)) != NULL)
	   return (value);

   return (NULL);
}

/*
 * Retrieve next values from the agent.
 *
 *        agent           snmp agent info
 *        oid             list of (partial) ObjectID's to send to agent
 *        n               number of ObjectID's in list
 *
 * Return the values (and their corresponding ObjectID's) that lexigraphically
 * succeeds those specified if successful, or NULL if the query failed. Check
 * errno to see what error occurred.
 */
snmp_next_value_t *snmp_get_next (snmp_agent_t *agent,uint32_t *const *oid,size_t n)
{
   uint16_t i;
   snmp_next_value_t *next;

   for (i = 0; i <= agent->retries; i++)
	 if ((next = getpdu (&agent->udp,&agent->community,oid,n,BER_GetNextRequest)) != NULL)
	   return (next);

   return (NULL);
}

static int get_first_entries (snmp_agent_t *agent,priv_table_t *priv)
{
   size_t i;
   snmp_next_value_t *next;

   if ((next = snmp_get_next (agent,priv->oid,priv->n)) == NULL)
	 return (-1);

   for (i = 0; i < priv->n; i++)
	 if (next[i].oid[0] != priv->oid[i][0] + 1 || memcmp (next[i].oid + 1,priv->oid[i] + 1,priv->oid[i][0] * sizeof (uint32_t)))
	   {
		  snmp_free_next_values (&next,priv->n);
		  errno = EINVAL;
		  return (-1);
	   }

   for (i = 0; i < priv->n; i++)
	 {
		size_t j;
		int saved;

		priv->table[i].type = next[i].value.type;
		priv->table[i].n = 1;

		if ((priv->table[i].data = (snmp_data_t *) mem_alloc (sizeof (snmp_data_t) * SIZE)) == NULL)
		  {
			 saved = errno;
			 snmp_free_next_values (&next,priv->n);
			 for (j = 0; j < i; j++) mem_free (priv->table[j].data);
			 errno = saved;
			 return (-1);
		  }

		priv->table[i].data[0] = next[i].value.data;

		if ((priv->table[i].oid = (uint32_t **) mem_alloc (sizeof (uint32_t *) * SIZE)) == NULL)
		  {
			 saved = errno;
			 snmp_free_next_values (&next,priv->n);
			 for (j = 0; j <= i; j++) mem_free (priv->table[j].data);
			 errno = saved;
			 return (-1);
		  }

		priv->oid[i] = priv->table[i].oid[0] = next[i].oid;
	 }

   mem_free (next);

   return (0);
}

static int get_next_entries (snmp_agent_t *agent,priv_table_t *priv)
{
   size_t i,mismatch = 0;
   snmp_next_value_t *next;

   if ((next = snmp_get_next (agent,priv->oid,priv->n)) == NULL)
	 return (-1);

   for (i = 0; i < priv->n; i++)
	 if (next[i].value.type != priv->table[i].type || next[i].oid[0] != priv->oid[i][0] ||
		 memcmp (next[i].oid + 1,priv->oid[i] + 1,(priv->oid[i][0] - 1) * sizeof (uint32_t)))
	   mismatch++;

   if (mismatch)
	 {
		snmp_free_next_values (&next,priv->n);
		if (mismatch != priv->n)
		  {
			 errno = EINVAL;
			 return (-1);
		  }
		return (0);
	 }

   for (i = 0; i < priv->n; i++)
	 {
		if (!(priv->table[i].n % SIZE))
		  {
			 snmp_data_t *data;
			 uint32_t **oid;
			 int failed = 1;

			 do
			   {
				  if ((data = (snmp_data_t *) mem_realloc (priv->table[i].data,sizeof (snmp_data_t) * (priv->table[i].n + SIZE))) == NULL)
					break;

				  priv->table[i].data = data;

				  if ((oid = (uint32_t **) mem_realloc (priv->table[i].oid,sizeof (uint32_t *) * (priv->table[i].n + SIZE))) == NULL)
					break;

				  priv->table[i].oid = oid;

				  failed = 0;
			   }
			 while (0);

			 if (failed)
			   {
				  int saved = errno;
				  snmp_free_next_values (&next,priv->n);
				  errno = saved;
				  return (-1);
			   }
		  }

		priv->table[i].data[priv->table[i].n] = next[i].value.data;
		priv->oid[i] = priv->table[i].oid[priv->table[i].n] = next[i].oid;
		priv->table[i].n++;
	 }

   mem_free (next);

   return (1);
}

/*
 * Retrieve tables of values from the agent. You should specify ObjectID's of the table
 * entries without their indexes. For example, if you want to retrieve the list of entries
 * in tableX.entryX, you should specify that, and this routine will return all the ObjectID's
 * of the form tableX.entryX.indexX. If the types of those entries differ, it is considered
 * to be an error.
 *
 *        agent           snmp agent info
 *        oid             list of table entry ObjectID's that define the tables
 *        n               number of ObjectID's in list
 *        rows            maximum number of rows to return (0 = unlimited)
 *
 * Return the entry lists that match specified entries if successful, or NULL if the query
 * failed. Check errno to see what error occurred.
 */
snmp_table_t *snmp_get_tables (snmp_agent_t *agent,uint32_t *const *oid,size_t n,size_t rows)
{
   priv_table_t priv;
   int result = 0;

   if (!n)
	 {
		errno = EINVAL;
		return (NULL);
	 }

   if ((priv.table = (snmp_table_t *) mem_alloc (sizeof (snmp_table_t) * n)) == NULL)
	 return (NULL);

   if ((priv.oid = (uint32_t **) mem_alloc (sizeof (uint32_t *) * n)) == NULL)
	 {
		result = errno;
		mem_free (priv.table);
		errno = result;
		return (NULL);
	 }

   priv.n = n;
   while (n--) priv.oid[n] = (uint32_t *) oid[n];

   if (get_first_entries (agent,&priv) < 0)
	 {
		result = errno;
		mem_free (priv.table);
		mem_free (priv.oid);
		errno = result;
		return (NULL);
	 }

   if (rows)
	 while (--rows > 0 && (result = get_next_entries (agent,&priv)) > 0) ;
   else
	 while ((result = get_next_entries (agent,&priv)) > 0) ;

   if (result < 0)
	 {
		result = errno;
		snmp_free_tables (&priv.table,priv.n);
		mem_free (priv.oid);
		errno = result;
		return (NULL);
	 }

   mem_free (priv.oid);

   return (priv.table);
}

