
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/utsname.h>

#include <debug/memory.h>

#include <abz/typedefs.h>
#include <abz/error.h>
#include <abz/tokens.h>

#include <tinysnmp/tinysnmp.h>
#include <tinysnmp/agent/odb.h>

#include "proc.h"
#include "update.h"
#include "system.h"

/*
 * TODO:
 *
 * sysORLastChange (remove dummy)
 * sysORTable
 */

#define LAYER(x) (1 << (x))

static char *contact = NULL;		/* contact person for this managed node		*/
static char *location = NULL;		/* the physical location of this node		*/

static void out_of_memory (void)
{
   abz_set_error ("failed to allocate memory: %m");
}

static void parse_error (const struct tokens *tokens,const char *suffix)
{
   abz_set_error ("usage: %s %s",tokens->argv[0],suffix);
}


static void already_defined (const struct tokens *tokens)
{
   abz_set_error ("`%s' already defined",tokens->argv[0]);
}

static int parse_contact (struct tokens *tokens)
{
   if (contact != NULL)
	 {
		already_defined (tokens);
		return (-1);
	 }

   if (tokens->argc != 2)
	 {
		parse_error (tokens,"<contact-string>");
		return (-1);
	 }

   if ((contact = mem_alloc (strlen (tokens->argv[1]) + 1)) == NULL)
	 {
		out_of_memory ();
		return (-1);
	 }

   strcpy (contact,tokens->argv[1]);

   return (1);
}

static int parse_location (struct tokens *tokens)
{
   if (location != NULL)
	 {
		already_defined (tokens);
		return (-1);
	 }

   if (tokens->argc != 2)
	 {
		parse_error (tokens,"<location-string>");
		return (-1);
	 }

   if ((location = mem_alloc (strlen (tokens->argv[1]) + 1)) == NULL)
	 {
		out_of_memory ();
		return (-1);
	 }

   strcpy (location,tokens->argv[1]);

   return (1);
}

int system_parse (struct tokens *tokens)
{
   static const struct
	 {
		const char *name;
		int (*parse) (struct tokens *);
	 } command[] =
	 {
		{ "contact", parse_contact },
		{ "location", parse_location }
	 };
   int i;

   abz_clear_error ();

   if (tokens == NULL)
	 {
		const char *missing =
		  contact == NULL ? "contact" :
		  location == NULL ? "location" :
		  NULL;

		if (missing != NULL)
		  {
			 abz_set_error ("unexpected end of file. `%s' statement missing",missing);
			 return (-1);
		  }

		return (0);
	 }

   for (i = 0; i < ARRAYSIZE (command); i++)
	 if (!strcmp (tokens->argv[0],command[i].name))
	   return (command[i].parse (tokens));

   return (0);
}

static int system_update_static (struct odb **odb)
{
   static int first = 1;
   /* iso.org.dod.internet.mgmt.mib-2.system.xx.xx */
   uint32_t oid[9] = { 8, 43, 6, 1, 2, 1, 1, 0, 0 };
   /* iso.org.dod.internet.private.enterprises.frogfoot.servers */
   uint32_t sysObjectID[8] = { 7, 43, 6, 1, 4, 1, 10002, 1 };
   struct utsname buf;
   octet_string_t str;
   int32_t sysServices;
   uint32_t sysORLastChange;

   if (!first)
	 return (0);

   first = 1;

   if (uname (&buf))
	 {
		abz_get_error ("failed to get information about current kernel: %m");
		return (-1);
	 }

   str.len = strlen (buf.sysname) +
	 strlen (buf.release) +
	 strlen (buf.version) +
	 strlen (buf.machine) + 3;

   if ((str.buf = mem_alloc (str.len + 1)) == NULL)
	 {
		out_of_memory ();
		return (-1);
	 }

   strcpy (str.buf,buf.sysname);
   strcat (str.buf," ");
   strcat (str.buf,buf.release);
   strcat (str.buf," ");
   strcat (str.buf,buf.version);
   strcat (str.buf," ");
   strcat (str.buf,buf.machine);

   oid[7] = 1;

   if (update (odb,oid,BER_OCTET_STRING,&str))
	 {
		mem_free (str.buf);
		return (-1);
	 }

   mem_free (str.buf);

   oid[7] = 2;

   if (update (odb,oid,BER_OID,sysObjectID))
	 return (-1);

   str.len = strlen (contact);
   str.buf = contact;

   oid[7] = 4;

   if (update (odb,oid,BER_OCTET_STRING,&str))
	 return (-1);

   str.len = strlen (location);
   str.buf = location;

   oid[7] = 6;

   if (update (odb,oid,BER_OCTET_STRING,&str))
	 return (-1);

   oid[7] = 7;

   /* Linux, FreeBSD, etc. can do layer 1-4, 7 */
   sysServices = LAYER(1) | LAYER(2) | LAYER(3) | LAYER(4) | LAYER(7);

   if (update (odb,oid,BER_INTEGER,&sysServices))
	 return (-1);

   oid[7] = 8;

   /* for now, we pretend life started in the beginning */
   sysORLastChange = 0;

   return (update (odb,oid,BER_TimeTicks,&sysORLastChange));
}

int system_update (struct odb **odb)
{
   /* iso.org.dod.internet.mgmt.mib-2.system.xx.xx */
   uint32_t oid[9] = { 8, 43, 6, 1, 2, 1, 1, 0, 0 };
   char hostname[MAXHOSTNAMELEN];
   octet_string_t str;
   uint32_t uptime;

   abz_clear_error ();

   if (system_update_static (odb))
	 return (-1);

   oid[7] = 3;

   if (getprocuptime (&uptime))
	 return (-1);

   uptime *= 100;

   if (update (odb,oid,BER_TimeTicks,&uptime))
	 return (-1);

   if (gethostname (hostname,MAXHOSTNAMELEN))
	 {
		abz_set_error ("failed to get hostname: %m");
		return (-1);
	 }

   hostname[MAXHOSTNAMELEN - 1] = '\0';

   str.len = strlen (hostname);
   str.buf = hostname;

   oid[7] = 5;

   return (update (odb,oid,BER_OCTET_STRING,&str));
}

void system_close (void)
{
   if (contact != NULL)
	 {
		mem_free (contact);
		contact = NULL;
	 }

   if (location != NULL)
	 {
		mem_free (location);
		location = NULL;
	 }
}

