# $Id: messages.tcl,v 1.67 2003/12/13 21:35:26 aleksey Exp $

namespace eval message {
    variable msgid 0
    variable headid 0

    variable headlines
    array set headlines {}

    variable options
    variable trees {}

    custom::defvar message_dest_list {} \
	[::msgcat::mc "List of message destination JIDs."] \
	-group Hidden

    custom::defvar headline_send_jids {} \
	[::msgcat::mc "List of JIDs to whom headlines have been sent."] \
	-group Hidden

    custom::defgroup Messages [::msgcat::mc "Message and Headline options."] \
	-group Tkabber

    custom::defvar options(headlines,cache) 0 \
	[::msgcat::mc "Cache headlines on exit and restore on start."] \
	-group Messages -type boolean

    custom::defvar options(headlines,multiple) 0 \
	[::msgcat::mc "Display headlines in single/multiple windows."] \
	-group Messages -type radio -layout vertical \
	-values [list 0 [::msgcat::mc "Single window"] \
		      1 [::msgcat::mc "One window per JID"] \
		      2 [::msgcat::mc "One window per JID/resource"]]

    custom::defvar options(headlines,display_subject_only) 1 \
	[::msgcat::mc "Do not display headline descriptions as tree nodes."] \
	-group Messages -type boolean
}

# TODO: connid
proc message::show_dialog {from type subject body thread priority x {replyP 1}} {
    variable msgid
    global font loginconf

    if {$type == "normal"} {
        ::message_archive::log_message \
                $from "$loginconf(user)@$loginconf(server)/$loginconf(resource)" $subject $body $x
    }

    set mw .msg$msgid
    toplevel $mw -class Message

    if {$replyP} {
        set title [format [::msgcat::mc "Message from %s"] $from]
    } else {
        set title [format [::msgcat::mc "Extras from %s"] $from]
    }
    wm title $mw $title
    wm iconname $mw $title
    
    frame $mw.bottom
    pack $mw.bottom -side bottom -fill x

    if {$replyP} {
        set bbox1 [ButtonBox $mw.bottom.buttons1 -spacing 0]
        if {![cequal [info commands ::ssj::sign:toggleP] ""]} {
            $bbox1 add -image [message::signed:icon] \
                    -helptype balloon -helptext [::msgcat::mc "Toggle signing"] \
                    -height 24 -width 24 -relief link -bd 1 \
                    -command ssj::sign:toggleP
            ssj::signed:trace "
                $mw.bottom.signed configure -image \[message::signed:icon\]
            "
        }
        if {![cequal [info commands ::ssj::encrypt:toggleP] ""]} {
            $bbox1 add -image [message::encrypted:icon $from] \
                    -helptype balloon -helptext [::msgcat::mc "Toggle encryption"] \
                    -height 24 -width 24 -relief link -bd 1 \
                    -command [list ssj::encrypt:toggleP $from]
            ssj::encrypted:trace "
                $mw.bottom.encrypted configure -image \[message::encrypted:icon $from\]
            " $from
        }
        pack $bbox1 -side left -fill x -padx 2m -pady 2m

        set bbox [ButtonBox $mw.bottom.buttons -spacing 10 -padx 10]
        $bbox add -text [::msgcat::mc "Reply"] -command [list message::send $mw]
        $bbox add -text [::msgcat::mc "Chat"] \
	    -command "chat::open_to_user {} [list $from]
                      destroy $mw"
        $bbox add -text [::msgcat::mc "Close"] -command [list destroy $mw]
        bind $mw <Control-Return> "ButtonBox::invoke $bbox 0"
        bind $mw <Escape> "ButtonBox::invoke $bbox 2"
        pack $bbox -side right -fill x -padx 2m -pady 2m
    } else {
        ButtonBox $mw.bottom.buttons -spacing 0 -padx 10
        $mw.bottom.buttons add -text [::msgcat::mc "Close"] -command [list destroy $mw]
        bind $mw <Return> "ButtonBox::invoke $mw.bottom.buttons 0"
        bind $mw <Escape> "ButtonBox::invoke $mw.bottom.buttons 0"
    }
    pack $mw.bottom.buttons -side right -fill x -padx 2m -pady 2m

    set sep [Separator::create $mw.sep -orient horizontal]
    pack $sep -pady 1m -fill x -side bottom

    frame $mw.frame
    pack $mw.frame -side top -fill both -expand yes -padx 2m -pady 2m

    if {$replyP} {
        set title [::msgcat::mc "Message from"]
    } else {
        set title [::msgcat::mc "Extras from"]
    }
    frame $mw.title
    label $mw.title.lab -text $title
    pack $mw.title.lab -side left
    menubutton $mw.title.mb -text $from -font $font -menu $mw.title.mb.menu
    subject_menu $mw.title.mb.menu $from message
    pack $mw.title.mb -side left
    pack $mw.title -side top -anchor w -in $mw.frame

    foreach tag [bind Menubutton] {
        if {[string first 1 $tag] >= 0} {
            regsub -all 1 $tag 3 new
            bind $mw.title.mb $new [bind Menubutton $tag]
        }
    }

    frame $mw.tspace
    pack $mw.tspace -side top -fill x -pady 0.5m -in $mw.frame

    frame $mw.rf
    grid columnconfigure $mw.rf 1 -weight 1

    set row 0

    if {$replyP || (![cequal $subject ""])} {
        label $mw.rf.lsubj -text [::msgcat::mc "Subject:"]
        entry $mw.rf.subj
        $mw.rf.subj insert 0 $subject
        if {[info tclversion] >= 8.4} {
            set bgcolor [lindex [$mw.rf.subj configure -background] 4]
            $mw.rf.subj configure -state readonly -readonlybackground $bgcolor
        } else {
            $mw.rf.subj configure -state disabled
        }
        grid $mw.rf.lsubj -row 0 -column 0 -sticky e
        grid $mw.rf.subj  -row 0 -column 1 -sticky ew
        incr row
    }

    pack $mw.rf -side top -anchor w -fill x -in $mw.frame

    frame $mw.rspace
    pack $mw.rspace -side top -fill x -in $mw.frame -pady 0.5m

    incr row
    set last $row
    #set x [list [read_file /tmp/alex/xdata.tcl]]
    #debugmsg message $x
    set namespaces {}
    foreach xe $x {
        lassign [process_x $mw.rf $xe $row $from $type $body $namespaces $replyP] row body namespaces
    }

    if {(!$replyP) && ($row == $last)} {
        destroy $mw
        return
    }

    ScrolledWindow $mw.rsw
    text $mw.rbody -width 60 -height 8 -wrap word
#       -yscrollcommand [list $mw.scroll set]
    #$mw.rbody insert 0.0 $body
    $mw.rbody tag configure emphasized -elide 1
    $mw.rbody tag configure nonemphasized -elide 0
    ::chat::add_emoteiconed_text $mw.rbody $body ""
    $mw.rbody configure -state disabled
    pack $mw.rsw -side top -fill both -expand yes -in $mw.frame
    pack $mw.rbody -side top -fill both -expand yes -in $mw.rsw
    $mw.rsw setwidget $mw.rbody
 
    if {$replyP} {
        button $mw.cite -text [::msgcat::mc "Quote"] -command [list message::quote $mw $body]
        pack $mw.cite -side top -anchor e -in $mw.frame

        frame $mw.f
        grid columnconfigure $mw.f 1 -weight 1

        #label $mw.f.lto -text To:
        Entry $mw.f.to -dropenabled 1 -droptypes {JID {}} \
                -dropcmd [list message::jiddropcmd]
        $mw.f.to insert 0 $from

        label $mw.f.lsubj -text [::msgcat::mc "Reply subject:"]
        entry $mw.f.subj
        regsub {(^Re: )*} $subject {Re: } subject
        $mw.f.subj insert 0 $subject

        #grid $mw.f.lto   -row 0 -column 0 -sticky e
        #grid $mw.f.to    -row 0 -column 1 -sticky ew
        grid $mw.f.lsubj -row 1 -column 0 -sticky e
        grid $mw.f.subj  -row 1 -column 1 -sticky ew

        pack $mw.f -side top -anchor w -fill x -in $mw.frame

        frame $mw.space
        pack $mw.f -side top -anchor w -fill x -in $mw.frame -pady 1m

        ScrolledWindow $mw.sw
        pack $mw.sw -in $mw.frame -side top -fill both -expand yes

        textUndoable $mw.body -width 60 -height 8 -wrap word
        pack $mw.body -side top -fill both -expand yes -in $mw.sw
        bind $mw.body <Control-z> { %W undo }
        bind $mw.body <Control-Z> { %W redo }
        bind $mw.body <Control-Return> "ButtonBox::invoke $bbox 0
                break"
        $mw.sw setwidget $mw.body

        focus $mw.body
    }
    incr msgid
}

proc message::quote {mw body} {
    regsub -all {(^|\n)} $body {\1> } body
    $mw.body insert 0.0 $body
    $mw.body insert insert "\n"
}

proc message::process_x {f x row from type body namespaces replyP} {
    jlib::wrapper:splitxml $x tag vars isempty chdata children

    set xmlns [jlib::wrapper:getattr $vars xmlns]
    lappend namespaces $xmlns
    switch -- $xmlns \
        jabber:x:roster {
            foreach child $children {
                process_x_roster $f $child $row $from
                incr row
            }
        } \
        jabber:x:oob {
            set row [process_x_oob $f $children $row $from]
        } \
        jabber:x:conference {
            if {![lcontain $namespaces $::NS(gc_user)]} {
                process_x_conference $f [jlib::wrapper:getattr $vars jid] $row
                incr row
                if {[cequal $body ""] && ![cequal $chdata ""]} {
                    set body $chdata
                }
            }
        } \
        jabber:x:signed {
            if {($replyP) && (![cequal $type error])} {
                process_x_signed $f $from $chdata $body
            }
        } \
        jabber:x:encrypted {
            if {($replyP) && (![cequal $type error])} {
                process_x_encrypted $f $from $chdata $body
            }
        } \
        jabber:x:data {
            process_x_data $f $from $x
        } \
        $::NS(gc_user) {
            foreach ch $children {
                jlib::wrapper:splitxml $ch tag1 vars1 isempty1 chdata1 children1

                switch -- $tag1 {
                    invite {
                        set inviter [jlib::wrapper:getattr $vars1 from]
                        if {![cequal $inviter ""]} {
                            process_x_conference $f $from $row
                            incr row

                            foreach c [jlib::connections] {
                                set name [roster::itemconfig $c \
                                              [roster::find_jid $c $inviter] \
                                              -name]
                                if {$name != ""} break
                            }
                            if {![cequal $name ""]} {
                                set inviter "$name ($inviter)"
                            }
                            set body [format [::msgcat::mc "%s invites you to conference room %s"] $inviter $from]

                            foreach ch1 $children1 {
                                jlib::wrapper:splitxml $ch1 tag2 vars2 isempty2 \
                                    chdata2 children2
                                if {[cequal $tag2 "reason"]} {
                                    append body [format [::msgcat::mc "\nReason is: %s" $chdata2]]
                                }
                            }
                        }
                    # TODO decline
                    }
                }
            }
        }

    return [list $row $body $namespaces]
}


proc message::process_x_roster {f x row from} {
    jlib::wrapper:splitxml $x tag vars isempty chdata children

    set jid [jlib::wrapper:getattr $vars jid]
    set name [jlib::wrapper:getattr $vars name]

    if {$name != ""} {
        set desc $name
    } else {
        set desc $jid
    }

    label $f.luser$row -text [::msgcat::mc "Attached user:"]
    set cb [button $f.user$row -text $desc \
                -command [list message::process_x_roster_user $jid \
                              "$from asked me to add you to my roster."]]
    if {![cequal $desc $jid]} {
        bind $cb <Any-Enter>  [list balloon::default_balloon $cb enter  %X %Y \
                                    $jid]
        bind $cb <Any-Motion> [list balloon::default_balloon $cb motion %X %Y \
                                    $jid]
        bind $cb <Any-Leave>  [list balloon::default_balloon $cb leave  %X %Y]
    }
    grid $f.luser$row -row $row -column 0 -sticky e
    grid $f.user$row  -row $row -column 1 -sticky ew
}

proc message::process_x_roster_user {jid body} {
    jlib::send_presence -to $jid -type subscribe -stat $body

    jlib::send_iq set \
        [jlib::wrapper:createtag query \
             -vars {xmlns jabber:iq:roster} \
             -subtags [list [jlib::wrapper:createtag item \
                                     -vars [list jid $jid]]]]
}

proc message::process_x_oob {f x row from} {
    set desc ""
    set url ""
    foreach item $x {
        jlib::wrapper:splitxml $item tag vars isempty chdata children   

        switch -- $tag {
            desc - url {
                set $tag $chdata
            }
        }
    }
    if {([cequal $desc ""]) || ([cequal $url ""])} {
        return $row
    }

    label $f.luser$row -text [::msgcat::mc "Attached file:"]
    set cb [button $f.user$row -text $url \
                -command [list ft::recv_file_dialog $from [list $url] $desc]]
    bind $cb <Any-Enter>  [list balloon::default_balloon $cb enter  %X %Y \
                                $desc]
    bind $cb <Any-Motion> [list balloon::default_balloon $cb motion %X %Y \
                                $desc]
    bind $cb <Any-Leave>  [list balloon::default_balloon $cb leave  %X %Y]
    grid $f.luser$row -row $row -column 0 -sticky e
    grid $f.user$row  -row $row -column 1 -sticky ew

    incr row

    return $row
}

proc message::process_x_conference {f conf row} {
    label $f.lconf$row -text [::msgcat::mc "Invited to:"]
    button $f.conf$row -text $conf -command [list join_group $conf -nick [get_group_nick $conf $::gr_nick]]
    
    grid $f.lconf$row -row $row -column 0 -sticky e
    grid $f.conf$row  -row $row -column 1 -sticky ew
}

set messageicon(badsigned)   [Bitmap::get [pixmap tkabber gpg-badsigned.gif]]
set messageicon(encrypted)   [Bitmap::get [pixmap tkabber gpg-encrypted.gif]]
set messageicon(signed)      [Bitmap::get [pixmap tkabber gpg-signed.gif]]
set messageicon(unencrypted) [Bitmap::get [pixmap tkabber gpg-unencrypted.gif]]
set messageicon(unsigned)    [Bitmap::get [pixmap tkabber gpg-unsigned.gif]]

proc message::process_x_signed {f from signature data} {
    global messageicon

# in case the sender didn't check the exit code from gpg...
    if {([cequal $signature ""]) \
            || ([cequal [info commands ::ssj::signed:input] ""])} {
        return
    }

    set lb [join [lrange [split $f .] 0 end-1] .].bottom.signed
    if {[winfo exists $lb]} {
        destroy $lb
    }

    ssj::signed:Label $lb \
       [ssj::signed:input $from $signature $data [::msgcat::mc "Message body"]]
    pack $lb -side left
}


proc message::process_x_encrypted {f from signature data} {
# we already deciphered it in client:message...

    global messageicon

    set lb [join [lrange [split $f .] 0 end-1] .].bottom.encrypted
    if {[winfo exists $lb]} {
        destroy $lb
    }

    Label $lb -image $messageicon(encrypted)
    pack $lb -side left
}

proc message::process_x_data {f from x} {
    data::draw_window [list $x] [list message::send_x_data $from]
}

proc message::send_x_data {to w restags} {
    #set f $w.fields
    #set restags [data::get_tags $f]
    jlib::send_msg $to -xlist $restags
    destroy $w
}

proc message::send_dialog {{to ""} {subject ""} {thread ""}} {
    global messageicon
    variable msgid
    variable message_dest_list

    set sendargs ""

    if {$thread != ""} {
        lappend sendargs -thread $thread
    }

    set mw .msg$msgid
    toplevel $mw

    if {$to != ""} {
        set title [format [::msgcat::mc "Send message to %s"] $to]
    } else {
        set title [::msgcat::mc "Send message"]
    }
    wm title $mw $title
    wm iconname $mw $title
    
    #frame $mw.subj
    #label $mw.subj.lab -text Subject:
    #entry $mw.subj.entry
    #$mw.subj.entry insert 0 $subject
    #pack $mw.subj.lab $mw.subj.entry -side left
    #pack $mw.subj -side top -anchor w

    frame $mw.bottom
    pack $mw.bottom -side bottom -fill x

    set bbox1 [ButtonBox $mw.bottom.buttons1 -spacing 0]
    if {![cequal [info commands ::ssj::sign:toggleP] ""]} {
        set b $bbox1.b[llength [winfo children $bbox1]]
        $bbox1 add -image [message::signed:icon] \
               -helptype balloon -helptext [::msgcat::mc "Toggle signing"] \
               -height 24 -width 24 -relief link -bd 1 \
               -command ssj::sign:toggleP
        ssj::signed:trace "
            $b configure -image \[message::signed:icon\]
        "
    }
    if {![cequal [info commands ::ssj::encrypt:toggleP] ""]} {
        set b $bbox1.b[llength [winfo children $bbox1]]
        $bbox1 add -image [message::encrypted:icon $to] \
               -helptype balloon -helptext [::msgcat::mc "Toggle encryption"] \
               -height 24 -width 24 -relief link -bd 1 \
               -command [list eval ssj::encrypt:toggleP $to]
        ssj::encrypted:trace "
            $b configure -image \[message::encrypted:icon $to\]
        " $to
    }
    pack $bbox1 -side left -fill x -padx 2m -pady 2m

    set bbox [ButtonBox $mw.bottom.buttons -spacing 10 -padx 10]
    $bbox add -text [::msgcat::mc "Send"]  -command [list message::send $mw]
    $bbox add -text [::msgcat::mc "Cancel"] -command [list destroy $mw]
    bind $mw <Control-Return> "ButtonBox::invoke $bbox 0"
    bind $mw <Escape> "ButtonBox::invoke $bbox 1"
    pack $bbox -side right -fill x -padx 2m -pady 2m

    set sep [Separator::create $mw.sep -orient horizontal]
    pack $sep -pady 1m -fill x -side bottom

    frame $mw.frame
    pack $mw.frame -side top -fill both -expand yes -padx 2m -pady 2m

    frame $mw.f
    grid columnconfigure $mw.f 1 -weight 1

    label $mw.f.lto -text [::msgcat::mc "To:"]
    #Entry $mw.f.to -dropenabled 1 -droptypes {JID {}} \
    #    -dropcmd [list message::jiddropcmd]
    ComboBox $mw.f.to -text $to \
	-dropenabled 1 -droptypes {JID {}} \
	-dropcmd [list message::jiddropcmd] \
	-values $message_dest_list
    #$mw.f.to insert 0 $to

    label $mw.f.lsubj -text [::msgcat::mc "Subject:"]
    entry $mw.f.subj
    $mw.f.subj insert 0 $subject

    grid $mw.f.lto   -row 0 -column 0 -sticky e
    grid $mw.f.to    -row 0 -column 1 -sticky ew
    grid $mw.f.lsubj -row 1 -column 0 -sticky e
    grid $mw.f.subj  -row 1 -column 1 -sticky ew

    pack $mw.f -side top -anchor w -fill x -in $mw.frame

    frame $mw.space
    pack $mw.f -side top -anchor w -fill x -in $mw.frame -pady 1m

    ScrolledWindow $mw.sw
    pack $mw.sw -in $mw.frame -side top -fill both -expand yes

    textUndoable $mw.body -width 60 -height 8 -wrap word
    pack $mw.body -side top -fill both -expand yes -in $mw.sw
    bind $mw.body <Control-z> { %W undo }
    bind $mw.body <Control-Z> { %W redo }
    bind $mw.body <Control-Return> "ButtonBox::invoke $bbox 0
            break"
    $mw.sw setwidget $mw.body

    if {$to != ""} {
        focus $mw.f.subj
    } else {
        focus $mw.f.to
    }

    incr msgid
}


proc message::send {mw} {
    variable message_dest_list

    set jid [$mw.f.to cget -text]
    set message_dest_list [update_combo_list $message_dest_list $jid 20]
    set custom::saved([namespace current]::message_dest_list) \
	$message_dest_list
    custom::store

    send_msg $jid -type normal \
        -subject [$mw.f.subj get] \
        -body [$mw.body get 1.0 {end -1 chars}]

    destroy $mw
}

proc message::send_msg {to args} {
    global loginconf

    array set params [list -xlist {}]
    array set params $args

    set command [list jlib::send_msg $to]
    set xs $params(-xlist)
    unset params(-xlist)

    if {[info exists params(-body)]} {
        set log_body $params(-body)
        foreach tag [list signed encrypted] {
            if {[cequal [info commands ::ssj::${tag}:output] ""]} {
                continue
            }

            if {[catch { ssj::${tag}:output $params(-body) $to } chdata]} {
                debugmsg message "ssj::${tag}:output: $chdata"
                return [list error ssj]
            }
            
            if {![cequal $chdata ""]} {
                lappend xs [jlib::wrapper:createtag x \
                                -vars "xmlns jabber:x:$tag" -chdata $chdata]
                if {[cequal $tag encrypted]} {
                    set params(-body) [::msgcat::mc "This message is encrypted."]
                }
            }
        }
    } else {
        set log_body ""
    }
    if {[info exists params(-subject)]} {
        set log_subject $params(-subject)
    } else {
        set log_subject ""
    }
    if {[llength $xs] > 0} {
        lappend command -xlist $xs
    }

    foreach {k v} [array get params] {
        lappend command $k $v
    }

    eval $command

    if {(![info exists params(-type)]) || ($params(-type) == "normal") } {
        ::message_archive::log_message \
                "$loginconf(user)@$loginconf(server)/$loginconf(resource)" $to $log_subject $log_body $xs
    }

    return [list success $xs]
}

proc message::jiddropcmd {target source pos op type data} {
    set jid [lindex $data 0]
    $target delete 0 end
    $target insert 0 $jid
}



proc message::show_subscribe_dialog {from x args} {
    variable msgid
    global font

    set status ""

    foreach {attr val} $args {
        switch -- $attr {
            -status {set status $val}
            default {debugmsg message "SHOW_SUBSCRIBE_MESSAGE: unknown attr $attr $val"}
        }
    }
    

    set mw .msg$msgid
    toplevel $mw -class Message

    set title [format [::msgcat::mc "Subscribe request from %s"] $from]
    wm title $mw $title
    wm iconname $mw $title

    set bbox [ButtonBox $mw.buttons -spacing 0 -padx 10 -default 0]
    # TODO: itemconfig $connid ...
    $bbox add -text [::msgcat::mc "Subscribe"] \
        -command "jlib::send_presence -to [list $from] -type subscribed
                  switch -- \
                      \[roster::itemconfig \[jlib::route [list $from]\] \
                          [list [tolower_node_and_domain $from]] -subsc\] {
                      {}   -
                      none -
                      from {
                          message::send_subscribe_dialog [list $from]
                      }
                  }
                  destroy [list $mw]"
        #-command [join [list [list jlib::send_presence -to $from \
        #                         -type subscribed] \
        #                   [list message::send_subscribe_dialog $from] \
        #                   [list destroy $mw]] \n]

    $bbox add -text [::msgcat::mc "Unsubscribe"] \
        -command [join [list [list jlib::send_presence -to $from \
                                  -type unsubscribed] \
                            [list destroy $mw]] \n]
    $bbox add -text [::msgcat::mc "Cancel"] -command [list destroy $mw]
    bind $mw <Return> "ButtonBox::invoke $bbox default"
    bind $mw <Escape> "ButtonBox::invoke $bbox 2"
    pack $bbox -side bottom -anchor e -padx 2m -pady 2m

    set sep [Separator::create $mw.sep -orient horizontal]
    pack $sep -pady 1m -fill x -side bottom

    frame $mw.frame
    pack $mw.frame -side top -fill both -expand yes -padx 2m -pady 2m

    frame $mw.subj
    label $mw.subj.lab -text [::msgcat::mc "Subscribe request from"]
    pack $mw.subj.lab -side left
    menubutton $mw.subj.mb -text $from -font $font -menu $mw.subj.mb.menu
    subject_menu $mw.subj.mb.menu $from subscribe
    pack $mw.subj.mb -side left
    pack $mw.subj -side top -anchor w -in $mw.frame

    foreach tag [bind Menubutton] {
        if {[string first 1 $tag] >= 0} {
            regsub -all 1 $tag 3 new
            bind $mw.subj.mb $new [bind Menubutton $tag]
        }
    }

    frame $mw.space
    pack $mw.space -side top -fill x -pady 0.5m -in $mw.frame

    ScrolledWindow $mw.sw
    pack $mw.sw -side top -fill both -expand yes -in $mw.frame

    text $mw.body -width 60 -height 8 -wrap word
    $mw.body insert 0.0 $status
    $mw.body configure -state disabled
    pack $mw.body -side bottom -fill both -expand yes -in $mw.sw
    $mw.sw setwidget $mw.body

    incr msgid
}

# TODO: connid
proc message::subject_menu {m jid type} {

    if {[winfo exists $m]} {
        destroy $m          
    }                       
    menu $m -tearoff 0
    $m add command -label [::msgcat::mc "Start chat"] \
        -command [list chat::open_to_user {} $jid]
    $m add command -label [::msgcat::mc "Send message..."] \
        -command [list message::send_dialog $jid]
    $m add command -label [::msgcat::mc "Invite to conference..."] \
        -command [list chat::invite_dialog $jid]
    $m add separator 
    $m add command -label [::msgcat::mc "Send users..."] \
        -command [list roster::send_users_dialog $jid]
    $m add command -label [::msgcat::mc "Send file..."] \
        -command [list ft::send_file_dialog $jid]
    $m add command -label [::msgcat::mc "Send file via Jidlink..."] \
        -command [list ftjl::send_file_dialog $jid]
    $m add separator
    if {$type == "message"} {
        $m add command -label [::msgcat::mc "Add user..."] \
            -command [list message::send_subscribe_dialog $jid]
    }
    $m add command -label [::msgcat::mc "Show info..."] \
        -command [list userinfo::open $jid]
    $m add command -label [::msgcat::mc "Show history..."] -command {} -state disabled
}   

# TODO: connid
proc message::send_subscribe_dialog {{to ""}} {
    variable msgid

    set mw .msg$msgid
    toplevel $mw

    if {[cequal $to ""]} {
        set title [::msgcat::mc "Send subscription"]
    } else {
        set title [format [::msgcat::mc "Send subscription to %s"] $to]
    }
    wm title $mw $title
    wm iconname $mw $title

    set bbox [ButtonBox $mw.buttons -spacing 0 -padx 10 -default 0]
    $bbox add -text [::msgcat::mc "Subscribe"] \
        -command [list message::send_subscribe $mw]
    $bbox add -text [::msgcat::mc "Cancel"] -command [list destroy $mw]
    bind $mw <Return> "ButtonBox::invoke $bbox default"
    bind $mw <Escape> "ButtonBox::invoke $bbox 1"
    pack $bbox -side bottom -anchor e -padx 2m -pady 2m

    set sep [Separator::create $mw.sep -orient horizontal]
    pack $sep -pady 1m -fill x -side bottom

    frame $mw.frame
    pack $mw.frame -side top -fill both -expand yes -padx 2m -pady 2m

    frame $mw.subj
    label $mw.subj.lab -text [::msgcat::mc "Send subscription to "]
    entry $mw.subj.entry
    $mw.subj.entry insert 0 $to
    pack $mw.subj.lab -side left
    pack $mw.subj.entry -side left -fill x -expand yes
    pack $mw.subj -side top -anchor w -fill x -expand yes -in $mw.frame

    frame $mw.space
    pack $mw.space -side top -fill x -in $mw.frame -pady 0.5m

    ScrolledWindow $mw.sw
    pack $mw.sw -side top -fill both -expand yes -in $mw.frame

    text $mw.body -width 60 -height 8 -wrap word
    $mw.body insert 0.0 "I would like to add you to my roster."
    pack $mw.body -side top -fill both -expand yes -in $mw.sw
    $mw.sw setwidget $mw.body

    focus $mw.subj.entry

    incr msgid
}

proc message::send_subscribe {mw} {
    set jid [$mw.subj.entry get]
    jlib::send_presence -to $jid -type subscribe \
        -stat [$mw.body get 1.0 end]
    jlib::send_iq set \
        [jlib::wrapper:createtag query \
             -vars {xmlns jabber:iq:roster} \
             -subtags [list [jlib::wrapper:createtag item \
                                 -vars [list jid $jid]]]] \
        -command "itemedit::show_dialog \
                      [list [jlib::route $jid] [tolower_node_and_domain $jid]] ;#"
    destroy $mw

}

package require md5

proc message::show_headline {from type subject body thread priority x {data {}}} {
    global tcl_platform
    variable headid
    variable headlines
    variable trees
    variable options

    global browsericon
    global font w

    set desc ""
    set url ""
    foreach extra $x {
        jlib::wrapper:splitxml $extra tag vars isempty chdata children
        if {![cequal [jlib::wrapper:getattr $vars xmlns] jabber:x:oob]} {
            continue
        }

        foreach item $children {
            jlib::wrapper:splitxml $item tag vars isempty chdata children   

            switch -- $tag {
                desc -
		url {
                    set $tag [string trim $chdata]
                }
            }
        }
    }
    if {[cequal $url ""]} {
        return
    }
    if {[cequal $subject ""]} {
	set subject $desc
    } else {
	if {$options(headlines,display_subject_only)} {
	    set desc $subject
	}
    }
    if {[cequal $subject ""]} {
	return
    }

    switch -- $options(headlines,multiple) {
        0 {
            set hw $w.headlines
            set title [::msgcat::mc "Headlines"]
            set tabtitle [::msgcat::mc "Headlines"]
        }

        1 {
            set hw $w.headlines_[jid_to_tag [set user [user_from_jid $from]]]
            set title [format [::msgcat::mc "%s Headlines"] $user]
            set tabtitle [node_from_jid $from]
        }

        default {
            set hw $w.headlines_[jid_to_tag $from]
            set title [format [::msgcat::mc "%s Headlines"] $from]
            set tabtitle [node_from_jid $from]/[resource_from_jid $from]
        }
    }
    if {[lsearch -exact $trees [set tw $hw.tree]] < 0} {
        lappend trees $tw
    }

    if {![winfo exists $hw]} {
        add_win $hw -title $title -tabtitle $tabtitle \
            -raisecmd "focus [list $hw.tree]
                       tab_set_updated [list $hw]" -class JBrowser

	if {![info exists options(seencolor)]} {
	    if {[cequal $tcl_platform(platform) unix]} {
		set options(seencolor) [option get $hw disabledForeground JBrowser]
	    } else {
		set options(seencolor) [option get $hw nscolor JBrowser]
	    }
	}
	if {![info exists options(unseencolor)]} {
	    set options(unseencolor) [option get $hw fill JBrowser]
	}

        set sw [ScrolledWindow $hw.sw]
	Tree $tw -deltax 16 -deltay 18
        $sw setwidget $tw

        pack $sw -side top -expand yes -fill both
        $tw bindText <ButtonPress-3> \
                [list message::headline_select_popup $hw]
        $tw bindText <Double-ButtonPress-1> \
                [list message::headline_action browse $hw]
        $tw bindText <Any-Enter>  \
                [list message::headline_balloon $hw enter  %X %Y]
        $tw bindText <Any-Motion> \
                [list message::headline_balloon $hw motion %X %Y]
        $tw bindText <Any-Leave>  \
                [list message::headline_balloon $hw leave  %X %Y]

        # HACK
        bind $tw.c <Return> \
            "message::headline_action browse $hw \[$tw selection get\]"
        bind $tw.c <4> {
            %W yview scroll -5 units
        }
        bind $tw.c <5> {
            %W yview scroll 5 units
        }
    }

    if {$options(headlines,multiple) > 1} {
        set text $subject
    } else {
        set text $from
    }
    set fnode [str2node $text]
    if {![$tw exists $fnode]} {
        $tw insert end root $fnode -text $text -open 1 \
            -image $browsericon(message) -font $font \
            -fill $options(seencolor) \
            -data [list type from text $text unseen 0]
    }

    if {($options(headlines,multiple) > 1) || ([cequal $subject $desc])} {
        set snode $fnode
    } else {
        set snode $fnode-subject-[str2node $subject]

        if {![$tw exists $snode]} {
            $tw insert end $fnode $snode -text $subject -open 1 \
                -image $browsericon(message) -font $font \
                -fill $options(seencolor) \
                -data [list type subject text $subject unseen 0]
        }
    }

    set anode $fnode-article-[incr headid]
    if {[$tw exists $anode]} {
        $tw delete $anode
    }
    array set props [list type article text $desc unseen 1 url $url body $body]
    array set props $data
    $tw insert end $snode $anode -text $desc -open 1 \
        -fill $options(seencolor) -font $font \
        -data [array get props]
    if {$props(unseen)} {
        $tw itemconfigure $anode -fill $options(unseencolor)
    }

    set headlines($anode) [list $from $type $subject $body $thread $priority $x]

    message::headline_update $tw $anode
    tab_set_updated $hw 1 message
}

proc message::str2node {string} {
    return [::md5::md5 [encoding convertto utf-8 $string]]
}

proc message::update_headlines_menu {menu num} {
    variable headline_send_jids

    set ind 3
    if {$num} {
	$menu delete $ind [expr $ind + $num - 1]
    }
    foreach jid $headline_send_jids {
	$menu insert $ind command \
	    -label [format [::msgcat::mc "Forward to %s"] $jid] \
	    -command "message::headline_forward3 $menu $jid \$::message::headwindow \$::message::headnode"
	incr ind
    }
}

if {[winfo exists [set m .h1popmenu]]} {
    destroy $m
}
menu $m -tearoff 0
$m add command -label [::msgcat::mc "Browse"] \
      -command {message::headline_action browse $::message::headwindow $::message::headnode}
$m add separator
$m add command -label [::msgcat::mc "Forward..."] \
      -command {message::headline_action forward $::message::headwindow $::message::headnode}
$m add separator
$m add command -label [::msgcat::mc "Toggle seen"] \
      -command {message::headline_action toggle $::message::headwindow $::message::headnode}
$m add command -label [::msgcat::mc "Delete"] \
      -command {message::headline_action delete $::message::headwindow $::message::headnode}
message::update_headlines_menu $m 0

    
if {[winfo exists [set m .h2popmenu]]} {
    destroy $m
}
menu $m -tearoff 0
$m add command -label [::msgcat::mc "Sort"] \
      -command {message::headline_action sort $::message::headwindow $::message::headnode}
$m add command -label [::msgcat::mc "Mark all seen"] \
      -command {message::headline_action markseen $::message::headwindow $::message::headnode}
$m add command -label [::msgcat::mc "Mark all unseen"] \
      -command {message::headline_action markunseen $::message::headwindow $::message::headnode}
$m add command -label [::msgcat::mc "Delete seen"] \
      -command {message::headline_action deleteseen $::message::headwindow $::message::headnode}
$m add command -label [::msgcat::mc "Delete all"] \
      -command {message::headline_action delete $::message::headwindow $::message::headnode}


proc message::headline_select_popup {hw node} {
    variable headwindow
    variable headnode

    $hw.tree selection set $node

    if {[catch { array set props [[set tw $hw.tree] itemcget $node -data] }]} {
        return
    }

    set headwindow $hw
    set headnode $node

    switch -- $props(type) {
        article {
           set hm .h1popmenu 
        }

        default {
           set hm .h2popmenu 
        }
    }

    tk_popup $hm [winfo pointerx .] [winfo pointery .]
}

proc message::headline_action {action hw node} {
    variable headlines
    variable options

    if {[catch { array set props [[set tw $hw.tree] itemcget $node -data] }]} {
        return
    }

    switch -glob -- $props(type)/$action {
        article/browse {
            browseurl $props(url)

            if {$props(unseen)} {
                set props(unseen) 0
                $tw itemconfigure $node -fill $options(seencolor) -data [array get props]
                message::headline_update $tw $node
            }
        }

	article/forward {
	     message::headline_forward .h1popmenu $tw $node
	}

        article/toggle {
            if {$props(unseen)} {
                set props(unseen) 0
                set myfill $options(seencolor)
            } else {
                set props(unseen) 1
                set myfill $options(unseencolor)
            }
            $tw itemconfigure $node -fill $myfill -data [array get props]
            message::headline_update $tw $node
        }

        article/markseen {
            set props(unseen) 0
            $tw itemconfigure $node -fill $options(seencolor) -data [array get props]
            message::headline_update $tw $node
        }

        article/markunseen {
            set props(unseen) 1
            $tw itemconfigure $node -fill $options(unseencolor) -data [array get props]
            message::headline_update $tw $node
        }

        */delete {
            set props(unseen) 0
            $tw itemconfigure $node -fill $options(seencolor) -data [array get props]
            message::headline_update $tw $node

            $tw delete $node
        }

        article/deleteseen {
            if {$props(unseen) == 0} {
		message::headline_action delete $hw $node
	    }
        }

	from/markseen -
	subject/markseen {
            foreach child [$tw nodes $node] {
		message::headline_action markseen $hw $child
            }
	}

	from/markunseen -
	subject/markunseen {
            foreach child [$tw nodes $node] {
		message::headline_action markunseen $hw $child
            }
	}

	from/deleteseen -
	subject/deleteseen {
	    if {$props(unseen) > 0} {
		foreach child [$tw nodes $node] {
		    message::headline_action deleteseen $hw $child
		}
	    } else {
		message::headline_action delete $hw $node
	    }
	}

        from/sort -
        subject/sort {
            set children {}
            foreach child [$tw nodes $node] {
                catch { unset props }
                array set props [$tw itemcget $child -data]

                lappend children [list $child $props(text)]
            }
            set neworder {}
            foreach child [lsort -index 1 $children] {
                lappend neworder [lindex $child 0]
            }
            $tw reorder $node $neworder

            foreach child [$tw nodes $node] {
                message::headline_action $action $hw $child
            }
        }

        default {
        }
    }
}

proc message::headline_update {tw node} {
    variable options

    for {set parent [$tw parent $node]} \
            {![cequal $parent root]} \
            {set parent [$tw parent $parent]} {
        set unseen 0

        foreach child [$tw nodes $parent] {
            catch { unset props }
            array set props [$tw itemcget $child -data]

            incr unseen $props(unseen)
        }

        catch { unset props }
        array set props [$tw itemcget $parent -data]
        set props(unseen) $unseen

        set text $props(text)
        set myfill $options(seencolor)
        if {$unseen > 0} {
            append text " ($unseen)"
            set myfill $options(unseencolor)
        }
        $tw itemconfigure $parent -text $text -fill $myfill \
                -data [array get props]
    }
}

proc message::headline_balloon {hw action X Y node} {
    if {[catch { array set props [$hw.tree itemcget $node -data] }]} {
        return
    }

    set width [expr {[winfo width $hw.tree] * 0.8}]
    if {$width < 400} {
	set width 400
    }

    switch -- $props(type) {
        article {
            if {![cequal $props(body) ""]} {
                balloon::default_balloon $hw:$node $action $X $Y $props(body) -width $width
            }
        }

        default {
        }
    }
}

proc message::save_headlines {} {
    global w

    variable options
    variable trees

    if {!$options(headlines,cache)} {
        return
    }

    if {[catch { open [set file1 ~/.tkabber/.headlines1.tcl] \
                      { WRONLY CREAT TRUNC } } fd]} {
        debugmsg message "unable to open $file: $fd"
        return
    }

    set code [catch {
        foreach tw $trees {
            save_headlines_aux $tw root $fd
        }
    } result]

    catch { close $fd }

    if {$code} {
        debugmsg message $result
        catch { file delete $file1 }

        return
    }

    set renameP 0
    if {![file exists [set file ~/.tkabber/.headlines.tcl]]} {
    } elseif {[file size $file] == 0} {
        catch { file delete -force $file }
    } else {
        set renameP 1
        catch { file rename -force $file \
                     [set file0 ~/.tkabber/.headlines0.tcl] }
    }

    if {![catch { file rename $file1 $file } result]} {
        return
    }

    debugmsg message "unable to rename $file1 to $file: $result"
    if {($renameP) && ([catch { file rename -force $file0 $file } result])} {
        debugmsg message "unable to rename $file0 back to $file: $result"
    }
    catch { file delete $file1 }

    return
}

proc message::save_headlines_aux {tw node fd} {
    variable headlines

    if {![winfo exists $tw]} {
        return
    }

    if {[llength [set children [$tw nodes $node]]] > 0} {
        foreach child $children {
            save_headlines_aux $tw $child $fd
        }
    } elseif {([info exists headlines($node)]) \
            && (![catch { array set props [$tw itemcget $node -data] }])} {
        puts $fd [concat [list [namespace current]::show_headline] \
                         $headlines($node) [list [array get props]]]
    }
}

proc message::restore_headlines {} {
    variable options

    if {($options(headlines,cache)) \
            && ([file exists [set file ~/.tkabber/.headlines.tcl]])} {
        source $file
    }

    return ""
}

proc message::headline_forward3 {menu to tw node} {
    variable headline_send_jids

    if {[catch { array set props [$tw.tree itemcget $node -data] } errmsg]} {
	return
    }

    message::send_msg $to -type headline \
	-subject $props(text) \
	-body $props(body) \
	-xlist [list [jlib::wrapper:createtag x \
			  -vars [list xmlns jabber:x:oob] \
			  -subtags [list [jlib::wrapper:createtag url \
			                      -chdata $props(url)] \
					 [jlib::wrapper:createtag desc \
					      -chdata $props(text)]]]]
    set len [llength $headline_send_jids]
    set headline_send_jids [update_combo_list $headline_send_jids $to 5]
    set custom::saved([namespace current]::headline_send_jids) \
	$headline_send_jids
    custom::store
    message::update_headlines_menu $menu $len
}

proc message::headline_forward2 {menu tw node} {
    global forward_hl
    variable headline_send_jids

    if {[catch { array set props [$tw itemcget $node -data] } errmsg]} {
	return
    }

    set len [llength $headline_send_jids]
    foreach choice [array names forward_hl] {
	if {$forward_hl($choice)} {
	    lassign $choice connid to
	    message::send_msg $to -type headline \
		-subject $props(text) \
		-body $props(body) \
		-xlist [list [jlib::wrapper:createtag x \
				  -vars [list xmlns jabber:x:oob] \
				  -subtags [list [jlib::wrapper:createtag url \
				                      -chdata $props(url)] \
						 [jlib::wrapper:createtag desc \
						      -chdata $props(text)]]]] \
		-connection $connid
	    set headline_send_jids [update_combo_list $headline_send_jids $to 5]
	}
    }
    set custom::saved([namespace current]::headline_send_jids) \
	$headline_send_jids
    custom::store
    message::update_headlines_menu $menu $len
}

proc message::headline_forward {menu tw node} {
    global forward_hl

    set gw .forward_headline
    catch { destroy $gw }

    set choices {}
    set balloons {}
    foreach c [jlib::connections] {
	foreach choice $roster::roster(jids,$c) {
	    if {![cequal $roster::roster(category,$c,$choice) conference]} {
		lappend choices [list $c $choice] [roster::get_label $c $choice]
		lappend balloons [list $c $choice] $choice
	    }
	}
    }
    if {[llength $choices] == 0} {
        MessageDlg ${gw}_err -aspect 50000 -icon info \
	    -message [::msgcat::mc "No users in roster..."] -type user \
	    -buttons ok -default 0 -cancel 0
        return
    }

    CbDialog $gw [::msgcat::mc "Forward headline"] \
	[list [::msgcat::mc "Send"] "message::headline_forward2 [list $menu] [list $tw] [list $node]
				     destroy $gw" \
	      [::msgcat::mc "Cancel"] [list destroy $gw]] \
	forward_hl $choices $balloons
}

proc message::signed:icon {} {
    global messageicon

    return [lindex [list $messageicon(unsigned) $messageicon(signed)] \
                   [ssj::signP]]
}

proc message::encrypted:icon {{arg ""}} {
    global messageicon

    return [lindex [list $messageicon(unencrypted) $messageicon(encrypted)] \
                   [ssj::encryptP $arg]]
}

hook::add finload_hook message::restore_headlines
hook::add quit_hook    message::save_headlines
