# $Id: roster.tcl,v 1.109 2003/12/13 21:35:26 aleksey Exp $

Tree .faketree
foreach {k v} [list background       White \
		    foreground       Black] {
    if {[cequal [set t$k [option get .faketree $k Tree]] ""]} {
	set t$k $v
    }
}
destroy .faketree
Button .fakebutton
foreach {k v} [list background       Gray \
		    activeBackground LightGray] {
    if {[cequal [set b$k [option get .fakebutton $k Button]] ""]} {
	set b$k $v
    }
}
destroy .fakebutton

option add *Roster.cbackground           $tbackground       widgetDefault
option add *Roster.groupindent           22                 widgetDefault
option add *Roster.jidindent             24                 widgetDefault
option add *Roster.jidmultindent         40                 widgetDefault
option add *Roster.subjidindent          34                 widgetDefault
option add *Roster.groupiconindent        2                 widgetDefault
option add *Roster.subgroupiconindent    22                 widgetDefault
option add *Roster.iconindent             3                 widgetDefault
option add *Roster.subitemtype            1                 widgetDefault
option add *Roster.subiconindent         13                 widgetDefault
option add *Roster.textuppad              1	            widgetDefault
option add *Roster.textdownpad            1	            widgetDefault
option add *Roster.linepad                2	            widgetDefault
option add *Roster.foreground            $tforeground       widgetDefault
option add *Roster.jidfill               $tbackground       widgetDefault
option add *Roster.jidhlfill             $bactiveBackground widgetDefault
option add *Roster.jidborder             $tbackground       widgetDefault
option add *Roster.groupfill             $bbackground       widgetDefault
option add *Roster.groupcfill            $bbackground       widgetDefault
option add *Roster.grouphlfill           $bactiveBackground widgetDefault
option add *Roster.groupborder           $tforeground       widgetDefault
option add *Roster.stalkerforeground     #663333            widgetDefault
option add *Roster.unavailableforeground #666666            widgetDefault
option add *Roster.dndforeground         #666633            widgetDefault
option add *Roster.xaforeground          #004d80            widgetDefault
option add *Roster.awayforeground        #004d80            widgetDefault
option add *Roster.availableforeground   #0066cc            widgetDefault
option add *Roster.chatforeground        #0099cc            widgetDefault

unset tbackground tforeground bbackground bactiveBackground

namespace eval roster {
    custom::defgroup Roster [::msgcat::mc "Roster options."] -group Tkabber
    custom::defvar show_only_online 0 \
	[::msgcat::mc "Show only online users in roster."] \
	-type boolean -group Roster -command roster::changed_only_online
    custom::defvar show_transport_icons 0 \
	[::msgcat::mc "Show native icons for transports/services in roster."] \
	-type boolean -group Roster -command roster::redraw_after_idle
    custom::defvar show_transport_user_icons 0 \
	[::msgcat::mc "Show native icons for contacts, connected to transports/services in roster."] \
	-type boolean -group Roster -command roster::redraw_after_idle
    custom::defvar options(nested) 0 \
	[::msgcat::mc "Enable nested roster groups."] \
	-type boolean -group Roster
    custom::defvar options(nested_delimiter) "::" \
	[::msgcat::mc "Default nested roster group delimiter."] \
	-type string -group Roster -command roster::redraw_after_idle
    custom::defvar options(chats_group) 0 \
	[::msgcat::mc "Add chats group in roster."] \
	-type boolean -group Roster -command roster::redraw_after_idle
    custom::defvar options(show_subscription) 0 \
	[::msgcat::mc "Show subscription type in roster item tooltips."] \
	-type boolean -group Roster
    custom::defvar options(show_conference_user_info) 0 \
	[::msgcat::mc "Show detailed info on conference room members in roster item tooltips."] \
	-type boolean -group Roster

    set menu_item_idx 0
    variable id ""
    set use_aliases 1

    set undef_group_name [::msgcat::mc "Undefined"]
    set chats_group_name [::msgcat::mc "Active Chats"]
}


proc roster::process_item {connid jid name groups subsc ask category subtype} {
    variable roster
    debugmsg roster "ROSTER_ITEM: $jid; $name; $groups; $subsc; $ask; $category; $subtype"

    set jid [tolower_node_and_domain $jid]


    if {$subsc != "remove"} {
	if {![lcontain $roster(jids,$connid) $jid]} {
	    lappend roster(jids,$connid) $jid
	}
	set roster(group,$connid,$jid)    $groups
	set roster(name,$connid,$jid)     $name
	set roster(subsc,$connid,$jid)    $subsc
	set roster(ask,$connid,$jid)      $ask
	set roster(category,$connid,$jid) $category
	set roster(subtype,$connid,$jid)  $subtype

	lassign [get_category_and_subtype $connid $jid] \
	    roster(ccategory,$connid,$jid) roster(csubtype,$connid,$jid)
	set roster(isuser,$connid,$jid) \
	    [cequal $roster(ccategory,$connid,$jid) user]
	catch {unset roster(cached_category_and_subtype,$connid,$jid)}
    } else {
	lvarpop roster(jids,$connid) [lsearch $roster(jids,$connid) $jid]

	unset roster(group,$connid,$jid)
	unset roster(name,$connid,$jid)
	unset roster(subsc,$connid,$jid)
	unset roster(ask,$connid,$jid)
	unset roster(category,$connid,$jid)
	unset roster(subtype,$connid,$jid)
	unset roster(ccategory,$connid,$jid)
	unset roster(csubtype,$connid,$jid)
	unset roster(isuser,$connid,$jid)
	catch {unset roster(cached_category_and_subtype,$connid,$jid)}
    }
    after cancel ::update_chat_titles
    after idle ::update_chat_titles
}

proc client:roster_item {connid jid name groups subsc ask category subtype} {
    roster::process_item \
	$connid $jid $name $groups $subsc $ask $category $subtype
}

proc client:roster_push {connid jid name groups subsc ask category subtype} {
    roster::process_item \
	$connid $jid $name $groups $subsc $ask $category $subtype
    roster::redraw_after_idle
}

proc client:roster_cmd {connid status} {
    debugmsg roster "ROSTER_CMD: $status"
    
    if {[cequal $status END_ROSTER]} {
	roster::redraw
    } else {
	global roster::roster
	#set roster::roster(jids) {}
	set roster::roster(jids,$connid) {}
	#roster::clear .roster
    }
}


proc roster::get_groups {connid} {
    variable roster

    set groups {}
    foreach jid $roster(jids,$connid) {
	set groups [concat $groups $roster(group,$connid,$jid)]
    }

    set groups [lrmdups $groups]
    return $groups
}

proc roster::itemconfig {connid jid args} {
    variable roster

    if {[llength $args] == 1} {
	lassign $args attr
	switch -- $attr {
	    -group    {set param group}
	    -name     {set param name}
	    -subsc    {set param subsc}
	    -ask      {set param ask}
	    -category {set param category}
	    -subtype  {set param subtype}
	    default {return}
	}
	if {[info exists roster($param,$connid,$jid)]} {
	    return $roster($param,$connid,$jid)
	} else {
	    return ""
	}
    } else {
	foreach {attr val} $args {
	    switch -- $attr {
		-group    {set param group}
		-name     {set param name}
		-subsc    {set param subsc}
		-ask      {set param ask}
		-category {set param category}
		-subtype  {set param subtype}
		default   {set param ""}
	    }
	    set roster($param,$connid,$jid) $val
	}
    }
}

proc roster::redraw {} {
    variable options
    variable roster
    variable config
    variable show_only_online
    variable aliases
    variable use_aliases
    variable show_transport_user_icons
    variable undef_group_name
    variable chats_group_name
    global rostericon

    clear .roster 0
    
    if {$use_aliases} {
	foreach jid [array names aliases] {
	    foreach alias $aliases($jid) {
		set ignore_jid($alias) ""
	    }
	}
    }

    set connections [jlib::connections]
    switch -- [llength $connections] {
	0 {
	    return
	}
	1 {
	    set draw_connection 0
	    set gindent 0
	}
	default {
	    set draw_connection 1
	    set gindent 1
	}
    }

    foreach connid $connections {
	if {$draw_connection} {
	    if {![info exists roster(collapsed,[list connid $connid])]} {
		set roster(collapsed,[list connid $connid]) 0
	    }
	    addline .roster group \
		[jlib::connection_jid $connid] [list connid $connid] 0
	    
	    if {$roster(collapsed,[list connid $connid])} {
		continue
	    }
	}

	if {![info exists roster(jids,$connid)]} {
	    continue
	}
	set groups {}
	array unset jidsingroup
	array unset jidsundergroup
	array unset groupsundergroup
	foreach jid $roster(jids,$connid) {
	    if {[info exists ignore_jid($jid)]} continue
	    if {![lempty $roster(group,$connid,$jid)]} {
		foreach group $roster(group,$connid,$jid) {
		    if {$options(nested)} {
			set sgroup [::textutil::splitx $group $options(nested_delimiter)]
		    } else {
			set sgroup [list $group]
		    }
		    lappend groups [join $sgroup "\u0000"]
		    lappend jidsingroup($sgroup) $jid
		    set deep [expr {[llength $sgroup] - 1}]
		    for {set i 0} {$i < $deep} {incr i} {
			set sgr [lrange $sgroup 0 $i]
			lappend groups [join $sgr "\u0000"]
			lappend jidsundergroup($sgr) $jid
			lappend groupsundergroup($sgr) $sgroup
			if {![info exists jidsingroup($sgr)]} {
			    set jidsingroup($sgr) {}
			}
		    }
		    if {![info exists jidsundergroup($sgroup)]} {
			set jidsundergroup($sgroup) {}
		    }
		    if {![info exists groupsundergroup($sgroup)]} {
			set groupsundergroup($sgroup) {}
		    }
		}
	    } else {
		set sgroup [list $undef_group_name]
		lappend jidsingroup($sgroup) $jid
		set groupsundergroup($sgroup) {}
		if {![info exists jidsundergroup($sgroup)]} {
		    set jidsundergroup($sgroup) {}
		}
	    }
	}
	set groups [lsort -unique -dictionary $groups]
	set ugroup [list $undef_group_name]
	if {[info exists jidsingroup($ugroup)]} {
	    lappend groups [join $ugroup "\u0000"]
	}
	if {$options(chats_group)} {
	    set cgroup [list $chats_group_name]
	    foreach chatid [array names chat::opened] {
		set cid [chat::get_connid $chatid]
		if {$cid == $connid} {
		    set jid [chat::get_jid $chatid]
		    lappend jidsingroup($cgroup) $jid
		    if {![info exists roster(isuser,$connid,$jid)]} {
			set roster(isuser,$connid,$jid) 1
			set roster(name,$connid,$jid) [chat::get_nick $jid chat]
			set roster(subsc,$connid,$jid) none
		    }
		}
	    }
	    if {[info exists jidsingroup($cgroup)]} {
		set groups [linsert $groups 0 [join $cgroup "\u0000"]]
	    }
	    set groupsundergroup($cgroup) {}
	    set jidsundergroup($cgroup) {}
	}
	foreach group $groups {
	    set group [split $group "\u0000"]
	    set gid [list $connid $group]
	    if {![info exists roster(show_offline,$gid)]} {
		if {$options(nested)} {
		    set gname [join $group $options(nested_delimiter)]
		} else {
		    set gname $group
		}
		if {[info exists roster(show_offline,$gname)]} {
		    set roster(show_offline,$gid) $roster(show_offline,$gname)
		} else {
		    set roster(show_offline,$gid) 0
		}
	    }
	}
	foreach group $groups {
	    set group [split $group "\u0000"]
	    set jidsingroup($group) [lrmdups $jidsingroup($group)]
	    set groupsundergroup($group) [lrmdups $groupsundergroup($group)]
	    set gid [list $connid $group]
	    if {![info exists roster(collapsed,$gid)]} {
		if {$options(nested)} {
		    set gname [join $group $options(nested_delimiter)]
		} else {
		    set gname $group
		}
		if {[info exists roster(collapsed,$gname)]} {
		    set roster(collapsed,$gid) $roster(collapsed,$gname)
		} else {
		    set roster(collapsed,$gid) 0
		}
	    }
	    set indent [expr {[llength $group] - 1}]
	    set collapsed 0
	    set show_offline 0
	    foreach undergroup $groupsundergroup($group) {
		if {$roster(show_offline,[list $connid $undergroup])} {
		    set show_offline 1
		    break
		}
	    }
	    for {set i 0} {$i < $indent} {incr i} {
		set sgr [list $connid [lrange $group 0 $i]]
		if {$roster(collapsed,$sgr)} {
		    set collapsed 1
		    break
		}
		if {$roster(show_offline,$sgr)} {
		    set show_offline 1
		}
	    }
	    incr indent $gindent
	    if {$collapsed} continue
	    set group_name "[lindex $group end]"
	    set online 0
	    set users 0
	    set not_users 0
	    set sub_jids 0
	    foreach jid [concat $jidsingroup($group) $jidsundergroup($group)] {
		if {$roster(isuser,$connid,$jid)} {
		    incr users
		    set status [get_user_aliases_status $connid $jid]
		    set jstat($jid) $status
		    if {$status != "unavailable"} {
			incr online
			set useronline($jid) 1
		    } else {
			set useronline($jid) 0
		    }
		} else {
		    incr not_users
		}
	    }
	    if {!$show_only_online || $show_offline || \
		    $roster(show_offline,$gid) || \
		    $online + $not_users + $sub_jids > 0} {
		if {$users} {
		    addline .roster group "$group_name ($online/$users)" \
			$gid $indent
		} else {
		    addline .roster group $group_name \
			$gid $indent
		}
	    }
	    if {!$roster(collapsed,$gid)} {
		set jid_names {}
		foreach jid $jidsingroup($group) {
		    lappend jid_names [list $jid [get_label $connid $jid]]
		}
		set jid_names [lsort -index 1 -dictionary $jid_names]
		foreach jid_name $jid_names {
		    lassign $jid_name jid name
		    if {$options(chats_group)} {
			set chatid [chat::chatid $connid $jid]
			if {[info exists chat::chats(messages,$chatid)] && \
				$chat::chats(messages,$chatid) > 0} {
			    append name " \[$chat::chats(messages,$chatid)\]"
			}
		    }
		    set cjid [list $connid $jid]
		    if {!$show_only_online || $show_offline || $roster(show_offline,$gid) || \
			    ![info exists useronline($jid)] || $useronline($jid)} {
			lassign [get_category_and_subtype $connid $jid] category type
			set jids [get_jids_of_user $connid $jid]
			set numjids [llength $jids]
			if {($numjids > 1) && ($config(subitemtype) > 0) && \
				$category == "user"} {
			    if {$config(subitemtype) & 1} {
				if {$category == "conference"} {
				    set numjids [expr {$numjids - 1}]
				}
				set label "$name ($numjids)"
			    } else {
				set label "$name"
			    }
			    addline .roster jid $label $cjid $indent $jids
			    changeicon .roster $cjid [get_jid_icon $connid $jid]
			    changeforeground .roster $cjid [get_jid_foreground $connid $jid]
			    
			    if {[info exists roster(collapsed,$cjid)] && \
				    !$roster(collapsed,$cjid)} {
				foreach subjid $jids {
				    set subjid_resource [resource_from_jid $subjid]
				    if {$subjid_resource != ""} {
					addline .roster jid2 \
					    $subjid_resource [list $connid $subjid] \
					    $indent \
					    [list $subjid]
					changeicon .roster \
					    [list $connid $subjid] [get_jid_icon $connid $subjid]
					changeforeground .roster \
					    [list $connid $subjid] [get_jid_foreground $connid $subjid]
				    }
				}
			    }
			} else {
			    if {$numjids <= 1 && $category == "user" && \
				    !$show_transport_user_icons} {
				#set user [node_and_server_from_jid $jid]
				set status $jstat($jid)
				
				if {([cequal $roster(subsc,$connid,$jid) from] || \
					 [cequal $roster(subsc,$connid,$jid) none]) && \
					$status == "unavailable"} {
				    set status stalker
				}
				addline .roster jid $name $cjid $indent \
				    $jids \
				    $rostericon(user,$status) \
				    $config(${status}foreground)
			    } else {
				addline .roster jid $name $cjid $indent $jids
				changeicon .roster $cjid [get_jid_icon $connid $jid]
				changeforeground .roster $cjid [get_jid_foreground $connid $jid]
			    }
			}
		    }
		}
	    }
	}
    }
    update_scrollregion .roster
}

proc roster::redraw_after_idle {args} {
    variable afterid

    if {[info exists afterid]} return

    if {![winfo exists .roster]} return

    set afterid [after idle {
	roster::redraw
	unset roster::afterid
    }]
}

proc roster::get_jids_of_user {connid user} {
    variable aliases
    variable use_aliases

    if {$use_aliases && [info exists aliases($user)]} {
	set jids [::get_jids_of_user $connid $user]
	foreach alias $aliases($user) {
	    set jids [concat $jids [::get_jids_of_user $connid $alias]]
	}
	return $jids
    } else {
	return [::get_jids_of_user $connid $user]
    }
}

proc roster::get_user_aliases_status {connid user} {
    variable aliases
    variable use_aliases

    if {$use_aliases && [info exists aliases($user)]} {
	set status [get_user_status $connid $user]

	foreach alias $aliases($user) {
	    set status [max_status $status [get_user_status $connid $alias]]
	}
	return $status
    } else {
	return [get_user_status $connid $user]
    }
}

proc roster::get_jid_foreground {connid jid} {
    lassign [get_category_and_subtype $connid $jid] category type

    switch -- $category {
	"" -
	user {
	    return [get_user_foreground $connid $jid]
	}
	conference {
	    set status [get_jid_status $connid $jid]
	    if {$status != "unavailable"} {
		return available
	    } else {
		return unavailable
	    }
	}
	service {
	    return [get_service_foreground $connid $jid $type]
	}
	default {
	    return ""
	}
    }
}

proc roster::get_jid_icon {connid jid} {
    variable roster
    global conferenceicon

    lassign [get_category_and_subtype $connid $jid] category type

    switch -- $category {
	"" {
	    return [get_user_icon $connid $jid]
	}
	user {
	    return [get_user_icon $connid $jid]
	}
	conference {
	    set show [get_jid_status $connid $jid]
	    if {$show != {unavailable}} {
		return $conferenceicon(available)
	    }
	    return $conferenceicon(unavailable)
	}
	service {
		return [get_service_icon $connid $jid $type]
	}
	default {
	    return ""
	}
    }
}

proc roster::get_service_icon {connid service type} {
    variable roster
    variable show_transport_icons
    global serviceicon rostericon browsericon

    if {$show_transport_icons} {
	switch -- $type {
	    jud {return $browsericon(jud)}
	    sms {return $serviceicon(sms)}
	}
	if {![cequal $roster(subsc,$connid,$service) none]} {
	    set status [get_user_status $connid $service]
	    set iconname serviceicon($type,$status)
	    if {[info exists $iconname]} {
		return [set $iconname]
	    } else {
		return $rostericon(user,$status)
	    }
	} else {
	    return $rostericon(user,stalker)
	}
    } else {
	if {![cequal $roster(subsc,$connid,$service) none]} {
	    return $rostericon(user,[get_user_status $connid $service])
	} else {
	    return $rostericon(user,stalker)
	}
    }
}

proc roster::get_service_foreground {connid service type} {
    variable roster

    switch -- $type {
	jud {return ""}
	}
    if {![cequal $roster(subsc,$connid,$service) none]} {
	return [get_user_status $connid $service]
    } else {
	return stalker
    }
}

proc roster::get_user_foreground {connid user} {
    variable roster
    global rostericon

    set status [get_user_aliases_status $connid $user]

    if {[info exists roster(subsc,$connid,$user)]} {
	if {([cequal $roster(subsc,$connid,$user) from] || \
		 [cequal $roster(subsc,$connid,$user) none]) && \
		$status == "unavailable"} {
	    return stalker
	} else {
	    return $status
	}
    } else {
	set user_without_resource [node_and_server_from_jid $user]
	if {([cequal $roster(subsc,$connid,$user_without_resource) from] || \
		 [cequal $roster(subsc,$connid,$user_without_resource) none]) && \
		$status == "unavailable"} {
	    return stalker
	} else {
	    return $status
	}
    }
}

proc roster::get_user_icon {connid user} {
    variable roster
    global rostericon serviceicon
    variable show_transport_user_icons

    set status [get_user_aliases_status $connid $user]

    if {[info exists roster(subsc,$connid,$user)]} {
	if {!([cequal $roster(subsc,$connid,$user) from] || \
		  [cequal $roster(subsc,$connid,$user) none]) || \
		$status != "unavailable"} {
	    if {$show_transport_user_icons} {
		set service [server_from_jid $user]
		lassign [get_category_and_subtype $connid $service] category type
		switch -- $type {
		    jud {return $browsericon(jud)}
		    sms {return $serviceicon(sms)}
		}
		set iconname serviceicon($type,$status)
		if {($type != "") && [info exists $iconname]} {
		    return [set $iconname]
		} else {
		    return $rostericon(user,$status)
		}
	    } else {
		return $rostericon(user,$status)
	    }
	} else {
	    return $rostericon(user,stalker)
	}
    } else {
	set user_without_resource [node_and_server_from_jid $user]
	if {!([cequal $roster(subsc,$connid,$user_without_resource) from] || \
		  [cequal $roster(subsc,$connid,$user_without_resource) none]) || \
		$status != "unavailable"} {
	    if {$show_transport_user_icons} {
		set service [server_from_jid $user]
		lassign [get_category_and_subtype $connid $service] category type
		switch -- $type {
		    jud {return $browsericon(jud)}
		    sms {return $serviceicon(sms)}
		}
		set iconname serviceicon($type,$status)
		if {($type != "") && [info exists $iconname]} {
		    return [set $iconname]
		} else {
		    return $rostericon(user,$status)
		}
	    } else {
		return $rostericon(user,$status)
	    }
	} else {
	    return $rostericon(user,stalker)
	}
    }
}

proc roster::on_change_jid_presence {connid jid} {
    variable roster
    
    set rjid [find_jid $connid $jid]
    debugmsg roster "$jid $rjid"

    if {$rjid != ""} {
	lassign [get_category_and_subtype $connid $rjid] category subtype
	
	if {$category == "user"} {
	    set_status [cconcat [get_label $connid $rjid] " " \
			    [::msgcat::mc "is now"] " " \
			    [::msgcat::mc [get_user_status $connid $rjid]]]
	    
	    hook::run on_change_user_presence_hook \
		[get_label $connid $rjid] [get_user_status $connid $rjid]
	}
    }
    redraw_after_idle
}

proc roster::find_jid {connid jid} {
    variable roster

    if {[info exists roster(category,$connid,$jid)]} {
	return $jid
    }

    lassign [heuristically_get_category_and_subtype $connid $jid] category subtype
    if {$category == "user"} {
	set rjid [node_and_server_from_jid $jid]
	if {[info exists roster(category,$connid,$rjid)]} {
	    lassign [get_category_and_subtype $connid $rjid] rcategory rsubtype
	    if {$category == $rcategory} {
		return $rjid
	    }
	}
    }
    return ""
}

proc roster::get_category_and_subtype {connid jid} {
    variable roster

    if {[info exists roster(cached_category_and_subtype,$connid,$jid)]} {
	return $roster(cached_category_and_subtype,$connid,$jid)
    }

    if {[info exists roster(category,$connid,$jid)]} {
	if {$roster(category,$connid,$jid) != ""} {
	    return [list $roster(category,$connid,$jid) $roster(subtype,$connid,$jid)]
	}
    }
    
    return [heuristically_get_category_and_subtype $connid $jid]
}

proc roster::heuristically_get_category_and_subtype {connid jid} {
    variable roster

    if {[info exists roster(cached_category_and_subtype,$connid,$jid)]} {
	return $roster(cached_category_and_subtype,$connid,$jid)
    }

    if {[node_from_jid $jid] == ""} {
	set category service

	set updomain [lindex [split [server_from_jid $jid] .] 0]
	if {[lcontain {aim icq irc jabber jud msn pager rss serverlist \
			   sms smtp yahoo} $updomain]} {
	    set subtype $updomain
	} elseif {[cequal icqv7 $updomain]} {
	    set subtype icq
	} elseif {[cequal gg $updomain]} {
	    set subtype x-gadugadu
	} elseif {[cequal pogoda $updomain]} {
	    set subtype x-weather
	} else {
	    set subtype ""
	}

	set roster(cached_category_and_subtype,$connid,$jid) [list $category $subtype]
	return [list $category $subtype]
    }

    if {[resource_from_jid $jid] == ""} {
	set updomain [lindex [split [server_from_jid $jid] .] 0]
	switch -- $updomain {
	    conference {
		set category conference
		set subtype ""
	    }
	    default {
		set category user
		set subtype ""
	    }
	}
	set roster(cached_category_and_subtype,$connid,$jid) [list $category $subtype]
	return [list $category $subtype]
    }
    set roster(cached_category_and_subtype,$connid,$jid) {user client}
    return {user client}
}

proc roster::changeicon {w jid icon} {
    set c $w.canvas
    set tag [jid_to_tag $jid]

    $c itemconfigure jid$tag&&icon -image $icon
}

proc roster::changeforeground {w jid fg} {
    variable config
    set c $w.canvas
    set tag [jid_to_tag $jid]
    $c itemconfigure jid$tag&&text -fill $config(${fg}foreground)
}

proc roster::create {w args} {
    variable roster
    variable config

    set c $w.canvas
    
    set width 150
    set height 100
    set popupproc {}
    set grouppopupproc {}
    foreach {attr val} $args {
	switch -- $attr {
	    -width {set width $val}
	    -height {set height $val}
	    -popup {set popupproc $val}
	    -grouppopup {set grouppopupproc $val}
	}
    }

    frame $w -relief sunken -borderwidth 1 -class Roster
    set sw [ScrolledWindow $w.sw]
    pack $sw -fill both -expand yes

    set config(background)  [option get $w cbackground Roster] 
    set config(groupindent) [option get $w groupindent Roster] 
    set config(jidindent)   [option get $w jidindent   Roster]
    set config(jidmultindent)   [option get $w jidmultindent   Roster]
    set config(jid2indent)  [option get $w subjidindent Roster]
    set config(groupiconindent) [option get $w groupiconindent  Roster]
    set config(subgroupiconindent) [option get $w subgroupiconindent  Roster]
    set config(iconindent)  [option get $w iconindent  Roster]
    set config(subitemtype)  [option get $w subitemtype  Roster]
    set config(subiconindent)  [option get $w subiconindent  Roster]
    set config(textuppad)   [option get $w textuppad   Roster]
    set config(textdownpad) [option get $w textdownpad Roster]
    set config(linepad)	    [option get $w linepad     Roster]
    set config(foreground)  [option get $w foreground  Roster]
    set config(jidfill)	    [option get $w jidfill     Roster]
    set config(jidhlfill)   [option get $w jidhlfill   Roster]
    set config(jidborder)   [option get $w jidborder   Roster]
    set config(jid2fill)    $config(jidfill)
    set config(jid2hlfill)  $config(jidhlfill)
    set config(jid2border)  $config(jidborder)
    set config(groupfill)   [option get $w groupfill   Roster]
    set config(groupcfill)  [option get $w groupcfill  Roster]
    set config(grouphlfill) [option get $w grouphlfill Roster]
    set config(groupborder) [option get $w groupborder Roster]
    set config(stalkerforeground) [option get $w stalkerforeground Roster]
    set config(unavailableforeground) [option get $w unavailableforeground Roster]
    set config(dndforeground) [option get $w dndforeground Roster]
    set config(xaforeground) [option get $w xaforeground Roster]
    set config(awayforeground) [option get $w awayforeground Roster]
    set config(availableforeground) [option get $w availableforeground Roster]
    set config(chatforeground) [option get $w chatforeground Roster]

    canvas $w.canvas -bg $config(background) \
	-highlightthickness 1 \
	-scrollregion {0 0 0 0} \
	-width $width -height $height

    $sw setwidget $c

    set roster($w,ypos) 1
    set roster($w,width) 0
    set roster($w,popup) $popupproc
    set roster($w,grouppopup) $grouppopupproc

    bindscroll $w.canvas

    if {$w == ".roster"} {
	DropSite::register .roster.canvas -dropcmd roster::dropcmd \
	    -droptypes {JID}
	DragSite::register .roster.canvas -draginitcmd roster::draginitcmd
    }

}


proc roster::addline {w type text jid indent {jids {}} {icon ""} {foreground ""}} {
    global font
    global rostericon
    variable roster
    variable config
    variable aliases
    variable use_aliases

    set c $w.canvas

    set tag [jid_to_tag $jid]

    set ypad 1
    set linespace [font metric $font -linespace]
    set lineheight [expr {$linespace + $ypad}]

    set uy $roster($w,ypos)
    set ly [expr {$uy + $lineheight + $config(textuppad) + \
		      $config(textdownpad)}]

    set levindent [expr $config(groupindent)*$indent]

    if {$type == "group" && [info exists roster(collapsed,$jid)] && \
	    $roster(collapsed,$jid)} {
	set rfill $config(groupcfill)
    } else {
	set rfill $config(${type}fill)
    }

    $c create rectangle [expr {1 + $levindent}] $uy 10000 $ly -fill $rfill \
	-outline $config(${type}border) \
	-tags [list jid$tag $type rect]


    if {[cequal $type jid]} {
	lassign $jid connid jjid
	set isuser \
	    [expr {![info exists roster(isuser,$connid,$jjid)] || $roster(isuser,$connid,$jjid)}]

	set y [expr {($uy + $ly)/2}]
	set x [expr {$config(iconindent) + $levindent}]

	if {$icon == ""} {
	    $c create image $x $y -image $rostericon(user,unavailable) \
		-anchor w \
		-tags [list jid$tag $type icon]
	} else {
	    $c create image $x $y -image $icon \
		-anchor w \
		-tags [list jid$tag $type icon]
	}
	#set jids [get_jids_of_user $jid]
	if {[llength $jids] > 1} {
	    if {[info exists roster(collapsed,$jid)] && !$roster(collapsed,$jid)} {
		set jid_state opened
	    } else {
		set roster(collapsed,$jid) 1
		set jid_state closed
	    }
	    if {$config(subitemtype) > 0} {
		if {($config(subitemtype) & 2) && $isuser} {
		    set y [expr {($uy + $ly)/2}]
		    set x [expr {$config(subgroupiconindent) + $levindent}]
		    $c create image $x $y -image $rostericon(group,$jid_state) -anchor w \
			-tags [list jid$tag $type group]
		}
	    }
	} else {
	    set roster(collapsed,$jid) 1
	}
    } elseif {[cequal $type jid2]} {
	#set jids [get_jids_of_user $jid]
	set y [expr {($uy + $ly)/2}]
	set x [expr {$config(subiconindent) + $levindent}]

	$c create image $x $y -image $rostericon(user,unavailable) -anchor w \
	    -tags [list jid$tag $type icon]
    } elseif {[cequal $type group]} {
	set y [expr {($uy + $ly)/2}]
	set x [expr {$config(groupiconindent) + $levindent}]
	if {[info exists roster(collapsed,$jid)] && $roster(collapsed,$jid)} {
	    set group_state closed
	} else {
	    set group_state opened
	}
	$c create image $x $y -image $rostericon(group,$group_state) -anchor w \
	    -tags [list jid$tag $type icon]
    }

    if {([cequal $type jid]) && ($config(subitemtype) > 0) && ($config(subitemtype) & 2)} {
	#set jids [get_jids_of_user $jid]
	if {$isuser && ([llength $jids] > 1)} {
	    set x [expr {$config(jidmultindent) + $levindent}]
	} else {
	    set x [expr {$config(jidindent) + $levindent}]
	}
    } else {
	set x [expr {$config(${type}indent) + $levindent}]
    }
    incr uy $config(textuppad)

    if {$foreground == ""} {
	if {[cequal $type jid] || [cequal $type jid2]} {
	    set foreground $config(unavailableforeground)
	} else {
	    set foreground $config(foreground)
	}
    }
    $c create text $x $uy -text $text -anchor nw -font $font \
	-fill $foreground -tags [list jid$tag $type text]

    set roster($w,width) [max $roster($w,width) \
			      [expr {$x + [font measure $font $text]}]]


    $c bind jid$tag <Any-Enter> \
	[list $c itemconfig jid$tag&&rect -fill $config(${type}hlfill)]
    $c bind jid$tag <Any-Leave> \
	[list $c itemconfig jid$tag&&rect -fill $rfill]

    set doubledjid  [double% $jid]


    $c bind jid$tag&&jid <Double-Button-1> \
	[list roster::jid_doubleclick $doubledjid]
    $c bind jid$tag&&jid2 <Double-Button-1> \
	[list roster::jid_doubleclick $doubledjid]

    set roster($w,ypos) [expr {$ly + $config(linepad)}]

    if {[cequal $type jid] || [cequal $type jid2]} {
	set doubledjids [double% $jids]
	$c bind jid$tag <Any-Enter> \
	    +[list eval balloon::set_text \
		  \[roster::jids_popup_info [list $doubledjid] [list $doubledjids]\]]

	$c bind jid$tag <Any-Motion> \
	    [list eval balloon::on_mouse_move \
		 \[roster::jids_popup_info [list $doubledjid] [list $doubledjids]\] %X %Y]




	#if {[llength $jids] > 0} {
	#    set doubledjids [double% $jids]
	#    $c bind jid$tag <Any-Enter> \
	#	+[list eval balloon::set_text \
	#	      \[roster::jids_popup_info [list $doubledjids]\]]
	#
	#    $c bind jid$tag <Any-Motion> \
	#	[list eval balloon::on_mouse_move \
	#	     \[roster::jids_popup_info [list $doubledjids]\] %X %Y]
	#} else {
	#    set jids [list $jid]
	#    if {$use_aliases && [info exists aliases($jid)]} {
	#	set jids [concat $jids $aliases($jid)]
	#    }
	#    set doubledjids [double% $jids]
	#    $c bind jid$tag <Any-Enter> \
	#	+[list eval balloon::set_text \
	#	      \[roster::jids_popup_info [list $doubledjids]\]]
	#
	#    $c bind jid$tag <Any-Motion> \
	#	[list eval balloon::on_mouse_move \
	#	     \[roster::jids_popup_info [list $doubledjids]\] %X %Y]
	#}
	
	$c bind jid$tag <Any-Leave> {+
	    balloon::destroy
	}
	
	if {![cequal $roster($w,popup) {}]} {
	    $c bind jid$tag <3> [list $roster($w,popup) $doubledjid]
	}
    } else {
	if {$w == ".roster"} {
	    $c bind jid$tag&&group <Button-1> \
		[list roster::group_click $doubledjid]
	}

	if {![cequal $roster($w,grouppopup) {}]} {
	    $c bind jid$tag&&group <3> \
		[list $roster($w,grouppopup) $doubledjid]
	}
    }

    if {[cequal $type jid]} {
	if {$isuser \
		&& ([llength $jids] > 1)} {
	    if {$w == ".roster"} {
		$c bind jid$tag <Button-1> \
		    [list roster::user_singleclick $jid]
	    }
	}
    }
    #$c configure -scrollregion [list 0 0 $roster($w,width) $roster($w,ypos)]
}


proc roster::clear {w {updatescroll 1}} {
    variable roster

    set c $w.canvas

    $c delete rect||icon||text||group

    set roster($w,ypos) 1
    set roster($w,width) 0
    if {$updatescroll} {
	update_scrollregion $w
    }
}

proc roster::clean {} {
    variable roster

    array unset roster jids,*
    array unset roster group,*
    array unset roster name,*
    array unset roster subsc,*
    array unset roster ask,*
    array unset roster category,*
    array unset roster subtype,*
    array unset roster ccategory,*
    array unset roster csubtype,*
    array unset roster isuser,*
    array unset roster cached_category_and_subtype,*
    redraw
}

proc roster::clean_connection {connid} {
    variable roster

    array unset roster jids,$connid
    array unset roster group,$connid,*
    array unset roster name,$connid,*
    array unset roster subsc,$connid,*
    array unset roster ask,$connid,*
    array unset roster category,$connid,*
    array unset roster subtype,$connid,*
    array unset roster ccategory,$connid,*
    array unset roster csubtype,$connid,*
    array unset roster isuser,$connid,*
    array unset roster cached_category_and_subtype,$connid,*

    redraw
}

proc roster::update_scrollregion {w} {
    variable roster
    set c $w.canvas
    $c configure -scrollregion [list 0 0 $roster($w,width) $roster($w,ypos)]
}

proc roster::jid_doubleclick {id} {
    variable roster

    lassign $id connid jid
    lassign [get_category_and_subtype $connid $jid] category subtype

    switch -- $category {
	conference {
	    global gr_nick
	
	    client:presence $connid $jid "" "" {}
	    join_group $jid \
		-nick [get_group_nick $jid $gr_nick] \
		-connection $connid
	}
	user -
	default {
	    if {[cequal $chat::options(default_message_type) chat]} {
		chat::open_to_user $connid $jid
	    } else {
		message::send_dialog $jid
	    }
	}
    }
}

proc roster::group_click {gid} {
    variable roster

    set roster(collapsed,$gid) [expr {!$roster::roster(collapsed,$gid)}]
    redraw
}

proc roster::jids_popup_info {id jids} {
    variable use_aliases

    lassign $id connid jid

    if {$jids == {}} {
	if {$use_aliases && [info exists aliases($jid)]} {
	    set jids [concat [list $jid] $aliases($jid)]
	} else {
	    set jids [list $jid]
	}
    }

    set text {}
    set i 0
    foreach j $jids {
	append text "\n[roster::user_popup_info $connid $j $i]"
	incr i
    }
    set text [string trimleft $text "\n"]
    return $text
}

proc roster::user_popup_info {connid user i} {
    variable options
    variable roster
    variable user_popup_info

    lassign [get_category_and_subtype $connid $user] category subtype
    set bare_user [node_and_server_from_jid $user]
    lassign [get_category_and_subtype $connid $bare_user] category1 subtype1

    set name $user
    switch -- $category {
	conference {
	    set status [get_jid_status $connid $user]
	    set desc ""
	}
	user -
	default {
	    set status [get_user_status $connid $user]
	    set desc   [get_user_status_desc $connid $user]
	    if {[cequal $category1 conference] && $i > 0} {
		if {$options(show_conference_user_info)} {
		    set name "     [resource_from_jid $user]"
		} else {
		    set name "\t[resource_from_jid $user]"
		}
	    }
	}
    }

    if {(![cequal $status [string tolower $desc]]) && (![cequal $desc ""])} {
	append status " ($desc)"
    }

    if {($options(show_subscription) && [info exists roster(subsc,$connid,$bare_user)]) &&
	    !([cequal $category1 conference] && [cequal $category user])} {
	set subsc " \[$roster(subsc,$connid,$bare_user)\]"
    } else {
	set subsc ""
    }

    set user_popup_info "$name$subsc: $status"

    if {!([cequal $category1 conference] && $i > 0) || $options(show_conference_user_info)} {
	hook::run roster_user_popup_info_hook \
	    [namespace which -variable user_popup_info] $connid $user
    }

    return $user_popup_info
}


proc roster::get_label {connid jid} {
    variable roster

    if {[lempty $roster(name,$connid,$jid)]} {
	return $jid
    } else {
	return $roster(name,$connid,$jid)
    }
}

proc roster::switch_only_online {} {
    variable show_only_online
    set show_only_online [expr !$show_only_online]
    changed_only_online
}

proc roster::changed_only_online {args} {
    global bbox
    global toolbaricon
    variable show_only_online

    if {![info exists bbox] || ![winfo exists $bbox]} return

    if {$show_only_online} {
	$bbox itemconfigure 3 -image $toolbaricon(online)
    } else {
	$bbox itemconfigure 3 -image $toolbaricon(offline)
    }
    redraw
}

proc roster::is_online {connid jid} {
    if {[is_user $connid $jid]} {
	switch -- [get_user_aliases_status $connid $jid] {
	    unavailable {return 0}
	    default {return 1}
	}
    } else {
	return 1
    }
}

proc roster::is_user_online {connid jid} {
    return [expr {![cequal [get_user_aliases_status $connid $jid] unavailable]}]
}

proc roster::is_user {connid jid} {
    return [cequal [lindex [get_category_and_subtype $connid $jid] 0] "user"]
}


proc roster::item_to_xml {connid jid} {
    variable roster

    set grtags {}
    foreach group $roster(group,$connid,$jid) {
	lappend grtags [jlib::wrapper:createtag group -chdata $group]
    }

    set vars [list jid $jid]

    if {$roster(name,$connid,$jid) != ""} {
	lappend vars name $roster(name,$connid,$jid)
    }

    if {$roster(category,$connid,$jid) != ""} {
	lappend vars category $roster(category,$connid,$jid)
	if {$roster(subtype,$connid,$jid) != ""} {
	    lappend vars type $roster(subtype,$connid,$jid)
	}
    }

    return [jlib::wrapper:createtag item \
		-vars $vars \
		-subtags $grtags]
}

proc roster::send_item {connid jid} {
    jlib::send_iq set \
	[jlib::wrapper:createtag query \
	     -vars {xmlns jabber:iq:roster} \
	     -subtags [list [roster::item_to_xml $connid $jid]]] \
	-connection $connid
}

proc roster::remove_item {connid jid} {
    jlib::send_iq set \
	[jlib::wrapper:createtag query \
	     -vars {xmlns jabber:iq:roster} \
	     -subtags [list [jlib::wrapper:createtag item \
				 -vars [list jid $jid \
					    subscription remove]]]] \
	-connection $connid
	
    jlib::send_presence -to $jid -type unsubscribe -connection $connid

    lassign [get_category_and_subtype $connid $jid] category subtype

    if {$category == "service"} {
	jlib::send_iq set \
	    [jlib::wrapper:createtag query \
		-vars {xmlns jabber:iq:register} \
		-subtags [list [jlib::wrapper:createtag remove]]] \
	    -to $jid \
	    -connection $connid
    }
}

proc roster::remove_item_dialog {connid jid} {
    set res [MessageDlg .remove_item -aspect 50000 -icon question -type user \
	-buttons {yes no} -default 0 -cancel 1 \
	-message [format [::msgcat::mc "Are you sure to remove %s from roster?"] $jid]]
    if {$res == 0} {
	roster::remove_item $connid $jid
    }
}


proc roster::update_chat_activity {args} {
    variable options
    if {$options(chats_group)} {
	redraw_after_idle
    }
}

hook::add open_chat_post_hook roster::redraw_after_idle
hook::add close_chat_post_hook roster::redraw_after_idle
hook::add draw_message_hook roster::update_chat_activity
hook::add raise_chat_tab_hook roster::update_chat_activity


###############################################################################

proc roster::dropcmd {target source X Y op type data} {
    variable roster
    debugmsg roster "$target $source $X $Y $op $type $data"

    set c .roster.canvas

    set x [expr {$X-[winfo rootx $c]}]
    set y [expr {$Y-[winfo rooty $c]}]
    set xc [$c canvasx $x]
    set yc [$c canvasy $y]

    set tags [$c gettags [lindex [$c find closest $xc $yc] 0]]
    if {[lcontain $tags group]} {
	set tag [crange [lindex $tags 0] 3 end]
	set cgr [tag_to_jid $tag]
	lassign $cgr connid gr
	if {$connid == "connid"} {
	    set connid $gr
	    set gr {}
	}
    } else {
	set connid [jlib::route ""]
	set gr {}
    }

    debugmsg roster "GG: $gr; $tags"

    lassign $data jid category type name version

    if {![lcontain $roster(jids,$connid) $jid]} {
	if {$gr != {}} {
	    itemconfig $connid $jid -category $category -subtype $type \
		-name $name -group [list $gr]
	} else {
	    itemconfig $connid $jid -category $category -subtype $type \
		-name $name -group {}
	}
	lassign [get_category_and_subtype $connid $jid] ccategory ctype
	switch -- $ccategory {
	    user {
		jlib::send_presence -to $jid -type subscribe -connection $connid
	    }
	}
    } else {
	set groups [itemconfig $connid $jid -group]
	if {$gr != ""} {
	    lappend groups $gr
	    set groups [lrmdups $groups]
	    debugmsg roster $groups
	}
	itemconfig $connid $jid -category $category -subtype $type \
	    -name $name -group $groups
    }
    send_item $connid $jid
}

proc roster::draginitcmd {target x y top} {
    variable roster
    debugmsg roster "$target $x $y $top"

    balloon::destroy
    set c .roster.canvas

    set tags [$c gettags current]
    if {[lcontain $tags jid]} {
	set tag [crange [lindex $tags 0] 3 end]
	set cjid [tag_to_jid $tag]
	lassign $cjid connid jid

	set data [list $jid \
		      [itemconfig $connid $jid -category] \
		      [itemconfig $connid $jid -subtype] \
		      [itemconfig $connid $jid -name] {}]

	debugmsg roster $data
	return [list JID {copy} $data]
    } else {
	return {}
    }
}

###############################################################################
###############################################################################

proc roster::popup_menu {id} {
    global curuser

    lassign $id connid jid
    set curuser $jid

    lassign [get_category_and_subtype $connid $jid] category subtype

    switch -- $category {
	conference {set menu [conference_popup_menu $connid $jid]}
	user {set menu [create_user_menu $connid $jid]}
	service {set menu [service_popup_menu $connid $jid]}
	default {set menu [jid_popup_menu $connid $jid]}
    }

    tk_popup $menu [winfo pointerx .] [winfo pointery .]
}


proc roster::group_popup_menu {id} {
    variable options

    lassign $id connid name
    if {$options(nested)} {
	set name [join $name $options(nested_delimiter)]
    }
    if {$connid != "connid"} {
	tk_popup [create_group_popup_menu $connid $name] \
	    [winfo pointerx .] [winfo pointery .]
    }
}


proc roster::groupchat_popup_menu {id} {
    lassign $id connid jid
    tk_popup [create_groupchat_user_menu $connid $jid] \
	[winfo pointerx .] [winfo pointery .]
}

proc roster::add_menu_item {m label command jids} {
    variable menu_item_idx
    if {[llength $jids] == 0} {
	$m add command -label $label -command $command
    } elseif {[llength $jids] == 1} {
	set curuser $jids
	set com [subst -nobackslashes -nocommands $command]
	$m add command -label $label -command $com
    } else {
	set m2 [menu $m.[incr menu_item_idx] -tearoff 0]
	$m add cascad -label $label -menu $m2
	foreach jid $jids {
	    set curuser [list $jid]
	    set com [subst -nobackslashes -nocommands $command]
	    $m2 add command -label $jid -command $com
	}
    }
}

proc roster::collapse_item {cjid} {
    variable roster
    variable id

    set id ""
    set roster(collapsed,$cjid) [expr !$roster::roster(collapsed,$cjid)]
    redraw
}

proc roster::user_singleclick {cjid} {
    variable id

    lassign $cjid connid jid
    if {$id == ""} {
	set id [after 300 [list roster::collapse_item $cjid]]
    } else {
	after cancel $id
	set id ""
    }
}

proc roster::create_user_menu {connid user} {
    # TODO: connid
    set jids [get_jids_of_user $connid $user]
    if {[winfo exists [set m .jidpopupmenu]]} {
	destroy $m
    }
    menu $m -tearoff 0
    add_menu_item $m [::msgcat::mc "Start chat"] \
	"chat::open_to_user [list $connid] \$curuser" $jids
    add_menu_item $m [::msgcat::mc "Send message..."] \
	{message::send_dialog $curuser} $jids
    add_menu_item $m [::msgcat::mc "Invite to conference..."] \
	{chat::invite_dialog $curuser} $jids
    $m add command -label [::msgcat::mc "Resubscribe"] -command {
	jlib::send_presence -to $curuser -type subscribe
    }

    hook::run roster_create_user_menu_hook $m $connid $jids

    $m add separator
    add_custom_presence_menu $m $jids
    add_menu_item $m [::msgcat::mc "Send users..."] \
	{roster::send_users_dialog $curuser} $jids
    add_menu_item $m [::msgcat::mc "Send file..."] \
	{ft::send_file_dialog $curuser} $jids
    add_menu_item $m [::msgcat::mc "Send file via Jidlink..."] \
	{ftjl::send_file_dialog $curuser} $jids


    $m add separator
    add_menu_item $m [::msgcat::mc "Show info..."] \
	"userinfo::open \$curuser -connection [list $connid]" $jids
    $m add command -label [::msgcat::mc "Show history..."] \
	-command {logger::show_log $curuser}
    $m add separator
    $m add command -label [::msgcat::mc "Edit item..."] \
	-command [list itemedit::show_dialog $connid $user]
    $m add command -label [::msgcat::mc "Edit security..."] \
	-command {ssj::prefs $curuser} \
	-state [lindex {normal disabled} \
		    [cequal [info commands ::ssj::prefs] ""]]
    $m add separator
    $m add command -label [::msgcat::mc "Remove..."] \
	-command [list roster::remove_item_dialog $connid $user]

    return $m
}


set rostericon(user,available)   [Bitmap::get [pixmap roster available.gif]]
set rostericon(user,away)        [Bitmap::get [pixmap roster available-away.gif]]
set rostericon(user,chat)        [Bitmap::get [pixmap roster available-chat.gif]]
set rostericon(user,dnd)         [Bitmap::get [pixmap roster available-dnd.gif]]
set rostericon(user,xa)          [Bitmap::get [pixmap roster available-xa.gif]]
set rostericon(user,unavailable) [Bitmap::get [pixmap roster unavailable.gif]]
catch {
set rostericon(user,invisible)   [Bitmap::get [pixmap roster invisible.gif]]
}
set rostericon(user,stalker)     [Bitmap::get [pixmap roster stalker.gif]]
set rostericon(user,error)       [Bitmap::get [pixmap roster unavailable.gif]]

set serviceicon(aim,available)   [Bitmap::get [pixmap services aim_online.xpm]]
set serviceicon(aim,chat)        [Bitmap::get [pixmap services aim_chat.xpm]]
set serviceicon(aim,away)        [Bitmap::get [pixmap services aim_away.xpm]]
set serviceicon(aim,xa)          [Bitmap::get [pixmap services aim_xa.xpm]]
set serviceicon(aim,dnd)         [Bitmap::get [pixmap services aim_dnd.xpm]]
set serviceicon(aim,unavailable) [Bitmap::get [pixmap services aim_offline.xpm]]

set serviceicon(icq,available)   [Bitmap::get [pixmap services icq_online.xpm]]
set serviceicon(icq,chat)        [Bitmap::get [pixmap services icq_chat.xpm]]
set serviceicon(icq,away)        [Bitmap::get [pixmap services icq_away.xpm]]
set serviceicon(icq,xa)          [Bitmap::get [pixmap services icq_xa.xpm]]
set serviceicon(icq,dnd)         [Bitmap::get [pixmap services icq_dnd.xpm]]
set serviceicon(icq,unavailable) [Bitmap::get [pixmap services icq_offline.xpm]]

set serviceicon(msn,available)   [Bitmap::get [pixmap services msn_online.xpm]]
set serviceicon(msn,chat)        [Bitmap::get [pixmap services msn_chat.xpm]]
set serviceicon(msn,away)        [Bitmap::get [pixmap services msn_away.xpm]]
set serviceicon(msn,xa)          [Bitmap::get [pixmap services msn_xa.xpm]]
set serviceicon(msn,dnd)         [Bitmap::get [pixmap services msn_dnd.xpm]]
set serviceicon(msn,unavailable) [Bitmap::get [pixmap services msn_offline.xpm]]

set serviceicon(x-gadugadu,available)   [Bitmap::get [pixmap services gg_online.gif]]
set serviceicon(x-gadugadu,chat)        [Bitmap::get [pixmap services gg_chat.gif]]
set serviceicon(x-gadugadu,away)        [Bitmap::get [pixmap services gg_away.gif]]
set serviceicon(x-gadugadu,xa)          [Bitmap::get [pixmap services gg_xa.gif]]
set serviceicon(x-gadugadu,dnd)         [Bitmap::get [pixmap services gg_dnd.gif]]
set serviceicon(x-gadugadu,unavailable) [Bitmap::get [pixmap services gg_offline.gif]]

set serviceicon(x-weather,available)   [Bitmap::get [pixmap services weather_online.gif]]
set serviceicon(x-weather,chat)        [Bitmap::get [pixmap services weather_chat.gif]]
set serviceicon(x-weather,away)        [Bitmap::get [pixmap services weather_away.gif]]
set serviceicon(x-weather,xa)          [Bitmap::get [pixmap services weather_xa.gif]]
set serviceicon(x-weather,dnd)         [Bitmap::get [pixmap services weather_dnd.gif]]
set serviceicon(x-weather,unavailable) [Bitmap::get [pixmap services weather_offline.gif]]

set serviceicon(yahoo,available)   [Bitmap::get [pixmap services yahoo_online.xpm]]
set serviceicon(yahoo,chat)        [Bitmap::get [pixmap services yahoo_chat.xpm]]
set serviceicon(yahoo,away)        [Bitmap::get [pixmap services yahoo_away.xpm]]
set serviceicon(yahoo,xa)          [Bitmap::get [pixmap services yahoo_xa.xpm]]
set serviceicon(yahoo,dnd)         [Bitmap::get [pixmap services yahoo_dnd.xpm]]
set serviceicon(yahoo,unavailable) [Bitmap::get [pixmap services yahoo_offline.xpm]]

set serviceicon(sms)   [Bitmap::get [pixmap services sms.xpm]]

set serviceicon(rss,available)   [Bitmap::get [pixmap services rss.xpm]]
set serviceicon(rss,away)        [Bitmap::get [pixmap services rss2.xpm]]
set serviceicon(rss,xa)          [Bitmap::get [pixmap services rss2.xpm]]
set serviceicon(rss,dnd)         [Bitmap::get [pixmap services rss3.xpm]]
set serviceicon(rss,unavailable) [Bitmap::get [pixmap services rss2.xpm]]

set conferenceicon(available)    [Bitmap::get [pixmap roster group_on.gif]]
set conferenceicon(unavailable)  [Bitmap::get [pixmap roster group_off.gif]]

set rostericon(group,opened)     [Bitmap::get [pixmap roster group-opened.gif]]
set rostericon(group,closed)     [Bitmap::get [pixmap roster group-closed.gif]]


proc roster::common_menu {m connid jid} {
    $m add separator
    $m add command -label [::msgcat::mc "Show info..."] \
	-command [list userinfo::open $jid -connection $connid]
    $m add command -label [::msgcat::mc "Show history..."] \
	-command [list logger::show_log $jid]
    $m add separator
    $m add command -label [::msgcat::mc "Edit item..."] \
	-command [list itemedit::show_dialog $connid $jid]
    # TODO: connid
    $m add command -label [::msgcat::mc "Edit security..."] \
	-command [list ssj::prefs $jid] \
	-state [lindex {normal disabled} \
		    [cequal [info commands ::ssj::prefs] ""]]
    $m add separator
    # TODO: connid
    $m add command -label [::msgcat::mc "Remove..."] \
	-command [list roster::remove_item_dialog $connid $jid]
    return $m
}


proc roster::jid_popup_menu {connid jid} {
    if {[winfo exists [set m .jidpopupmenu]]} {
	destroy $m
    }
    menu $m -tearoff 0
    $m add command -label [::msgcat::mc "Start chat"] \
	-command [list chat::open_to_user $connid $jid]
    # TODO: connid
    $m add command -label [::msgcat::mc "Send message..."] \
	-command [list message::send_dialog $jid]
    # TODO: connid
    $m add command -label [::msgcat::mc "Invite to conference..."] \
	-command [list chat::invite_dialog $jid]
    $m add command -label [::msgcat::mc "Resubscribe"] \
	-command [list jlib::send_presence \
		      -to $jid \
		      -type subscribe \
		      -connection $connid]
    $m add separator
    # TODO: connid
    $m add command -label [::msgcat::mc "Send users..."] \
	-command [list roster::send_users_dialog $jid]
    # TODO: connid
    $m add command -label [::msgcat::mc "Send file..."] \
	-command [list ft::send_file_dialog $jid]
    # TODO: connid
    $m add command -label [::msgcat::mc "Send file via Jidlink..."] \
	-command [list ftjl::send_file_dialog $jid]
    common_menu $m $connid $jid
}

proc roster::conference_popup_menu {connid jid} {
    if {[winfo exists [set m .confpopupmenu]]} {
	destroy $m
    }
    menu $m -tearoff 0
    $m add command -label [::msgcat::mc "Join..."] \
	-command [list join_group_dialog \
		      -server [server_from_jid $jid] \
		      -group [node_from_jid $jid] \
		      -connection $connid]
    common_menu $m $connid $jid
}

proc roster::service_popup_menu {connid jid} {
    if {[winfo exists [set m .servicepopupmenu]]} {
	destroy $m
    }
    menu $m -tearoff 0
    # TODO
    $m add command -label [::msgcat::mc "Log in"] -command {
	switch -- $userstatus {
	    available { jlib::send_presence -to $curuser }
	    invisible { jlib::send_presence -to $curuser -type $userstatus }
	    default   { jlib::send_presence -to $curuser -show $userstatus }
	}
    }
    # TODO
    $m add command -label [::msgcat::mc "Log out"] -command {
	jlib::send_presence -to $curuser -type unavailable
    }
    common_menu $m $connid $jid
}

proc roster::create_groupchat_user_menu {connid jid} {
    if {[winfo exists [set m .groupchatpopupmenu]]} {
	destroy $m
    }
    # TODO: connid
    menu $m -tearoff 0
    $m add command -label [::msgcat::mc "Start chat"] \
	-command [list chat::open_to_user $connid $jid]
    #$m add command -label "Send message..." -command {}
    hook::run roster_create_groupchat_user_menu_hook $m $connid $jid
    $m add separator
    $m add command -label [::msgcat::mc "Send users..."] \
	-command [list roster::send_users_dialog $jid]
    $m add command -label [::msgcat::mc "Send file..."] \
	-command [list ft::send_file_dialog $jid]
    $m add command -label [::msgcat::mc "Send file via Jidlink..."] \
	-command [list ftjl::send_file_dialog $jid]
    $m add command -label [::msgcat::mc "Invite to conference..."] \
	-command [list chat::invite_dialog $jid]
    $m add separator
    $m add command -label [::msgcat::mc "Show info..."] \
	-command [list userinfo::open $jid -connection $connid]
    $m add command -label [::msgcat::mc "Show history..."] \
	-command {} -state disabled
    return $m
}


proc roster::create_group_popup_menu {connid name} {
    variable options

    if {[winfo exists [set m .grouppopupmenu]]} {
	destroy $m
    }
    if {$options(nested)} {
	set oname [::textutil::splitx $name $options(nested_delimiter)]
    } else {
	set oname $name
    }
    menu $m -tearoff 0
    $m add command -label [::msgcat::mc "Rename..."] \
	-command [list roster::rename_group_dialog $connid $name]
    $m add command \
	-label [::msgcat::mc "Resubscribe to all users in group..."] \
	-command [list roster::resubscribe_group $connid $name]
    $m add checkbutton -label [::msgcat::mc "Show offline users"] \
	-variable [namespace current]::roster(show_offline,[list $connid $oname]) \
	-command [list roster::redraw_after_idle]
    $m add command -label [::msgcat::mc "Remove..."] \
	-command [list roster::remove_group_dialog $connid $name]
    return $m
}

###############################################################################

proc roster::remove_group_dialog {connid name} {
    set res [MessageDlg .remove_item -aspect 50000 -icon question -type user \
		 -buttons {yes no} -default 0 -cancel 1 \
		 -message [format [::msgcat::mc "Are you sure to remove group '%s' from roster?"] $name]]

    if {$res == 0} {
	send_rename_group $connid $name ""
    }
}

proc roster::rename_group_dialog {connid name} {
    global new_roster_group_name

    set new_roster_group_name $name

    set w .roster_group_rename
    if {[winfo exists $w]} {
	destroy $w
    }
    
    Dialog $w -title [::msgcat::mc "Rename roster group"] \
	-separator 1 -anchor e -default 0 -cancel 1

    $w add -text [::msgcat::mc "OK"] -command "
	destroy [list $w]
	roster::send_rename_group [list $connid $name] \$new_roster_group_name
    "
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]


    set p [$w getframe]
    
    label $p.lgroupname -text [::msgcat::mc "New group name:"]
    ecursor_entry [entry $p.groupname -textvariable new_roster_group_name]

    grid $p.lgroupname  -row 0 -column 0 -sticky e
    grid $p.groupname   -row 0 -column 1 -sticky ew

    focus $p.groupname
    $w draw
}

proc roster::send_rename_group {connid name new_name} {
    variable roster
    variable undef_group_name

    if {$new_name == $name} return

    set items {}

    foreach jid $roster(jids,$connid) {
	if {[lcontain $roster(group,$connid,$jid) $name] || \
		($name == $undef_group_name && \
		     $roster(group,$connid,$jid) == {})} {
	    set idx [lsearch -exact $roster(group,$connid,$jid) $name]
	    if {$new_name != ""} {
		set roster(group,$connid,$jid) \
		    [lreplace $roster(group,$connid,$jid) $idx $idx $new_name]
	    } else {
		set roster(group,$connid,$jid) \
		    [lreplace $roster(group,$connid,$jid) $idx $idx]
	    }
	    set roster(group,$connid,$jid) [lrmdups $roster(group,$connid,$jid)]
	    lappend items [item_to_xml $connid $jid]
	}
    }

    if {$items != {}} {
	jlib::send_iq set \
	    [jlib::wrapper:createtag query \
		 -vars {xmlns jabber:iq:roster} \
		 -subtags $items] \
	    -connection $connid
    }
}

proc roster::resubscribe_group {connid name} {
    variable roster

    foreach jid $roster(jids,$connid) {
	if {[lcontain $roster(group,$connid,$jid) $name]} {
	    lassign [get_category_and_subtype $connid $jid] category type
	    if {$category == "user"} {
		jlib::send_presence \
		    -to $jid \
		    -connection $connid \
		    -type subscribe
	    }
	}
    }
}


proc roster::add_group_by_jid_regexp_dialog {} {
    global new_roster_group_rname
    global new_roster_group_regexp

    set w .roster_group_add_by_jid_regexp
    if {[winfo exists $w]} {
	destroy $w
    }
    
    Dialog $w -title [::msgcat::mc "Add roster group by JID regexp"] \
	-separator 1 -anchor e -default 0 -cancel 1

    $w add -text [::msgcat::mc "OK"] -command "
	destroy [list $w]
	roster::add_group_by_jid_regexp \
	    \$new_roster_group_rname \$new_roster_group_regexp
    "
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]


    set p [$w getframe]
    
    label $p.lgroupname -text [::msgcat::mc "New group name:"]
    ecursor_entry [entry $p.groupname -textvariable new_roster_group_rname]
    label $p.lregexp -text [::msgcat::mc "JID regexp:"]
    ecursor_entry [entry $p.regexp -textvariable new_roster_group_regexp]

    grid $p.lgroupname -row 0 -column 0 -sticky e
    grid $p.groupname  -row 0 -column 1 -sticky ew
    grid $p.lregexp    -row 1 -column 0 -sticky e
    grid $p.regexp     -row 1 -column 1 -sticky ew

    focus $p.groupname
    $w draw
}

proc roster::add_group_by_jid_regexp {name regexp} {
    variable roster

    # TODO: connid
    if {$name == ""} return

    foreach connid [jlib::connections] {
	set items {}

	foreach jid $roster(jids,$connid) {
	    if {[regexp $regexp $jid]} {
		set idx [lsearch -exact $roster(group,$connid,$jid) $name]
		lappend roster(group,$connid,$jid) $name
		set roster(group,$connid,$jid) \
		    [lrmdups $roster(group,$connid,$jid)]
		lappend items [item_to_xml $connid $jid]
	    }
	}

	if {$items != {}} {
	    jlib::send_iq set \
		[jlib::wrapper:createtag query \
		     -vars {xmlns jabber:iq:roster} \
		     -subtags $items] \
		-connection $connid
	}
    }
}



###############################################################################

proc roster::send_users_dialog {user} {
    global send_uc

    # TODO
    set connid [jlib::route $user]
    set jid [get_jid_of_user $connid $user]

    if {[cequal $jid ""]} {
        set jid $user
    }

    set gw .contacts
    catch { destroy $gw }

    if {[catch { set nick [get_label $connid $user] }]} {
	if {[catch { set nick [chat::get_nick $user groupchat] }]} {
	    set nick $user
	}
    }

    set choices {}
    set balloons {}
    foreach c [jlib::connections] {
	foreach choice $roster::roster(jids,$c) {
	    if {![cequal $roster::roster(category,$c,$choice) conference]} {
		lappend choices [list $c $choice] [roster::get_label $c $choice]
		lappend balloons [list $c $choice] $choice
	    }
	}
    }
    if {[llength $choices] == 0} {
        MessageDlg ${gw}_err -aspect 50000 -icon info \
	    -message [::msgcat::mc "No users in roster..."] -type user \
	    -buttons ok -default 0 -cancel 0
        return
    }

    CbDialog $gw [format [::msgcat::mc "Send contacts to %s"] $nick] \
	[list [::msgcat::mc "Send"] [list roster::send_users $gw $jid] \
	      [::msgcat::mc "Cancel"] [list destroy $gw]] \
	send_uc $choices $balloons
}

proc roster::send_users {gw jid} {
    variable roster
    global send_uc

    set sf [$gw getframe].sw.sf
    set choices {}
    foreach uc [array names send_uc] {
        if {$send_uc($uc)} {
            lappend choices $uc
        }
    }

    destroy $gw

    set subtags {}
    set body [::msgcat::mc "Contact Information"]
    foreach choice $choices {
	lassign $choice con uc
	lappend subtags [item_to_xml $con $uc]
	set nick [roster::get_label $con $uc]
        append body "\n$nick - xmpp:$uc"
    }

    message::send_msg $jid -type normal -body $body \
	        -xlist [list [jlib::wrapper:createtag x \
	                          -vars [list xmlns jabber:x:roster] \
	                          -subtags $subtags]]
}

###############################################################################

proc roster::export_to_file {connid} {
    variable roster

    set filename [tk_getSaveFile \
		      -initialdir ~/.tkabber/ \
		      -initialfile $::loginconf(user).roster \
		      -filetypes [list \
				      [list [::msgcat::mc "Roster Files"] \
					   .roster] \
				      [list [::msgcat::mc "All Files"] *]]]
    if {$filename != ""} {
	set items {}

	foreach jid $roster(jids,$connid) {
	    lappend items [item_to_xml $connid $jid]
	}

	set fd [open $filename w]
	fconfigure $fd -encoding utf-8
	puts $fd $items
	close $fd
    }
}

proc roster::import_from_file {connid} {
    variable roster

    set filename [tk_getOpenFile \
		      -initialdir ~/.tkabber/ \
		      -initialfile $::loginconf(user).roster \
		      -filetypes [list \
				      [list [::msgcat::mc "Roster Files"] \
					   .roster] \
				      [list [::msgcat::mc "All Files"] *]]]
    if {$filename != ""} {
	set fd [open $filename r]
	fconfigure $fd -encoding utf-8
	set items [read $fd]
	close $fd

	if {$items != {}} {
	    jlib::send_iq set \
		[jlib::wrapper:createtag query \
		     -vars [list xmlns "jabber:iq:roster"] \
		     -subtags $items] \
		-connection $connid
	}
    }
}

proc roster::setup_import_export_menus {args} {
    set emenu [.mainframe getmenu export_roster]
    set imenu [.mainframe getmenu import_roster]

    if {[winfo exists $emenu]} {
	destroy $emenu
    }
    menu $emenu -tearoff 0

    if {[winfo exists $imenu]} {
	destroy $imenu
    }
    menu $imenu -tearoff 0

    if {[jlib::connections] == {}} {
	.mainframe setmenustate export_roster disabled
	.mainframe setmenustate import_roster disabled
    } else {
	.mainframe setmenustate export_roster normal
	.mainframe setmenustate import_roster normal
    }

    foreach c [jlib::connections] {
	set jid [jlib::connection_jid $c]
	set label [format [::msgcat::mc "Roster of %s"] $jid]
	set ecommand [list roster::export_to_file $c]
	set icommand [list roster::import_from_file $c]
	$emenu add command -label $label -command $ecommand
	$imenu add command -label $label -command $icommand
    }
}
hook::add connected_hook roster::setup_import_export_menus
hook::add disconnected_hook roster::setup_import_export_menus
hook::add finload_hook roster::setup_import_export_menus

###############################################################################

proc roster::add_custom_presence_menu {m jids} {
    set mm [menu $m.custom_presence -tearoff 0]

    add_menu_item $mm [::msgcat::mc "Online"] \
	{send_custom_presence $curuser available} $jids
    add_menu_item $mm [::msgcat::mc "Free to chat"] \
	{send_custom_presence $curuser chat} $jids
    add_menu_item $mm [::msgcat::mc "Away"] \
	{send_custom_presence $curuser away} $jids
    add_menu_item $mm [::msgcat::mc "Extended Away"] \
	{send_custom_presence $curuser xa} $jids
    add_menu_item $mm [::msgcat::mc "Do not disturb"] \
	{send_custom_presence $curuser dnd} $jids
    add_menu_item $mm [::msgcat::mc "Offline"] \
	{send_custom_presence $curuser unavailable} $jids

    $m add cascad -label [::msgcat::mc "Send custom presence"] -menu $mm
}

###############################################################################

set roster_main_menu \
    [list cascad [::msgcat::mc "Roster"] {} roster 1 \
	 [list \
	      [list command [::msgcat::mc "Add user..."] {} {} {} \
		   -command message::send_subscribe_dialog] \
	      [list command [::msgcat::mc "Add conference..."] {} {} {} \
		   -command {add_group_dialog}] \
	      [list command [::msgcat::mc "Add group by regexp on JIDs..."] {} {} {} \
		   -command {roster::add_group_by_jid_regexp_dialog}] \
	      [list checkbutton [::msgcat::mc "Show online users only"] \
		   {} {} {} \
		   -variable roster::show_only_online \
		   -command roster::changed_only_online] \
	      [list checkbutton [::msgcat::mc "Use aliases"] {} {} {} \
		   -variable roster::use_aliases] \
	      [list cascad [::msgcat::mc "Export roster..."] \
		   export_roster export_roster 0 {}] \
	      [list cascad [::msgcat::mc "Import roster..."] \
		   import_roster import_roster 0 {}] \
	     ]]

trace variable roster::use_aliases w "roster::redraw ; #"
trace variable roster::options(nested) w "roster::redraw ; #"

