#
# This script is part of TkKasse. You may use, modify and distribute it under the terms
# of the GNU General Public License, Version 2.
# See details at http://www.fsf.org/licenses/gpl.txt
# (C) by Jan Kandziora <tkkasse@users.sf.net> 
#


## Article list class:
option add *ArticlesView.labelText [ msgcat::mc <ArticlesView.labelText> ] startupFile

itk::usual ArticlesView \
{
	keep -background -cursor -font -foreground
}

itcl::class ArticlesView \
{
	inherit ::itk::Widget

	variable ARTICLES
	variable SELECTED_ARTICLE {}

	constructor {args} {}
	
	itk_option define -articles articles Articles {}
	itk_option define -newcommand newCommand NewCommand {}
	itk_option define -newagaincommand newAgainCommand NewAgainCommand {}

	public method select      {KEY}
	public method subselect   {KEY}
	public method deselect    {}
	public method mouseselect {YPOS}
	public method get         {}

	protected method _reconfigure {}
}

## Article list class constructor:
itcl::body ArticlesView::constructor {args} \
{
	## Compose the widget.
	itk_component add label \
	{
		label $itk_interior.label -relief groove
	} \
	{
		rename -font -labelfont labelFont LabelFont
		rename -text -labeltext labelText LabelText
	}

	itk_component add list \
	{
		tixHList $itk_interior.list \
			-relief flat \
			-borderwidth 20 \
			-drawbranch false \
			-selectmode browse -indent 20 \
			-yscrollcommand "$itk_interior.sy set"
	} \
	{
	}

	itk_component add sy \
	{
		scrollbar $itk_interior.sy \
			-orient vertical \
			-command "$itk_interior.list yview"
	} \
	{
	}

	## Pack it.
	pack $itk_component(label) -side top   -fill x
	pack $itk_component(sy)    -side right -fill y
	pack $itk_component(list)  -side left  -fill both -expand yes

	## Key bindings:
	bind $itk_component(list) <KeyPress-0>         "$this subselect 0"
	bind $itk_component(list) <KeyPress-KP_Insert> "$this subselect 0"
	bind $itk_component(list) <KeyPress-1>         "$this subselect 1"
	bind $itk_component(list) <KeyPress-KP_End>    "$this subselect 1"
	bind $itk_component(list) <KeyPress-2>         "$this subselect 2"
	bind $itk_component(list) <KeyPress-KP_Down>   "$this subselect 2"
	bind $itk_component(list) <KeyPress-3>         "$this subselect 3"
	bind $itk_component(list) <KeyPress-KP_Next>   "$this subselect 3"
	bind $itk_component(list) <KeyPress-4>         "$this subselect 4"
	bind $itk_component(list) <KeyPress-KP_Left>   "$this subselect 4"
	bind $itk_component(list) <KeyPress-5>         "$this subselect 5"
	bind $itk_component(list) <KeyPress-KP_Begin>  "$this subselect 5"
	bind $itk_component(list) <KeyPress-6>         "$this subselect 6"
	bind $itk_component(list) <KeyPress-KP_Right>  "$this subselect 6"
	bind $itk_component(list) <KeyPress-7>         "$this subselect 7"
	bind $itk_component(list) <KeyPress-KP_Home>   "$this subselect 7"
	bind $itk_component(list) <KeyPress-8>         "$this subselect 8"
	bind $itk_component(list) <KeyPress-KP_Up>     "$this subselect 8"
	bind $itk_component(list) <KeyPress-9>         "$this subselect 9"
	bind $itk_component(list) <KeyPress-KP_Prior>  "$this subselect 9"

	bind $itk_component(list) <KeyPress-plus>      "eval \[ $this cget -newagaincommand \]"
	bind $itk_component(list) <KeyPress-KP_Add>    "eval \[ $this cget -newagaincommand \]"
	bind $itk_component(list) <KeyPress-Return>    "eval \[ $this cget -newcommand \]; $this deselect"
	bind $itk_component(list) <KeyPress-KP_Enter>  "eval \[ $this cget -newcommand \]; $this deselect"
	
	bind $itk_component(list) <KeyPress-KP_Delete> "$this deselect"
	bind $itk_component(list) <KeyPress-BackSpace> "$this deselect"
	bind $itk_component(list) <KeyPress-Escape>    "$this deselect"

	bind $itk_component(list) <KeyPress-Up>    "$itk_component(list) yview scroll -1 units ; break"
	bind $itk_component(list) <KeyPress-Down>  "$itk_component(list) yview scroll +1 units ; break"
	bind $itk_component(list) <KeyPress-Left>  break
	bind $itk_component(list) <KeyPress-Right> break

	## Mouse bindings:
	bind $itk_component(list) <ButtonPress-1>        "$this mouseselect %y ; break"
	bind $itk_component(list) <Double-ButtonPress-1> "$this mouseselect %y ; eval \[ $this cget -newcommand \] ; break"
	bind $itk_component(list) <ButtonPress-3>        "$this deselect"

	## Now initialize the widget.
	eval itk_initialize $args
}

## Configure the widget as the "articles" option is configured.
itcl::configbody ArticlesView::articles \
{
	## Set the articles array
	catch { unset ARTICLES }
	array set ARTICLES {}
	foreach ARTICLE_INFO $itk_option(-articles) \
	{
		## Get the key and the name of the current article and make an array entry of it.
		set ARTICLES([ lindex $ARTICLE_INFO [ expr [ lsearch -exact $ARTICLE_INFO "-key" ] + 1 ] ]) [ lindex $ARTICLE_INFO [ expr [ lsearch -exact $ARTICLE_INFO "-name" ] + 1 ] ]
	}

	## Delete all current entries in the listbox.
	$itk_component(list) delete all

	## Adjust the listbox.
	foreach KEY [ lsort [ array names ARTICLES ] ] \
	{
		$itk_component(list) add [ join [ split $KEY {} ] . ] -text "$KEY   $ARTICLES($KEY)"
	}

	## Remove any selection.
	deselect
}

## Reconfigure the widget.
itcl::body ArticlesView::_reconfigure {} \
{
	## Open the list.
	set X {}
	foreach SUBKEY [ split $SELECTED_ARTICLE {} ] \
	{
		## Only show subentries of the selected entry.
		foreach Y [ $itk_component(list) info children $X ] \
		{
			$itk_component(list) show entry $Y
			foreach Z [ $itk_component(list) info children $Y ] \
			{
				$itk_component(list) hide entry $Z
			}
		}

		## Next subentry.
		set X [ join [ concat $X $SUBKEY ] . ]
	}

	## Only show subentries of the selected entry.
	foreach Y [ $itk_component(list) info children $X ] \
	{
		$itk_component(list) show entry $Y
		foreach Z [ $itk_component(list) info children $Y ] \
		{
			$itk_component(list) hide entry $Z
		}
	}

	## Return if no article is selected.
	if { $SELECTED_ARTICLE == "" } return

	## Show the selected article and put a mark on it
	$itk_component(list) selection clear
	$itk_component(list) selection set [ join [ split $SELECTED_ARTICLE {} ] . ] 
	$itk_component(list) yview [ join [ split $SELECTED_ARTICLE {} ] . ]
}

## Select an article by its key.
itcl::body ArticlesView::select {KEY} \
{
	## Check if the key exists.
	if [ catch { set ARTICLES($KEY) } ] return

	## Select the article.
	set SELECTED_ARTICLE $KEY
	_reconfigure
}

## Select a subentry.
itcl::body ArticlesView::subselect {KEY} \
{
	select "$SELECTED_ARTICLE$KEY"
}

## Remove any selection.
itcl::body ArticlesView::deselect {} \
{
	set SELECTED_ARTICLE {}
	_reconfigure
}

## Select a subentry at mouse position.
itcl::body ArticlesView::mouseselect { YPOS } \
{
	## Focus the articles list.		
	focus $itk_component(list)

	## Select the entry.
	select [ join [ split [ $itk_component(list) nearest $YPOS ] . ] {} ]
}

## Get the selected article key.
itcl::body ArticlesView::get {} \
{
	return $SELECTED_ARTICLE
}



## Bills view class:
itk::usual BillsView \
{
	keep -background -cursor -font -foreground
}

itcl::class BillsView \
{
	inherit ::itk::Widget

	constructor { args } {}
	
	itk_option define -userlogin userLogin UserLogin {}
	itk_option define -username userName UserName {}

	method addView { TAG_ARG TAG_TEXT_ARG } 
	method configureView { TAG_ARG BILL_INFO_ARG }
	method getView { TAG_ARG }
	method removeView { TAG_ARG } 
	method seeView { TAG_ARG }
	method getViewTags {}
}

## Bills view class constructor
itcl::body BillsView::constructor { args } \
{
	## Compose the widget.
	itk_component add label \
	{
		label $itk_interior.label \
			-relief raised
	} \
	{
		rename -background -labelbackground labelBackground LabelBackground
		rename -font -labelfont labelFont LabelFont
	}

	itk_component add bills \
	{
		frame $itk_interior.bills
	}
	blt::tabset $itk_component(bills).tabset \
		-relief flat -gap 1 -selectpad 0  \
		-samewidth 0 -side left \
		-takefocus 0

	## Pack it.
	pack $itk_interior.label        -fill x 
	pack $itk_interior.bills.tabset -fill both -expand yes -pady 5
	pack $itk_interior.bills        -fill both -expand yes

	## Now initialize the widget.
	eval itk_initialize $args
}

## Configure the widget as the "username" option is configured.
itcl::configbody BillsView::username \
{
	$itk_component(label) configure -text $itk_option(-username)
}

## Add a bill view.
itcl::body BillsView::addView { TAG_ARG TAG_TEXT_ARG } \
{
	## Create a bill view object.
	BillView $itk_component(bills).tabset.$TAG_ARG 

	## Put the bill view window into a new tab. 
	$itk_component(bills).tabset insert 0 $TAG_ARG -window $itk_component(bills).tabset.$TAG_ARG -text $TAG_TEXT_ARG -fill both 

	## Let the billsview window take the keyboard focus.
	## This is only allowed if there is at least one tab, as Blt::tabset crashes Tk otherwise.
	## Remember to take the keyboard focus away when the last bill view is removed.
	$itk_component(bills).tabset configure -takefocus 1
}

## Configure a bill view.
itcl::body BillsView::configureView { TAG_ARG BILL_INFO_ARG } \
{
	## Check if the tab is present.
	if [ catch { $itk_component(bills).tabset index -name $TAG_ARG } ] {} \
	{
		## Yes. Configure it.
		$itk_component(bills).tabset.$TAG_ARG configure -billinfo $BILL_INFO_ARG
	}
}

## Get a bill view window by its tag.
itcl::body BillsView::getView { TAG_ARG } \
{
	$itk_component(bills).tabset tab cget $TAG_ARG -window
}

## Remove a bill view:
itcl::body BillsView::removeView { TAG_ARG } \
{
	## Delete the tab.
	$itk_component(bills).tabset delete [ $itk_component(bills).tabset index -name $TAG_ARG ]

	## If there is no tab left, don't take the keyboard focus anymore and focus the main window immediately.
	if { [ getViewTags ] == {} } \
	{
		$itk_component(bills).tabset configure -takefocus 0
		focus .main
	}

	## Delete the bill view object.
	itcl::delete object $itk_component(bills).tabset.$TAG_ARG
}

## Let the user see the view with the given tag.
itcl::body BillsView::seeView { TAG_ARG } \
{
	## Select the tab.
	blt::SelectTab $itk_component(bills).tabset [ $itk_component(bills).tabset index -name $TAG_ARG ]

	## Move the view to the first entry.
	$itk_component(bills).tabset view moveto 0.0
}

## Get a list of all bill tags in this users view.
itcl::body BillsView::getViewTags {} \
{
	$itk_component(bills).tabset tab names
}



## Bill view class:
option add *BillView.createTimeFormat [ msgcat::mc <BillView.createTimeFormat> ] startupFile
option add *BillView.createTimeLabel [ msgcat::mc <BillView.createTimeLabel> ] startupFile
option add *BillView.descriptionLabel [ msgcat::mc <BillView.descriptionLabel> ] startupFile
option add *BillView.deliveredSumLabel [ msgcat::mc <BillView.deliveredSumLabel> ] startupFile
option add *BillView.keyLabel [ msgcat::mc <BillView.keyLabel> ] startupFile
option add *BillView.markLabel [ msgcat::mc <BillView.markLabel> ] startupFile
option add *BillView.markOff [ msgcat::mc <BillView.markOff> ] startupFile
option add *BillView.markOn [ msgcat::mc <BillView.markOn> ] startupFile
option add *BillView.orderedSumLabel [ msgcat::mc <BillView.orderedSumLabel> ] startupFile
option add *BillView.priceLabel [ msgcat::mc <BillView.priceLabel> ] startupFile
option add *BillView.serialNumberLabel [ msgcat::mc <BillView.serialNumberLabel> ] startupFile
option add *BillView.serviceLabel [ msgcat::mc <BillView.serviceLabel> ] startupFile
option add *BillView.serviceBar [ msgcat::mc <BillView.serviceBar> ] startupFile
option add *BillView.serviceKitchen [ msgcat::mc <BillView.serviceKitchen> ] startupFile
option add *BillView.showedSumLabel [ msgcat::mc <BillView.showedSumLabel> ] startupFile
option add *BillView.showedSumLabel [ msgcat::mc <BillView.showedSumLabel> ] startupFile
option add *BillView.specialLabel [ msgcat::mc <BillView.specialLabel> ] startupFile
option add *BillView.stateCancel [ msgcat::mc <BillView.stateCancel> ] startupFile
option add *BillView.stateCancelledDelivered [ msgcat::mc <BillView.stateCancelledDelivered> ] startupFile
option add *BillView.stateCancelledOrdered [ msgcat::mc <BillView.stateCancelledOrdered> ] startupFile
option add *BillView.stateDelivered [ msgcat::mc <BillView.stateDelivered> ] startupFile
option add *BillView.stateLabel [ msgcat::mc <BillView.stateLabel> ] startupFile
option add *BillView.stateOrdered [ msgcat::mc <BillView.stateOrdered> ] startupFile
option add *BillView.stateOrderedCancel [ msgcat::mc <BillView.stateOrderedCancel> ] startupFile
option add *BillView.taxLabel [ msgcat::mc <BillView.taxLabel> ] startupFile
option add *BillView.taxOff [ msgcat::mc <BillView.taxOff> ] startupFile
option add *BillView.taxOn [ msgcat::mc <BillView.taxOn> ] startupFile

itk::usual BillView \
{
	keep -background -cursor -font -foreground
}

itcl::class BillView \
{
	inherit ::itk::Widget

	constructor { args } {}

	itk_option define -billinfo billInfo BillInfo {-showed_sum 0 -delivered_sum 0 -ordered_sum 0 -create_time 0 -items {} -tag 0 }
	itk_option define -createtimeformat createTimeFormat Text ""
	itk_option define -descriptionlabel descriptionLabel Text ""
	itk_option define -keylabel keyLabel Text ""
	itk_option define -marklabel markLabel Text ""
	itk_option define -markoff markOff Text ""
	itk_option define -markon markOn Text ""
	itk_option define -pricelabel priceLabel Text ""
	itk_option define -servicelabel serviceLabel Text ""
	itk_option define -servicebar serviceBar Text ""
	itk_option define -servicekitchen serviceKitchen Text ""
	itk_option define -speciallabel specialLabel Text ""
	itk_option define -statelabel stateLabel Text ""
	itk_option define -statecancel stateCancel Text ""
	itk_option define -statecancelleddelivered stateCancelledDelivered Text ""
	itk_option define -statecancelledordered stateCancelledOrdered Text ""
	itk_option define -statedelivered stateDelivered Text ""
	itk_option define -stateordered stateOrdered Text ""
	itk_option define -stateorderedcancel stateOrderedCancel Text ""
	itk_option define -taxlabel taxLabel Text ""
	itk_option define -taxoff taxOff Text ""
	itk_option define -taxon taxOn Text ""
	itk_option define -tkkassecommand tkKasseCommand TkKasseCommand {}

	method clearSelection {}
	method getSelection {}
	method getTag {}
	method handleEvent { EVENT_TYPE_ARG args }

	protected variable selectable_items_unique_array
}

itcl::body BillView::constructor { args } \
{
	## Compose the widget.
	itk_component add serialnumberlabel \
	{
		label $itk_interior.serialnumberlabel
	} \
	{
		rename -text -serialnumberlabel serialNumberLabel SerialNumberLabel
		rename -font -infofont infoFont InfoFont
		rename -foreground -infoforeground infoForeground InfoForeground
		rename -background -infobackground infoBackground InfoBackground
	}
	itk_component add serialnumber \
	{
		label $itk_interior.serialnumber
	} \
	{
		rename -text -serialnumber serialNumber SerialNumber
		rename -font -infofont infoFont InfoFont
		rename -foreground -infoforeground infoForeground InfoForeground
		rename -background -infobackground infoBackground InfoBackground
	}

	itk_component add createtimelabel \
	{
		label $itk_interior.createtimelabel
	} \
	{
		rename -text -createtimelabel createTimeLabel CreateTimeLabel
		rename -font -infofont infoFont InfoFont
		rename -foreground -infoforeground infoForeground InfoForeground
		rename -background -infobackground infoBackground InfoBackground
	}
	itk_component add createtime \
	{
		label $itk_interior.createtime
	} \
	{
		rename -text -createtime createTime CreateTime
		rename -font -infofont infoFont InfoFont
		rename -foreground -infoforeground infoForeground InfoForeground
		rename -background -infobackground infoBackground InfoBackground
	}

	itk_component add list \
	{
		tixHList $itk_interior.list \
			-width 70 -height 25 \
			-columns 8 -header true \
			-indicator false -drawbranch false \
			-selectmode extended -relief groove \
			-yscrollcommand "$itk_interior.sy set"
	} \
	{
		rename -font -itemsfont itemsFont ArticlesFont
		rename -foreground -itemsforeground itemsForeground ArticlesForeground
		rename -background -itemsbackground itemsBackground ArticlesBackground
		rename -highlightbackground -itemshighlightbackground itemsHighlightBackground ArticlesHighlightBackground
		rename -selectbackground -itemsselectbackground itemsSelectBackground ArticlesSelectBackground
	}

	$itk_component(list) column width 0 16
	$itk_component(list) column width 1 16
	$itk_component(list) column width 2 16
	$itk_component(list) column width 3 16
	$itk_component(list) column width 4 16
	$itk_component(list) column width 5 48
	$itk_component(list) column width 6 260
	$itk_component(list) column width 7 88 

	$itk_component(list) header create 0 -headerbackground $itk_option(-itemsbackground) -relief groove -itemtype text
	$itk_component(list) header create 1 -headerbackground $itk_option(-itemsbackground) -relief groove -itemtype text
	$itk_component(list) header create 2 -headerbackground $itk_option(-itemsbackground) -relief groove -itemtype text
	$itk_component(list) header create 3 -headerbackground $itk_option(-itemsbackground) -relief groove -itemtype text
	$itk_component(list) header create 4 -headerbackground $itk_option(-itemsbackground) -relief groove -itemtype text
	$itk_component(list) header create 5 -headerbackground $itk_option(-itemsbackground) -relief groove -itemtype text
	$itk_component(list) header create 6 -headerbackground $itk_option(-itemsbackground) -relief groove -itemtype text
	$itk_component(list) header create 7 -headerbackground $itk_option(-itemsbackground) -relief groove -itemtype text

	itk_component add sy \
	{
		scrollbar $itk_interior.sy \
			-orient vertical \
			-command "$itk_interior.list yview"
	} \
	{
		rename -background -itemsbackground itemsBackground ArticlesBackground
		rename -highlightbackground -itemshighlightbackground itemsHighlightBackground ArticlesHighlightBackground
	}

	itk_component add showedsumlabel \
	{
		label $itk_interior.showedsumlabel
	} \
	{
		rename -text -showedsumlabel showedSumLabel ShowedSumLabel
		rename -font -infofont infoFont InfoFont
		rename -foreground -infoforeground infoForeground InfoForeground
		rename -background -infobackground infoBackground InfoBackground
	}
	itk_component add showedsum \
	{
		label $itk_interior.showedsum
	} \
	{
		rename -text -showedsum showedSum ShowedSum
		rename -font -infofont infoFont InfoFont
		rename -foreground -infoforeground infoForeground InfoForeground
		rename -background -infobackground infoBackground InfoBackground
	}

	itk_component add orderedsumlabel \
	{
		label $itk_interior.orderedsumlabel
	} \
	{
		rename -text -orderedsumlabel orderedSumLabel OrderedSumLabel
		rename -font -infofont infoFont InfoFont
		rename -foreground -infoforeground infoForeground InfoForeground
		rename -background -infobackground infoBackground InfoBackground
	}
	itk_component add orderedsum \
	{
		label $itk_interior.orderedsum
	} \
	{
		rename -text -orderedsum orderedSum OrderedSum
		rename -font -infofont infoFont InfoFont
		rename -foreground -infoforeground infoForeground InfoForeground
		rename -background -infobackground infoBackground InfoBackground
	}

	itk_component add deliveredsumlabel \
	{
		label $itk_interior.deliveredsumlabel
	} \
	{
		rename -text -deliveredsumlabe deliveredSumLabel DeliveredSumLabel
		rename -font -infofont infoFont InfoFont
		rename -foreground -infoforeground infoForeground InfoForeground
		rename -background -infobackground infoBackground InfoBackground
	}
	itk_component add deliveredsum \
	{
		label $itk_interior.deliveredsum
	} \
	{
		rename -text -deliveredsum deliveredSum DeliveredSum
		rename -font -infofont infoFont InfoFont
		rename -foreground -infoforeground infoForeground InfoForeground
		rename -background -infobackground infoBackground InfoBackground
	}

	## Pack it.
	grid $itk_component(serialnumberlabel) -row 0 -column 0 -sticky w -padx 5 -pady 3
	grid $itk_component(serialnumber) -row 0 -column 1 -sticky w
	grid $itk_component(createtimelabel) -row 1 -column 0 -sticky w -padx 5 -pady 3
	grid $itk_component(createtime) -row 1 -column 1 -sticky w

	grid $itk_component(sy) -row 2 -column 2 -sticky ns
	grid $itk_component(list) -row 2 -column 0 -columnspan 2 -sticky nsew

	grid $itk_component(showedsumlabel) -row 3 -column 0 -sticky w -padx 5 -pady 3
	grid $itk_component(showedsum) -row 3 -column 1 -sticky e -padx 10
	grid $itk_component(orderedsumlabel) -row 4 -column 0 -sticky w -padx 5 -pady 3
	grid $itk_component(orderedsum) -row 4 -column 1 -sticky e -padx 10
	grid $itk_component(deliveredsumlabel) -row 5 -column 0 -sticky w -padx 5 -pady 3
	grid $itk_component(deliveredsum) -row 5 -column 1 -sticky e -padx 10
	
	grid columnconfigure $itk_interior 1 -weight 1
	grid rowconfigure $itk_interior 2 -weight 1

	## Keyboard bindings:	
	bind $itk_component(list) <KeyPress-plus> "$this handleEvent add"
	bind $itk_component(list) <KeyPress-KP_Add> "$this handleEvent add"
	bind $itk_component(list) <KeyPress-minus> "$this handleEvent subtract"
	bind $itk_component(list) <KeyPress-KP_Subtract> "$this handleEvent subtract"
	bind $itk_component(list) <KeyPress-Escape> "$this clearSelection"
	bind $itk_component(list) <KeyPress-BackSpace> "$this clearSelection"
	bind $itk_component(list) <KeyPress-KP_Delete> "$this clearSelection"

	## Mouse bindings:	
	bind $itk_component(list) <ButtonPress-3> "$this clearSelection"

	## Now initialize the widget.
	eval itk_initialize $args

	## Define a special style for this window's hlist price component.
	tixDisplayStyle text -stylename $this.anchoreast -refwindow $itk_component(list) -anchor e
}

## Configure widget by option.
itcl::configbody BillView::descriptionlabel \
{
	$itk_component(list) header configure 6 -text $itk_option(-descriptionlabel)
}
itcl::configbody BillView::keylabel \
{
	$itk_component(list) header configure 5 -text $itk_option(-keylabel)
}
itcl::configbody BillView::marklabel \
{
	$itk_component(list) header configure 2 -text $itk_option(-marklabel)
}
itcl::configbody BillView::pricelabel \
{
	$itk_component(list) header configure 7 -text $itk_option(-pricelabel)
}
itcl::configbody BillView::servicelabel \
{
	$itk_component(list) header configure 1 -text $itk_option(-servicelabel)
}
itcl::configbody BillView::speciallabel \
{
	$itk_component(list) header configure 4 -text $itk_option(-speciallabel)
}
itcl::configbody BillView::statelabel \
{
	$itk_component(list) header configure 0 -text $itk_option(-statelabel)
}
itcl::configbody BillView::taxlabel \
{
	$itk_component(list) header configure 3 -text $itk_option(-taxlabel)
}

## Configure the widget as the "billinfo" option is configured.
itcl::configbody BillView::billinfo \
{
	## Put the bill info into an array, we need all the data.
	array set BILL_INFO $itk_option(-billinfo)

	## Apply the serial number, create time and the sums 
	## of the bill info to the widget.
	configure \
		-serialnumber $BILL_INFO(-tag) \
		-createtime [ clock format $BILL_INFO(-create_time) -format $itk_option(-createtimeformat) ] \
		-showedsum [ formatCurrency $BILL_INFO(-showed_sum) ] \
		-orderedsum [ formatCurrency $BILL_INFO(-ordered_sum) ] \
		-deliveredsum [ formatCurrency $BILL_INFO(-delivered_sum) ]

	## Save the anchor and the selection of the item list.
	set ANCHOR [ $itk_component(list) info anchor ]
	set SELECTION [ $itk_component(list) selection get ]
	
	## Delete the item list contents.
	$itk_component(list) delete all

	## Add all the given items to the list.
	set ENTRY_PATH 0
	catch { unset selectable_items_unique_array }
	array set selectable_items_unique_array {}
	foreach ITEM $BILL_INFO(-items) \
	{
		## Make an array of the current item.
		array set ITEM_ARRAY $ITEM

		## Add this item to the array of selectable items.
		set selectable_items_unique_array($ENTRY_PATH) [ lindex $ITEM [ expr [ lsearch -exact $ITEM "-unique" ] +1 ] ]

		## Add an item view.
		$itk_component(list) add $ENTRY_PATH

		## Configure the item view state column.
		set STATE ""	
		if $ITEM_ARRAY(-cancelled) \
		{
			## The item is cancelled.
			if $ITEM_ARRAY(-delivered) \
			{
				## The item was delivered, but cancelled later.
				set STATE $itk_option(-statecancelleddelivered)
			} \
			{
				## The item was ordered, but cancelled later.
				set STATE $itk_option(-statecancelledordered)
			}
		} \
		{
			## The item is not cancelled.
			if $ITEM_ARRAY(-ordered) \
			{
				## The item is ordered.
				if $ITEM_ARRAY(-cancel) \
				{
					## The Item is a ordered cancel entry.
					set STATE $itk_option(-stateorderedcancel)
				} \
				{
					if $ITEM_ARRAY(-delivered) \
					{
						## The item is ordered and delivered.
						set STATE $itk_option(-statedelivered)
					} \
					{
						## The item is ordered.
						set STATE $itk_option(-stateordered) 
					}
				}
			} \
			{
				## The item is not ordered.
				if $ITEM_ARRAY(-cancel) \
				{
					## The Item is a cancel entry.
					set STATE $itk_option(-statecancel)
				}
			}
		}

		$itk_component(list) item create $ENTRY_PATH 0 -itemtype text -text $STATE

		## Configure the item view service column
		switch $ITEM_ARRAY(-service) \
		{
			bar     { $itk_component(list) item create $ENTRY_PATH 1 -itemtype text -text $itk_option(-servicebar) } 
			kitchen { $itk_component(list) item create $ENTRY_PATH 1 -itemtype text -text $itk_option(-servicekitchen) } 
		}

		## Configure the item view order column.
		switch $ITEM_ARRAY(-order) \
		{
			0 { $itk_component(list) item create $ENTRY_PATH 2 -itemtype text -text $itk_option(-markoff) } 
			1 { $itk_component(list) item create $ENTRY_PATH 2 -itemtype text -text $itk_option(-markon) } 
		}

		## Configure the item view tax column
		switch $ITEM_ARRAY(-tax) \
		{
			0 { $itk_component(list) item create $ENTRY_PATH 3 -itemtype text -text $itk_option(-taxoff) } 
			1 { $itk_component(list) item create $ENTRY_PATH 3 -itemtype text -text $itk_option(-taxon) } 
		}

		## Configure the item view key column
		$itk_component(list) item create $ENTRY_PATH 5 -itemtype text -text $ITEM_ARRAY(-key)

		## Configure the item view name column
		$itk_component(list) item create $ENTRY_PATH 6 -itemtype text -text $ITEM_ARRAY(-name) 

		## Configure the item view price column.
		set PRICE_FORMATTED [ formatCurrency [ expr $ITEM_ARRAY(-price) ] ]
		if { $ITEM_ARRAY(-multiplier) < 2 } \
		{
			## The multiplier is less than two. 
			$itk_component(list) item create $ENTRY_PATH 7 -itemtype text -text " $PRICE_FORMATTED " -style $this.anchoreast

			## Add item extras.
			set EXTRA_COUNT 0
			foreach EXTRA $ITEM_ARRAY(-extras) \
			{
				## Add this item extra to the array of selectable items.
				set selectable_items_unique_array(${ENTRY_PATH}.$EXTRA_COUNT) [ lindex $EXTRA [ expr [ lsearch -exact $EXTRA "-unique" ] +1 ] ]

				## Add another line to this item.
				$itk_component(list) add ${ENTRY_PATH}.$EXTRA_COUNT 
		
				## Configure the latest extra line.
				$itk_component(list) item create ${ENTRY_PATH}.$EXTRA_COUNT 6 -itemtype text -text [ lindex $EXTRA [ expr [ lsearch -exact $EXTRA "-name" ] + 1 ] ]

				set PRICE_FORMATTED [ formatCurrency [ expr [ lindex $EXTRA [ expr [ lsearch -exact $EXTRA "-price" ] + 1 ] ] ] ]
				$itk_component(list) item create ${ENTRY_PATH}.$EXTRA_COUNT 7 -itemtype text -text " $PRICE_FORMATTED " -style $this.anchoreast

				## Next extra.
				incr EXTRA_COUNT
			}
		} \
		{
			## The multiplier is greater than one.
			$itk_component(list) item create $ENTRY_PATH 7 -itemtype text -text "(${PRICE_FORMATTED})" -style $this.anchoreast

			## Add item extras.
			set EXTRA_COUNT 0
			foreach EXTRA $ITEM_ARRAY(-extras) \
			{
				## Add this item extra to the array of selectable items.
				set selectable_items_unique_array(${ENTRY_PATH}.$EXTRA_COUNT) [ lindex $EXTRA [ expr [ lsearch -exact $EXTRA "-unique" ] +1 ] ]

				## Add another line to this item.
				$itk_component(list) add ${ENTRY_PATH}.$EXTRA_COUNT 
		
				## Configure the latest extra line.
				$itk_component(list) item create ${ENTRY_PATH}.$EXTRA_COUNT 6 -itemtype text -text [ lindex $EXTRA [ expr [ lsearch -exact $EXTRA "-name" ] + 1 ] ]

				set PRICE_FORMATTED [ formatCurrency [ expr [ lindex $EXTRA [ expr [ lsearch -exact $EXTRA "-price" ] + 1 ] ] ] ]
				$itk_component(list) item create ${ENTRY_PATH}.$EXTRA_COUNT 7 -itemtype text -text "($PRICE_FORMATTED)" -style $this.anchoreast

				## Next extra.
				incr EXTRA_COUNT
			}

			## Add another line to this item.
			$itk_component(list) add ${ENTRY_PATH}.multiplier

			## Configure the multiplier line.
			$itk_component(list) item create ${ENTRY_PATH}.multiplier 6 -itemtype text -text "x $ITEM_ARRAY(-multiplier)"

			## The price is that of the article plus all extras, multiplied.
			set PRICE $ITEM_ARRAY(-price)
			foreach EXTRA $ITEM_ARRAY(-extras) \
			{
				## Add the price of the extra.
				set PRICE [ expr $PRICE + [ lindex $EXTRA [ expr [ lsearch -exact $EXTRA "-price" ] + 1 ] ] ]
			}

			## Actually configure the item view price column.
			set PRICE_FORMATTED [ formatCurrency [ expr $PRICE * $ITEM_ARRAY(-multiplier) ] ]
			$itk_component(list) item create ${ENTRY_PATH}.multiplier 7 -itemtype text -text "$PRICE_FORMATTED " -style $this.anchoreast 
		}

		## Scroll to the last item.
		$itk_component(list) see $ENTRY_PATH

		## Next item.
		incr ENTRY_PATH
		unset ITEM_ARRAY
	}

	## Restore the anchor and the selection of the item list.
	catch { $itk_component(list) anchor set $ANCHOR }
	foreach SELECTED_ITEM $SELECTION { catch { $itk_component(list) selection set $SELECTED_ITEM } }
}

## Clear the selection.
itcl::body BillView::clearSelection {} \
{
	$itk_component(list) selection clear
	$itk_component(list) anchor clear
}

## Return the selection.
itcl::body BillView::getSelection {} \
{
	## Get the selection of the indices in the list.
	set SELECTION [ $itk_component(list) selection get ]

	## Go through all the selected items.
	set RESULT {}
	foreach INDEX $SELECTION \
	{
		## Separate the entry and subentry field.
		set INDEX_LIST [ split $INDEX . ]
		set ENTRY [ lindex $INDEX_LIST 0 ]
		set SUBENTRY [ lindex $INDEX_LIST 1 ]

		## Get the unique field of the item.
		set ITEM_UNIQUE $selectable_items_unique_array($ENTRY)

		## Switch by the subentry type.
		switch $SUBENTRY \
		{
			{}         { lappend RESULT $ITEM_UNIQUE } 
 			multiplier { lappend RESULT [ join [ list $ITEM_UNIQUE multiplier ] . ] } 
			default \
			{
				##  Get the unique field of the extra.
				set EXTRA_UNIQUE $selectable_items_unique_array(${ENTRY}.$SUBENTRY)

				## Append the newly formed selection.
				lappend RESULT [ join [ list $ITEM_UNIQUE $EXTRA_UNIQUE ] . ]
			}
		}
	}

	## Return the complete reformed selection.
	return $RESULT
}

## Get the tag of this bill view object.
itcl::body BillView::getTag {} \
{
	lindex $itk_option(-billinfo) [ expr [ lsearch -exact $itk_option(-billinfo) "-tag" ] + 1 ] 
}

## Handle an event by this object.
itcl::body BillView::handleEvent { EVENT_TYPE_ARG args } \
{
	## Switch by event type.
	switch $EVENT_TYPE_ARG \
	{
		add \
		{
			## Get the selection.
			set SELECTION [ getSelection ]

			## Check all the selection entries are item main or multiplier entries.
			foreach ENTRY $SELECTION \
			{
				## Get the subentry.
				set SUBENTRY [ lindex [ split $ENTRY . ] 1 ]

				## If there is a wrong type ignore the user event.
				if { ( $SUBENTRY != "" ) && ( $SUBENTRY != "multiplier" ) } return
			}

			## They are. Go through all entries.
			set ITEM_UNIQUE_LIST {}
			foreach ENTRY $SELECTION { lappend ITEM_UNIQUE_LIST [ lindex [ split $ENTRY . ] 0 ] }
		
			## Increment the multiplier entries:
			$itk_option(-tkkassecommand) incrementBillItemMultiplier [ getTag ] $ITEM_UNIQUE_LIST
		}
		changeOrder \
		{
			## Get the selection.
			set SELECTION [ getSelection ]
	
			## Go through all items in the selection.
			set ITEM_UNIQUE_LIST {}
			foreach ENTRY $SELECTION \
			{
				## Get the subentries.
				set ENTRY_LIST [ split $ENTRY . ]

				## Ignore all non-main items.
				if { [ lindex $ENTRY_LIST 1 ] == "" } { lappend ITEM_UNIQUE_LIST [ lindex $ENTRY_LIST 0 ] }
			}

			## Change the order counter.
			$itk_option(-tkkassecommand) changeOrderOfItemFromBill [ getTag ] $ITEM_UNIQUE_LIST
		}
		changeService \
		{
			## Get the selection.
			set SELECTION [ getSelection ]
	
			## Go through all items in the selection.
			set ITEM_UNIQUE_LIST {}
			foreach ENTRY $SELECTION \
			{
				## Get the subentries.
				set ENTRY_LIST [ split $ENTRY . ]

				## Ignore all non-main items.
				if { [ lindex $ENTRY_LIST 1 ] == "" } { lappend ITEM_UNIQUE_LIST [ lindex $ENTRY_LIST 0 ] }
			}

			## Change the service.
			$itk_option(-tkkassecommand) changeServiceOfItemFromBill [ getTag ] $ITEM_UNIQUE_LIST
		}
		changeTax \
		{
			## Get the selection.
			set SELECTION [ getSelection ]
	
			## Go through all items in the selection.
			set ITEM_UNIQUE_LIST {}
			foreach ENTRY $SELECTION \
			{
				## Get the subentries.
				set ENTRY_LIST [ split $ENTRY . ]

				## Ignore all non-main items.
				if { [ lindex $ENTRY_LIST 1 ] == "" } { lappend ITEM_UNIQUE_LIST [ lindex $ENTRY_LIST 0 ] }
			}

			## Change the tax.
			$itk_option(-tkkassecommand) changeTaxOfItemFromBill [ getTag ] $ITEM_UNIQUE_LIST
		}
		deliver \
		{
			## Get the selection.
			set SELECTION [ getSelection ]
	
			## Go through all items in the selection.
			set ITEM_UNIQUE_LIST {}
			foreach ENTRY $SELECTION \
			{
				## Get the subentries.
				set ENTRY_LIST [ split $ENTRY . ]

				## Ignore all non-main items.
				if { [ lindex $ENTRY_LIST 1 ] == "" } { lappend ITEM_UNIQUE_LIST [ lindex $ENTRY_LIST 0 ] }
			}

			## Deliver the items.
			$itk_option(-tkkassecommand) deliverItemFromBill [ getTag ] $ITEM_UNIQUE_LIST
		}
		deliver \
		{
			## Get the selection.
			set SELECTION [ getSelection ]
	
			## Go through all items in the selection.
			set ITEM_UNIQUE_LIST {}
			foreach ENTRY $SELECTION \
			{
				## Get the subentries.
				set ENTRY_LIST [ split $ENTRY . ]

				## Ignore all non-main items.
				if { [ lindex $ENTRY_LIST 1 ] == "" } { lappend ITEM_UNIQUE_LIST [ lindex $ENTRY_LIST 0 ] }
			}

			## Deliver the items.
			$itk_option(-tkkassecommand) deliverItemFromBill [ getTag ] $ITEM_UNIQUE_LIST
		}
		order \
		{
			## Get the selection.
			set SELECTION [ getSelection ]
	
			## Go through all items in the selection.
			set ITEM_UNIQUE_LIST {}
			foreach ENTRY $SELECTION \
			{
				## Get the subentries.
				set ENTRY_LIST [ split $ENTRY . ]

				## Ignore all non-main items.
				if { [ lindex $ENTRY_LIST 1 ] == "" } { lappend ITEM_UNIQUE_LIST [ lindex $ENTRY_LIST 0 ] }
			}

			## Order the items.
			$itk_option(-tkkassecommand) orderItemFromBill [ getTag ] $ITEM_UNIQUE_LIST
		}
		subtract \
		{
			## Get the selection.
			set SELECTION [ getSelection ]

			## Check all the selection entries are all the same type. Get the type
			set SELECTION_TYPE {}
			foreach ENTRY $SELECTION \
			{
				## Get the subentry.
				set SUBENTRY [ lindex [ split $ENTRY . ] 1 ]

				## Get the type of the current subentry.
				switch $SUBENTRY \
				{
					""  \
					{
						## Main entry.
						set SUBENTRY_TYPE "main"
					}
					"multiplier" \
					{
						## Multiplier entry.
						set SUBENTRY_TYPE "multiplier"
					}
					default \
					{
						## Article extra entry.
						set SUBENTRY_TYPE "extra"
					}
				}

				## Check the type, if a type is already assigned.
				## If the type has changed, ignore the user event.
				if { ( $SELECTION_TYPE != {} ) && ( $SELECTION_TYPE != $SUBENTRY_TYPE ) } return
				
				## The type is ok or not set yet. Set it.
				set SELECTION_TYPE $SUBENTRY_TYPE
			}

			## Switch by the type of the selection.
			switch $SELECTION_TYPE \
			{
				"main"  \
				{
					## Main entries. Go through all entries.
					set ITEM_UNIQUE_LIST {}
					foreach ENTRY $SELECTION { lappend ITEM_UNIQUE_LIST [ lindex [ split $ENTRY . ] 0 ] }

					## Cancel items.
					$itk_option(-tkkassecommand) cancelItemFromBill [ getTag ] $ITEM_UNIQUE_LIST
				}
				"multiplier" \
				{
					## Multiplier entries. Go through all entries.
					set ITEM_UNIQUE_LIST {}
					foreach ENTRY $SELECTION { lappend ITEM_UNIQUE_LIST [ lindex [ split $ENTRY . ] 0 ] }

					## Decrement the multiplier entries.
					$itk_option(-tkkassecommand) decrementBillItemMultiplier [ getTag ] $ITEM_UNIQUE_LIST
				}
				"extra" \
				{
					## Go through all entries.
					foreach ENTRY $SELECTION \
					{
						## Get the subentries.
						set ENTRY_LIST [ split $ENTRY . ]
 
						## Article extra entry. Remove the extra.
						$itk_option(-tkkassecommand) removeExtraFromBillItem [ getTag ] [ lindex $ENTRY_LIST 0 ] [ lindex $ENTRY_LIST 1 ]
					}
				}
			}
		}
		addExtra \
		{
			## Get the selection.
			set SELECTION [ getSelection ]

			## Check for no selection.
			if { $SELECTION == {} } \
			{ 
				## Yes. Add the extra to the last entry.
				eval $itk_option(-tkkassecommand) addExtraToBillItem [ getTag ] end $args
			}

			## Check for a single selection.
			if { [ llength $SELECTION ] == 1 } \
			{
				## Get the subentries.
				set ENTRY_LIST [ split $SELECTION . ]
				set ITEM_UNIQUE [ lindex $ENTRY_LIST 0 ]
				set SUBENTRY [ lindex $ENTRY_LIST 1 ]
 
				## The following operations only work for single selections.
				## Is this a main entry?
				if { $SUBENTRY == "" } \
				{
					## Yes. Add the extra to the selected entry.
					eval $itk_option(-tkkassecommand) addExtraToBillItem [ getTag ] $ITEM_UNIQUE $args
				}
			}
		}
		default { $itk_option(-tkkassecommand) $EVENT_TYPE_ARG [ getTag ] }
	}
}



## Select frame class:
itk::usual SelectFrame \
{
	keep -background -cursor -foreground
}

itcl::class SelectFrame \
{
	inherit ::itk::Widget

	private variable UNIQUE 0
	private variable FRAMELIST {}
	private variable SELECTED {}

	constructor {args} {}
	
	public method index     {args}
	public method childsite {TAG}
	public method add       {args}
	public method insert    {args}
	public method delete    {args}
	public method flush     {}
	public method deselect  {}
	public method select    {TAG}
	public method get       {}

	protected method _reconfigure
}

## Select frame class constructor:
itcl::body SelectFrame::constructor {args} \
{
	## Now initialize the widget.
	eval itk_initialize $args
}


## Get the index of the frame.
itcl::body SelectFrame::index {INDEX} \
{
	if { [ llength $FRAMELIST ] > 0 } \
	{
		if { [ regexp {(^[0-9]+$)} $INDEX ] } \
		{
			if { $INDEX < [ llength $FRAMELIST ] } \
			{
				return $INDEX
			} \
			{
				error "SelectFrame index \"$index\" is out of range"
			}
	    
		} \
		{
			if { $INDEX == "end" } \
			{
				return [ expr [ llength $FRAMELIST ] - 1 ]
			} \
			{
				if { [ set IDX [ lsearch $FRAMELIST $INDEX ] ] != -1} \
				{
					return $IDX
				}
				error "bad SelectFrame index \"$INDEX\": must be number, end, or pattern"
			}
			error "SelectFrame \"$itk_component(hull)\" has no frames"
		}
	}
}

## Return the childsite window
itcl::body SelectFrame::childsite {TAG} \
{
	if [ catch { set itk_component($TAG) } RESULT ] \
	{
		return
	} \
	{
		return $RESULT
	}
}

## Add a frame to select.
itcl::body SelectFrame::add {TAG args} \
{
	itk_component add $TAG \
	{
		frame $itk_component(hull).[ incr UNIQUE ]
	} \
	{
		usual
	}
    
	if { $args != "" } \
	{
		uplevel $itk_component($TAG) configure $args
	}
    
	lappend FRAMELIST $TAG

	## Return the frame.
	return $itk_component($TAG)
}

## Insert a frame to select at a distinct position.
itcl::body SelectFrame::insert {INDEX TAG args} \
{
	itk_component add $TAG \
	{
		frame $itk_component(hull).[ incr UNIQUE ]
	} \
	{
		usual
	}
    
	if { $args != "" } \
	{
		uplevel $itk_component($TAG) configure $args
	}
    
	set INDEX [ index $INDEX ]
	set FRAMELIST [ linsert $FRAMELIST $INDEX $TAG ]

	## Return the frame.
	return $itk_component($TAG)
}

## Delete a frame.
itcl::body SelectFrame::delete {INDEX} \
{
	set INDEX [ index $INDEX ]
	set TAG [ lindex $FRAMELIST $INDEX ]

	deselect
	destroy $itk_component($TAG)
    
	set FRAMELIST [ lreplace $FRAMELIST $INDEX $INDEX ]
    
	update idletasks
}

## Clear the select frame widget.
itcl::body SelectFrame::flush {} \
{
}

## Deselect all frames inside the select frame widget.
itcl::body SelectFrame::deselect {} \
{
	catch { pack forget $itk_component($SELECTED) }
	set SELECTED {}
}

## Select a distinct frame inside the select frame widget.
itcl::body SelectFrame::select {TAG} \
{
	deselect

	set SELECTED $TAG
	catch { pack $itk_component($TAG) -fill both -expand 1 }
	
	update idletasks
}

## Get the index of the selected frame.
itcl::body SelectFrame::get {} \
{
	return $SELECTED
}



## List of users view widget class:
option add *UsersView.labelText [ msgcat::mc <UsersView.labelText> ] startupFile

itk::usual UsersView \
{
	keep -background -cursor -font -foreground
}

itcl::class UsersView \
{
	inherit ::itk::Widget

	constructor {args} {}
	
	itk_option define -users users Users {}
	itk_option define -command command Command {}

	public method getLogin { args }	
	public method getName { args }	
	public method select {}
	public method mouseselect { Y }
}

## List of users view widget class constructor:
itcl::body UsersView::constructor {args} \
{
	## Compose the widget.
	itk_component add label \
	{
		label $itk_interior.label -relief groove
	} \
	{
		rename -font -labelfont labelFont LabelFont
		rename -text -labeltext labelText LabelText
	}

	itk_component add list \
	{
		listbox $itk_interior.list \
			-height 3 \
			-yscrollcommand "$itk_interior.sy set"
	} \
	{
	}

	itk_component add sy \
	{
		scrollbar $itk_interior.sy \
			-orient vertical \
			-command "$itk_interior.list yview"
	} \
	{
	}

	## Pack it.
	pack $itk_component(label) -side top   -fill x
	pack $itk_component(sy)    -side right -fill y
	pack $itk_component(list)  -side left  -fill both -expand yes -padx 1 -pady 1

	## Key bindings:
	bind $itk_component(list) <KeyRelease-Up>   "$this select"
	bind $itk_component(list) <KeyRelease-Down> "$this select"

	## Mouse bindings:
	bind $itk_component(list) <ButtonPress-1> "$this mouseselect %y"

	## Additional bindings:
	bind $itk_component(list) <FocusIn>  "$this select"
	bind $itk_component(list) <FocusOut> "$this select"

	## Now initialize the widget.
	eval itk_initialize $args
}

## Configure the widget as the "users" option is configured.
itcl::configbody UsersView::users \
{
	## Delete all current entries in the listbox.
	$itk_component(list) delete 0 end

	## Adjust the listbox.
	foreach USER_INFO $itk_option(-users) \
	{
		## Put the full name of the current user into the listbox.
		$itk_component(list) insert end [ lindex $USER_INFO [ expr [ lsearch -exact $USER_INFO "-name" ] + 1 ] ]
	}
}

## Get the login name for an entry.
itcl::body UsersView::getLogin { args } \
{
	## Get the user info record of the selected element.
	set USER_INFO [ lindex $itk_option(-users) [ $itk_component(list) index $args ] ]

	## Return the login name of this user.
	lindex $USER_INFO [ expr [ lsearch -exact $USER_INFO "-login" ] + 1 ]
}

## Get the full name for an entry.
itcl::body UsersView::getName { args } \
{
	## Get the user info record of the selected element.
	set USER_INFO [ lindex $itk_option(-users) [ $itk_component(list) index $args ] ]

	## Return the full name of this user.
	lindex $USER_INFO [ expr [ lsearch -exact $USER_INFO "-name" ] + 1 ]
}

## Select an entry:
itcl::body UsersView::select {} \
{
	## Get the active user entry.
	## Do some substitutions with the select handler and evaluate the result.
	regsub -all {\\|&} [ getLogin active ] {\\\0} SUBST
	regsub -all "%n" $itk_option(-command) $SUBST COMMAND
	eval $COMMAND
}

## Select an entry with the mouse
itcl::body UsersView::mouseselect { Y } \
{
	## Get the user info record of the selected element.
	set USER_INFO [ lindex $itk_option(-users) [ $itk_component(list) nearest $Y ] ]

	## Get the login name for this user.
	set LOGIN_NAME [ lindex $USER_INFO [ expr [ lsearch -exact $USER_INFO "-login" ] + 1 ] ]

	## Do some substitutions with the select handler and evaluate the result.
	regsub -all {\\|&} $LOGIN_NAME {\\\0} SUBST
	regsub -all "%n" $itk_option(-command) $SUBST COMMAND
	eval $COMMAND
}

