/*
    Copyright (C) 2000  Dennis Roddeman
    email: dennis.roddeman@feat.nl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation 
    59 Temple Place, Suite 330, Boston, MA, 02111-1307, USA
*/

#include "tnsuplu.h"

#if ( SUPERLU_USE || SUPERLU_MT_USE || SUPERLU_DIST_USE )

#if SUPERLU_USE 
#include "dsp_defs.h"
#elif SUPERLU_MT_USE
#include "pdsp_defs.h"
#elif SUPERLU_DIST_USE
#include <math.h>
#include "superlu_ddefs.h"
#endif
#include "util.h"
 
long int solve_superlu( double *superlu_A, int *superlu_asub,
  int *superlu_xa, int superlu_nnz, double solve_b[], long int solve_nlocal,
  long int nthread )

{
  int n ,info;

#if SUPERLU_DIST_USE
  SuperMatrix A;
  int_t nprow=1, npcol=1, nrhs=1;
  SuperLUStat_t stat;
  superlu_options_t options;
  ScalePermstruct_t ScalePermstruct;
  LUstruct_t LUstruct;
  gridinfo_t grid;
  double *berr;
  int iam, count;
#else
  SuperMatrix A, B, L, U;
  int *perm_r, *perm_c;
  int perm_spec;
  int nrhs;
#endif

  /* Create Matrix A in the format expected by SuperLU */
#if SUPERLU_DIST_USE
    /*  get number of processors nprow, npcol */
  MPI_Comm_size(MPI_COMM_WORLD, &count);
  nprow=count; npcol=1;
    /* Initialize superlu process grid */
  superlu_gridinit(MPI_COMM_WORLD,nprow,npcol,&grid);
  iam  = grid.iam;
    /* if we dont belong to this grid , bug out  */
  if (iam >= nprow*npcol ) goto out;
#endif
  n = (int) solve_nlocal;
#if SUPERLU_DIST_USE
  if ( !iam ) {  
      /* broadcast matrices to other processes */
    MPI_Bcast(&n,1,mpi_int_t,0,grid.comm);
    MPI_Bcast(&superlu_nnz,1,mpi_int_t,0,grid.comm);
    MPI_Bcast(superlu_A,superlu_nnz,MPI_DOUBLE,0,grid.comm);
    MPI_Bcast(superlu_asub,superlu_nnz,mpi_int_t,0,grid.comm);
    MPI_Bcast(superlu_xa,n+1,mpi_int_t,0,grid.comm);
  } else {
      /* receive matrices from proc 0  */
    MPI_Bcast(&n,1,mpi_int_t,0,grid.comm);
    MPI_Bcast(&superlu_nnz,1,mpi_int_t,0,grid.comm);
      /* allocate storage for compressed column rep */
    dallocateA(n,superlu_nnz,&superlu_A,&superlu_asub,&superlu_xa);
    MPI_Bcast(superlu_A,superlu_nnz,MPI_DOUBLE,0,grid.comm);
    MPI_Bcast(superlu_asub,superlu_nnz,mpi_int_t,0,grid.comm);
    MPI_Bcast(superlu_xa,n+1,mpi_int_t,0,grid.comm);
  }
#endif
    /*  create compressed column matrix for A  */
  dCreate_CompCol_Matrix( &A, n, n, superlu_nnz, superlu_A, superlu_asub,
    superlu_xa, NC, _D, GE );

    /* Create right-hand side matrix B. */
  nrhs = 1;
#if ( SUPERLU_USE || SUPERLU_MT_USE )
  dCreate_Dense_Matrix( &B, n, nrhs, solve_b, n, DN, _D, GE ); 
  if ( !(perm_r = intMalloc(1000+n)) ) ABORT("Not enough memory for SuperLU.");
  if ( !(perm_c = intMalloc(1000+n)) ) ABORT("Not enough memory for SuperLU.");
    /* Get column permutation vector perm_c[] */
  perm_spec = 1;
  get_perm_c( perm_spec, &A, perm_c );
#endif

  /* Solve */
#if SUPERLU_USE 
  dgssv( &A, perm_c, perm_r, &L, &U, &B, &info );
#elif SUPERLU_MT_USE
  pdgssv( nthread, &A, perm_c, perm_r, &L, &U, &B, &info );
#elif SUPERLU_DIST_USE
    /* set the default input options */ 
  set_default_options(&options);
    /*  Initialize ScalePermstruct and LUstruct */
  ScalePermstructInit(n,n,&ScalePermstruct);
  LUstructInit(n,n,&LUstruct);
    /*  Initialize Stats vars */
  PStatInit(&stat); 
    /*  call the linear eqn solver */
 if (!(berr = doubleMalloc(nrhs)) ) ABORT("Malloc fails for berr[].");
 pdgssvx_ABglobal(&options, &A, &ScalePermstruct, solve_b, n, nrhs,
                  &grid, &LUstruct, berr, &stat, &info); 
    /*  print the stats */
 PStatPrint(&stat,&grid);
#endif

  /* De-allocate storage */
#if ( SUPERLU_USE || SUPERLU_MT_USE )
  SUPERLU_FREE(perm_r);
  SUPERLU_FREE(perm_c);
#elif SUPERLU_DIST_USE
  PStatFree(&stat);
  Destroy_LU(n,&grid, &LUstruct);
  ScalePermstructFree(&ScalePermstruct);
  LUstructFree(&LUstruct);
  SUPERLU_FREE(berr);
#endif
  Destroy_CompCol_Matrix(&A);
#if ( SUPERLU_USE || SUPERLU_MT_USE )
  Destroy_SuperMatrix_Store(&B);
  Destroy_SuperNode_Matrix(&L);
  Destroy_CompCol_Matrix(&U); 
#endif

#if SUPERLU_DIST_USE
/* release the process grid */
out:
  superlu_gridexit(&grid);
#endif

  if ( info ) {
    if ( info<n ) {
      printf( "\nError: SuperLU solver detected zero diagonal term for equation %d\n", info );
      exit(1);
    }
    else if ( info>=n ) {
      printf( "\nError: not enough memory for SuperLU solver.\n" );
      exit(1);
    }
  }
  return 1;
}

#endif
