// Copyright (C) 2002 Yoshua Bengio (bengioy@iro.umontreal.ca)
//                and Samy Bengio (bengio@idiap.ch)
//                and Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef RANDOM_INC
#define RANDOM_INC

#include "general.h"

namespace Torch {

/** Random functions.
    
    @author Samy Bengio (bengio@idiap.ch)
    @author Yoshua Bengio (bengioy@iro.umontreal.ca)
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
//@{

/*  
    Special functions.
    -----------------
*/

/// Returns the natural logarithm of the gamma function.
real  gammln(real xx);
/// Returns in #indices# #n_indices# shuffled. (between 0 and #n_indices#).
void getShuffledIndices(int *indices, int n_indices);
/// Shuffles tabular, which contains #n_elems# of size #size_elem#.
void shuffle(void *tabular, int size_elem, int n_elems);

/*   
     Utilities for random numbers generation. 
     ---------------------------------------
*/

/// Initializes the random number generator with the cpu time.
void  seed();
/// Initialzes the random number generator with the given long "x".
void  manual_seed(long x);
/// Returns the current seed used by the random number generator.
long  get_seed();
/// Returns a random number uniformly distributed between 0 and 1.
real  uniform();
/// Returns a random number uniformly distributed between a and b.
real  bounded_uniform(real a,real b);
/// Returns an exponential distributed random number.
real  expdev();
/// Returns a random number gaussian with mean 0 and standard deviation 1.
real  gaussian_01();
/// Returns a random number gaussian with mean "mu" and standard dev "sigma".
real  gaussian_mu_sigma(real mu, real sigma);
/// Returns a gamma distributed random number.
real  gamdev(int ia);
/// Returns a poisson random number with lambda = "xm".
real  poidev(real xm);
/// Returns a binomial random number with probability = 'pp' and trials number = 'n'.
real  bnldev(real pp, int n);

//@}


}

#endif


