/***********************************************************************
* File name: ALPS.c                                                    *
* Written by: Bruce Kall                                               *
* Created:  2/2000                                                     *
* Language: C                                                          *
* Version: 1.0                                                         *
* Purpose: Routines for manipulating the ALPS Glidepad/Stickpointer    *
*                                                                      *
***********************************************************************/


 
/* Copyright (c) 1998-2000 Bruce Kalk <kall@compass.com>
 *
 * Currently maintained by: Bruce Kall, <kall@compass.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   $Id: ALPS.c,v 1.4 2000/10/31 18:02:45 cph Exp bruce $
 */
 

/*$Log: ALPS.c,v $
 *Revision 1.4  2000/10/31 18:02:45  cph
 *Move copyright into --help screen.
 *
 *Revision 1.3  2000/10/31 17:43:12  cph
 *ACK now AUX_ACK.
 *
 *Revision 1.2  2000/09/29 20:18:13  bruce
 *Added in code for ALPS_GLIDEPAD (single pad, no stick).
 *
 *Revision 1.1  2000/09/28 13:44:35  bruce
 *Initial revision
 **/

static char rcsid[]="$Header: /home/bruce/linux-stuff/tpconfig/tpconfig-3.1.1/RCS/ALPS.c,v 1.4 2000/10/31 18:02:45 cph Exp bruce $";


#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <assert.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <getopt.h>
#include <termios.h>
#include "tpconfig.h"

extern int ps2_write(int fd,char *buffer,int num_bytes);
extern int ps2_read(int fd,char *buffer,int num_bytes);
extern void version_info(void);
extern void putbyte(int fd, byte b);
extern byte getbyte(int fd);



int ALPS_status(int fd,char *status)
{ /* ALPS_status */
byte c;
int num_written;
int num_read;
int i;

for(i = 0;i < 3;i++)
  {
  c = 0xf5;
  putbyte(fd,c);
  }
c = 0xe9;
if(DEBUG_LEVEL)
  printf("Writing [%x]\n",c);
num_written = ps2_write(fd,&c,1);
if(num_written != 1)
  {
  fprintf(stderr,"Error writing byte\n");
  return(ERROR);
  }
num_read = ps2_read(fd,status,4);
if(num_read != 4)
  {
  fprintf(stderr,"Error reading initial 4 byte response\n");
  return(ERROR);
  }

/* resend enable command for xmission of external mouse data */
c = 0xf4;
if(DEBUG_LEVEL)
  {
  printf("Resending Enable command\n");
  printf("Writing [%x]\n",c);
  }
num_written = ps2_write(fd,&c,1);
if(num_written != 1)
  {
  fprintf(stderr,"Error writing byte\n");
  return(0);
  }
if(DEBUG_LEVEL)
  {
  printf("Response 0 [%x]\n",status[0]);
  printf("Response 1 [%x]\n",status[1]);
  printf("Response 2 [%x]\n",status[2]);
  printf("Response 3 [%x]\n",status[3]);
  }
return(!ERROR);
} /* ALPS_status */

int is_ALPS_tap_enabled(int fd)
{ /* is_ALPS_tap_enabled */
byte response[4];
int tap_on = TRUE;
int error;

if(DEBUG_LEVEL)
  printf("\n\nChecking Whether ALPS Tap status is enabled or disabled\n");


error = ALPS_status(fd,response);
if(error)
  return(0);

if(touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT)
  {
  if(response[3] & 0x4)
    tap_on = TRUE;
  else
    tap_on = FALSE;
  }
else if(touchpad_type == ALPS_GLIDEPAD)
  {
  if(response[1] & 0x4)
    tap_on = TRUE;
  else
    tap_on = FALSE;
  }


if(DEBUG_LEVEL)
  {
  if(tap_on)
    printf("Done Checking Status of Tap Enabled/Disable, Tap is ON \n");
  else
    printf("Done Checking Status of Tap Enabled/Disable, Tap is OFF \n");
  }
return(tap_on);
} /* is_ALPS_tap_enabled */


int reset_ALPS(int fd)
{ /* reset_ALPS */
byte c;
int num_written;

c = 0xff;
if(DEBUG_LEVEL)
  printf("\nWriting Reset [%x], Reading Nothing\n",(int)c);
num_written = ps2_write(fd,&c,1);
if(num_written != 1)
  {
  fprintf(stderr,"Error writing reset byte\n");
  return(FALSE);
  }
/* need to wait 1.5 sec here for reset to complete */
usleep(1600000);
return(TRUE);
} /* reset_ALPS */

int ALPS_SP_tap(int fd,char enable)
{ /* ALPS_SP_tap */
byte c;
int i;
if(DEBUG_LEVEL)
  {
  if(enable)
    printf("\n\nEnabling SP tap\n");
  else
    printf("\n\nDisabling SP tap\n");
  }

for(i = 0;i < 3;i++)
  {
  c = 0xE6;
  putbyte(fd,c);
  }
c = 0xF3;
putbyte(fd,c);
if(enable)
  c = 0x14;
else
  c = 0x0a;

putbyte(fd,c);
if(DEBUG_LEVEL)
  printf("Done Enabling/Disabling SP tap\n");
return(!ERROR);
} /* ALPS_SP_tap */


int ALPS_through_mode(int fd,char set)
{ /* ALPS_through_mode */
byte c;
int i;
if(DEBUG_LEVEL)
  {
  if(set)
    printf("\n\nEnabling ALPS_through_mode\n");
  else
    printf("\n\nDisabling ALPS_through_mode\n");
  }

for(i = 0;i < 3;i++)
  {
  if(set)
    c = 0xE7;
  else
    c = 0xE6;
  putbyte(fd,c);
  }

c = 0xF5;
putbyte(fd,c);

/* We may receive 3 more bytes, ignore them */
tcflush(fd,TCIOFLUSH);
if(DEBUG_LEVEL)
  {
  if(set)
    printf("Done Enabling ALPS_through_mode\n\n\n");
  else
    printf("Done Disabling ALPS_through_mode\n\n\n");
  }
return(0);
} /* ALPS_through_mode */


int ALPS_GP_tap(int fd,char enable)
{ /* ALPS_GP_tap */
byte c;
byte response[4];
int i;
int num_read;
int num_written;
int error;

if(DEBUG_LEVEL)
  {
  if(enable)
    printf("\n\nEnabling GP tap\n");
  else
    printf("\n\nDisabling GP tap\n");
  }

if(touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT)
  {
  error = ALPS_through_mode(fd,(char) 1);
  if(error)
    {
    printf("Error turning on through mode\n");
    return(ERROR);
    }
  }

c = 0xE9;
if(DEBUG_LEVEL)
  printf("Writing Initial [%x]\n",(int)c);
num_written = ps2_write(fd,&c,1);
num_read = ps2_read(fd,response,4);
if(DEBUG_LEVEL)
  {
  printf("Just Read num = [%d] bytes",num_read);
  printf("\n");
  }
if((num_read != 4) || (response[0] != AUX_ACK))
    {
    printf("-----> Invalid response from Alps Glidepad/Glidepoint [%x] \n",(int)response[0]);
    return(ERROR);
    }
if(DEBUG_LEVEL)
  printf("Current Settings [%x] [%x] [%x]\n",response[1],response[2],response[3]);

for(i = 0;i < 2;i++)
  {
  c = 0xF5;
  putbyte(fd,c);
  }

if(enable)
  {
  if(DEBUG_LEVEL)
    printf("Enabling Tap\n");
  c = 0xF3;
  }
else
  {
  if(DEBUG_LEVEL)
    printf("Disabling Tap\n");
  c = 0xE8;
  }

putbyte(fd,c);

if(enable)
  c = 0x0A;
else
  c = 0x00;
putbyte(fd,c);

if(touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT)
  {
  error = ALPS_through_mode(fd,(char) 0);
  if(error)
    {
    printf("Error turning off through mode\n");
    return(ERROR);
    }
  if(DEBUG_LEVEL)
    printf("Done Enabling/Disabling GP tap\n");
  }

return(NO_ERROR);
} /* ALPS_GP_tap */

int is_ALPS(int fd)
{
#define NUM_SINGLES 10
static int singles[NUM_SINGLES * 3] ={
  0x33,0x2,0xa,
  0x53,0x2,0x0a,
  0x53,0x2,0x14,
  0x63,0x2,0xa,
  0x63,0x2,0x14,
  0x73,0x2,0xa,
  0x63,0x2,0x28,
  0x63,0x2,0x3c,
  0x63,0x2,0x50,
  0x63,0x2,0x64};
#define NUM_DUALS 3
static int duals[NUM_DUALS * 3]={
  0x20,0x2,0xe, /* as reported by William Moran, billm@adv-techcenter.com CPxH*/
  0x22,0x2,0xa,
  0x22,0x2,0x14};

byte c;
byte response[4];
int i;
int num_read,num_written;
int return_value = 0;
int tap_on;
int error;

  
tap_on = TRUE;
/* Get initial statis of ALPS device so we can reset after reset below */
/* tap_on = is_ALPS_tap_enabled(fd);*/

for(i = 0;i < 3;i++)
  {
  c = 0xe7;
  putbyte(fd,c);
  }
c = 0xe9;
if(DEBUG_LEVEL)
  printf("Writing Initial [%x] in is_ALPS\n",(int)c);
num_written = ps2_write(fd,&c,1);
if(num_written != 1)
  {
  fprintf(stderr,"Error writing Secondary Initial response\n");
  return(0);
  }
num_read = ps2_read(fd,response,4);
if(num_read != 4)
  {
  fprintf(stderr,"Error reading Initial 4 byte response\n");
  return(0);
  }

if(DEBUG_LEVEL)
  printf("ALPS Configuration Info [%2x][%2x][%2x]\n",response[1],response[2],response[3]);

return_value = 0;
for(i = 0;i < NUM_SINGLES;i++)
  {
  if((response[1] == singles[i * 3] && (response[2] == singles[i * 3 + 1]) &&
    response[3] == singles[i * 3 + 2]))
    {
    touchpad_type = ALPS_GLIDEPAD;
    return_value = 1;
    }
  }
if(return_value == 0)
  {
  for(i = 0;i < NUM_DUALS;i++)
    {
    if((response[1] == duals[i * 3]) && (response[2] == duals[i * 3 + 1]) &&
      (response[3] == duals[i * 3 + 2]))
      {
      touchpad_type = ALPS_STICKPOINTER_AND_GLIDEPOINT;
      return_value = 1;
      }
    }
  }

/* Older kernels may need this .. <2.4.x */
/* Newer kernels get an assertion error if this reset_ALPS */
/* is performed.  With 2.4.9 commenting this out fixes the */
/* assertion error on ALPS dual (touchpad/glidestick) */
/* I don't have one, but a single ALPS touchpad may need this */
/*
reset_ALPS(fd);
*/

if(return_value)
  { 
  if(!tap_on)
    { /* reset tap_on to off since we just had to do a reset */
    if(DEBUG_LEVEL)
      printf("Turning ALPS Tap Back OFF\n");
    if(touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT)
      {
      error = ALPS_GP_tap(fd,(char) 0);
      error = ALPS_SP_tap(fd,(char) 0);
      }
    else if(touchpad_type == ALPS_GLIDEPAD)
      error = ALPS_GP_tap(fd,(char) 0);
    }
  }

return(return_value);
} /* is_ALPS */




void ALPS_usage(char *progname) 
{
  copyright ();
  printf ("Usage: %s [OPTION]...\n", progname);
  printf ("Configure an ALPS GlidePad/GlidePoint.\n"
	  "\n"
	  "  -i, --info                    display current TouchPad configuration\n"
	  "  -t, --tapmode=[0-1]           display/set tapping mode:\n"
	  "                                   0 = tapping off\n"
	  "                                   1 = tapping on\n"
	  "  -r, --reset                   reset ALPS device\n"
	  "      --help                    display this help and exit\n"
	  "      --version                 output version information\n"
	  "\n"
	  "Report bugs to <kall@compass.com\n");
  exit(0);
}


void alps_functions(int c,int fd,char **argv)
  { /* alps_functions */
  int error;
  int status;
  int mode;
  switch (c) 
    {
    case 'h': ALPS_usage(argv[0]);
              break;
    case 'i': if(touchpad_type == ALPS_GLIDEPAD)
               printf("\nFound ALPS GlidePad\n\n");
             else if (touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT)
               printf("\nFound ALPS Dual StickPoint/Glidepad\n\n");
             else
               {
               fprintf(stderr,"Undefined touchpad found[%d]\n",touchpad_type);
               exit(1);
               }
              if(is_ALPS_tap_enabled(fd))
                printf("Tapping is ENABLED\n");
              else
                printf("Tapping is DISABLED\n");
              break;
    case 'x': reset_ALPS(fd);
              break;
    case 't': /* --tapmode */
  	      if (optarg) 
                {
  	        if((optarg[0]<'0') || (optarg[0]>'3'))
  	          fprintf(stderr, "Invalid tap mode. [Use 0-3]\n");
                else
                  {
  	          mode = optarg[0]-'0';
                  if(DEBUG_LEVEL)
                    printf("User Asked to set Tap mode to [%d]\n",mode);
                  if(mode > 0)
                    {
                    if(DEBUG_LEVEL)
                      printf("Turning ALPS Tap ON\n");
                    if(touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT)
                      {
                      error = ALPS_GP_tap(fd,(char) 1);
                      error = ALPS_SP_tap(fd,(char) 1);
                      }
                    else if(touchpad_type == ALPS_GLIDEPAD)
                      error = ALPS_GP_tap(fd,(char) 1);
                    }
                  else
                    { /* disable tap mode */
                    if(DEBUG_LEVEL)
                      printf("Turning ALPS Tap OFF\n");
                    if(touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT)
                      {
                      error = ALPS_GP_tap(fd,(char) 0);
                      error = ALPS_SP_tap(fd,(char) 0);
                      }
                    else if(touchpad_type == ALPS_GLIDEPAD)
                      error = ALPS_GP_tap(fd,(char) 0);
                    }
                  status = is_ALPS_tap_enabled(fd);
                  if(status)
                    printf("ALPS Tap is ON\n");
                  else
                    printf("ALPS Tap is OFF\n");
                  }
                }
              break;
    case 'v': /* --version */
      version_info();
      break;

    default:
      fprintf(stderr, "Unknown option %c.\n", (char)c);
      break;
    }

  }


