/***********************************************************************
 * 
 * File: pointing_device.c
 *
 * Author: Martin Fluch
 *
 *     Copyright (c) 1999 Martin Fluch, All rights reserved
 *     
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *     GNU General Public License for more details.
 * 
 *     To receive a copy of the GNU General Public License, please write
 *     to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *     Boston, MA 02111-1307 USA
 *
 ***********************************************************************/

#include <unistd.h>
#include <stdlib.h>
#include <linux/types.h>
#include <stdio.h>
#include <string.h>

#include <curses.h>
#include <menu.h>

#include "ntpctl.h"


#define REQTYPE_CMOS             0x01
#define REQTYPE_current_hardware 0x00

/***********************************************************************
 *
 * handle_internal_pd()
 *
 * see TP 770 Tech Ref page 101
 *
 ***********************************************************************/

int handle_internal_pd(struct menuinfo *mi, int action, int reqtype)
{
  smapi_outparm_t *result;
  static int tmp;
  int chconfig=FALSE;
  char *tmp_str = NULL;

  if((result=do_smapi(0x11,0x02,reqtype << 8,0,0,0,0,FALSE))==NULL) return 1;
 
  switch(action)
    {
    case ACTION_KEY_D:
    case ACTION_KEY_N:
      tmp=(result->wParm2 & 0x0200) | 0x0000;
      chconfig=TRUE;
      break;

    case ACTION_KEY_E:
    case ACTION_KEY_Y:
      tmp=(result->wParm2 & 0x0200) | 0x0100;
      chconfig=TRUE;
      break;

    case ACTION_KEY_A:
      tmp=(result->wParm2 & 0x0200) | 0x0400;
      chconfig=TRUE;
      break;
    }

  if(chconfig)
    {
      if(do_smapi(0x11,0x03,tmp | ((reqtype==REQTYPE_CMOS) ? 0x02 : 0x01),
		  0 ,0,0,0,TRUE)==NULL) return 1;
      if((result=do_smapi(0x11,0x02,reqtype << 8,0,0,0,0,FALSE))==NULL)
	return 1;
    }

  switch((result->wParm2 >> 8)& 0x05)
    {
    case 0x00: tmp_str="disabled"; break;
    case 0x01: tmp_str="enabled "; break;
    case 0x04: tmp_str="auto    "; break;
    }

  snprintf(mi->desc,MAXDESCSIZE,"%s  %s",tmp_str,
	   (result->wParm2 & 0x0001) ? "(ctrable)    " : "(not ctrable)");
 
  return 0;
}

/***********************************************************************
 *
 * handle_external_pd()
 *
 * see TP 770 Tech Ref page 101 (or better not :-)
 *
 * According to Thomas Hood 'Get Pointing Device State' returns
 * the information in parameter 2 and *not* 1. This seems to work on 
 * my TP 770.
 *
 * On the other hand 'Set Pointing Device State' seems to expect the
 * information in parameter 1 and not 2. This seems to be right for
 * the TP 770, too.
 *
 * And finaly, bit 0 and bit 1 in parameter 1 indicates where to
 * uodate: the first for the current hardware, the later for CMOS. 
 *
 ************************************************************************/

int handle_external_pd(struct menuinfo *mi, int action, int reqtype)
{
  smapi_outparm_t *result;
  int tmp = 0;
  int chconfig=FALSE;

  if((result=do_smapi(0x11,0x02,reqtype << 8,0,0,0,0,FALSE))==NULL) return 1;

  if((action==ACTION_NEXT)||(action==ACTION_PREV))
    action=(result->wParm2 & 0x0200) ? ACTION_KEY_D : ACTION_KEY_E;
  
  switch(action)
    {
    case ACTION_KEY_D:
    case ACTION_KEY_N:
      tmp=result->wParm2 & 0x0500;
      chconfig=TRUE;
      break;

    case ACTION_KEY_E:
    case ACTION_KEY_Y:
      tmp=(result->wParm2 | 0x0200) & 0x0700;
      chconfig=TRUE;
      break;
    }

  if(chconfig)
    {
      if(do_smapi(0x11,0x03,tmp | ((reqtype==REQTYPE_CMOS) ? 0x02 : 0x01),
		  0,0,0,0,TRUE)==NULL) return 1;
      if((result=do_smapi(0x11,0x02,reqtype << 8,0,0,0,0,FALSE))==NULL)
	return 1;
    }

  snprintf(mi->desc,MAXDESCSIZE,"%s  %s",
	   (result->wParm2 & 0x0200) ? "enabled "      : "disabled",
	   (result->wParm2 & 0x0002) ? "(ctrable)    " : "(not ctrable)");
 
  return 0;
}

/***********************************************************************
 *
 * pd_external_ch()
 *
 ***********************************************************************/

int pd_external_ch(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Ext. Pointing device (current)",MAXNAMESIZE);
      mi->help=helptext[HELP_ed];
    }

  return handle_external_pd(mi, action, REQTYPE_current_hardware);
}

/***********************************************************************
 *
 * pd_external_CMOS()
 *
 ***********************************************************************/

int pd_external_CMOS(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Ext. pointing device (CMOS)",MAXNAMESIZE);
      mi->help=helptext[HELP_ed];
    }

  return handle_external_pd(mi, action, REQTYPE_CMOS);
}

/***********************************************************************
 *
 * pd_internal_ch()
 *
 ***********************************************************************/

int pd_internal_ch(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Int. pointing device (current)",MAXNAMESIZE);
      mi->help=helptext[HELP_eda];
    }

  return handle_internal_pd(mi, action, REQTYPE_current_hardware);
}

/***********************************************************************
 *
 * pd_internal_CMOS()
 *
 ***********************************************************************/

int pd_internal_CMOS(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Int. Pointing device (CMOS)",MAXNAMESIZE);
      mi->help=helptext[HELP_eda];
    }

  return handle_internal_pd(mi, action, REQTYPE_CMOS);
}
