/*
	**
	** textfe.c 
	**
	** Text display frontend to network traffic visualiser
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: textfe.c,v 1.2 1999/02/04 10:22:42 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <glib.h>

#include "textfe.h"
#include "util.h"
#include "report.h"

static char			rcsid[] = "$Id: textfe.c,v 1.2 1999/02/04 10:22:42 dmiller Exp $";

/* Prototypes */
static void print_host(FILE *out, host_t *h);
static void print_peers(FILE *out, peer_t *p);
static void format_hostname(host_t *host, char *s, size_t s_len);
static const char *format_time(time_t t);

/* Displays summary of network traffic */
void text_display_summary(report_t *r, const char *output_file)
{
	host_t	*h;
	FILE		*out;
	
	if (output_file == NULL)
	{
		setlinebuf(stdout);
		out = stdout;
	} else
	{
		/* Open output file */
		out = fopen(output_file, "w");
		if (out == NULL)
		{
			fprintf(stderr, "Couldn't open file '%s' for writing.\n", output_file);
			return;
		}
	}
	
	fprintf(out, "*** Collection started at %s\n", format_time(r->summary_start));
	fprintf(out, "*** Collection concluded at %s\n", format_time(r->summary_finish));
	fprintf(out, "\n");

	h = r->hosts;
	while(h != NULL)
	{
		print_host(out, h);
		h = h->next;
	}

	fprintf(out, "Total active hosts:        %u\n", r->active_hosts);
	fprintf(out, "Total bytes:               %llu\n", r->total_bytes);
	fprintf(out, "Total packets:             %llu\n", r->total_packets);
	fprintf(out, "Total connection requests: %llu\n", r->total_connections);
	fprintf(out, "\n");
}

static void print_host(FILE *out, host_t *h)
{
	char			hostname[1024];

	format_hostname(h, hostname, sizeof(hostname));
	fprintf(out, "%s\n", hostname);

	if (h->n_peers != 0)
		print_peers(out, h->peers);
	
	fprintf(out, " [ %llu bytes sent, %llu bytes received ]\n", h->bytes_sent, h->bytes_received);
	fprintf(out, " [ %llu packets sent, %llu packets received ]\n", h->packets_sent, h->packets_received);
	fprintf(out, " [ %llu connection requests sent, %llu connection requests received ]\n", h->connections_sent, h->connections_received);
	fprintf(out, " [ First traffic recorded at %s ]\n", format_time(h->first_seen));
	fprintf(out, " [ Last traffic recorded at %s ]\n", format_time(h->last_seen));

	fprintf(out, "\n");
}

static void print_peers(FILE *out, peer_t *p)
{
	char			src_hostname[1024];
	char			dst_hostname[1024];
	
	while(p != NULL)
	{
		format_hostname(p->src, src_hostname, sizeof(src_hostname));
		format_hostname(p->dst, dst_hostname, sizeof(dst_hostname));
		
		fprintf(out, "\t%s -> %s (%llu bytes, %llu packets)\n", 
				 src_hostname, dst_hostname, p->bytes_sent, p->packets_sent);
		
		p = p->next;
	}
}

static void format_hostname(host_t *host, char *s, size_t s_len)
{
	static struct	in_addr i;

	i.s_addr = host->ip_addr;
	
	if (host->hostname != NULL)
		g_snprintf(s, s_len, "%s", host->hostname);
	else	
		g_snprintf(s, s_len, "Unknown (%s)", inet_ntoa(i));
}

static const char *format_time(time_t t)
{
	static char	ts[128];
	struct tm	*tm;
	
	tm = localtime(&t);
	
	strftime(ts, sizeof(ts), "%a, %d %b %Y %H:%M:%S %Z", tm);
	
	return(ts);
}
