/*
	**
	** read-report.c
	**
	** Routines to read a report generated by traffic-collect 
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: read-report.c,v 1.2 1999/02/04 10:22:42 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <glib.h>

#include "report.h"
#include "read-report.h"

#include "util.h"

static char rcsid[] = "$Id: read-report.c,v 1.2 1999/02/04 10:22:42 dmiller Exp $";

/* Prototypes */
void move_peers_to_hosts(void);
void copy_peer_to_host(peer_t *p, host_t *h);

/* Parser action function prototypes */
static int set_finish_time(const char *value);
static int end_host(const char *value);
static int end_peer(const char *value);
static int new_host(const char *value);
static int set_host_addr(const char *value);
static int set_host_bytes_received(const char *value);
static int set_host_bytes_sent(const char *value);
static int set_host_conreqs_received(const char *value);
static int set_host_conreqs_sent(const char *value);
static int set_host_first_traffic_time(const char *value);
static int set_host_last_traffic_time(const char *value);
static int set_host_name(const char *value);
static int set_host_packets_received(const char *value);
static int set_host_packets_sent(const char *value);
static int new_peer(const char *value);
static int set_peer_bytes(const char *value);
static int set_peer_conreqs(const char *value);
static int set_peer_dst_addr(const char *value);
static int set_peer_packets(const char *value);
static int set_peer_src_addr(const char *value);
static int set_start_time(const char *value);
static int set_total_bytes(const char *value);
static int set_total_conreqs(const char *value);
static int set_total_hosts(const char *value);
static int set_total_packets(const char *value);

/* Parser key->action table */
typedef int (*action_func_t)(const char *value);
struct parser_tag
{
	const char		*tag;
	action_func_t	action;
};
static struct parser_tag parser_tags[] = 
{
	{	"FINISH_TIME", 				set_finish_time	},
	{	"END_HOST", 					end_host	},
	{	"END_PEER", 					end_peer	},
	{	"HOST_ADDR", 					set_host_addr	},
	{	"HOST_BYTES_RECEIVED", 		set_host_bytes_received	},
	{	"HOST_BYTES_SENT", 			set_host_bytes_sent	},
	{	"HOST_CONREQS_RECEIVED", 	set_host_conreqs_received	},
	{	"HOST_CONREQS_SENT", 		set_host_conreqs_sent	},
	{	"HOST_FIRST_TRAFFIC_TIME", set_host_first_traffic_time	},
	{	"HOST_LAST_TRAFFIC_TIME", 	set_host_last_traffic_time	},
	{	"HOST_NAME", 					set_host_name	},
	{	"HOST_PACKETS_RECEIVED", 	set_host_packets_received	},
	{	"HOST_PACKETS_SENT", 		set_host_packets_sent	},
	{	"NEW_HOST", 					new_host	},
	{	"NEW_PEER", 					new_peer	},
	{	"PEER_BYTES", 					set_peer_bytes	},
	{	"PEER_CONREQS", 				set_peer_conreqs	},
	{	"PEER_DST_ADDR",				set_peer_dst_addr	},
	{	"PEER_PACKETS", 				set_peer_packets	},
	{	"PEER_SRC_ADDR",				set_peer_src_addr	},
	{	"START_TIME", 					set_start_time	},
	{	"TOTAL_BYTES", 				set_total_bytes	},
	{	"TOTAL_CONREQS", 				set_total_conreqs	},
	{	"TOTAL_HOSTS", 				set_total_hosts	},
	{	"TOTAL_PACKETS", 				set_total_packets	},
	{	(const char *)NULL,			(action_func_t)NULL }
};

/* Module-global variables */
static report_t		*report;			/* Report we are building */
static host_t			*host;			/* Host we are currently building */
static host_t			*last_host;		/* Last host in list */
static peer_t			*peer;			/* Peer we are currently building */
static peer_t			*peer_list;		/* List of all peers read from file */
static GHashTable		*host_addrs;	/* Hash table of host addresses */

/* Parse specified report file into report_t structure */
/* FIXME: rewrite parser using lex/yacc? */
report_t *parse_report(const char *report_file)
{
	FILE				*report_h;
	char				buffer[2048];
	char				key[128];
	char				value[2048];
	int				c;
	u_int64_t		line;
	int				items;
	
	if (report_file == NULL)
	{
		report_h = stdin;
	} else
	{
		/* Open report file */
		report_h = fopen(report_file, "r");
		if (report_h == NULL)
		{
			fprintf(stderr, "Couldn't open report file '%s' for reading.\n", report_file);
			return(NULL);
		}
	}
		
	/* Initialise report structures */
	report = util_zmalloc(sizeof(*report));
	host = NULL;
	last_host = NULL;
	peer = NULL;
	peer_list = NULL;

	/* Initialise host address hash table for */
	host_addrs = g_hash_table_new((GHashFunc)host_hash, (GCompareFunc)host_compare);
	
	/* Read file one line at a time */
	line = 0;
	while(fgets(buffer, sizeof(buffer), report_h) != NULL)
	{
		line++;
		
		/* Skip blank lines */
		if (buffer[0] == '\n')
			continue;
		
		/* Skip comments */
		if (buffer[0] == '#')
			continue;
		
		/* Parse a single line of the report into key/value */
		key[0] = '\0';
		value[0] = '\0';
		items = sscanf(buffer, " %s %s", key, value);
		if ((items < 1) || (items > 2))
		{
			fprintf(stderr, "Parse error at report line %llu.\n", line);
			fclose(report_h);
			return(NULL);
		}
		
		/* Check key against each key in key/action array */
		/* FIXME: rewrite using hash tables / gperf? */
		c = 0;
		while(parser_tags[c].tag != NULL)
		{
			/* Compare key */
			if (strcmp(parser_tags[c].tag, key) == 0)
			{
				/* If match, execute action - check for success */
				if (!parser_tags[c].action(value))
				{
					fprintf(stderr, "Parse error at report line %llu.\n", line);
					fclose(report_h);
					return(NULL);
				}
				break;
			}
			c++;
		}
		
		/* Report error if key not found */
		if (parser_tags[c].tag == NULL)
		{
			fprintf(stderr, "Parse error at report line %llu.\n", line);
			fclose(report_h);
			return(NULL);
		}
	}
	
	if (report_file != NULL)
		fclose(report_h);
	
	/* Move peers from big list to final locations in host entries */
	move_peers_to_hosts();

	/* Free hash table */
	g_hash_table_destroy(host_addrs);
	
	return(report);
}

/* Move peers from big list of all peers to final location in host */
/* entries. This cannot be done as the file is read as a peer may */
/* reference a host which is not in memory yet. */
/* This routine also fills in the pointers to source and destination */
/* hosts in the peer data structure */
void move_peers_to_hosts(void)
{
	host_t *src_h;
	host_t *dst_h;
	peer_t *p;
	peer_t *last_p;
	
	/* For each host, find relevant peers */
	p = peer_list;
	while (p != NULL)
	{
		src_h = g_hash_table_lookup(host_addrs, (gpointer)&(p->src_addr));
		dst_h = g_hash_table_lookup(host_addrs, (gpointer)&(p->dst_addr));

#ifdef DEBUG
		if ((src_h == NULL) || (dst_h == NULL))
		{
			if (src_h == NULL)
				fprintf(stderr, "Missing source address %x\n", p->src_addr);
			if (dst_h == NULL)
				fprintf(stderr, "Missing dest address %x\n", p->dst_addr);
			abort();
		}
#endif /* DEBUG */

		if ((src_h != NULL) && (dst_h != NULL))
		{
			p->src = src_h;
			p->dst = dst_h;
			src_h->n_peers++;
			dst_h->n_peers++;

			copy_peer_to_host(p, src_h);
			copy_peer_to_host(p, dst_h);
		}
		
		last_p = p;
		p = p->next;

		free(last_p);
	}
}

/* Copies a peer entry to a host record */
void copy_peer_to_host(peer_t *p, host_t *h)
{
	peer_t	*new_peer;

	new_peer = util_malloc(sizeof(*new_peer));
	memcpy(new_peer, p, sizeof(*new_peer));
	new_peer->next = NULL;
		
	/* Place peer at end of list */
	if (h->peers == NULL)
	{
		h->peers = new_peer;
	} else
	{
		p = h->peers;
		while(p->next != NULL)
			p = p->next;

		p->next = new_peer;
	}
}

/* Action functions for keys follow */

static int set_finish_time(const char *value)
{
	/* Parse finish time from report */
	if (sscanf(value, "%ld", &(report->summary_finish)) != 1)
		return(0);

#ifdef DEBUG
	 fprintf(stderr, "Finish time: %s", ctime(&(report->summary_finish)));
#endif /* DEBUG */

	return(1);
}

static int end_host(const char *value)
{
	/* Ensure that a host entry currently exists */
	if (host == NULL)
		return(0);
		
	/* Add host to end of report to preserve existing order */
	if (last_host == NULL)
		report->hosts = host;
	else
		last_host->next = host;

	last_host = host;

	/* Add host to address hash table */
	g_hash_table_insert(host_addrs, (gpointer)&(host->ip_addr), (gpointer)host);
	
	/* Invalidate current host entry */
	host = NULL;
	
#ifdef DEBUG
	 fprintf(stderr, "Finished host\n");
#endif /* DEBUG */

	return(1);
}

static int end_peer(const char *value)
{
	peer_t	*p;
	peer_t	*last_p;

#ifdef DEBUG
	 fprintf(stderr, "Finished peer\n");
#endif /* DEBUG */

	/* Ensure there is a current peer entry */
	if (peer == NULL)
		return(0);
		
	/* Check for duplicate entry */
	p = peer_list;
	last_p = NULL;
	while(p != NULL)
	{
		/* If duplicate found. Free and invalidate current peer */
		if ((p->src_addr == peer->src_addr) && 
			 (p->dst_addr == peer->dst_addr))
		{

#ifdef DEBUG
			printf("Nuking dup peer\n");
			if (p->bytes_sent != peer->bytes_sent)
				fprintf(stderr, "bad dup peer BYTES SENT\n");
			if (p->packets_sent != peer->packets_sent)
				fprintf(stderr, "bad dup peer BYTES SENT\n");
			if (p->connections_sent != peer->connections_sent)
				fprintf(stderr, "bad dup peer BYTES SENT\n");
#endif /* DEBUG */

			free(peer);
			peer = NULL;
			return(1);
		}
		last_p = p;
		p = p->next;
	}

	/* Add peer to list of all peers, see comments before */
	/* move_peers_to_hosts for rationale */
	/* Place peer at end of list, to preserve order */
	if (last_p == NULL)
		peer_list = peer;
	else
		last_p->next = peer;
	
	/* Invalidate current peer entry */
	peer = NULL;
	
	return(1);
}

static int new_host(const char *value)
{
	/* Ensure that there is no current host entry */
	if (host != NULL)
		return(0);
		
	/* Allocate a new host entry */
	host = util_zmalloc(sizeof(*host));

#ifdef DEBUG
	 fprintf(stderr, "New host\n");
#endif /* DEBUG */

	return(1);
}

static int set_host_addr(const char *value)
{
	struct in_addr i_addr;
	
	/* Ensure that there is a current host entry */
	if (host == NULL)
		return(0);

	/* Convert dotted-quad to integer address */
	if (inet_aton(value, &i_addr) == 0)
		return(0);
	
	host->ip_addr = i_addr.s_addr;	

#ifdef DEBUG
	 fprintf(stderr, "Host address: %s\n", value);
#endif /* DEBUG */

	return(1);
}

static int set_host_bytes_received(const char *value)
{
	/* Ensure that there is a current host entry */
	if (host == NULL)
		return(0);
	
	if (sscanf(value, "%llu", &(host->bytes_received)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Host bytes received: %llu\n", host->bytes_received);
#endif /* DEBUG */

	return(1);
}

static int set_host_bytes_sent(const char *value)
{
	/* Ensure that there is a current host entry */
	if (host == NULL)
		return(0);
		
	if (sscanf(value, "%llu", &(host->bytes_sent)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Host bytes sent: %llu\n", host->bytes_sent);
#endif /* DEBUG */

	return(1);
}

static int set_host_conreqs_received(const char *value)
{
	/* Ensure that there is a current host entry */
	if (host == NULL)
		return(0);
		
	if (sscanf(value, "%llu", &(host->connections_received)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Host conreq received: %llu\n", host->connections_received);
#endif /* DEBUG */

	return(1);
}

static int set_host_conreqs_sent(const char *value)
{
	/* Ensure that there is a current host entry */
	if (host == NULL)
		return(0);
		
	if (sscanf(value, "%llu", &(host->connections_sent)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Host conreq sent: %llu\n", host->connections_sent);
#endif /* DEBUG */

	return(1);
}

static int set_host_first_traffic_time(const char *value)
{
	/* Ensure that there is a current host entry */
	if (host == NULL)
		return(0);
		
	if (sscanf(value, "%ld", &(host->first_seen)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Host first traffic: %s", ctime(&(host->first_seen)));
#endif /* DEBUG */

	return(1);
}

static int set_host_last_traffic_time(const char *value)
{
	/* Ensure that there is a current host entry */
	if (host == NULL)
		return(0);
		
	if (sscanf(value, "%ld", &(host->last_seen)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Host last traffic: %s", ctime(&(host->last_seen)));
#endif /* DEBUG */

	return(1);
}

static int set_host_name(const char *value)
{
	/* Ensure that there is a current host entry */
	if (host == NULL)
		return(0);

	host->hostname = util_strdup(value);
		
	return(1);
}

static int set_host_packets_received(const char *value)
{
	/* Ensure that there is a current host entry */
	if (host == NULL)
		return(0);
		
	if (sscanf(value, "%lld", &(host->packets_received)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Host packets received: %s\n", value);
#endif /* DEBUG */

	return(1);
}

static int set_host_packets_sent(const char *value)
{
	/* Ensure that there is a current host entry */
	if (host == NULL)
		return(0);
		
	if (sscanf(value, "%lld", &(host->packets_sent)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Host packets sent: %s\n", value);
#endif /* DEBUG */

	return(1);
}

static int new_peer(const char *value)
{
#ifdef DEBUG
	 fprintf(stderr, "New peer\n");
#endif /* DEBUG */

	/* Ensure that there is no current peer entry */
	if (peer != NULL)
		return(0);
		
	/* Allocate a new peer entry */
	peer = util_zmalloc(sizeof(*peer));
	return(1);
}

static int set_peer_src_addr(const char *value)
{
	struct in_addr i_addr;
	
	/* Ensure there is a current peer entry */
	if (peer == NULL)
		return(0);
		
	/* Convert dotted-quad to integer address */
	if (inet_aton(value, &i_addr) == 0)
		return(0);
	
	peer->src_addr = i_addr.s_addr;	

#ifdef DEBUG
	 fprintf(stderr, "Peer source addr: %s\n", value);
#endif /* DEBUG */

	return(1);
}

static int set_peer_dst_addr(const char *value)
{
	struct in_addr i_addr;
	
	/* Ensure there is a current peer entry */
	if (peer == NULL)
		return(0);
		
	/* Convert dotted-quad to integer address */
	if (inet_aton(value, &i_addr) == 0)
		return(0);
	
	peer->dst_addr = i_addr.s_addr;	

#ifdef DEBUG
	 fprintf(stderr, "Peer dest addr: %s\n", value);
#endif /* DEBUG */

	return(1);
}

static int set_peer_bytes(const char *value)
{
	/* Ensure there is a current peer entry */
	if (peer == NULL)
		return(0);
		
	if (sscanf(value, "%lld", &(peer->bytes_sent)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Peer bytes: %llu\n", peer->bytes_sent);
#endif /* DEBUG */

	return(1);
}

static int set_peer_conreqs(const char *value)
{
	/* Ensure there is a current peer entry */
	if (peer == NULL)
		return(0);
		
	if (sscanf(value, "%lld", &(peer->connections_sent)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Peer conreqs: %llu\n", peer->connections_sent);
#endif /* DEBUG */

	return(1);
}

static int set_peer_packets(const char *value)
{
	/* Ensure there is a current peer entry */
	if (peer == NULL)
		return(0);
		
	if (sscanf(value, "%lld", &(peer->packets_sent)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Peer packets: %llu\n", peer->packets_sent);
#endif /* DEBUG */

	return(1);
}

static int set_start_time(const char *value)
{
	/* Parse start time from report */
	if (sscanf(value, "%ld", &(report->summary_start)) != 1)
		return(0);
	else	
		return(1);
}

static int set_total_bytes(const char *value)
{
	if (sscanf(value, "%lld", &(report->total_bytes)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Total bytes: %llu\n", report->total_bytes);
#endif /* DEBUG */

	return(1);
}

static int set_total_conreqs(const char *value)
{
	if (sscanf(value, "%lld", &(report->total_connections)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Total conreqs: %llu\n", report->total_connections);
#endif /* DEBUG */

	return(1);
}

static int set_total_hosts(const char *value)
{
	if (sscanf(value, "%d", &(report->active_hosts)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Total hosts: %u\n", report->active_hosts);
#endif /* DEBUG */

	return(1);
}

static int set_total_packets(const char *value)
{
	if (sscanf(value, "%lld", &(report->total_packets)) != 1)
		return(0);
	
#ifdef DEBUG
	 fprintf(stderr, "Total packets: %llu\n", report->total_packets);
#endif /* DEBUG */

	return(1);
}
