/*
 * Copyright (C) 2007 Arnaud Ysmal <wiidevel@stacktic.org>
 *
 * transfermii_cli.c (TransferMii)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include "tmii/tmii.h"

#define OPT_UL 0x1
#define OPT_DL (OPT_UL+1)
#define OPT_LS (OPT_DL+1)
#define OPT_FI (OPT_LS+1)

static void help(const char *name)
{
	printf("Usage : %s OPTION [FILE [FILE]]\n", name);
	printf("  -b\t Specify bluetooth device (%s -b XX:XX:XX:XX:XX:XX"
	       " ...)\n", name);
	printf("  -d\t Download miis from wiimote (%s -d [-s begin [-t end]]"
	       ")\n    \t   -s\t slot or first slot to download "
	       "(default 1)\n"
	       "    \t   -t\t last slot to download (default 10)\n", 
	       name);
	printf("  -h\t Show this message\n");	
	printf("  -i\t Display creator and mii's name of miifile (%s -i"
	       " miifile [FILE...])\n", name);
	printf("  -l\t List all miis on the wiimote\n");
	printf("  -m\t Set the wii mac address (3 last bytes) (%s -m XX:XX:XX "
	       "...)\n", name);
	printf("  -u\t Upload miis to wiimote\n"
	       "    \t   (%s -u filename [-s slot] [FILE])\n"
	       "    \t   FILE : mii files to upload to the next empty slot "
	       "if possible\n", name);
	exit(EXIT_SUCCESS);
}

int main(int argc, char **argv)
{
      	char *filename;
	char option = 0, fname[11] = "slot0X.mii";
	int arg, slot = 0, slot2 = 0, i;
	extern char *optarg;
	extern int optind;
	extern bdaddr_t ba;

	if(argc == 1)
		help(*argv);
	
     	while((arg=getopt(argc, argv, "b:dhi:lm:s:t:u:")) != EOF) {
		switch(arg) {
		case 'b' : str2ba(optarg, &ba); break;
		case 'd' :
			if(option)
				help(*argv);
			option = OPT_DL;
			break;
		case 'h' : help(*argv); break;
		case 'i' : 
			if(option)
				help(*argv);
			option = OPT_FI;
			filename = optarg;
			break;
		case 'l' :
			if(option)
				help(*argv);
			option = OPT_LS;
			break;
		case 'm' : parseMacAddress(optarg); break;
		case 's' : if(!slot) slot=atoi(optarg); break;
		case 't' : if(!slot2) slot2=atoi(optarg); break;
		case 'u' : 
			if(option) 
				help(*argv);
			option = OPT_UL;
			filename = optarg;
			break;
		default : help(*argv);
		}
	}
	switch(option) {
	case OPT_FI :
		fileInfo(filename);
		for(i=optind; i<argc; i++)
			fileInfo(argv[i]);
		break;
	case OPT_DL :
		if(slot==0 && slot2==0) {
			slot=1;
			slot2=10;
		} else if(slot2==0) {
			slot2=slot;
		} else if(slot==0) {
			slot=slot2;
		} else if(slot<1 || slot>10 || slot2<1 || slot2>10) {
			fprintf(stderr, "slot must be between 1 and 10\n");
			exit(EXIT_FAILURE);			
		}
		for(; slot<=slot2 && slot<=MII_NB; slot++) {
			if(slot<10)
				fname[5]=slot+'0';
			else {
				fname[4]='1';
				fname[5]='0';
			}
			fileWrite(fname, slot);
		}
		break;
	case OPT_LS : listMiis(); break;
	case OPT_UL :
		wm_auto_update=TRUE;
		addMii(filename, slot);
		if(optind>argc) 
			exit(EXIT_SUCCESS);
		for(i=optind; i<argc && addMii(argv[i], 0); i++);
		for(; i<argc; i++)
			fprintf(stderr, "No place left for %s\n", argv[i]);
		break;
	default : help(*argv);
	}
	exit(EXIT_SUCCESS);
}
