/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2013 Martin Brehm
                  2012-2013 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "react.h"
#include "travis.h"
#include "maintools.h"


CReactTraj::CReactTraj()
{
	m_fFile = NULL;
	m_bTDO = false;
	m_waAtoms.SetName("CReactTraj::m_waAtoms");
}


CReactTraj::~CReactTraj()
{
}


CReactMol::CReactMol()
{
	m_bPrinted = false;
	m_oaParents.SetName("CReactMol::m_oaParents");
	m_oaChildren.SetName("CReactMol::m_oaChildren");
	m_waAtoms.SetName("CReactMol::m_waAtoms");
	m_waBonds.SetName("CReactMol::m_waBonds");
}


CReactMol::~CReactMol()
{
}


CReact::CReact()
{
	m_oaReactTrajectories.SetName("CReact::m_oaReactTrajectories");
	m_oaMolecules.SetName("CReact::m_oaMolecules");
	m_waAtomMol.SetName("CReact::m_waAtomMol");
}


CReact::~CReact()
{
}


void CReact::Create()
{
	int z;

	try { m_bBonds = new bool[g_iGesAtomCount*g_iGesAtomCount]; } catch(...) { m_bBonds = NULL; }
	if (m_bBonds == NULL) NewException((double)g_iGesAtomCount*g_iGesAtomCount*sizeof(bool),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { m_iBondCounter = new int[g_iGesAtomCount*g_iGesAtomCount]; } catch(...) { m_iBondCounter = NULL; }
	if (m_iBondCounter == NULL) NewException((double)g_iGesAtomCount*g_iGesAtomCount*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_waAtomMol.SetSize(g_iGesAtomCount);
	for (z=0;z<g_iGesAtomCount*g_iGesAtomCount;z++)
	{
		m_bBonds[z] = false;
		m_iBondCounter[z] = 0;
	}
}


void CReact::Parse()
{
	int z;
	mprintf(WHITE,"\n>>> Reaction Tracking >>>\n\n");
	m_fBondFactor = AskFloat("    Enter bonding factor: [1.3] ",1.3f);
	m_iWaitSteps = AskUnsignedInteger("    Wait n steps before new bonds are recognized: [100] ",100);
	m_iColorScheme = AskUnsignedInteger("    Coloring after lifetime (0), atom trace (1) or uniform (2)? [0] ",0);
	if (m_iColorScheme == 1)
	{
		mprintf("    Elements: ");
		for (z=0;z<g_oaAtoms.GetSize()-1;z++)
		{
			mprintf("%s=%d",((CAtom*)g_oaAtoms[z])->m_sName,z+1);
			if (z < g_oaAtoms.GetSize()-2)
				mprintf(", ");
		}
		mprintf(".\n");
		m_iColorElement = AskUnsignedInteger_ND("    Highlight molecules that contain this element (enter number from list above): ")-1;
	}
	m_bLeaveInert = AskYesNo("    Leave out non-reacting educts in the reaction pathway (y/n)? [yes] ",true);
	m_bIgnoreRearrange = AskYesNo("    Ignore intramolecular rearrangements (y/n)? [no] ",false);
	m_bOnlyElemReact = AskYesNo("    Only draw reactions that involve a specific element (y/n)? [no] ",false);
	if (m_bOnlyElemReact)
	{
		mprintf("    Elements: ");
		for (z=0;z<g_oaAtoms.GetSize()-1;z++)
		{
			mprintf("%s=%d",((CAtom*)g_oaAtoms[z])->m_sName,z+1);
			if (z < g_oaAtoms.GetSize()-2)
				mprintf(", ");
		}
		mprintf(".\n");
		m_iOnlyElem = AskUnsignedInteger_ND("    Which element should this be (enter number from list above): ")-1;
	}
	m_bWriteSnapshots = AskYesNo("    Write out snapshots of each molecule (y/n)? [yes] ",true);
	m_bWriteMolTrajec = AskYesNo("    Write out lifetime trajectory of each molecule (y/n)? [no] ",false);
	m_bWriteReactTrajec = AskYesNo("    Write out trajectory of each reaction (y/n)? [no] ",false);
	if (m_bWriteReactTrajec)
	{
		m_iReactTrajecRange = AskUnsignedInteger("    How many steps should reaction trajectories contain? [400] ",400);
		m_bCreateTDO = AskYesNo("    Create Temporal Development Overlay (TDO) plot for each reaction (y/n)? [yes] ",true);
		if (m_bCreateTDO)
		{
			m_iTDOSteps = AskUnsignedInteger("    How many snapshots should the TDO plot contain? [5] ",5);
			m_iTDORange = AskUnsignedInteger("    Over how many time steps the TDO plot should span? [200] ",200);
			m_fTDOBleaching = AskRangeFloat("    Enter the TDO bleaching grade (0-1): [0.8] ",0.0f,1.0f,0.8f);
		}
	} else m_bCreateTDO = false;
	mprintf(WHITE,"\n<<< End of Reaction Tracking <<<\n\n");
}


void CReact::ScanBonds(CTimeStep *ts, bool instant)
{
	int z, z2;
	float f;

	for (z=0;z<g_iGesAtomCount;z++)
	{
		for (z2=z+1;z2<g_iGesAtomCount;z2++)
		{
			f = FoldVector(ts->m_vaCoords[z]-ts->m_vaCoords[z2]).GetLength() / m_fBondFactor;
			if (f <= ((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_pElement->m_fRadius + ((CAtom*)g_oaAtoms[g_waAtomRealElement[z2]])->m_pElement->m_fRadius)
			{
				if (instant)
				{
					m_bBonds[z*g_iGesAtomCount+z2] = true;
					ChangeBond(z,z2,0,true);
				} else
				{
					if (!m_bBonds[z*g_iGesAtomCount+z2])
					{
						if (m_iBondCounter[z*g_iGesAtomCount+z2] != 0)
						{ // Erst kuerzlich gebrochen: Sofort neu bilden
							m_bBonds[z*g_iGesAtomCount+z2] = true;
							m_iBondCounter[z*g_iGesAtomCount+z2] = 0;
						} else
						{
							m_bBonds[z*g_iGesAtomCount+z2] = true;
							m_iBondCounter[z*g_iGesAtomCount+z2] = 1;
						}
					} else
					{
						if (m_iBondCounter[z*g_iGesAtomCount+z2] != 0)
						{
							m_iBondCounter[z*g_iGesAtomCount+z2]++;
							if (m_iBondCounter[z*g_iGesAtomCount+z2] > m_iWaitSteps)
							{
								m_iBondCounter[z*g_iGesAtomCount+z2] = 0;
								ChangeBond(z,z2,g_iSteps-m_iWaitSteps,true);
								mprintf("\n%6d: ",g_iSteps-m_iWaitSteps);
								PrintName(z);
								mprintf(" - ");
								PrintName(z2);
								mprintf(" established.");
							}
						}
					}
				}
			} else
			{
				if (m_bBonds[z*g_iGesAtomCount+z2])
				{
					if (m_iBondCounter[z*g_iGesAtomCount+z2] != 0)
					{ // Erst kuerzlich gebildet: Sofort brechen
						m_bBonds[z*g_iGesAtomCount+z2] = false;
						m_iBondCounter[z*g_iGesAtomCount+z2] = 0;
					} else
					{
						m_bBonds[z*g_iGesAtomCount+z2] = false;
						m_iBondCounter[z*g_iGesAtomCount+z2] = 1;
					}
				} else
				{
					if (m_iBondCounter[z*g_iGesAtomCount+z2] != 0)
					{
						m_iBondCounter[z*g_iGesAtomCount+z2]++;
						if (m_iBondCounter[z*g_iGesAtomCount+z2] > m_iWaitSteps)
						{
							m_iBondCounter[z*g_iGesAtomCount+z2] = 0;
							ChangeBond(z,z2,g_iSteps-m_iWaitSteps,false);
							mprintf("\n%6d: ",g_iSteps-m_iWaitSteps);
							PrintName(z);
							mprintf(" - ");
							PrintName(z2);
							mprintf(" broken.");
						}
					}
				}
			}
		}
	}
}


void CReact::PrintName(int i)
{
	CMolecule *m;
	CSingleMolecule *sm;

//	mprintf("%s%-2d",((CAtom*)g_oaAtoms[g_waAtomRealElement[i]])->m_sName,i+1);
	if (g_iScanMolStep == -1)
	{
		sm = (CSingleMolecule*)g_oaSingleMolecules[g_laAtomSMIndex[i]];
		m = (CMolecule*)g_oaMolecules[sm->m_iMolType];
		mprintf("%s%-2d",((CAtom*)g_oaAtoms[g_waAtomRealElement[i]])->m_sName,sm->m_iMolSMIndex+1);
	} else
	{
		sm = (CSingleMolecule*)g_oaSingleMolecules[g_laAtomSMIndex[i]];
		m = (CMolecule*)g_oaMolecules[sm->m_iMolType];
		mprintf("[%d.%d]%s%-2d",m->m_iIndex+1,sm->m_iMolSMIndex+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[i]])->m_sName,g_waAtomMolNumber[i]+1);
	}
}


void CReact::InitMolecules(CTimeStep *ts)
{
	int z, z2;
	CReactMol *m;

	for (z=0;z<g_iGesAtomCount;z++)
	{
		try { m = new CReactMol(); } catch(...) { m = NULL; }
		if (m == NULL) NewException((double)sizeof(CReactMol),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m->m_iCreationTime = 0;
		m->m_iDestructionTime = -1;
		m->m_waAtoms.Add(z);
		m_waAtomMol[z] = z;
		m->BuildName();
		m->m_iIndex = z;
		m_oaMolecules.Add(m);
	}
	ScanBonds(ts,true);
	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		((CReactMol*)m_oaMolecules[z])->m_oaParents.RemoveAll();
		if (((CReactMol*)m_oaMolecules[z])->m_oaChildren.GetSize() != 0)
		{
			delete (CReactMol*)m_oaMolecules[z];
			m_oaMolecules.RemoveAt(z,1);
			z--;
			continue;
		}
		((CReactMol*)m_oaMolecules[z])->m_iIndex = z;
		for (z2=0;z2<((CReactMol*)m_oaMolecules[z])->m_waAtoms.GetSize();z2++)
			m_waAtomMol[((CReactMol*)m_oaMolecules[z])->m_waAtoms[z2]] = z;
	}
	for (z=0;z<g_iGesAtomCount*g_iGesAtomCount;z++)
		m_iBondCounter[z] = 0;
}


void CReact::ChangeBond(int i, int j, int t, bool add)
{
	int mi, mj, ti, tj, z;
	CReactMol *moli, *molj, *molk, *moll;

	ti = -1;
	tj = -1;

	mi = m_waAtomMol[i];
	mj = m_waAtomMol[j];
	moli = (CReactMol*)m_oaMolecules[mi];
	molj = (CReactMol*)m_oaMolecules[mj];

	if (add)
	{
		if (mi == mj)
		{
			if (m_bIgnoreRearrange)
			{
				moli->m_waBonds.Add(i);
				moli->m_waBonds.Add(j);
			} else
			{
				try { molk = new CReactMol(); } catch(...) { molk = NULL; }
				if (molk == NULL) NewException((double)sizeof(CReactMol),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				molk->m_iCreationTime = t;
				molk->m_iDestructionTime = -1;
				molk->m_oaParents.Add(moli);
				moli->m_oaChildren.Add(molk);
				moli->m_iDestructionTime = t;
				for (z=0;z<moli->m_waAtoms.GetSize();z++)
					molk->m_waAtoms.Add(moli->m_waAtoms[z]);
				for (z=0;z<moli->m_waBonds.GetSize();z++)
					molk->m_waBonds.Add(moli->m_waBonds[z]);
				molk->m_waBonds.Add(i);
				molk->m_waBonds.Add(j);
				molk->BuildName();
				for (z=0;z<molk->m_waAtoms.GetSize();z++)
					m_waAtomMol[molk->m_waAtoms[z]] = m_oaMolecules.GetSize();
				molk->m_iIndex = m_oaMolecules.GetSize();
				m_oaMolecules.Add(molk);
			}
		} else
		{
			try { molk = new CReactMol(); } catch(...) { molk = NULL; }
			if (molk == NULL) NewException((double)sizeof(CReactMol),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			molk->m_iCreationTime = t;
			molk->m_iDestructionTime = -1;
			molk->m_oaParents.Add(moli);
			molk->m_oaParents.Add(molj);
			moli->m_oaChildren.Add(molk);
			molj->m_oaChildren.Add(molk);
			moli->m_iDestructionTime = t;
			molj->m_iDestructionTime = t;
			for (z=0;z<moli->m_waAtoms.GetSize();z++)
				molk->m_waAtoms.Add(moli->m_waAtoms[z]);
			for (z=0;z<molj->m_waAtoms.GetSize();z++)
				molk->m_waAtoms.Add(molj->m_waAtoms[z]);
			for (z=0;z<moli->m_waBonds.GetSize();z++)
				molk->m_waBonds.Add(moli->m_waBonds[z]);
			for (z=0;z<molj->m_waBonds.GetSize();z++)
				molk->m_waBonds.Add(molj->m_waBonds[z]);
			molk->m_waBonds.Add(i);
			molk->m_waBonds.Add(j);
			molk->BuildName();
			for (z=0;z<molk->m_waAtoms.GetSize();z++)
				m_waAtomMol[molk->m_waAtoms[z]] = m_oaMolecules.GetSize();
			molk->m_iIndex = m_oaMolecules.GetSize();
			m_oaMolecules.Add(molk);
		}
	} else
	{
		if (mi == mj)
		{
			try { molk = new CReactMol(); } catch(...) { molk = NULL; }
			if (molk == NULL) NewException((double)sizeof(CReactMol),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			molk->m_iCreationTime = t;
			molk->m_iDestructionTime = -1;
			for (z=0;z<moli->m_waAtoms.GetSize();z++)
			{
				molk->m_waAtoms.Add(moli->m_waAtoms[z]);
				if (moli->m_waAtoms[z] == i)
					ti = z;
				if (moli->m_waAtoms[z] == j)
					tj = z;
			}
			for (z=0;z<moli->m_waBonds.GetSize()/2;z++)
				if ((moli->m_waBonds[z*2] != i) || (moli->m_waBonds[z*2+1] != j))
				{
					molk->m_waBonds.Add(moli->m_waBonds[z*2]);
					molk->m_waBonds.Add(moli->m_waBonds[z*2+1]);
				}

			try { molk->m_bAtomMarked = new bool[molk->m_waAtoms.GetSize()]; } catch(...) { molk->m_bAtomMarked = NULL; }
			if (molk->m_bAtomMarked == NULL) NewException((double)molk->m_waAtoms.GetSize()*sizeof(bool),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			try { molk->m_bBondMarked = new bool[molk->m_waBonds.GetSize()]; } catch(...) { molk->m_bBondMarked = NULL; }
			if (molk->m_bBondMarked == NULL) NewException((double)molk->m_waBonds.GetSize()*sizeof(bool),__FILE__,__LINE__,__PRETTY_FUNCTION__);

			if ((ti == -1) || (tj == -1))
				abort();
			if (molk->IsConnected(ti,tj))
			{
				if (m_bIgnoreRearrange)
				{
					for (z=0;z<moli->m_waBonds.GetSize()/2;z++)
						if ((moli->m_waBonds[z*2] == i) && (moli->m_waBonds[z*2+1] == j))
						{
							moli->m_waBonds.RemoveAt(z*2,2);
							z--;
						}
				} else
				{
					moli->m_iDestructionTime = t;
					moli->m_oaChildren.Add(molk);
					molk->m_oaParents.Add(moli);
					molk->BuildName();
					for (z=0;z<molk->m_waAtoms.GetSize();z++)
						m_waAtomMol[molk->m_waAtoms[z]] = m_oaMolecules.GetSize();
					molk->m_iIndex = m_oaMolecules.GetSize();
					m_oaMolecules.Add(molk);
				}
			} else
			{
				moli->m_iDestructionTime = t;

				try { moll = new CReactMol(); } catch(...) { moll = NULL; }
				if (moll == NULL) NewException((double)sizeof(CReactMol),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				moll->m_iCreationTime = t;
				moll->m_iDestructionTime = -1;
				moll->m_oaParents.Add(moli);
				moli->m_oaChildren.Add(moll);
				moli->m_iDestructionTime = t;
				for (z=0;z<molk->m_waAtoms.GetSize();z++)
				{
					if (molk->m_bAtomMarked[z])
						continue;
					moll->m_waAtoms.Add(moli->m_waAtoms[z]);
				}
				for (z=0;z<molk->m_waBonds.GetSize()/2;z++)
				{
					if (molk->m_bBondMarked[z])
						continue;
					moll->m_waBonds.Add(molk->m_waBonds[z*2]);
					moll->m_waBonds.Add(molk->m_waBonds[z*2+1]);
				}
				moll->BuildName();
				for (z=0;z<moll->m_waAtoms.GetSize();z++)
					m_waAtomMol[moll->m_waAtoms[z]] = m_oaMolecules.GetSize();
				moll->m_iIndex = m_oaMolecules.GetSize();
				m_oaMolecules.Add(moll);

				try { moll = new CReactMol(); } catch(...) { moll = NULL; }
				if (moll == NULL) NewException((double)sizeof(CReactMol),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				moll->m_iCreationTime = t;
				moll->m_iDestructionTime = -1;
				moll->m_oaParents.Add(moli);
				moli->m_oaChildren.Add(moll);
				moli->m_iDestructionTime = t;
				for (z=0;z<molk->m_waAtoms.GetSize();z++)
				{
					if (!molk->m_bAtomMarked[z])
						continue;
					moll->m_waAtoms.Add(moli->m_waAtoms[z]);
				}
				for (z=0;z<molk->m_waBonds.GetSize()/2;z++)
				{
					if (!molk->m_bBondMarked[z])
						continue;
					moll->m_waBonds.Add(molk->m_waBonds[z*2]);
					moll->m_waBonds.Add(molk->m_waBonds[z*2+1]);
				}
				moll->BuildName();
				for (z=0;z<moll->m_waAtoms.GetSize();z++)
					m_waAtomMol[moll->m_waAtoms[z]] = m_oaMolecules.GetSize();
				moll->m_iIndex = m_oaMolecules.GetSize();
				m_oaMolecules.Add(moll);
			}
		} else
		{
			eprintf("Weird error.\n");
		}
	}
}


void CReactMol::BuildName()
{
	int z, z2, i;
	char buf[256], buf2[256];

	buf2[0] = 0;
	for (z=0;z<g_oaAtoms.GetSize();z++)
	{
		i = 0;
		for (z2=0;z2<m_waAtoms.GetSize();z2++)
			if (g_waAtomRealElement[m_waAtoms[z2]] == z)
				i++;
		if (i != 0)
		{
			if (i == 1)
				sprintf(buf,"%s",((CAtom*)g_oaAtoms[z])->m_sName);
					else sprintf(buf,"%s%d",((CAtom*)g_oaAtoms[z])->m_sName,i);
			strcat(buf2,buf);
		}
	}

	try { m_sName = new char[strlen(buf2)+1]; } catch(...) { m_sName = NULL; }
	if (m_sName == NULL) NewException((double)(strlen(buf2)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	strcpy(m_sName,buf2);
}


void CReact::DumpMolecules()
{
	int z, z2, i;
	CReactMol *m, *m2;
	CxPtrArray pa;
	char buf[256], *p;

	mprintf(WHITE,"\n>>> Reaction Analysis Information >>>\n\n");
	mprintf(WHITE,"In the beginning we had the following molecules:\n");
	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		m = (CReactMol*)m_oaMolecules[z];
		if (m->m_oaParents.GetSize() != 0)
			continue;
		mprintf("  - %s[%d]\n",m->m_sName,m->m_iIndex+1);
	}
	mprintf("\n");

	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		m = (CReactMol*)m_oaMolecules[z];
		if (m->m_oaParents.GetSize() == 0)
			continue;
		if (m->m_bPrinted)
			continue;
		if (m->m_iCreationTime <= 1)
			continue;
		m->m_bPrinted = true;
		mprintf("%6d: ",m->m_iCreationTime);
		if (m->m_oaParents.GetSize() == 1)
		{
			m2 = (CReactMol*)m->m_oaParents[0];
			if (m2->m_oaChildren.GetSize() == 1)
			{
				mprintf("%s[%d] --> %s[%d] (rearrangement - ",m2->m_sName,m2->m_iIndex+1,m->m_sName,m->m_iIndex+1);
				if (m->m_waBonds.GetSize() > m2->m_waBonds.GetSize())
					mprintf("established bond)");
						else if (m->m_waBonds.GetSize() < m2->m_waBonds.GetSize()) 
							mprintf("broke bond)");
								else mprintf("changed bond)");
				mprintf(".\n");
			} else
			{
				mprintf("%s[%d] --> ",m2->m_sName,m2->m_iIndex+1);
				for (z2=0;z2<m2->m_oaChildren.GetSize();z2++)
				{
					mprintf("%s[%d]",((CReactMol*)m2->m_oaChildren[z2])->m_sName,((CReactMol*)m2->m_oaChildren[z2])->m_iIndex+1);
					((CReactMol*)m2->m_oaChildren[z2])->m_bPrinted = true;
					if (z2 < m2->m_oaChildren.GetSize()-1)
						mprintf(" + ");
				}
				mprintf("\n");
			}
		} else
		{
			for (z2=0;z2<m->m_oaParents.GetSize();z2++)
			{
				mprintf("%s[%d]",((CReactMol*)m->m_oaParents[z2])->m_sName,((CReactMol*)m->m_oaParents[z2])->m_iIndex+1);
				if (z2 < m->m_oaParents.GetSize()-1)
					mprintf(" + ");
			}
			mprintf(" --> %s[%d]\n",m->m_sName,m->m_iIndex+1);
		}
	}

	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		m = (CReactMol*)m_oaMolecules[z];
		if (m->m_oaChildren.GetSize() != 0)
			continue;
		sprintf(buf,"%s[%d]",m->m_sName,m->m_iIndex+1);

		try { p = new char[strlen(buf)+1]; } catch(...) { p = NULL; }
		if (p == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		strcpy(p,buf);
		pa.Add(p);
	}

	for (z=0;z<pa.GetSize();z++)
	{
		p = (char*)"";
		i = -1;
		for (z2=z;z2<pa.GetSize();z2++)
		{
			if (strcmp(p,(char*)pa[z2]) < 0)
			{
				p = (char*)pa[z2];
				i = z2;
			}
		}
		if (i == -1)
			abort();
		p = (char*)pa[z];
		pa[z] = pa[i];
		pa[i] = p;
	}

	mprintf(WHITE,"\nIn the end we have the following molecules:\n");
	for (z=0;z<pa.GetSize();z++)
		mprintf("  - %s\n",pa[z]);
	mprintf(WHITE,"\n\n<<< End of Reaction Analysis Information <<<\n\n");
}


bool CReactMol::IsConnected(int i, int j)
{
	int z;

	for (z=0;z<m_waAtoms.GetSize();z++)
		m_bAtomMarked[z] = false;
	for (z=0;z<m_waBonds.GetSize();z++)
		m_bBondMarked[z] = false;
	REC_IsConnected(i);
	if (m_bAtomMarked[j])
		return true;
			else return false;
}


void CReactMol::REC_IsConnected(int i)
{
	int z, z2;

	m_bAtomMarked[i] = true;

	for (z=0;z<m_waBonds.GetSize();z+=2)
	{
		if (m_waBonds[z] == m_waAtoms[i])
		{
			for (z2=0;z2<m_waAtoms.GetSize();z2++)
				if (m_waAtoms[z2] == m_waBonds[z+1])
				{
					m_bBondMarked[z/2] = true;
					if (!m_bAtomMarked[z2])
						REC_IsConnected(z2);
				}
		}
		if (m_waBonds[z+1] == m_waAtoms[i])
		{
			for (z2=0;z2<m_waAtoms.GetSize();z2++)
				if (m_waAtoms[z2] == m_waBonds[z])
				{
					m_bBondMarked[z/2] = true;
					if (!m_bAtomMarked[z2])
						REC_IsConnected(z2);
				}
		}
	}
}


void CReact::CreateOutput()
{
	int z, z2;
	CReactMol *m, *m2;
	char buf[256], buf2[256];
	FILE *a;
	double tf;
	CReactTraj *rt;
	unsigned long col;

	mprintf(WHITE,"\n>>> Reaction Analysis Output >>>\n\n");
	mprintf("Creating output diagrams...\n");

	a = OpenFileWrite("reactions.html",true);
	mfprintf(a,"<HTML>\n<HEAD>\n<TITLE>\nTRAVIS Reaction Diagram\n</TITLE>\n<meta HTTP-EQUIV=\"REFRESH\" content=\"0; url=reactions.svg\">\n</HEAD>\n");
	mfprintf(a,"<BODY>\n<H1>Redirection to <A HREF=\"reactions.svg\"></H1>\n</BODY>\n</HTML>\n");
	fclose(a);

	a = OpenFileWrite("reactions.dot",true);
	mfprintf(a,"digraph react {\n");
	mfprintf(a,"  graph [pack=true,splines=true,overlap=false,concentrate=true];\n");
	mfprintf(a,"  node [style=\"filled\",fontsize=16,fontname=\"Arial\",margin=0];\n");

	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		m = (CReactMol*)m_oaMolecules[z];
		if (!m->m_bEnabled)
			continue;
		if (m_bWriteMolTrajec)
		{
			try { rt = new CReactTraj(); } catch(...) { rt = NULL; }
			if (rt == NULL) NewException((double)sizeof(CReactTraj),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			sprintf(buf,"traj_%04d_%s.xyz",m->m_iIndex,m->m_sName);

			try { rt->m_sFileName = new char[strlen(buf)+1]; } catch(...) { rt->m_sFileName = NULL; }
			if (rt->m_sFileName == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			strcpy(rt->m_sFileName,buf);
			rt->m_waAtoms.CopyFrom(&m->m_waAtoms);
			rt->m_iBeginStep = m->m_iCreationTime;
			rt->m_iEndStep = m->m_iDestructionTime;
			m_oaReactTrajectories.Add(rt);
		}
		if (m_bWriteSnapshots)
		{
			try { rt = new CReactTraj(); } catch(...) { rt = NULL; }
			if (rt == NULL) NewException((double)sizeof(CReactTraj),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			sprintf(buf,"snapshot_%04d_%s.xyz",m->m_iIndex,m->m_sName);

			try { rt->m_sFileName = new char[strlen(buf)+1]; } catch(...) { rt->m_sFileName = NULL; }
			if (rt->m_sFileName == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			strcpy(rt->m_sFileName,buf);
			rt->m_waAtoms.CopyFrom(&m->m_waAtoms);
			rt->m_iBeginStep = (m->m_iCreationTime+m->m_iDestructionTime)/2;
			rt->m_iEndStep = rt->m_iBeginStep;
			m_oaReactTrajectories.Add(rt);
		}
		if (m_bLeaveInert && (m->m_oaParents.GetSize() == 0) && (m->m_oaChildren.GetSize() == 0))
			continue;
		if (m_iColorScheme == 0)
			col = m->m_iColor;
		else if (m_iColorScheme == 1)
		{
			for (z2=0;z2<m->m_waAtoms.GetSize();z2++)
				if (g_waAtomRealElement[m->m_waAtoms[z2]] == m_iColorElement)
				{
					col = 0xFF9090;
					goto _elemfound;
				}
			col = 0xE0E0E0;
_elemfound:;
		} else col = 0xE0E0E0;

		if (m->m_oaParents.GetSize() == 0)
			mfprintf(a,"  _%d [label=\"%s\",URL=\"topimg_%04d_%s.svg\",fillcolor=\"#%06lX\",style=\"filled,bold\"] ; \n",z+1,m->m_sName,m->m_iIndex+1,m->m_sName,col);
				else mfprintf(a,"  _%d [label=\"%s\",URL=\"topimg_%04d_%s.svg\",fillcolor=\"#%06lX\"] ; \n",z+1,m->m_sName,m->m_iIndex+1,m->m_sName,col);
	}

	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		m = (CReactMol*)m_oaMolecules[z];
		m->m_bPrinted = false;
	}

	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		m = (CReactMol*)m_oaMolecules[z];
		if (!m->m_bEnabled)
			goto _nextmol;
		for (z2=0;z2<m->m_oaParents.GetSize();z2++)
			if (!((CReactMol*)m->m_oaParents[z2])->m_bEnabled)
				goto _nextmol;
		if (m->m_oaParents.GetSize() == 0)
			goto _nextmol;
		if (m->m_bPrinted)
			goto _nextmol;
		if (m->m_iCreationTime <= 1)
			goto _nextmol;
		m->m_bPrinted = true;
		if (m->m_oaParents.GetSize() == 1)
		{
			m2 = (CReactMol*)m->m_oaParents[0];
			if (m2->m_oaChildren.GetSize() == 1)
			{
				if (m_bWriteReactTrajec)
				{
					try { rt = new CReactTraj(); } catch(...) { rt = NULL; }
					if (rt == NULL) NewException((double)sizeof(CReactTraj),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					sprintf(buf,"react_%04d_to_%04d.xyz",m2->m_iIndex+1,m->m_iIndex+1);

					try { rt->m_sFileName = new char[strlen(buf)+1]; } catch(...) { rt->m_sFileName = NULL; }
					if (rt->m_sFileName == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					strcpy(rt->m_sFileName,buf);
					rt->m_waAtoms.CopyFrom(&m2->m_waAtoms);
					rt->m_iBeginStep = m2->m_iDestructionTime - m_iReactTrajecRange/2;
					rt->m_iEndStep = m2->m_iDestructionTime + m_iReactTrajecRange/2;
					m_oaReactTrajectories.Add(rt);
					if (m_bCreateTDO)
					{
						try { rt = new CReactTraj(); } catch(...) { rt = NULL; }
						if (rt == NULL) NewException((double)sizeof(CReactTraj),__FILE__,__LINE__,__PRETTY_FUNCTION__);
						
						rt->m_bTDO = true;
						sprintf(buf,"tdo_%04d_to_%04d",m2->m_iIndex+1,m->m_iIndex+1);

						try { rt->m_sFileName = new char[strlen(buf)+1]; } catch(...) { rt->m_sFileName = NULL; }
						if (rt->m_sFileName == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
						
						strcpy(rt->m_sFileName,buf);
						rt->m_waAtoms.CopyFrom(&m2->m_waAtoms);
						rt->m_iBeginStep = m2->m_iDestructionTime - m_iTDORange/2;
						rt->m_iEndStep = m2->m_iDestructionTime + m_iTDORange/2;
						m_oaReactTrajectories.Add(rt);
					}
				}
				mfprintf(a,"  _%d -> _%d [weight=%d];\n",m2->m_iIndex+1,m->m_iIndex+1,(int)(m2->m_fMass+m->m_fMass));
			} else
			{
				if (m_bWriteReactTrajec)
				{
					try { rt = new CReactTraj(); } catch(...) { rt = NULL; }
					if (rt == NULL) NewException((double)sizeof(CReactTraj),__FILE__,__LINE__,__PRETTY_FUNCTION__);

					sprintf(buf,"react_%04d_to",m2->m_iIndex+1);
					for (z2=0;z2<m2->m_oaChildren.GetSize();z2++)
					{
						sprintf(buf2,"_%04d",((CReactMol*)m2->m_oaChildren[z2])->m_iIndex+1);
						strcat(buf,buf2);
					}
					strcat(buf,".xyz");

					try { rt->m_sFileName = new char[strlen(buf)+1]; } catch(...) { rt->m_sFileName = NULL; }
					if (rt->m_sFileName == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					strcpy(rt->m_sFileName,buf);
					rt->m_waAtoms.CopyFrom(&m2->m_waAtoms);
					rt->m_iBeginStep = m2->m_iDestructionTime - m_iReactTrajecRange/2;
					rt->m_iEndStep = m2->m_iDestructionTime + m_iReactTrajecRange/2;
					m_oaReactTrajectories.Add(rt);
					if (m_bCreateTDO)
					{
						try { rt = new CReactTraj(); } catch(...) { rt = NULL; }
						if (rt == NULL) NewException((double)sizeof(CReactTraj),__FILE__,__LINE__,__PRETTY_FUNCTION__);
						
						rt->m_bTDO = true;
						sprintf(buf,"tdo_%04d_to",m2->m_iIndex+1);
						for (z2=0;z2<m2->m_oaChildren.GetSize();z2++)
						{
							sprintf(buf2,"_%04d",((CReactMol*)m2->m_oaChildren[z2])->m_iIndex+1);
							strcat(buf,buf2);
						}

						try { rt->m_sFileName = new char[strlen(buf)+1]; } catch(...) { rt->m_sFileName = NULL; }
						if (rt->m_sFileName == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
						
						strcpy(rt->m_sFileName,buf);
						rt->m_waAtoms.CopyFrom(&m2->m_waAtoms);
						rt->m_iBeginStep = m2->m_iDestructionTime - m_iTDORange/2;
						rt->m_iEndStep = m2->m_iDestructionTime + m_iTDORange/2;
						m_oaReactTrajectories.Add(rt);
					}
				}
				for (z2=0;z2<m2->m_oaChildren.GetSize();z2++)
				{
					mfprintf(a,"  _%d -> _%d [weight=%d];\n",m2->m_iIndex+1,((CReactMol*)m2->m_oaChildren[z2])->m_iIndex+1,(int)(m2->m_fMass+((CReactMol*)m2->m_oaChildren[z2])->m_fMass));
					((CReactMol*)m2->m_oaChildren[z2])->m_bPrinted = true;
				}
			}
		} else
		{
				if (m_bWriteReactTrajec)
				{
					try { rt = new CReactTraj(); } catch(...) { rt = NULL; }
					if (rt == NULL) NewException((double)sizeof(CReactTraj),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					sprintf(buf,"react");
					for (z2=0;z2<m->m_oaParents.GetSize();z2++)
					{
						sprintf(buf2,"_%04d",((CReactMol*)m->m_oaParents[z2])->m_iIndex+1);
						strcat(buf,buf2);
					}
					sprintf(buf2,"_to_%04d.xyz",m->m_iIndex+1);
					strcat(buf,buf2);

					try { rt->m_sFileName = new char[strlen(buf)+1]; } catch(...) { rt->m_sFileName = NULL; }
					if (rt->m_sFileName == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					strcpy(rt->m_sFileName,buf);
					rt->m_waAtoms.CopyFrom(&m->m_waAtoms);
					rt->m_iBeginStep = m->m_iCreationTime - m_iReactTrajecRange/2;
					rt->m_iEndStep = m->m_iCreationTime + m_iReactTrajecRange/2;
					m_oaReactTrajectories.Add(rt);
					if (m_bCreateTDO)
					{
						try { rt = new CReactTraj(); } catch(...) { rt = NULL; }
						if (rt == NULL) NewException((double)sizeof(CReactTraj),__FILE__,__LINE__,__PRETTY_FUNCTION__);
						
						rt->m_bTDO = true;
						sprintf(buf,"tdo");
						for (z2=0;z2<m->m_oaParents.GetSize();z2++)
						{
							sprintf(buf2,"_%04d",((CReactMol*)m->m_oaParents[z2])->m_iIndex+1);
							strcat(buf,buf2);
						}
						sprintf(buf2,"_to_%04d.xyz",m->m_iIndex+1);
						strcat(buf,buf2);

						try { rt->m_sFileName = new char[strlen(buf)+1]; } catch(...) { rt->m_sFileName = NULL; }
						if (rt->m_sFileName == NULL) NewException((double)(strlen(buf)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
						
						strcpy(rt->m_sFileName,buf);
						rt->m_waAtoms.CopyFrom(&m->m_waAtoms);
						rt->m_iBeginStep = m->m_iCreationTime - m_iTDORange/2;
						rt->m_iEndStep = m->m_iCreationTime + m_iTDORange/2;
						m_oaReactTrajectories.Add(rt);
					}
				}
			for (z2=0;z2<m->m_oaParents.GetSize();z2++)
				mfprintf(a,"  _%d -> _%d [weight=%d];\n",((CReactMol*)m->m_oaParents[z2])->m_iIndex+1,m->m_iIndex+1,(int)(((CReactMol*)m->m_oaParents[z2])->m_fMass+m->m_fMass));
		}
_nextmol:;
	}
	mfprintf(a,"}\n");
	fclose(a);

	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		m = (CReactMol*)m_oaMolecules[z];
		if (!m->m_bEnabled)
			continue;
		sprintf(buf,"topol_%04d_%s.dot",z+1,m->m_sName);
		a = OpenFileWrite(buf,true);
		mfprintf(a,"graph molecule {\n");
		mfprintf(a,"  graph [pack=true,splines=true,overlap=false];\n");
		mfprintf(a,"  node [shape=none,fontsize=20,fontname=\"Arial\",margin=0,fixedsize,height=0.45];\n");
		mfprintf(a,"  edge [style=bold,len=0.65];\n");
		for (z2=0;z2<m->m_waAtoms.GetSize();z2++)
			mfprintf(a,"  %s%d [label=\"%s\",width=%.2f];\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[m->m_waAtoms[z2]]])->m_sName,m->m_waAtoms[z2],((CAtom*)g_oaAtoms[g_waAtomRealElement[m->m_waAtoms[z2]]])->m_sName,strlen(((CAtom*)g_oaAtoms[g_waAtomRealElement[m->m_waAtoms[z2]]])->m_sName)*0.3f);
		for (z2=0;z2<m->m_waBonds.GetSize()/2;z2++)
			mfprintf(a,"  %s%d -- %s%d [weight=%d];\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[m->m_waBonds[z2*2]]])->m_sName,m->m_waBonds[z2*2],((CAtom*)g_oaAtoms[g_waAtomRealElement[m->m_waBonds[z2*2+1]]])->m_sName,m->m_waBonds[z2*2+1],(int)(((CAtom*)g_oaAtoms[g_waAtomRealElement[m->m_waBonds[z2*2]]])->m_pElement->m_fMass+((CAtom*)g_oaAtoms[g_waAtomRealElement[m->m_waBonds[z2*2+1]]])->m_pElement->m_fMass));
		mfprintf(a,"}\n");
		fclose(a);
	}

	mprintf("Rendering reaction diagram (may take a while)...\n");
	system("unflatten reactions.dot > react_u.dot");
	system("dot react_u.dot -Tsvg -oreactions.svg -Kdot -Gnslimit=5000 -Gmclimit=5000");
	mprintf("Rendering molecule topologies...\n");
	mprintf(WHITE,"  [");

	tf = m_oaMolecules.GetSize() / 60.0;

	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		if (fmod(z,tf) < 1.0)
			mprintf(WHITE,"#");
		m = (CReactMol*)m_oaMolecules[z];
		if (!m->m_bEnabled)
			continue;
		sprintf(buf,"dot topol_%04d_%s.dot -Tsvg -otopimg_%04d_%s.svg -Kneato -Gepsilon=0.000001 -Gnslimit=5000 -Gmclimit=5000",z+1,m->m_sName,z+1,m->m_sName);
		system(buf);
	}
	mprintf(WHITE,"]\n");
	mprintf("\nPlease open reactions.html\n");
	mprintf(WHITE,"\n<<< End of Reaction Analysis Output <<<\n\n");
}


void CReact::Finish()
{
	int z, z2;
	CReactMol *m;
	int l;

	l = 0;
	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		m = (CReactMol*)m_oaMolecules[z];
		m->CalcMass();
		if (m->m_iDestructionTime == -1)
			m->m_iDestructionTime = g_iSteps-m_iWaitSteps;
		if (m->m_iDestructionTime-m->m_iCreationTime > l)
			l = m->m_iDestructionTime-m->m_iCreationTime;
	}

	for (z=0;z<m_oaMolecules.GetSize();z++)
	{
		m = (CReactMol*)m_oaMolecules[z];
		m->m_iColor = ColorFunction(((float)m->m_iDestructionTime-m->m_iCreationTime)/l,0.5);
	}

	if (m_bOnlyElemReact)
	{
		for (z=0;z<m_oaMolecules.GetSize();z++)
		{
			m = (CReactMol*)m_oaMolecules[z];
			m->m_bEnabled = false;
			if (m->ContainsElement(m_iOnlyElem))
			{
				m->m_bEnabled = true;
				goto _done;
			}
			for (z2=0;z2<m->m_oaParents.GetSize();z2++)
			{
				if (((CReactMol*)m->m_oaParents[z2])->ContainsElement(m_iOnlyElem))
				{
					m->m_bEnabled = true;
					goto _done;
				}
			}
			for (z2=0;z2<m->m_oaChildren.GetSize();z2++)
			{
				if (((CReactMol*)m->m_oaChildren[z2])->ContainsElement(m_iOnlyElem))
				{
					m->m_bEnabled = true;
					goto _done;
				}
			}
_done:;
		}
	} else
	{
		for (z=0;z<m_oaMolecules.GetSize();z++)
		{
			m = (CReactMol*)m_oaMolecules[z];
			m->m_bEnabled = true;
		}
	}
}


unsigned long CReact::ColorFunction(float val, float bleach)
{
	int r, g, b;
	if (val < 0.5)
	{
		r = (int)(bleach*255.0 + (1-bleach)*255.0*(1.0-val*2.0));
		g = (int)(bleach*255.0 + (1-bleach)*255.0*val*2.0);
		b = (int)(bleach*255.0);
	} else
	{
		g = (int)(bleach*255.0 + (1-bleach)*255.0*(1.0-(val-0.5)*2.0));
		b = (int)(bleach*255.0 + (1-bleach)*255.0*(val-0.5)*2.0);
		r = (int)(bleach*255.0);
	}
	if (r > 255)
		r = 255;
	if (r < 0)
		r = 0;
	if (g > 255)
		g = 255;
	if (g < 0)
		g = 0;
	if (b > 255)
		b = 255;
	if (b < 0)
		b = 0;
	return r*0x10000 + g*0x100 + b;
}


void CReact::WriteTrajec(CTimeStep *ts)
{
	int z, z2, i;
	CReactTraj *rt;
	float c;
	static CxVec3Array tv;
	CxVector3 vc, v;
	char buf[256];
	FILE *a;

	for (z=0;z<m_oaReactTrajectories.GetSize();z++)
	{
		rt = (CReactTraj*)m_oaReactTrajectories[z];
		if ((rt->m_iBeginStep <= (int)g_iSteps) && (rt->m_iEndStep >= (int)g_iSteps))
		{
			if (rt->m_bTDO)
			{
				if (rt->m_fFile == NULL)
				{
#ifdef TARGET_WINDOWS
					sprintf(buf,"%s.bat",rt->m_sFileName);
#else
					sprintf(buf,"%s.sh",rt->m_sFileName);
#endif
					rt->m_fFile = OpenFileWrite(buf,true);
#ifdef TARGET_WINDOWS
					mfprintf(rt->m_fFile,"pymolwin ");
#else
					mfprintf(rt->m_fFile,"pymol ");
#endif
					for (z=0;z<m_iTDOSteps;z++)
						mfprintf(rt->m_fFile,"%s_%02d.xyz ",rt->m_sFileName,z+1);
					mfprintf(rt->m_fFile,"-d \"set sphere_scale,0.25\" ");
					mfprintf(rt->m_fFile,"-d \"set stick_radius,0.17\" ");
					mfprintf(rt->m_fFile,"-d \"set ray_trace_mode,1\" ");
					mfprintf(rt->m_fFile,"-d \"set fog,0\" ");
					mfprintf(rt->m_fFile,"-d \"set ray_trace_fog,0\" ");
					mfprintf(rt->m_fFile,"-d \"set bg_rgb,(1,1,1)\" ");
					mfprintf(rt->m_fFile,"-d \"set ray_shadow,0\" ");
					mfprintf(rt->m_fFile,"-d \"set ray_shadows,0\" ");
					mfprintf(rt->m_fFile,"-d \"set ray_interior_shadows,0\" ");
					mfprintf(rt->m_fFile,"-d \"show sticks; show spheres\" ");
					
					for (z=0;z<m_iTDOSteps;z++)
						for (z2=0;z2<g_oaAtoms.GetSize();z2++)
						{
							if (z2 == g_iVirtAtomType)
								continue;
							mfprintf(rt->m_fFile,"-d \"set_color mol_%s_%d, [%d,%d,%d]\" ",((CAtom*)g_oaAtoms[z2])->m_sName,z+1,((CAtom*)g_oaAtoms[z2])->m_pElement->ColorR((float)z/(m_iTDOSteps-1)*m_fTDOBleaching),((CAtom*)g_oaAtoms[z2])->m_pElement->ColorG((float)z/(m_iTDOSteps-1)*m_fTDOBleaching),((CAtom*)g_oaAtoms[z2])->m_pElement->ColorB((float)z/(m_iTDOSteps-1)*m_fTDOBleaching));
						}
					for (z=0;z<m_iTDOSteps;z++)
						for (z2=0;z2<g_oaAtoms.GetSize();z2++)
						{
							if (z2 == g_iVirtAtomType)
								continue;
							mfprintf(rt->m_fFile,"-d \"color mol_%s_%d, %s_%02d and name %s\" ",((CAtom*)g_oaAtoms[z2])->m_sName,z+1,rt->m_sFileName,z+1,((CAtom*)g_oaAtoms[z2])->m_sName);
						}
					mfprintf(rt->m_fFile,"\n");
				}
				for (i=0;i<m_iTDOSteps;i++)
				{
					if ((int)floor((float)i/(m_iTDOSteps-1.0f)*(rt->m_iEndStep-rt->m_iBeginStep)+rt->m_iBeginStep) != (int)g_iSteps)
						continue;

					sprintf(buf,"%s_%02d.xyz",rt->m_sFileName,i+1);
					a = OpenFileWrite(buf,true);

					mfprintf(a,"  %d\nStep %lu (TDO Step %d/%d)\n",rt->m_waAtoms.GetSize(),g_iSteps,i+1,m_iTDOSteps);
					for (z2=1;z2<rt->m_waAtoms.GetSize();z2++)
						tv[rt->m_waAtoms[z2]] = tv[rt->m_waAtoms[0]] + FoldVector(tv[rt->m_waAtoms[z2]] - tv[rt->m_waAtoms[0]]);
					vc = CxVector3(0,0,0);
					c = 0;
					for (z2=0;z2<rt->m_waAtoms.GetSize();z2++)
					{
						vc[0] += tv[rt->m_waAtoms[z2]][0] * ((CAtom*)g_oaAtoms[g_waAtomRealElement[rt->m_waAtoms[z2]]])->m_pElement->m_fMass;
						vc[1] += tv[rt->m_waAtoms[z2]][1] * ((CAtom*)g_oaAtoms[g_waAtomRealElement[rt->m_waAtoms[z2]]])->m_pElement->m_fMass;
						vc[2] += tv[rt->m_waAtoms[z2]][2] * ((CAtom*)g_oaAtoms[g_waAtomRealElement[rt->m_waAtoms[z2]]])->m_pElement->m_fMass;
						c += ((CAtom*)g_oaAtoms[g_waAtomRealElement[rt->m_waAtoms[z2]]])->m_pElement->m_fMass;
					}
					vc /= c;
					for (z2=0;z2<rt->m_waAtoms.GetSize();z2++)
						mfprintf(a," %2s  % 10.4f  % 10.4f  % 10.4f\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[rt->m_waAtoms[z2]]])->m_sName,(tv[rt->m_waAtoms[z2]][0]-vc[0])/100.0f,(tv[rt->m_waAtoms[z2]][1]-vc[1])/100.0f,(tv[rt->m_waAtoms[z2]][2]-vc[2])/100.0f);

					fclose(a);
				}
			} else
			{
				tv.CopyFrom(&ts->m_vaCoords);
				if (rt->m_fFile == NULL)
					rt->m_fFile = OpenFileWrite(rt->m_sFileName,true);
				mfprintf(rt->m_fFile,"  %d\nStep %lu\n",rt->m_waAtoms.GetSize(),g_iSteps);
				for (z2=1;z2<rt->m_waAtoms.GetSize();z2++)
					tv[rt->m_waAtoms[z2]] = tv[rt->m_waAtoms[0]] + FoldVector(tv[rt->m_waAtoms[z2]] - tv[rt->m_waAtoms[0]]);
				vc = CxVector3(0,0,0);
				c = 0;
				for (z2=0;z2<rt->m_waAtoms.GetSize();z2++)
				{
					vc[0] += tv[rt->m_waAtoms[z2]][0] * ((CAtom*)g_oaAtoms[g_waAtomRealElement[rt->m_waAtoms[z2]]])->m_pElement->m_fMass;
					vc[1] += tv[rt->m_waAtoms[z2]][1] * ((CAtom*)g_oaAtoms[g_waAtomRealElement[rt->m_waAtoms[z2]]])->m_pElement->m_fMass;
					vc[2] += tv[rt->m_waAtoms[z2]][2] * ((CAtom*)g_oaAtoms[g_waAtomRealElement[rt->m_waAtoms[z2]]])->m_pElement->m_fMass;
					c += ((CAtom*)g_oaAtoms[g_waAtomRealElement[rt->m_waAtoms[z2]]])->m_pElement->m_fMass;
				}
				vc /= c;
				for (z2=0;z2<rt->m_waAtoms.GetSize();z2++)
					mfprintf(rt->m_fFile," %2s  % 10.4f  % 10.4f  % 10.4f\n",((CAtom*)g_oaAtoms[g_waAtomRealElement[rt->m_waAtoms[z2]]])->m_sName,(tv[rt->m_waAtoms[z2]][0]-vc[0])/100.0f,(tv[rt->m_waAtoms[z2]][1]-vc[1])/100.0f,(tv[rt->m_waAtoms[z2]][2]-vc[2])/100.0f);
			}
		} else
		{
			if (rt->m_fFile != NULL)
			{
				fclose(rt->m_fFile);
				rt->m_fFile = NULL;
#ifdef TARGET_LINUX
				if (rt->m_bTDO)
				{						
					sprintf(buf,"chmod 755 %s.sh",rt->m_sFileName);
					system(buf);
				}
#endif
			}
		}
	}

}


void CReactMol::CalcMass()
{
	int z;

	m_fMass = 0;
	for (z=0;z<m_waAtoms.GetSize();z++)
		m_fMass += ((CAtom*)g_oaAtoms[g_waAtomRealElement[m_waAtoms[z]]])->m_pElement->m_fMass;
}


bool CReactMol::ContainsElement(int i)
{
	int z;

	for (z=0;z<m_waAtoms.GetSize();z++)
		if (g_waAtomRealElement[m_waAtoms[z]] == i)
			return true;

	return false;
}
