/* Copyright (C) 2006,2007 Daiki Ueno <ueno@unixuser.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA */

#include <stdlib.h>
#include <stdio.h>
#include "config.h"
#include "treil.h"
#include "heap.h"

struct node {
  long rank;
  treil_tree_t tree;
  struct node *node0, *node1;
};

static int fill_by_tree (treil_tree_t tree, treil_rect_t rect, heap_t heap);

treil_rect_t
treil_rect_new (double x, double y, double width, double height)
{
  treil_rect_t rect = malloc (sizeof(struct treil_rect));
  if (!rect)
    return NULL;
  rect->x = x;
  rect->y = y;
  rect->width = width;
  rect->height = height;
  rect->rect0 = NULL;
  rect->rect1 = NULL;
  rect->tree = NULL;
  return rect;
}

void
treil_rect_free (treil_rect_t rect)
{
  if (rect->rect0)
    treil_rect_free (rect->rect0);
  if (rect->rect1)
    treil_rect_free (rect->rect1);
  free (rect);
}

static void
node_free (struct node *node)
{
  if (node->node0)
    node_free (node->node0);
  if (node->node1)
    node_free (node->node1);
  free (node);
}

static struct node *
encode_tree (treil_tree_t tree, heap_t heap)
{
  treil_tree_t p;
  struct node *node;

  for (p = tree->children; p; p = p->sibling)
    {
      if (!p->size)
	continue;
      node = malloc (sizeof(struct node));
      if (!node)
	return NULL;
      node->rank = p->size;
      node->tree = p;
      node->node0 = NULL;
      node->node1 = NULL;
      heap_insert (heap, (heap_entry_t)node);
    }

  if (heap->len < 2)
    {
      node = (struct node *)heap_delete (heap);
      return node;
    }
  while (heap->len > 1)
    {
      struct node *node0, *node1;

      node0 = (struct node *)heap_delete (heap);
      node1 = (struct node *)heap_delete (heap);

      node = malloc (sizeof(struct node));
      if (!node)
	return NULL;
      node->rank = node0->rank + node1->rank;
      node->tree = NULL;
      node->node0 = node0;
      node->node1 = node1;
      heap_insert (heap, (heap_entry_t)node);
    }
  node = (struct node *)heap_delete (heap);
  node->tree = tree;
  return node;
}

static int
fill_by_node (struct node *node, treil_rect_t rect, heap_t heap)
{
  rect->tree = node->tree;
  if (node->node0 && node->node1)
    {
      treil_rect_t rect0, rect1;
      double ratio = node->node0->rank / (1.0 * node->rank);

      if (rect->height > rect->width)
	{
	  rect0 = treil_rect_new (rect->x, rect->y, rect->width,
				  rect->height * ratio);
	  rect1 = treil_rect_new (rect->x, rect->y + rect->height * ratio,
				  rect->width, rect->height * (1.0 - ratio));
	}
      else
	{
	  rect0 = treil_rect_new (rect->x, rect->y, rect->width * ratio,
				  rect->height);
	  rect1 = treil_rect_new (rect->x + rect->width * ratio, rect->y,
				  rect->width * (1.0 - ratio), rect->height);
	}
      rect->rect0 = rect0;
      rect->rect1 = rect1;
      if (fill_by_node (node->node0, rect0, heap) < 0)
	return -1;
      if (fill_by_node (node->node1, rect1, heap) < 0)
	return -1;
    }
  else if (node->tree)
    return fill_by_tree (node->tree, rect, heap);
  return 0;
}

static int
fill_by_tree (treil_tree_t tree, treil_rect_t rect, heap_t heap)
{
  if (tree->children)
    {
      struct node *node;
	
      node = encode_tree (tree, heap);
      if (!node)
	return -1;
      if (fill_by_node (node, rect, heap) < 0)
	return -1;
      node_free (node);
    }
  return 0;
}

int
treil_rect_fill (treil_rect_t rect, treil_tree_t tree)
{
  heap_t heap;
  int err;

  heap = heap_new (1024);
  if (!heap)
    return -1;
  err = fill_by_tree (tree, rect, heap);
  heap_free (heap);
  return err;
}
