// $Id$
// $Source$

//@HEADER
// ************************************************************************
// 
//            NOX: An Object-Oriented Nonlinear Solver Package
//                 Copyright (2002) Sandia Corporation
// 
//            LOCA: Library of Continuation Algorithms Package
//                 Copyright (2005) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
// 
// Questions? Contact Roger Pawlowski (rppawlo@sandia.gov) or 
// Eric Phipps (etphipp@sandia.gov), Sandia National Laboratories.
// ************************************************************************
//  CVS Information
//  $Source$
//  $Author$
//  $Date$
//  $Revision$
// ************************************************************************
//@HEADER

#ifndef LOCA_TURNINGPOINT_MOORESPENCE_SOLVERSTRATEGY_H
#define LOCA_TURNINGPOINT_MOORESPENCE_SOLVERSTRATEGY_H

#include "Teuchos_RCP.hpp" 

#include "NOX_Abstract_Group.H"  // for ReturnType

// forward declarations
namespace LOCA {
  namespace TurningPoint {
    namespace MooreSpence {
      class AbstractGroup;
      class ExtendedGroup;
      class ExtendedMultiVector;
    }
  }
}

namespace LOCA {

  namespace TurningPoint {

    namespace MooreSpence {
 
      /*! 
       * \brief Abstract strategy for solving the Moore-Spence turning point
       * equations.
       */
      /*!
       * This class provides an abstract interface for solver strategies to
       * solve the Moore-Spence turning point Newton system:
       * \f[
       *     \begin{bmatrix}
       *          J   &    0   &   f_p \\
       *       (Jv)_x &    J   & (Jv)_p \\
       *          0   & \phi^T &   0
       *     \end{bmatrix}
       *     \begin{bmatrix}
       *        X \\
       *        Y \\
       *        z
       *     \end{bmatrix} = 
       *     \begin{bmatrix}
       *        F \\
       *        G \\
       *        h
       *     \end{bmatrix}.
       *  \f]
       * After instantiating a solver (via
       * LOCA::TurningPoint::MooreSpence::SolverFactory), the linear
       * system is set up by setBlocks() and can then be solved by solve().
       */
      class SolverStrategy {

      public:

	//! Constructor
	SolverStrategy() {}

	//! Destructor
	virtual ~SolverStrategy() {}

	//! Set blocks in extended linear system
	/*!
	 * \param group [in] Underlying group representing J
	 * \param tpGroup [in] Turning point group representing the turning
	 *        point equations.  
	 * \param nullVector [in] Vector representing v
	 * \param JnVector [in] Vector representing Jv
	 * \param dfdp [in] Vector representing df/dp
	 * \param dJndp [in] Vector representing d(Jv)/dp
	 */
	virtual void setBlocks(
	  const Teuchos::RCP<LOCA::TurningPoint::MooreSpence::AbstractGroup>& group,
	  const Teuchos::RCP<LOCA::TurningPoint::MooreSpence::ExtendedGroup>& tpGroup,
	  const Teuchos::RCP<const NOX::Abstract::Vector>& nullVector,
	  const Teuchos::RCP<const NOX::Abstract::Vector>& JnVector,
	  const Teuchos::RCP<const NOX::Abstract::MultiVector>& dfdp,
	  const Teuchos::RCP<const NOX::Abstract::MultiVector>& dJndp) = 0;

	/*! 
	 * \brief Solves the extended system as defined above
	 */
	/*!
	 * The \em params argument is the linear solver parameters. 
	 */
	virtual NOX::Abstract::Group::ReturnType 
	solve(
	  Teuchos::ParameterList& params,
	  const LOCA::TurningPoint::MooreSpence::ExtendedMultiVector& input,
	  LOCA::TurningPoint::MooreSpence::ExtendedMultiVector& result) const = 0;

	/*! 
	 * \brief Solves the transpose of the extended system as defined above
	 */
	/*!
	 * The \em params argument is the linear solver parameters. 
	 */
	virtual NOX::Abstract::Group::ReturnType 
	solveTranspose(
	  Teuchos::ParameterList& params,
	  const LOCA::TurningPoint::MooreSpence::ExtendedMultiVector& input,
	  LOCA::TurningPoint::MooreSpence::ExtendedMultiVector& result) const {
	  return NOX::Abstract::Group::NotDefined; 
	}

      private:

	//! Private to prohibit copying
	SolverStrategy(const SolverStrategy&);
	
	//! Private to prohibit copying
	SolverStrategy& operator = (const SolverStrategy&);

      }; // Class SolverStrategy
      
    } // Namespace MooreSpence

  } // Namespace TurningPoint
    
} // Namespace LOCA

#endif // LOCA_TURNINGPOINT_MOORESPENCE_SOLVERSTRATEGY_H
