
/*
 * Licensed Materials - Property of IBM
 *
 * trousers - An open source TCG Software Stack
 *
 * (C) Copyright International Business Machines Corp. 2004-2006
 *
 */


#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <pthread.h>

#include "trousers/tss.h"
#include "trousers/trousers.h"
#include "spi_internal_types.h"
#include "spi_utils.h"
#include "capabilities.h"
#include "tsplog.h"
#include "obj.h"

UINT32 nextObjectHandle = 0xC0000000;

pthread_mutex_t keylist_lock = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t handle_lock = PTHREAD_MUTEX_INITIALIZER;

struct obj_list tpm_list;
struct obj_list context_list;
struct obj_list hash_list;
struct obj_list pcrs_list;
struct obj_list policy_list;
struct obj_list rsakey_list;
struct obj_list encdata_list;

void
list_init(struct obj_list *list)
{
	list->head = NULL;
	pthread_mutex_init(&list->lock, NULL);
}

void
obj_list_init()
{
	list_init(&tpm_list);
	list_init(&context_list);
	list_init(&hash_list);
	list_init(&pcrs_list);
	list_init(&policy_list);
	list_init(&rsakey_list);
	list_init(&encdata_list);
}

TSS_HOBJECT
obj_get_next_handle()
{
	pthread_mutex_lock(&handle_lock);

	/* return any object handle except NULL_HOBJECT */
	do {
		nextObjectHandle++;
	} while (nextObjectHandle == NULL_HOBJECT);

	pthread_mutex_unlock(&handle_lock);

	return nextObjectHandle;
}

/* search through the provided list for an object with handle matching
 * @handle. If found, return a pointer to the object with the list
 * locked, else return NULL.  To release the lock, caller should
 * call obj_list_put() after manipulating the object.
 */
struct tsp_object *
obj_list_get_obj(struct obj_list *list, UINT32 handle)
{
	struct tsp_object *obj;

	pthread_mutex_lock(&list->lock);

	for (obj = list->head; obj; obj = obj->next) {
		if (obj->handle == handle)
			break;
	}

	if (obj == NULL)
		pthread_mutex_unlock(&list->lock);

	return obj;
}

/* search through the provided list for an object with TSP context
 * matching @tspContext. If found, return a pointer to the object
 * with the list locked, else return NULL.  To release the lock,
 * caller should call obj_list_put() after manipulating the object.
 */
struct tsp_object *
obj_list_get_tspcontext(struct obj_list *list, UINT32 tspContext)
{
	struct tsp_object *obj;

	pthread_mutex_lock(&list->lock);

	for (obj = list->head; obj; obj = obj->next) {
		if (obj->tspContext == tspContext)
			break;
	}

	return obj;
}

/* release a list whose handle was returned by obj_list_get_obj() */
void
obj_list_put(struct obj_list *list)
{
	pthread_mutex_unlock(&list->lock);
}

TSS_RESULT
obj_list_add(struct obj_list *list, UINT32 tsp_context, TSS_FLAG flags, void *data,
	     TSS_HOBJECT *phObject)
{
        struct tsp_object *new_obj, *tmp;

        new_obj = calloc(1, sizeof(struct tsp_object));
        if (new_obj == NULL) {
		LogError("malloc of %zd bytes failed.", sizeof(struct tsp_object));
                return TSPERR(TSS_E_OUTOFMEMORY);
        }

        new_obj->handle = obj_get_next_handle();
	new_obj->flags = flags;
        new_obj->data = data;

	if (list == &context_list)
		new_obj->tspContext = new_obj->handle;
	else
		new_obj->tspContext = tsp_context;

        pthread_mutex_lock(&list->lock);

        if (list->head == NULL) {
                list->head = new_obj;
        } else {
                tmp = list->head;
                list->head = new_obj;
                new_obj->next = tmp;
        }

        pthread_mutex_unlock(&list->lock);

        *phObject = new_obj->handle;

        return TSS_SUCCESS;
}

TSS_RESULT
obj_list_remove(struct obj_list *list, TSS_HOBJECT hObject, TSS_HCONTEXT tspContext)
{
	struct tsp_object *obj, *prev = NULL;
	TSS_RESULT result = TSPERR(TSS_E_INVALID_HANDLE);

	pthread_mutex_lock(&list->lock);

	for (obj = list->head; obj; prev = obj, obj = obj->next) {
		if (obj->handle == hObject) {
			/* validate tspContext */
			if (obj->tspContext != tspContext)
				break;

			free(obj->data);
			if (prev)
				prev->next = obj->next;
			else
				list->head = obj->next;
			free(obj);
			result = TSS_SUCCESS;
			break;
		}
	}

	pthread_mutex_unlock(&list->lock);

	return result;
}

/* a generic routine for removing all members of a list who's tsp context
 * matches @tspContext */
void
obj_list_close(struct obj_list *list, TSS_HCONTEXT tspContext)
{
	struct tsp_object *index;
	struct tsp_object *next = NULL;
	struct tsp_object *toKill;
	struct tsp_object *prev = NULL;

	pthread_mutex_lock(&list->lock);

	for (index = list->head; index; ) {
		next = index->next;
		if (index->tspContext == tspContext) {
			toKill = index;
			if (prev == NULL) {
				list->head = toKill->next;
			} else {
				prev->next = toKill->next;
			}

			free(toKill->data);
			free(toKill);

			index = next;
		} else {
			prev = index;
			index = next;
		}
	}

	pthread_mutex_unlock(&list->lock);
}

void
obj_close_context(TSS_HCONTEXT tspContext)
{
	obj_list_close(&tpm_list, tspContext);
	obj_list_close(&context_list, tspContext);
	obj_list_close(&pcrs_list, tspContext);
	obj_list_close(&policy_list, tspContext);

	/* these three must be custom due to the need to free members of their
	 * private data areas. */
	obj_list_hash_close(&hash_list, tspContext);
	obj_list_rsakey_close(&rsakey_list, tspContext);
	obj_list_encdata_close(&encdata_list, tspContext);
}

/* When a policy object is closed, all references to it must be removed. This function
 * calls the object specific routines for each working object type to remove all refs to the
 * policy */
void
obj_lists_remove_policy_refs(TSS_HPOLICY hPolicy, TSS_HCONTEXT tspContext)
{
	obj_rsakey_remove_policy_refs(hPolicy, tspContext);
	obj_encdata_remove_policy_refs(hPolicy, tspContext);
	obj_tpm_remove_policy_refs(hPolicy, tspContext);
}

