/*
 * Copyright (c) Tomas Znamenacek
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

package net.sf.turkey;

import java.io.File;
import java.io.FileInputStream;

import java.util.Vector;

/**
 * The high-level generator interface.
 * This is the class You use for text generation---it wraps all the
 * dictionaries and settings. Just a single instance of this class exists
 * (having more instances of it just does not make sense), You can get it by
 * calling the getInstance() method.
 */
public class Generator {

	/** Minimum number of sentences in a paragraph. */
	public static final int SENTENCES_MIN = 1;
	
	/** Maximum number of sentences in a paragraph. */
	public static final int SENTENCES_MAX = 20;

	/** Minimum number of total paragraphs. */
	public static final int PARAGRAPHS_MIN = 1;
	
	/** Maximum number of total paragraphs. */
	public static final int PARAGRAPHS_MAX = 100;

	/**
	 * The samples used for building the vocabulary.
	 * The parser in the ProbabilityTable class should chew up
	 * pretty much anything, but you can achieve best results
	 * by saving in the format illustrated by default Turkey
	 * samples --- plain text in UTF-8, one word per line,
	 * case does not matter.
	 * 
	 * There's a simple Perl script in the samples/ directory
	 * that can do the dirty conversion work for you.
	 *
	 * I know this sample loading scheme is not an optimal solution, I'll
	 * try to come up with user-installed samples in the next version.
	 */
	protected static final String[] samples = new String[] {
		"Default",
		"Czech",
		"Spanish",
		"Danish",
		"German",
		"Bass"	
	};
	
	/** The list of available probability tables. */
	protected Vector tables;

	/** The table used for text generation. */
	protected ProbabilityTable currentTable;
	
	/** The instance of Generator class. */
	protected static Generator instance = new Generator();

	/** Use HTML tags in the generated text? */
	protected boolean htmlOutput = false;

	/** The number of sentences per paragraph. */
	protected int sentencesPerPar = 5;

	/** The total number of paragraphs. */
	protected int paragraphsTotal = 10;

	/**
	 * Loads the text samples.
	 * This method is protected, to make sure nobody creates more instances
	 * of this class. If You want to get the Generator instance, use the
	 * getInstance() method.
	 */
	protected Generator() {
		
		tables = new Vector();
				
		for (int f = 0; f < samples.length; f++) {

			try {
				tables.add(new ProbabilityTable(getClass().getResourceAsStream(samples[f]), samples[f]));

			} catch (Exception e) {

				System.err.println("Error loading sample " + samples[f] + ": " + e);
			}
		}

		if (tables.size() == 0) {

			System.err.println("No usable samples found, exiting.");
			System.exit(2);
		}

		currentTable = (ProbabilityTable)tables.elementAt(0);
	}

	/** Returns the Generator class instance. */
	public static Generator getInstance() {

		return instance;
	}
	
	/** Returns generated text. */
	public String getText() {

		StringBuffer s = new StringBuffer();

		for (int p = 0; p < paragraphsTotal; p++) {

			if (htmlOutput) s.append("<p>");
			
			for (int v = 0; v < sentencesPerPar; v++) {

				s.append(currentTable.getSentence() + " ");
			}

			s.deleteCharAt(s.length()-1);
			if (htmlOutput) s.append("</p>");
			s.append("\n\n");
		}

		s.deleteCharAt(s.length()-1);
		return s.toString();
	}

	/** 
	 * Selects dictionary by name, falls back to default dictionary
	 * if no dictionary with given name exists.
	 */
	public void selectDictionary(String name) {

		currentTable = (ProbabilityTable)tables.elementAt(0);

		for (int i=0; i < tables.size(); i++) {
			
			if (tables.elementAt(i).toString().equals(name))
				currentTable = (ProbabilityTable)tables.elementAt(i);
		}
	}

	/**
	 * Returns the names of available dictionaries.
	 */
	public Vector getDictionaryNames() {

		Vector n = new Vector();
	
		for (int i=0; i < tables.size(); i++) {
			n.add(tables.elementAt(i).toString());
		}
		
		return n;
	}

	public ProbabilityTable getCurrentDictionary() {

		return currentTable;
	}
	
	/**
	 * Sets the number of sentences in generated paragraph.
	 * The number is kept between SENTENCES_MIN and SENTENCES_MAX.
	 */
	public void setSentencesPerPar(int s) {

		sentencesPerPar = (s < SENTENCES_MIN) ? SENTENCES_MIN : (s > SENTENCES_MAX) ? SENTENCES_MAX : s;
	}

	public int getSentencesPerPar() {

		return sentencesPerPar;
	}

	/**
	 * Sets the total number of paragraphs in generated text.
	 * The number is kept between PARAGRAPHS_MIN and PARAGRAPHS_MAX.
	 */
	public void setParagraphsTotal(int p) {

		paragraphsTotal = (p < PARAGRAPHS_MIN) ? PARAGRAPHS_MIN : (p > PARAGRAPHS_MAX) ? PARAGRAPHS_MAX : p;
	}

	public int getParagraphsTotal() {

		return paragraphsTotal;
	}

	public void setHTMLOutput(boolean h) {

		htmlOutput = h;
	}
	
	public boolean isHTMLOutput() {

		return htmlOutput;
	}
}
