#ifndef ARRAYTYPE_HH
#define ARRAYTYPE_HH

//---------------------------------------------------------------------------
// Copyright (c) 1995-1997 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.
//---------------------------------------------------------------------------

#include "tyvis/VHDLType.hh"
#include "tyvis/TypeInfo.hh"
#include "tyvis/ElementAssociation.hh"
#include <vector>
using std::vector;
class ScalarType;
class SignalBase;
class AccessType;
class IntegerType;

/** The class ArrayType.

    This class represents all the array types and array subtypes that get used
    in a VHDL simulation.  The primary functionality of this class is to act as
    a place holder for the elements contained by an array type.  The basic
    structure for the array type is provided by the ArrayTypeInfo class that
    presents the necessary type information for each array_object
    (signal/variable).  Hence, the ArrayTypeInfo object associated with the
    array type is a very critical structure.
*/    
    
class ArrayType : public VHDLType {
public:
  /**@name Public Class Methods of ArrayType. */
  //@{

  /** This is the default constructor.  However, this constructor is rather evil
      because it let's the user build ArrayType object's that may not have
      sufficient information (such as TypeInfo data) for their working.  Right
      now, it is not very clear if this constructor is essential.  It should be
      removed if it is not necessary.

      @param alias A flag (true/flase) to indicate if this object is an alias.
  */
  ArrayType(); 
  ArrayType(bool alias); 

  /** This is the basic constructor for the array type.  This constructor should
      be used by all applications to build default ArrayType objects.

      @param objType  The type of the object (variable/signal etc.).
      @param typeInfo The ArrayTypeInfo structure associated with this object.
      @param resolveFnId Resolution function id, if any (-1 for invalid).
  */
  ArrayType( ObjectBase::ObjectType objType,
	     const ArrayTypeInfo &typeInfo,
	     ResolutionFnId_t resolveFnId );

  /** This constructor is used to build Single Dimensional ArrayType objects
      with initial value.  This constructor is a convinience constructor that is
      used to build objects of single dimension and the object is initialized
      with the string provided in the parameter.  This constructor is useful for
      building objects of type bit_vector, string.

      @param objType The type of the object (variable/signal etc.).
      @param typeInfo The ArrayTypeInfo structure associated with this object.
      @param resolveFnId The resolution function id, if any (-1 for invalid).
      @param left The left bounds for this object
      @param direction The direction value for this type
      @param right THe right bounds for this object
      @param value The initial value to be assigned to this object.
  */
  ArrayType( ObjectBase::ObjectType objType, 
	     const ArrayTypeInfo& typeInfo, 
	     ResolutionFnId_t resolveFnId, 
	     int left, 
	     ArrayInfo::ArrayDirn_t direction, 
	     int right, 
	     const char *value );

  /** This constructor is used to build Single Dimensional ArrayType objects
      with initial value.  This constructor is a convinience constructor that is
      used to build objects of single dimension and the object is initialized
      with the string provided in the parameter.  This constructor is useful for
      building objects of type bit_vector, string.  If the type in an
      unconstrained array type, then the constraints of the new type is set
      based on the length of "value" .ie. the newObject'left ==
      typeInfo.get_left(1), newObject'direction == typeInfo.get_direction(1),
      newObject'right == newObject'left "+ or -" strlen(value).

      @param objType The type of the object (variable/signal etc.).
      @param typeInfo The ArrayTypeInfo structure associated with this object.
      @param resolveFnId The resolution function id, if any (-1 for invalid).
      @param value The initial value to be assigned to this object.
  */  
   ArrayType( ObjectBase::ObjectType objType, 
	      const ArrayTypeInfo& typeInfo, 
	      ResolutionFnId_t resolveFnId, 
	      const char *value );

  /** This constructor is used to easily build an alias for a given type of
      ArrayType object.  The constructor utilizes a given ArrayType object to
      build another ArrayType object (an alias) with the same bounds as that of
      the original.

      @param alias A flag to indicate if the new array type is an alias or not.
      @param objType The type of the objects (variable/signal).
      @param source The source ArrayType object (which we are aliasing).
  */
  ArrayType(bool alias, ObjectBase::ObjectType objType, const ArrayType& source);
  
  /** This constructor is used to build an alias for a given type of ArrayType
      object.  The constructor utilizes a given ArrayType object to build
      another ArrayType object (an alias) with a given bounds (.ie. left,
      direction, right).

      @param alias A flag to indicate if the new array type is an alias or not.
      @param objType The type of the objects (variable/signal).
      @param left  The left most range/index of this object.
      @param direction The direction of definition (to/downto).
      @param right The right most range/index of this object.
      @param source The source ArrayType object (which we are aliasing).
      @param bounds The bounds of the source ArrayType object (if known else pass nullInfo).
  */
  ArrayType(bool alias, ObjectBase::ObjectType objType, int left, ArrayInfo::ArrayDirn_t direction, int right, const ArrayType& source, const ArrayInfo& bounds);
  
  /** Constructor for aggregate initialization.  This constructor is used to
      initialize a given one dimensional unconstrained (or constrained) array
      type with the given bounds.  The ArrayType object is initialized using the
      AssociationElements (pointers) specified in the constructor.  The
      constructor deletes the pointers at the end (so the caller must "new" the
      pointers during the constructor call).  Note that the bounds specified in
      the parameter list override the bounds defined in the typeInfo structure
      (in the case of constrained array types).

      @param objType The type objects contained by this object (signal/variable).
      @param typeInfo The ArrayTypeInfo object associated with this array.
      @param left The 'left bounds for this object
      @param direction The 'direction for this object (.ie. to/downto)
      @param right The 'right bounds for this object
      @param noOfElements The number of ElementAssociations.
  */
  ArrayType(ObjectBase::ObjectType objType, const ArrayTypeInfo& typeInfo, ResolutionFnId_t resolveFnId, int left, ArrayInfo::ArrayDirn_t direction, int right, int noOfElements = 0, ...);

  /** Constructor for aggregate initialization of constrained array types (both
      1 and n dimensional arrays).  This constructor is used to initialize a
      given one dimensional constrained array type.  The ArrayType object is
      initialized using the AssociationElements (pointers) specified in the
      constructor.  The constructor deletes the pointers at the end (so the
      caller must "new" the pointers during the constructor call).

      @param objType The type objects contained by this object (signal/variable).
      @param typeInfo The ArrayTypeInfo object associated with this array.
      @param resolveFnId The resolution function id, if any (-1 if not available).
      @param noOfElements THe number of ElementAssociations
  */
  ArrayType(ObjectBase::ObjectType objType, const ArrayTypeInfo& typeInfo, ResolutionFnId_t resolveFnId, int noOfElements, ...);


  /** Constructor for 2+ dimensional constrained/unconstrained arrayType Objects.
      This constructor is used to create a two dimensional ArrayType object
      whose constraints are specified at the time object creation.  An optional
      resoltion function may also be specified during object creation.  Such a
      feature is useful when objects are associated with resolution function and
      no explict resolved type has been defined.  This constructor can be used
      for objects that do not have a association list or initializers specified
      for them, by specifying the number of element associations in the list to
      be 0 (zero).

      @param objType The type objects contained by this object (signal/variable)
      @param typeInfo the ArrayTypeInfo object associates with this array.
      @param resolveFnId The resolution function id, if any (-1 if not present).
      @param left1 The 'left for the first dimension.
      @param direction1 The 'direction for the first dimension.
      @param right1 The 'right for the second dimension.
      @param left2 The 'left for the second dimension.
      @param direction2 The 'direction for the second dimension.
      @param right2 The 'right for the second dimension.
      @param noOfElements The number of ElementAssociation in the list
  */
  ArrayType(ObjectBase::ObjectType objType, const ArrayTypeInfo& typeInfo, ResolutionFnId_t resolveFnId, int left1, ArrayInfo::ArrayDirn_t direction1, int right1, int left1, ArrayInfo::ArrayDirn_t direction2, int right2, ...);

  /** A handy constructor for creating temporary objects.  This constructor is
      typically used by the frontend code-generator to build temporary variables
      during (in the body of) subprogram calls.

      @param objType The type objects contained by this object (signal/variable)
      @param typeInfo the ArrayTypeInfo object associated with this array.
      @param resolveFnId The resolution function for this type.
      @param source The source ArrayType object.
  */
  ArrayType(ObjectBase::ObjectType objType, const ArrayTypeInfo& typeInfo, ResolutionFnId_t resolveFnId, const ArrayType &source);

  /** A handy constructor for creating temporary objects.  This constructor is
      typically used by the frontend code-generator to build  temporary array
      slices.
 
      @param objType The type objects contained by this object (signal/variable)
      @param source The source ArrayType object.
  */
  ArrayType(ObjectBase::ObjectType objType, const ArrayType &source);

  /** A handy constructor for creating temporary objects.  This constructor is
      typically used by the frontend code-generator to build objects of
      unconstrained array type, that get their constraints at the time of their
      declaration along with their initial value.

      @param objType The type objects contained by this object (signal/variable)
      @param typeInfo The array type info for the new object.
      @param resolveFnId The resolution function id, if any (-1 if not present).
      @param left1 The 'left for the first dimension.
      @param direction1 The 'direction for the first dimension.
      @param right1 The 'right for the second dimension.
      @param source The source ArrayType object.
  */
  ArrayType(ObjectBase::ObjectType objType, const ArrayTypeInfo &typeInfo, ResolutionFnId_t resolveFnId, int left1, ArrayInfo::ArrayDirn_t direction1, int right1, const ArrayType &source);

  
  /** A rather handy copy constructor.
      @param source The source array type object.  
  */
  ArrayType( const ArrayType &source );
  
  /// The destructor
  virtual ~ArrayType();

  /** This method is used to identify the type.  Note: The method "get_kind()"
      MUST NOT be overloaded by a derived type (if any).  This is to ensure that
      the type is recognized correctly.

      @return The kind of this type/object (ARRAY_TYPE)
  */
  Type get_kind() const;

  /** This method is used to print a ArrayType object.  This method should not
      be used to dump ArrayType objects to achieve texio support.  This method
      is just a utility method.  More for debugging and maybe for tracing
      signals.  In any case, the use of textio methods is encouraged.

      @param os The output stream to which the data must be dumped.
  */
  virtual void print(ostream& os) const;

  /** Method to access a given element at the given position.  This is more of
      an internal access method for convinience.  This method should be moved to
      protected or private section of the class later on, if its visibility is
      not necessary.  In general direct use of this method is discouraged.

      @param pos The offset/index within the array.
      @return The VHDLType constained at the given "pos".
  */
  VHDLType& operator[](const int pos) const;

  /** Method to access a given element at the given position.  This method can
      be invoked by all classes to access a given element at a given poisition.
      This method should be used as the primary interface for ArrayTypes.

      @param pos The position/offset/index within the array
      @return The VHDLType constained at the given "pos"
  */      
  VHDLType& operator[](const ScalarType& pos) const;

 /** Method to access a given element at the given position.  This method can
      be invoked by all classes to access a given element at a given poisition.
      This method is a convinience method ArrayTypes.

      @param pos The position/offset/index within the array
      @return The VHDLType constained at the given "pos"
  */      
  VHDLType& operator[](const VHDLType& pos) const;
  
  /** Assignment operator for array types.  NOTE that this is more of an
      internal method.  It should NOT be used to assign values for variables and
      signals.  The assignVal method should be used for that purpose.

      @param source The source ArrayType object.
      @return The ArrayType object with new value (*this)
  */
  VHDLType& operator=(const VHDLType& source);

  /** Assignment operator for array types.

      @param source The source ArrayType object.
      @return The ArrayType object with new value (*this)
  */
  ArrayType& operator=(const ArrayType& source);
  
  /** This method is used to assigna the driving value for the array type object.

      @param The source ArrayType object
      @return The ArrayType object with new value (*this)
  */
  VHDLType &assignVal(const VHDLType& val);

  /** This method is used to compare if the given ArrayType object is operator== to
      another.  Two ArrayType objects are operator== if and only if: (i) They are of
      the same length (actual bounds are not considered in this method); (ii)
      Each corresponding element in the two objects are operator==.

      @param val The ArrayType object that is being compared.
      @return true/false indicating if the object is operator== or not.
  */
  bool operator==(const RValue &val ) const;
  bool operator!=(const RValue &val ) const;
  bool operator<(const RValue &val ) const;
  bool operator<=(const RValue &val ) const;
  bool operator>(const RValue &val ) const;
  bool operator>=(const RValue &val ) const;

  ArrayType vhdlAnd( const ArrayType &rhs) const; 
  ArrayType vhdlOr( const ArrayType &rhs) const; 
  ArrayType vhdlNand( const ArrayType &rhs) const; 
  ArrayType vhdlNor( const ArrayType &rhs) const;
  ArrayType vhdlXor( const ArrayType &rhs) const; 
  ArrayType vhdlXnor( const ArrayType &rhs) const; 
  ArrayType vhdlNot() const; 


  ArrayType vhdlSll( const RValue &rhs ) const;
  ArrayType vhdlSrl( const RValue &rhs ) const;
  ArrayType vhdlSla( const RValue &rhs ) const;
  ArrayType vhdlSra( const RValue &rhs ) const;
  ArrayType vhdlRol( const RValue &rhs ) const;
  ArrayType vhdlRor( const RValue &rhs ) const;

  /** A utility method that does not do bounds translation.  This is a purely
      internal method.  This should NOT be utilized by application programs.

      @param index The offset within the array
      @return The element at the given offset.
  */
  VHDLType &get_element(const int index) const;

  /** Obtain a pointer to the objects contained by this array object.

      @return The list of objects maintained by this ArrayType
  */
  virtual const vector<VHDLType*> &get_array() const;

  virtual unsigned int getNumberOfDimensions() const { return bounds.size(); }

  /** Obtain the number of elements contained by a given dimension.

      @param dimension The dimension of interest (1 or 2).
      @return The number of elements in that given dimension.
  */
  virtual int get_number_of_elements(int dimension = 0) const;


  /** The bounds for a given dimension of the object.

      @param dimension The dimension of interest (1 or 2).
      @return The bounds for the given dimension.
  */
  const ArrayInfo &get_bounds(int dimension) const;

  /** The string representation for the elements for the given ArrayType.  This
      method is valid only for one-dimensional ArrayTypes whose elements are of
      EnumerationType.

      @return The "c" string representation for the elements of the ArrayType object
  */
  virtual char *getString() const;

  /** Objain a slice of a one dimensional array type.  The return value is a array
      type object with its bounds suitably set.

      @newBounds The bounds of the array type object returned by this method
      @return    A slice of the object based on the newBounds
  */
  virtual ArrayType &getSlice(const ArrayInfo& newBounds) const;
      
  /** Returns a clone of the object.  All the fields, dimensions, and elements
      are duplicated.  The return value is a fresh/new duplicate and the caller
      must ensure that the return value is deleted (otherwise, memory will leak)

      @return A pointer to the clone of the object.
  */
  virtual VHDLType *clone() const;

  /** A convinience method to determine the objecttype of the elements
      constained by this array type.

      @return The type of the objects contained in the array (signal/variable etc.).
  */
  virtual ObjectBase::ObjectType getKind() const;

  /** Obtain the left bounds for this object.  It must be noted that the bounds
      for an object and bounds for a type are essentially different.  However,
      they could be the same.

      @param dimension The dimension of interest (0/1).
      @return The left bound for the requested dimension.
  */
  int left(const int dimension) const;

  /** Obtain the left bounds based on the dimension.  It must be noted that the
      bounds for an object and bounds for a type are essentially different.
      However, they could be the same.

      @param dimension The dimension of interest (0/1).
      @return The left bound for the requested dimension.
  */
  VHDLType &left(const ScalarType& dimension) const;

  /** Obtain the right bounds for this object.  It must be noted that the bounds
      for an object and bounds for a type are essentially different.  However,
      they could be the same.

      @param dimension The dimension of interest (0/1).
      @return The left bound for the requested dimension.
  */
  int right(const int dimension) const;

  /** Obtain the right bounds based on the dimension.  It must be noted that the
      bounds for an object and bounds for a type are essentially different.
      However, they could be the same.

      @param dimension The dimension of interest (0/1).
      @return The left bound for the requested dimension.
  */
  VHDLType &right(const ScalarType& dimension) const;

  /** Obtain the direction for a given dimension of the ArrayType.

      @param dimension The dimension of interest (0/1).
      @return The direction for the specified dimension (to/downto).
  */
  ArrayInfo::ArrayDirn_t dirn(const int dimension) const;

  /** Obtain the number of elements in a given dimension of the ArrayType.
      
      @param dimension The dimension of interest (0/1).
      @return The direction for the specified dimension (to/downto).
  */
  int length(const int dimension) const;

  /** Method to access the array type info for this object.

      @return The array type info for this object.
  */
  const ArrayTypeInfo& getArrayTypeInfo() const;

  /** Method to access the secondary type info for this object.

      @return The array type info for the 2nd dimension.
  */
  const ArrayTypeInfo& getSubArrayTypeInfo() const;
  
  /**  The resolution interface for a ArrayType.  The TYPE's resolve is called
       only for composite resolved signals. This resolve goes down to first
       sub-element of the VHDLType and calls the sub-elements resolve, but
       which actually does the resolution for the whole composite type.

       The TYPE's resolve is called only for composite resolved signals This
       resolve goes down to first sub-element of the VHDLType and calls the
       sub-elements resolve, but which actually does the resolution for the
       whole composite type.  The above given reason was the original reason for
       writing resolve in the ArrayTypes.  But when TypeConversion came into
       picture, we needed resolve to be written for ArrayTypes even if the
       ArrayType was not a composite resolved signal.  In case of not a
       composite resolved type, resolve method just goes down to the sub
       elements and resolves the individual elements and then updates once the
       resolved value is obtained.

       @param process A pointer to the process containing this object.
       @return The resolved value for this object.
  */
  virtual VHDLType *resolve( VHDLKernel * );

  /** This method is used to update the effective value of the signal objects
      contained by this array type.  The source for this method is ususally
      obtained by a call to resolve.

      @param source The source Array that contains the effective value.
  */
  virtual void updateEffVal(const VHDLType* source);

  /** This method is used to set the resolutionFnId for all elements that 
      contain this array object.

      @param resolutionFnId The resolution function id.
  */
  virtual void setResolutionFunctionId(ResolutionFnId_t resolutionFnId);

  /** This method can be used to access the resolution function id for this
      array type.

      @param The resolution function id (-1 if none present)
  */
  ResolutionFnId_t getResolutionFunctionId() const { return resolutionFunctionId; }
      
  /** This method is used to set the Typeconversion function id.

      @param typeConversionFnId The type conversion function id.  */
  virtual void setTypeConversionFunctionId(TypeConversionFnId_t typeConversionFnId);

  /** This method is used to set the parent pointer to each element of the array
      type object.

      @param parent The pointer to the parent for each element.
  */
  virtual void setParentCompositeType(VHDLType *parent);

  /** This method is used to indicate that the array type object is a composite
      resolved signal.  This method updates the corresponding fields of all the
      elements contained by this array type.

      @param flag Boolean value indicating whether the signal is composite-resolved or not.
  */
  virtual void setCompositeResolvedSignal(bool flag);

  /** This method is invoked to update the elaboration information for each
      signal object contained by this array type.  The source must be a
      equivalent array type object whose elements are of type SignalNetinfo.
      This method is a part of the elaboration routine and is usually invoked
      from the VHDLKernelState::initiState().  The elaboration information
      contained in the source is filled in during elaboration.

      @param source The signal net for this array.
  */
  void setElaborationInfo(const VHDLType& source);

  /** This method is used to initialize/set the attribute object corresponding
      to the specific attribute types defined for this signal.  The source must
      be a equivalent ArrayType.

      @param aType The attribute type being initialized.
      @param source The ArrayType object containing the attributes.
  */
  void setAttrib(AttribType aType, VHDLType& source);

  /** This method is used to locate the driver object corresponding to the
      signalId passed in as the parameter.  This method searches all the
      elements (recursively) to locate the driver corresponding to each
      signalId.

      @param sigId The id of the signal whose dirver needs to be located.
      @return A pointer to the dirver (NULL if not found).
  */
  virtual SignalBase* locateSig(int sigId);


  /** This method is used to locate the driver for a signal within a given
      process.

      @param sigId The id of the signal whose driver is required.
      @param srcId The object/process corresponding to the driver.
      @return A pointer to the driver (NULL if not found).
  */
  virtual SignalBase* findSigInBlock(int sigId, VHDLKernel *srcId);

  /** Utility method to determine if the given array type object is a compositve
      resolved signal or not.

      @return Boolean value (true/false) indicating if signal is a composite resolved signal or not.
  */
  virtual bool _is_composite_resolved_type() const { return isCompositeResolvedType; }

  /** This method is used to dump the connectivity information for the signals
      contained in a array type object.  The connectivity information may be
      used for serveral purposes.  One of them being partitioning of VHDL
      processes during parallel simulation (in order to improve performance).

      @param ofs The output file stream to which the information will be dumped.
  */
  virtual void dump_connectivity_info(ofstream& ofs);

  /** A utility method to verify if the array type object is a signal or not.

      @return Returns true/false to indicate wether the object is a signal or not.
  */
  virtual bool _is_signal() const;

  /** Method to write the elements contained by the array type into the
      specified line.  This is a internal method and its direct use is
      discouraged.  This method is "deprecated" (will be removed in later release) 

      @param line The line that is going to contain the information.
      @return The return type (NORMAL_RETURN etc.)
  */
  int savantwrite( AccessVariable &line ) const;

  /** Method to write the elements contained by the array type object into the
      specified accesstype variable.  The access type variable must be of "line"
      type.

      @param line A access variable of "line" type.
      @return The return type (NORTMAL_RETURN etc.)
  */
  int savantwrite(AccessType &line) const;

  /** Method to read the data for the elements in the array type from the
      specified input line.  The elements in the line are altered (or consumed)
      during read.  This method is an internal method.  Its direct use is
      discouraged.  This method is "deprecated" (it will be removed in future
      release).

      @param line The line variable containing the data.
      @return The exit/return status (NORMAL_RETURN etc.)
  */
  int savantread(AccessVariable &line);

  /** Method to read the elements contained by the array type object from the
      specified accesstype variable.  The access type variable must be of "line"
      type.
      
      @param line A access variable of "line" type.
      @return The return type (NORTMAL_RETURN etc.)
  */
  int savantread(AccessType &line);

  /** This method can be used to determine the Object's LEFT attribute.  This
      method returns the LEFT attribute an IntegerType.
      
      @param dimension The dimension of interest.
      @return The Object'LEFT(dimension).
  */
  IntegerType LEFT_O(const IntegerType& dimension) const;

  /** This method can be used to determine the Object's LEFT attribute.  This
      method returns the LEFT attribute an EnumerationType.
      
      @param dimension The dimension of interest.
      @param ti A type info. structure that is not used.
      @return The Object'LEFT(dimension).
  */
  EnumerationType LEFT_O(const IntegerType& dimension, const TypeInfo& ti) const;

  /** This method can be used to determine the Object's RIGHT attribute.  This
      method returns the RIGHT attribute an IntegerType.
      
      @param dimension The dimension of interest.
      @return The Object'RIGHT(dimension).
  */
  
  IntegerType RIGHT_O(const IntegerType& dimension) const;

  /** This method can be used to determine the Object's RIGHT attribute.  This
      method returns the RIGHT attribute an EnumerationType.
      
      @param dimension The dimension of interest.
      @param ti A type info. structure that is not used.
      @return The Object'RIGHT(dimension).
  */
  EnumerationType RIGHT_O(const IntegerType& dimension, const TypeInfo& ti) const;


  /** This method can be used to determine the Object's HIGH attribute.  This
      method returns the HIGH attribute an IntegerType.
      
      @param dimension The dimension of interest.
      @return The Object'HIGH(dimension).
  */
  IntegerType HIGH_O(const IntegerType& dimension) const;

  /** This method can be used to determine the Object's HIGH attribute.  This
      method returns the HIGH attribute an EnumerationType.
      
      @param dimension The dimension of interest.
      @param ti A type info. structure that is not used.
      @return The Object'HIGH(dimension).
  */
  EnumerationType HIGH_O(const IntegerType& dimension, const TypeInfo& ti) const;
 

  /** This method can be used to determine the Object's LOW attribute.  This
      method returns the LOW attribute an IntegerType.
      
      @param dimension The dimension of interest.
      @return The Object'LOW(dimension).
  */
  IntegerType LOW_O(const IntegerType& dimension) const;

  /** This method can be used to determine the Object's LOW attribute.  This
      method returns the LOW attribute an EnumerationType.
      
      @param dimension The dimension of interest.
      @param ti A type info. structure that is not used.
      @return The Object'HIGH(dimension).
  */
  EnumerationType LOW_O(const IntegerType& dimension, const TypeInfo& ti) const;

 
  /** This method can be used to determine the Object's LENGTH attribute.  This
      method returns the LENGTH attribute an IntegerType.
      
      @param dimension The dimension of interest.
      @return The Object'LENGTH(dimension).
  */
  IntegerType LENGTH_O(const IntegerType& dimension) const;

  /** This method can be used to determine the Object's ASCENDING attribute.  This
      method returns the ASCENDING attribute an EnumerationType.
      
      @param dimension The dimension of interest.
      @return The Object'ASCENDING (dimension).
  */ 
  EnumerationType ASCENDING_O(const IntegerType& dimension) const;


  /** This method can be used to determine the Object's RANGE for a particular
      dimension.  This method returns the RANGE attribute an IntegerType.
      
      @param dimension The dimension of interest.
      @return The Object'RANGE(dimension).  */
  IntegerType RANGE_O(const IntegerType& dimension) const;

  /** This method can be used to determine the Object's RANGE for a particular
      dimension. This method returns the RANGE attribute an EnumerationType.
      
      @param dimension The dimension of interest.
      @param ti A type info. structure that is not used.      
      @return The Object'RANGE(dimension).  */
  EnumerationType RANGE_O(const IntegerType& dimension, const TypeInfo& ti) const;

  /** This method can be used to determine the Object's REVERSE_RANGE for a
      particular dimension.  This method returns the REVERSE_RANGE attribute an
      IntegerType.
      
      @param dimension The dimension of interest.
      @return The Object'REVERSE_RANGE(dimension).  */
  IntegerType REVERSE_RANGE_O(const IntegerType& dimension) const;

  /** This method can be used to obtain the Object's REVERSE_RANGE for a particular
      dimension. This method returns the REVERSE_RANGE attribute an EnumerationType.
      
      @param dimension The dimension of interest.
      @param ti A type info. structure that is not used.      
      @return The Object'REVERSE_RANGE(dimension).  */
  EnumerationType REVERSE_RANGE_O(const IntegerType& dimension, const TypeInfo& ti) const;
  
  /** This method is used to determine the Type'LEFT for a ArrayType.  The 'LEFT
      is determined from the type info structure.

      @param dimension The dimension of interest.
      @param typeInfo  Type array type info structure that needs to be used.
      @return The Type'LEFT */
  static IntegerType LEFT(const IntegerType& dimension, const TypeInfo& typeInfo);

  /** This method is used to determine the Type'LEFT for a ArrayType.  The 'LEFT
      is determined from the type info structure.

      @param dimension The dimension of interest.
      @param typeInfo  Type array type info structure that needs to be used.
      @param ti A type info. structure (unused -- can be TypeInfo::nullTypeInfo).
      @return The Type'LEFT */
  static EnumerationType LEFT(const IntegerType& dimension, const TypeInfo& typeInfom, const TypeInfo& ti);
  
  /** This method is used to determine the Type'RIGHT for a ArrayType.  The
      'RIGHT is determined from the type info structure.

      @param dim The dimension of interest.
      @param typeInfo  Type array type info structure that needs to be used.
      @return The Type'RIGHT
  */
  static IntegerType RIGHT(const IntegerType& dim, const TypeInfo& typeInfo);

  /** This method is used to determine the Type'RIGHT for a ArrayType.  The
      'RIGHT is determined from the type info structure.

      @param dim The dimension of interest.
      @param typeInfo  Type array type info structure that needs to be used.
      @param ti A type info. structure (unused -- can be TypeInfo::nullTypeInfo).   
      @return The Type'RIGHT
  */
  static EnumerationType RIGHT(const IntegerType& dim, const TypeInfo& typeInfo, const TypeInfo& ti);
  
  /** This method is used to determine the Type'HIGH for a ArrayType.  The
      'HIGH is determined from the type info structure.

      @param dim The dimension of interest.
      @param typeInfo  Type array type info structure that needs to be used.
      @return The Type'HIGH
  */  
  static IntegerType HIGH(const IntegerType& dim, const TypeInfo& typeInfo);


  /** This method is used to determine the Type'HIGH for a ArrayType.  The
      'HIGH is determined from the type info structure.

      @param dim The dimension of interest.
      @param typeInfo  Type array type info structure that needs to be used.
      @param ti A type info. structure (unused -- can be TypeInfo::nullTypeInfo).   
      @return The Type'HIGH
  */
  static EnumerationType HIGH(const IntegerType& dim, const TypeInfo& typeInfo, const TypeInfo& ti);
  
  /** This method is used to determine the Type'LOW for a ArrayType.  The
      'LOW is determined from the type info structure.

      @param dim The dimension of interest.
      @param typeInfo  Type array type info structure that needs to be used.
      @return The Type'LOW
  */  
  static IntegerType LOW(const IntegerType& dim, const TypeInfo& typeInfo);

  /** This method is used to determine the Type'LOW for a ArrayType.  The
      'LOW is determined from the type info structure.

      @param dim The dimension of interest.
      @param typeInfo  Type array type info structure that needs to be used.
      @param ti A type info. structure (unused -- can be TypeInfo::nullTypeInfo).   
      @return The Type'LOW
  */
  static EnumerationType LOW(const IntegerType& dim, const TypeInfo& typeInfo, const TypeInfo& ti);
  
  /** This method is used to determine the Type'LENGTH for a ArrayType.  The
      'LENGTH is determined from the type info structure.

      @param dim The dimension of interest.
      @param typeInfo  Type array type info structure that needs to be used.
      @return The Type'LENGTH
  */
  static IntegerType LENGTH(const IntegerType& dim, const TypeInfo& typeInfo);

 /** This method is used to determine the Type'ASCENDING for a ArrayType.  The
      'ASCENDING is determined from the type info structure.

      @param dim The dimension of interest.
      @param ti  Type array type info structure that needs to be used.
      @return The Type'ASCENDING
  */
  static EnumerationType ASCENDING(const IntegerType& dim, const TypeInfo& ti);

  /** The following pointer is used to specify the bounds of an unconstrained
      array that is element off an array type.  */
  static ArrayInfo* rangeInfo;

  void Add( const VHDLType & );
  void Add( const SignalNetinfo *rhs );
  void Add( VHDLKernel *processPtr );
  void Add( VHDLKernel *processPtr, int sigId );

  void copyId( const VHDLType &src );
  void resolveAndUpdate( VHDLKernel * );
  bool is_driver_already_set() const;
  bool is_resolved_signal() const;
  void set_sourcebase_delete_flag( bool flag ) const;

  void assignSignal( VHDLKernel *destProcess, 
		     VHDLKernel *srcId, 
		     const VHDLType& src,
		     const PhysicalType& delay, 
		     const PhysicalType& rejTime, 
		     const ArrayInfo& dinfo,
		     const ArrayInfo& sinfo );

  //@} // End of Public class Methods of ArrayType
  
private:
  /**@name Private Class Methods of ArrayType. */
  //@{
  
  /// The list of objects contained by this array type
  vector<VHDLType *> object;

  /// The bounds for this object
  vector<ArrayInfo> bounds;

  /**
     Returns the array index for the element index and the dimension passed
     in.
  */
  int storageIndex( int dimension, int elementIndex ) const;
  
  /** This flag is to check whether this ArrayType is a composite type.  This
      flag will be set in setCompositeResolvedSignal. It can also be set in
      SetParentCompositeType method of ArrayType.
  */
  bool isCompositeResolvedType;

  /// This object holds the resolution function id associated with this object
  ResolutionFnId_t resolutionFunctionId;
  
  /// This holds the information about the ranges and dimensions
  ArrayTypeInfo arrayInfo;

  /// This holds information of sub-arrays for 2-d array types
  ArrayTypeInfo subArrayInfo;

  void buildArray( ObjectBase::ObjectType );

  /**
     Performs a dyadic operation on the array, with the array passed in as
     the argument.
  */
  ArrayType bitOperation( const ArrayType &rhs,
			  const EnumerationType &(EnumerationType::*operation)(const EnumerationType& ) const )const;

  /**
     Performs a logical shift on this array.  Positive amounts shift left,
     negative amounts shift right.  Assumes the ArrayType is one dimensional.
  */
  void shiftLogical( int shiftAmount );

  /**
     Prints one dimension - used by "print".
  */
  void printDimension( ostream &os, int dimension ) const;

  
  //@} // End of Private class Methods of ArrayType  
};


ArrayType savantConcatenate(const ArrayType  &, const ArrayType  &);
ArrayType savantConcatenate(const ArrayType  &, const VHDLType &);
ArrayType savantConcatenate(const VHDLType &, const ArrayType  &);
ArrayType savantConcatenate(const VHDLType &, const VHDLType &, const ArrayTypeInfo &);

extern "C" bool eatwhite( AccessVariable &line );
typedef ArrayType ArrayType_event;
typedef ArrayType ArrayType_lastevent;

extern ostream& operator<<(ostream &, const ArrayType &);

#endif

