//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.0.1, Copyright (C) Peter A. Buhr 2001
// 
// pthread.cc -- 
// 
// Author           : Peter A. Buhr
// Created On       : Sun Dec  9 21:38:53 2001
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Aug 31 23:47:18 2004
// Update Count     : 551
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#include <uC++.h>
#include <uSystemTask.h>
#include <uSequence.h>
#include <csignal>					// access: sigset_t
#include <cerrno>					// access: EBUSY, ETIMEDOUT
#include <pthread.h>
#include <limits.h>					// access: PTHREAD_KEYS_MAX

//#include <uDebug.h>

#if defined( __irix__ )					// TEMPORARY
#undef PTHREAD_THREADS_MAX
#define PTHREAD_THREADS_MAX _POSIX_THREAD_THREADS_MAX
#undef PTHREAD_KEYS_MAX
#define PTHREAD_KEYS_MAX _POSIX_THREAD_KEYS_MAX
#define __PID_TABLE__					// must use pid table 8-(
extern "C" long _sysconf( int );			// TEMPORARY: unnecessary
#endif // __irix__


// Only use the pid table if the sizeof(pthread_t) < sizeof(void *).  Pointers
// versus integers allows creation of pthreads to be limited by the size of
// memory versus the size of the pid table.
#ifdef __PID_TABLE__
static pthread_mutex_t u_pthread_create_lock = PTHREAD_MUTEX_INITIALIZER;

static uBaseTask *u_pthread_pidtable[PTHREAD_THREADS_MAX] = { 0 }; // set all elements to zero
static int u_pthread_max = 0;

static int u_pthread_hash( uBaseTask *tid ) {
    int posn = (size_t)tid % 1009 /* prime number */ % PTHREAD_THREADS_MAX;
    if ( u_pthread_pidtable[posn] != tid ) {
	for ( int i = (posn + 1) % PTHREAD_THREADS_MAX; ; i = (i + 1) % PTHREAD_THREADS_MAX ) {
	  if ( i == posn ) {				// no empty slots
		posn = -1;
		break;
	    } // exit
	    uBaseTask *v = u_pthread_pidtable[i];
	  if ( v == tid || v == 0 ) {			// found or next empty slot
		posn = i;
		break;
	    } // exit
	} // for
    } // if
    return posn;
} // u_pthread_hash

#define u_pthread_lookup( name ) ((uPThread *)(u_pthread_pidtable[name]))
#else
#define u_pthread_hash( name ) name
#define u_pthread_lookup( name ) ((uPThread *)name)
#endif // __PID_TABLE__


uThrowEvent uPThreadExit {
  public:
    uPThreadExit( void *joinval ) : uEHM::uThrowClass( "pthread exit" ), joinval( joinval ) {}
    void *joinval;
}; // uPThreadExit
uInitEvent( uPThreadExit );

uThrowEvent uPThreadCancel {
  public:
    uPThreadCancel() : uEHM::uThrowClass( "pthread cancellation" ) {}
}; // uPThreadCancel
uInitEvent( uPThreadCancel );


static struct u_pthread_attr_t {			// thread attributes
    int contentionscope;
    int detachstate;
    size_t stacksize;
    void *stackaddr;
    int policy;
    int inheritsched;
    struct sched_param param;
} u_pthread_attr_defaults = { PTHREAD_SCOPE_SYSTEM,
			      PTHREAD_CREATE_JOINABLE,
			      __U_DEFAULT_STACK_SIZE__,
			      NULL,
			      0,
			      PTHREAD_EXPLICIT_SCHED,
			      { 0 },
};


struct pthread_values : public uSeqable {		// thread specific data
    bool in_use;
    void *value;
};
struct pthread_key_struct {				// all of these fields are initialized with zero
    bool in_use;
    void (*destructor)( void * );
    uSequence<pthread_values> threads;
};
// Create storage separately to ensure no constructors are called.
char u_pthread_keys_storage[sizeof(pthread_key_struct) * PTHREAD_KEYS_MAX] __attribute__ ((aligned (16))) = {0};
#define u_pthread_keys ((pthread_key_struct *)u_pthread_keys_storage)

static pthread_mutex_t u_pthread_mutex_init = PTHREAD_MUTEX_INITIALIZER; // dynamic initialization value

static pthread_mutex_t u_pthread_keys_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t u_pthread_once_lock = PTHREAD_MUTEX_INITIALIZER;


uTask uPThread;						// forward declarations
extern "C" {
    void _pthread_cancel_check( uPThread *p );
}


uTask uPThread {
    friend int pthread_join( pthread_t threadID, void **status ); // access: attr
    friend int pthread_detach( pthread_t threadID );	// access: attr
    friend int pthread_setspecific( pthread_key_t key, const void *value ); // access: values
    friend void *pthread_getspecific( pthread_key_t key ); // access: values

    friend int pthread_cancel( pthread_t thread );	// access: cancelled, cancel_state
    friend int pthread_setcancelstate( int state, int *oldstate ); // access: cancel_state
    friend int pthread_setcanceltype( int type, int *oldtype ); // access: cancel_type
    friend void _pthread_cancel_check( uPThread *p );	// access: cancelled, cancel_state, cancel_type

    u_pthread_attr_t attr;				// pthread attributes
    void *(*start_func)( void * );
    void *arg, *joinval;				// thread parameter and join value

    bool cancelled;
    int cancel_state, cancel_type;
    pthread_mutex_t cancel_lock;

    void main() {
	try {
	    uEnable {
		joinval = (*start_func)( arg );
	    } // uEnable
	} catch( uPThreadExit pthexit ) {		// sync thrown by pthread_exit
	    joinval = pthexit.joinval;
	} catch( uPThreadCancel ) {			// async thrown by pthread_cancel
	} // try

	if ( attr.detachstate == PTHREAD_CREATE_DETACHED ) {
	    uKernelModule::uTaskSystem->pthreadDetachEnd( *this );
	} // if
    } // uPThread::main
  public:
    uPThread( void * (*start_func)( void * ), void *arg, const pthread_attr_t *attr, size_t stacksize ) :
		uBaseTask( stacksize ), start_func( start_func ), arg( arg ) {
	if ( attr == NULL ) {				// construct default attribute
	    uPThread::attr = u_pthread_attr_defaults;	// copy default attributes
	    uPThread::attr.stacksize = stacksize;
	} else {
	    uPThread::attr = **((u_pthread_attr_t **)attr); // copy attributes
	    if ( uPThread::attr.detachstate == PTHREAD_CREATE_DETACHED ) {
		uKernelModule::uTaskSystem->pthreadDetachStart();
	    } // if
	} // if

	cancelled = false;
	cancel_state = PTHREAD_CANCEL_ENABLE;
	cancel_type  = PTHREAD_CANCEL_DEFERRED;
	cancel_lock = u_pthread_mutex_init;
    } // uPThread::uPThread

    void *join() {
	return joinval;
    } // uPThread::join
}; // uPThread


extern "C" {
    // Creation
    int pthread_create( pthread_t *new_thread_id, const pthread_attr_t *attr, void * (*start_func)( void * ), void *arg ) {
	int stacksize = ( attr != NULL ) ? (*(u_pthread_attr_t **)attr)->stacksize : uDefaultStackSize();
#ifdef __PID_TABLE__
	// storage for a pthread_t must be < uBaseTask *
	pthread_mutex_lock( &u_pthread_create_lock );
      if ( u_pthread_max == PTHREAD_THREADS_MAX ) {
	    pthread_mutex_unlock( &u_pthread_create_lock );
	    return EAGAIN;
	} // if
	uBaseTask *tid = new uPThread( start_func, arg, attr, stacksize );
	pthread_t posn = u_pthread_hash( tid );		// not called if no space remaining
	u_pthread_max += 1;
	u_pthread_pidtable[posn] = tid;
	pthread_mutex_unlock( &u_pthread_create_lock );
	*new_thread_id = posn;
#else
	// storage for a pthread_t must be >= uBaseTask *
	_STATIC_ASSERT_( sizeof(pthread_t) >= sizeof(uBaseTask *) );
	*new_thread_id = (pthread_t)new uPThread( start_func, arg, attr, stacksize );
#endif // __PID_TABLE__
	return 0;
    } // pthread_create

    int pthread_attr_init( pthread_attr_t *attr ) {
	// storage for pthread_attr_t must be >= void *
	_STATIC_ASSERT_( sizeof(pthread_attr_t) >= sizeof(void *) );
	*((u_pthread_attr_t **)attr) = new u_pthread_attr_t;
	**((u_pthread_attr_t **)attr) = u_pthread_attr_defaults; // set all fields to zero
	return 0;
    } //  pthread_attr_init

    int pthread_attr_destroy( pthread_attr_t *attr ) {
	delete *((u_pthread_attr_t **)attr);
	return 0;
    } // pthread_attr_destroy

    int pthread_attr_setscope( pthread_attr_t *attr, int contentionscope ) {
	(*(u_pthread_attr_t **)attr)->contentionscope = contentionscope;
	return 0;
    } // pthread_attr_setscope

    int pthread_attr_getscope( const pthread_attr_t *attr, int *contentionscope ) {
	*contentionscope = (*(u_pthread_attr_t **)attr)->contentionscope;
	return 0;
    } // pthread_attr_getscope

    int pthread_attr_setdetachstate( pthread_attr_t *attr, int detachstate ) {
	(*(u_pthread_attr_t **)attr)->detachstate = detachstate;
	return 0;
    } // pthread_attr_setdetachstate

    int pthread_attr_getdetachstate( const pthread_attr_t *attr, int *detachstate ) {
	*detachstate = (*(u_pthread_attr_t **)attr)->detachstate;
	return 0;
    } // pthread_attr_getdetachstate

    int pthread_attr_setstacksize( pthread_attr_t *attr, size_t stacksize ) {
	(*(u_pthread_attr_t **)attr)->stacksize = stacksize;
	return 0;
    } // pthread_attr_setstacksize

    int pthread_attr_getstacksize( const pthread_attr_t *attr, size_t *stacksize ) {
	*stacksize = (*(u_pthread_attr_t **)attr)->stacksize;
	return 0;
    } // pthread_attr_getstacksize

    int pthread_attr_setstackaddr( pthread_attr_t *attr, void *stackaddr ) {
	(*(u_pthread_attr_t **)attr)->stackaddr = stackaddr;
	return ENOSYS;					// unsupported
    } // pthread_attr_setstackaddr

    int pthread_attr_getstackaddr( const pthread_attr_t *attr, void **stackaddr ) {
	*stackaddr = (*(u_pthread_attr_t **)attr)->stackaddr;
	return ENOSYS;					// unsupported
    } // pthread_attr_getstackaddr

    int pthread_attr_setschedpolicy( pthread_attr_t *attr, int policy ) {
	(*(u_pthread_attr_t **)attr)->policy = policy;
	return ENOSYS;					// unsupported
    } // pthread_attr_setschedpolicy

    int pthread_attr_getschedpolicy( const pthread_attr_t *attr, int *policy ) {
	*policy = (*(u_pthread_attr_t **)attr)->policy;
	return ENOSYS;
    } // pthread_attr_getschedpolicy

    int pthread_attr_setinheritsched( pthread_attr_t *attr, int inheritsched ) {
	(*(u_pthread_attr_t **)attr)->inheritsched = inheritsched;
	return ENOSYS;
    } // pthread_attr_setinheritsched

    int pthread_attr_getinheritsched( const pthread_attr_t *attr, int *inheritsched ) {
	*inheritsched = (*(u_pthread_attr_t **)attr)->inheritsched;
	return ENOSYS;
    } // pthread_attr_getinheritsched

    int pthread_attr_setschedparam( pthread_attr_t *attr, const struct sched_param *param ) {
	(*(u_pthread_attr_t **)attr)->param = *param;
	return ENOSYS;
    } // pthread_attr_setschedparam

    int pthread_attr_getschedparam( const pthread_attr_t *attr, struct sched_param *param ) {
	*param = ((u_pthread_attr_t *)attr)->param;
	return ENOSYS;
    } // pthread_attr_getschedparam


    // Exit
    void pthread_exit( void *status ) {
	// Exception caught in "main" routine of current task.
	uThrow uPThreadExit( status );
	// CONTROL NEVER REACHES HERE!
	abort();
    } // pthread_exit

    int pthread_join( pthread_t threadID, void **status ) {
      if ( u_pthread_lookup( threadID )->attr.detachstate != PTHREAD_CREATE_JOINABLE ) return EINVAL;
	if ( status != NULL ) *status = u_pthread_lookup( threadID )->join();
#ifdef __PID_TABLE__
	pthread_mutex_lock( &u_pthread_create_lock );
	delete u_pthread_lookup( threadID );
	u_pthread_pidtable[threadID] = 0;
	u_pthread_max -= 1;
	pthread_mutex_unlock( &u_pthread_create_lock );
#else
	delete u_pthread_lookup( threadID );
#endif // __PID_TABLE__
	return 0;
    } // pthread_join

    int pthread_detach( pthread_t threadID ) {
	// There is always a race condition in setting the detach flag, even if
	// "detach" is accessed with mutual exclusion.
	u_pthread_lookup( threadID )->attr.detachstate = PTHREAD_CREATE_DETACHED;
	return 0;
    } // pthread_detach


    // Thread Specific Data
    int pthread_key_create( pthread_key_t *key, void (*destructor)( void * ) ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_key_create(key:0x%p, destructor:0x%p) enter task:0x%p\n", key, destructor, &uThisTask() );
#endif // __U_DEBUG_H__
	pthread_mutex_lock( &u_pthread_keys_lock );
	for ( int i = 0; i < PTHREAD_KEYS_MAX; i += 1 ) {
 	    if ( ! u_pthread_keys[i].in_use ) {
		u_pthread_keys[i].in_use = true;
		u_pthread_keys[i].destructor = destructor;
		pthread_mutex_unlock( &u_pthread_keys_lock );
		*key = i;
#ifdef __U_DEBUG_H__
		uDebugPrt( "pthread_key_create(key:%d, destructor:0x%p) exit task:0x%p\n", *key, destructor, &uThisTask() );
#endif // __U_DEBUG_H__
		return 0;
	    } // if
	} // for
	pthread_mutex_unlock( &u_pthread_keys_lock );
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_key_create(key:0x%p, destructor:0x%p) try again!!! task:0x%p\n", key, destructor, &uThisTask() );
#endif // __U_DEBUG_H__
	return EAGAIN;
    } // pthread_key_create

    int pthread_key_delete( pthread_key_t key ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_key_delete(key:0x%p) enter task:0x%p\n", key, &uThisTask() );
#endif // __U_DEBUG_H__
	pthread_mutex_lock( &u_pthread_keys_lock );
      if ( key >= PTHREAD_KEYS_MAX || ! u_pthread_keys[key].in_use ) {
	    pthread_mutex_unlock( &u_pthread_keys_lock );
	    return EINVAL;
	} // if
	u_pthread_keys[key].in_use = false;
	u_pthread_keys[key].destructor = NULL;
	
	// Remove key from all threads with a value.

	pthread_values *p;
	uSequence<pthread_values> &head = u_pthread_keys[key].threads;
	for ( uSeqGen<pthread_values> gen(head); gen >> p; ) {
	    p->in_use = false;
	    head.uRemove( p ); 
	} // for

	pthread_mutex_unlock( &u_pthread_keys_lock );
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_key_delete(key:0x%p) exit task:0x%p\n", key, &uThisTask() );
#endif // __U_DEBUG_H__
	return 0;
    } // pthread_key_delete

    int pthread_setspecific( pthread_key_t key, const void *value ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_setspecific(key:0x%p, value:0x%p) enter task:0x%p\n", key, value, &uThisTask() );
#endif // __U_DEBUG_H__
	uBaseTask &t = uThisTask();

	pthread_values *values;
	if ( t.pthreadData == NULL ) {
	    values = new pthread_values[PTHREAD_KEYS_MAX];
	    t.pthreadData = values;
	    for ( int i = 0; i < PTHREAD_KEYS_MAX; i += 1 ) {
		values[i].in_use = false;
	    } // for
	} else {
	    values = (pthread_values *)t.pthreadData;
	} // if

	pthread_mutex_lock( &u_pthread_keys_lock );
      if ( key >= PTHREAD_KEYS_MAX || ! u_pthread_keys[key].in_use ) {
	    pthread_mutex_unlock( &u_pthread_keys_lock );
	    return EINVAL;
	} // if

	pthread_values &entry = values[key];
	if ( ! entry.in_use ) {
	    entry.in_use = true;
	    u_pthread_keys[key].threads.uAdd( &entry );
	} // if
	entry.value = (void *)value;
	pthread_mutex_unlock( &u_pthread_keys_lock );
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_setspecific(key:0x%p, value:0x%p) exit task:0x%p\n", key, value, &uThisTask() );
#endif // __U_DEBUG_H__
	return 0;
    } // pthread_setspecific

    void *pthread_getspecific( pthread_key_t key ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_getspecific(key:0x%p) enter task:0x%p\n", key, &uThisTask() );
#endif // __U_DEBUG_H__
      if ( key >= PTHREAD_KEYS_MAX ) return NULL;

	uBaseTask &t = uThisTask();
      if ( t.pthreadData == NULL ) return NULL;

	pthread_mutex_lock( &u_pthread_keys_lock );
	pthread_values &entry = ((pthread_values *)t.pthreadData)[key];
      if ( ! entry.in_use ) {
	    pthread_mutex_unlock( &u_pthread_keys_lock );
	    return NULL;
	} // if
	void *value = entry.value;
	pthread_mutex_unlock( &u_pthread_keys_lock );
#ifdef __U_DEBUG_H__
	uDebugPrt( "0x%p = pthread_getspecific(key:0x%p) exit task:0x%p\n", value, key, &uThisTask() );
#endif // __U_DEBUG_H__
	return value;
    } // pthread_getspecific

    void _pthread_deletespecific( void *pthreadData ) {
	pthread_mutex_lock( &u_pthread_keys_lock );
	pthread_values *values = (pthread_values *)pthreadData;

	// If, after all the destructors have been called for all non-NULL
	// values with associated destructors, there are still some non-NULL
	// values with associated destructors, then the process is
	// repeated. If, after at least PTHREAD_DESTRUCTOR_ITERATIONS
	// iterations of destructor calls for outstanding non-NULL values,
	// there are still some non-NULL values with associated destructors,
	// implementations may stop calling destructors, or they may continue
	// calling destructors until no non-NULL values with associated
	// destructors exist, even though this might result in an infinite
	// loop.
				   
	bool destcalled = true;
	for ( int attempts = 0; attempts < PTHREAD_DESTRUCTOR_ITERATIONS && destcalled ; attempts += 1 ) {
	    destcalled = false;
	    for ( int i = 0; i < PTHREAD_KEYS_MAX; i += 1 ) { // remove each value from key list
#ifdef __U_DEBUG_H__
		uDebugPrt( "_pthread_deletespecific, value[%d].in_use:%d, value:0x%p\n",
			   i, values[i].in_use, values[i].value );
#endif // __U_DEBUG_H__
		if ( values[i].in_use ) {
		    pthread_values &entry = values[i];
		    entry.in_use = false;
		    u_pthread_keys[i].threads.uRemove( &entry );
		    if ( u_pthread_keys[i].destructor != NULL && entry.value != NULL ) {
			void *data = entry.value;
			entry.value = NULL;
#ifdef __U_DEBUG_H__
			uDebugPrt( "_pthread_deletespecific, destructor:0x%p, value:0x%p\n",
				   u_pthread_keys[i].destructor, data );
#endif // __U_DEBUG_H__
			destcalled = true;
			u_pthread_keys[i].destructor( data );
		    } // if
		} // if
	    } // for
	} // for
	pthread_mutex_unlock( &u_pthread_keys_lock );
	delete [] values;
    } // pthread_deletespecific


    // Signal
    int pthread_sigmask( int how, const sigset_t *set, sigset_t *oset ) {
	uAbort( "pthread_sigmask : not implemented" );
	return 0;
    } // pthread_sigmask

    int pthread_kill( pthread_t thread, int sig ) {
	uAbort( "pthread_kill : not implemented" );
	return 0;
    } // pthread_kill


    // ID
    pthread_t pthread_self( void ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_self enter task:0x%p\n", &uThisTask() );
#endif // __U_DEBUG_H__
#ifdef __U_DEBUG__
	uPThread *p = dynamic_cast<uPThread *>(&uThisTask());
      if ( p == NULL ) {
#if ! defined( __irix__ )
	    uAbort( "pthread_self performed on non-pthread task" );
#else
	    // TEMPORARY: IRIX libc calls pthread_self, even for non-pthread tasks
	    return 65536;
#endif // ! __irix__
	} // if
#endif // __U_DEBUG__
	return (pthread_t)u_pthread_hash( &uThisTask() );
    } // pthread_self

#if ! defined( __irix__ )				// TEMPORARY: done with macro: bad dog!
    int pthread_equal( pthread_t t1, pthread_t t2 ) {
	return t1 == t2;
    } // pthread_equal
#endif // ! __irix__

    int pthread_once( pthread_once_t *once_control, void (*init_routine)( void ) ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_once(once_control:0x%p, init_routine:0x%p) enter task:0x%p\n", once_control, init_routine, &uThisTask() );
#endif // __U_DEBUG_H__
	// storage for pthread_once_t must be >= int and initialized to zero by
	// PTHREAD_ONCE_INIT
	_STATIC_ASSERT_( sizeof(pthread_once_t) >= sizeof(int) );
	pthread_mutex_lock( &u_pthread_once_lock );
	if ( *((int *)once_control) == 0 ) {
	    init_routine();
	    *((int *)once_control) = 1;
	} // if
	pthread_mutex_unlock( &u_pthread_once_lock );
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_once(once_control:0x%p, init_routine:0x%p) exit task:0x%p\n", once_control, init_routine, &uThisTask() );
#endif // __U_DEBUG_H__
	return 0;
    } // pthread_once


    // Scheduling
    int pthread_setschedparam( pthread_t thread, int policy, const struct sched_param *param ) {
	uAbort( "pthread_setschedparam : not implemented" );
	return 0;
    }
    int pthread_getschedparam( pthread_t thread, int *policy, struct sched_param *param ) {
	uAbort( "pthread_getschedparam : not implemented" );
	return 0;
    }


    // Cancellation
    int pthread_cancel( pthread_t threadID ) {
	uPThread *p = dynamic_cast<uPThread *>(&uThisTask());
#ifdef __U_DEBUG__
	if ( p == NULL ) uAbort( "pthread_setcancelstate performed on non-pthread task" );
#endif // __U_DEBUG__
	pthread_mutex_lock( &(p->cancel_lock) );
      if ( p->cancel_state == PTHREAD_CANCEL_DISABLE || p->cancelled ) {
	    pthread_mutex_unlock( &(p->cancel_lock) );
	    return 0;
	} // if
	p->cancelled = true;
	pthread_mutex_unlock( &(p->cancel_lock) );
	uThrow uPThreadCancel() uAt *u_pthread_lookup(threadID);
	return 0;
    } // pthread_cancel

    int pthread_setcancelstate( int state, int *oldstate ) {
      if ( state != PTHREAD_CANCEL_ENABLE && state != PTHREAD_CANCEL_DISABLE ) return EINVAL;
	uPThread *p = dynamic_cast<uPThread *>(&uThisTask());
#ifdef __U_DEBUG__
	if ( p == NULL ) uAbort( "pthread_setcancelstate performed on non-pthread task" );
#endif // __U_DEBUG__
	pthread_mutex_lock( &(p->cancel_lock) );
	if ( oldstate != NULL ) *oldstate = p->cancel_state;
	p->cancel_state = state;

	_pthread_cancel_check( p );			// unlocks p->cancel_lock
	return 0;
    } // pthread_setcancelstate

    int pthread_setcanceltype( int type, int *oldtype ) {
      if ( type != PTHREAD_CANCEL_DEFERRED && type != PTHREAD_CANCEL_ASYNCHRONOUS ) return EINVAL;
	uPThread *p = dynamic_cast<uPThread *>(&uThisTask());
#ifdef __U_DEBUG__
	if ( p == NULL ) uAbort( "pthread_setcancelstate performed on non-pthread task" );
#endif // __U_DEBUG__
	pthread_mutex_lock( &(p->cancel_lock) );
	if ( oldtype != NULL ) *oldtype = p->cancel_type;
	p->cancel_type = type;

	_pthread_cancel_check( p );			// unlocks p->cancel_lock
	return 0;
    } // pthread_setcanceltype

    void pthread_testcancel( void ) {
	uEHM::uPoll();					// check for concurrent exceptions
    } // pthread_testcancel

    void _pthread_cancel_check( uPThread *p ) {
	// PRECONDITION: must be called with thread's cancel_lock acquired.
	if ( ! p->cancelled && p->cancel_state == PTHREAD_CANCEL_ENABLE && p->cancel_type == PTHREAD_CANCEL_ASYNCHRONOUS ) {
	    pthread_mutex_unlock( &(p->cancel_lock) );
	    pthread_cancel( pthread_self() );
	} else {
	    pthread_mutex_unlock( &(p->cancel_lock) );
	} // if
    } // _pthread_cancel_check


    // Mutex
    int pthread_mutex_init( pthread_mutex_t *mutex, const pthread_mutexattr_t *attr ) {
	// storage for a pthread_mutex_t must be >= uOwnerLock
	_STATIC_ASSERT_( sizeof(pthread_mutex_t) >= sizeof(uOwnerLock) );
	new( mutex ) uOwnerLock;			// run constructor on supplied storage
	return 0;
    } // pthread_mutex_init

    int pthread_mutex_destroy( pthread_mutex_t *mutex ) {
	((uOwnerLock *)mutex)->~uOwnerLock();		// run destructor on supplied storage
	return 0;
    } // pthread_mutex_destroy

    int pthread_mutex_lock( pthread_mutex_t *mutex ) {
	if ( ! uKernelModule::uKernelModuleInitialized ) {
	    uKernelModule::startup();
	} // if

      if ( THREAD_GETMEM( uDisableInt) ) return 0; // TEMPORARY: profiler allocating memory from the kernel issue
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_mutex_lock(mutex:0x%p) enter task:0x%p\n", mutex, &uThisTask() );
#endif // __U_DEBUG_H__
	((uOwnerLock *)mutex)->acquire();
	return 0;
    } // pthread_mutex_lock

    int pthread_mutex_trylock( pthread_mutex_t *mutex ) {
      if ( THREAD_GETMEM( uDisableInt) ) return 0; // TEMPORARY: profiler allocating memory from the kernel issue
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_mutex_trylock(mutex:0x%p) enter task:0x%p\n", mutex, &uThisTask() );
#endif // __U_DEBUG_H__
	return ((uOwnerLock *)mutex)->tryacquire() ? 0 : EBUSY;
    } // pthread_mutex_trylock

    int pthread_mutex_unlock( pthread_mutex_t *mutex ) {
      if ( THREAD_GETMEM( uDisableInt) ) return 0; // TEMPORARY: profiler allocating memory from the kernel issue
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_mutex_unlock(mutex:0x%p) enter task:0x%p\n", mutex, &uThisTask() );
#endif // __U_DEBUG_H__
	((uOwnerLock *)mutex)->release();
	return 0;
    } // pthread_mutex_unlock


    int pthread_mutexattr_init( pthread_mutexattr_t *attr ) {
	return 0;
    } // pthread_mutexattr_init

    int pthread_mutexattr_destroy( pthread_mutexattr_t *attr ) {
	return 0;
    } // pthread_mutexattr_destroy

    int pthread_mutexattr_setpshared( pthread_mutexattr_t *attr, int pshared ) {
	return 0;
    } // pthread_mutexattr_setpshared

    int pthread_mutexattr_getpshared( const pthread_mutexattr_t *attr, int *pshared ) {
	return 0;
    } // pthread_mutexattr_getpshared

    int pthread_mutexattr_setprotocol( pthread_mutexattr_t *attr, int protocol ) {
	return 0;
    } // pthread_mutexattr_setprotocol

    int pthread_mutexattr_getprotocol( const pthread_mutexattr_t *attr, int *protocol ) {
	return 0;
    } // pthread_mutexattr_getprotocol

    int pthread_mutexattr_setprioceiling( pthread_mutexattr_t *attr, int prioceiling ) {
	return 0;
    } // pthread_mutexattr_setprioceiling

    int pthread_mutexattr_getprioceiling( const pthread_mutexattr_t *attr, int *ceiling ) {
	return 0;
    } // pthread_mutexattr_getprioceiling

    int pthread_mutex_setprioceiling( pthread_mutex_t *mutex, int prioceiling, int *old_ceiling ) {
	return 0;
    } // pthread_mutex_setprioceiling

    int pthread_mutex_getprioceiling( const pthread_mutex_t *mutex, int *ceiling ) {
	return 0;
    } // pthread_mutex_getprioceiling


    // Condition Variable
    int pthread_cond_init( pthread_cond_t *cond, const pthread_condattr_t *attr ) {
	// storage for a pthread_mutex_t must be >= uCondLock
	_STATIC_ASSERT_( sizeof(pthread_mutex_t) >= sizeof(uCondLock) );
	new( cond ) uCondLock;				// run constructor on supplied storage
	return 0;
    } // pthread_cond_init

    int pthread_cond_destroy( pthread_cond_t *cond ) {
	((uCondLock *)cond)->~uCondLock();		// run destructor on supplied storage
	return 0;
    } // pthread_cond_destroy

    int pthread_cond_wait( pthread_cond_t *cond, pthread_mutex_t *mutex ) {
	((uCondLock *)cond)->wait( *(uOwnerLock *)mutex );
	return 0;
    } // pthread_cond_wait

    int pthread_cond_timedwait( pthread_cond_t *cond, pthread_mutex_t *mutex, const struct timespec *abstime ) {
	return ((uCondLock *)cond)->timedwait( *(uOwnerLock *)mutex, uTime( abstime->tv_sec, abstime->tv_nsec ) ) ? 0 : ETIMEDOUT;
    } // pthread_cond_timedwait

    int pthread_cond_signal( pthread_cond_t *cond ) {
	((uCondLock *)cond)->signal();
	return 0;
    } // pthread_cond_signal

    int pthread_cond_broadcast( pthread_cond_t *cond ) {
	((uCondLock *)cond)->broadcast();
	return 0;
    } // pthread_cond_broadcast


    int pthread_condattr_init( pthread_condattr_t *attr ) {
	// storage for pthread_condattr_t must be >= int
	return 0;
    } // pthread_condattr_init

    int pthread_condattr_destroy( pthread_condattr_t *attr ) {
	return 0;
    } // pthread_condattr_destroy

    int pthread_condattr_setpshared( pthread_condattr_t *attr, int pshared ) {
	*((int *)attr) = pshared;
	return 0;
    } // pthread_condattr_setpshared

    int pthread_condattr_getpshared( const pthread_condattr_t *attr, int *pshared ) {
	*pshared = *((int *)attr);
	return 0;
    } // pthread_condattr_getpshared
} // extern "C"


#if defined( __solaris__ )
#include <thread.h>

extern "C" {
    // Creation
    int _thr_create( void *stackaddr, size_t stacksize, void *(*start_func)( void* ), void *arg, long flags, thread_t *new_thread_id ) {
	pthread_attr_t attr;
	pthread_attr_init( &attr );
	pthread_attr_setstacksize( &attr, stacksize );
	pthread_attr_setstackaddr( &attr, stackaddr );
	pthread_attr_setdetachstate( &attr, flags | THR_DETACHED ? PTHREAD_CREATE_DETACHED : PTHREAD_CREATE_JOINABLE );
	pthread_attr_setscope( &attr, flags | THR_BOUND ? PTHREAD_SCOPE_SYSTEM : PTHREAD_SCOPE_PROCESS );
	int code = pthread_create( (pthread_t *)new_thread_id, &attr, start_func, arg );
	pthread_attr_destroy( &attr );
	if ( code != 0 ) {
	    errno = code;
	    code = -1;
	} // if
	return code;
    } // thr_create


    // Mutex
    int _mutex_init( mutex_t *mutex, int type, void * arg ) {
	return pthread_mutex_init( (pthread_mutex_t *)mutex, NULL );
    } // mutex_init

    int _mutex_destroy( mutex_t *mutex ) {
	return pthread_mutex_destroy( (pthread_mutex_t *)mutex );
    } // mutex_destroy

    int _mutex_lock( mutex_t *mutex ) {
	if ( mutex->flags.magic == MUTEX_MAGIC ) pthread_mutex_init( (pthread_mutex_t *)mutex, NULL );
	return pthread_mutex_lock( (pthread_mutex_t *)mutex );
    } // mutex_lock

    int _mutex_trylock( mutex_t *mutex ) {
	if ( mutex->flags.magic == MUTEX_MAGIC ) pthread_mutex_init( (pthread_mutex_t *)mutex, NULL );
	return pthread_mutex_trylock( (pthread_mutex_t *)mutex );
    } // mutex_trylock

    int _mutex_unlock( mutex_t *mutex ) {
	return pthread_mutex_unlock( (pthread_mutex_t *)mutex );
    } // mutex_unlock

    // Condition Variable
    int cond_init( cond_t *cond, int type, void *arg ) {
	pthread_condattr_t attr;
	pthread_condattr_init( &attr );
	pthread_condattr_setpshared( &attr, type == USYNC_PROCESS ? PTHREAD_PROCESS_SHARED : PTHREAD_PROCESS_PRIVATE );
	pthread_cond_init( (pthread_cond_t *)cond, &attr );
	pthread_condattr_destroy( &attr );
	return 0;
    } // cond_init

    int cond_destroy( cond_t *cond ) {
	return pthread_cond_destroy( (pthread_cond_t *)cond );
    } // cond_destroy

    int cond_wait( cond_t *cond, mutex_t *mutex ) {
	return pthread_cond_wait( (pthread_cond_t *)cond, (pthread_mutex_t *)mutex );
    } // cond_wait

    int cond_timedwait( cond_t *cond, mutex_t *mutex, timestruc_t *abstime ) {
	return pthread_cond_timedwait( (pthread_cond_t *)cond, (pthread_mutex_t *)mutex, abstime );
    } // cond_timedwait

    int cond_signal( cond_t *cond ) {
	return pthread_cond_signal( (pthread_cond_t *)cond );
    } // cond_signal

    int cond_broadcast( cond_t *cond ) {
	return pthread_cond_broadcast( (pthread_cond_t *)cond );
    } // cond_broadcast
}
#endif // __solaris__

// Local Variables: //
// compile-command: "gmake install" //
// End: //
