//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.0.1, Copyright (C) Peter A. Buhr 1994
// 
// uFile.h -- Nonblocking UNIX I/O library
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 16:38:54 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Aug 11 14:11:57 2004
// Update Count     : 109
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#ifndef __U_FILE_H__
#define __U_FILE_H__


#include <uIOcntl.h>
#include <uIOEvent.h>


#pragma __U_NOT_USER_CODE__


#include <fcntl.h>					// open, mode flags
#include <sys/stat.h>					// stat


//######################### uFile #########################


class uFile {
    friend class uFileWrapper;
    friend class uFileAccess;
    friend class uFilebuf;

    char *uName;
    int accessCnt;

    void uCreateFile( const char *name );
    uMutex void access();
    uMutex void unaccess();
  public:
    uFile( const char *name );
    virtual ~uFile();

    const char *uGetName() const;
    void status( struct stat &buf );

    uDualEvent uFailure;
    uDualEvent uTerminateFailure;
    uDualEvent uStatusFailure;
}; // uFile


//######################### uFileIO #########################


class uFileIO {
    int _read( char *buf, int len );
    int _readv( const struct iovec *iov, int iovcnt );
    int _write( char *buf, int len );
    int _writev( const struct iovec *iov, int iovcnt );
  protected:
    uIOaccess &access;

    uFileIO( uIOaccess &acc );
    virtual ~uFileIO();
  public:
    int read( char *buf, int len, uDuration *timeout = NULL );
    int readv( const struct iovec *iov, int iovcnt, uDuration *timeout = NULL );
    uMutex int write( char *buf, int len, uDuration *timeout = NULL );
    int writev( const struct iovec *iov, int iovcnt, uDuration *timeout = NULL );
    int fd();
}; // uFileIO


//######################### uFileAccess #########################


class uFileAccess : public uFileIO {		// monitor
    friend class uFilebuf;

    uFile &ufile;
    uIOaccess access;

    uFileAccess( int fd, uFile &f );
  public:
    uFileAccess( uFile &f, int flags, int mode = 0644 );
    uMutex virtual ~uFileAccess();

    off_t lseek( off_t offset, int whence );
    int fsync();

    uDualEvent uFailure;
    friend uDualEvent uFailure;
    uDualEvent uOpenFailure;
    uDualEvent uCloseFailure;
    uDualEvent uSeekFailure;
    uDualEvent uSyncFailure;
    uDualEvent uReadFailure;
    uDualEvent uReadTimeout;
    uDualEvent uWriteFailure;
    uDualEvent uWriteTimeout;
}; // uFileAccess


//######################### uFile (cont) #########################


uDualEvent uFile::uFailure : public uIOFailure {
    const uFile &f;
    char uName[uEHMMaxName + sizeof("...")];		// room for "...\0"
  public:
    uFailure( const uFile &f, const char *const msg );
    ~uFailure();
    const uFile &file() const;
    const char *uGetName() const;
    virtual void defaultTerminate() const;
}; // uFile::uFailure


uDualEvent uFile::uTerminateFailure : public uFile::uFailure {
    const int accessCnt;
  public:
    uTerminateFailure( const uFile &f, const int accessCnt, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFile::uTerminateFailure


uDualEvent uFile::uStatusFailure : public uFile::uFailure {
    const struct stat &buf;
  public:
    uStatusFailure( const uFile &f, const struct stat &buf, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFile::uStatusFailure


//######################### uFileAccess (cont) #########################


uDualEvent uFileAccess::uFailure : public uFile::uFailure {
    const uFileAccess &fa;
    int fd;
  public:
    uFailure( const uFileAccess &fa, const char *const msg );
    const uFileAccess &fileAccess() const;
    int fileDescriptor() const;
    virtual void defaultTerminate() const;
}; // uFileAccess::uFailure


uDualEvent uFileAccess::uOpenFailure : public uFileAccess::uFailure {
    const int flags;
    const int mode;
  public:
    uOpenFailure( uFileAccess &fa, int flags, int mode, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFileAccess::uOpenFailure


uDualEvent uFileAccess::uCloseFailure : public uFileAccess::uFailure {
  public:
    uCloseFailure( uFileAccess &fa, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFileAccess::uCloseFailure


uDualEvent uFileAccess::uSeekFailure : public uFileAccess::uFailure {
    const off_t offset;
    const int whence;
  public:
    uSeekFailure( const uFileAccess &fa, const off_t offset, const int whence, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFileAccess::uSeekFailure


uDualEvent uFileAccess::uSyncFailure : public uFileAccess::uFailure {
  public:
    uSyncFailure( const uFileAccess &fa, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFileAccess::uSyncFailure


uDualEvent uFileAccess::uReadFailure : public uFileAccess::uFailure {
  protected:
    const char *buf;
    const int len;
    const uDuration *timeout;
  public:
    uReadFailure( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFileAccess::uReadFailure


uDualEvent uFileAccess::uReadTimeout : public uFileAccess::uReadFailure {
  public:
    uReadTimeout( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
}; // uFileAccess::uReadTimeout


uDualEvent uFileAccess::uWriteFailure : public uFileAccess::uFailure {
  protected:
    const char *buf;
    const int len;
    const uDuration *timeout;
  public:
    uWriteFailure( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
    virtual void defaultResume() const;			// handle special case when errno == EIO
    virtual void defaultTerminate() const;
}; // uFileAccess::uWriteFailure


uDualEvent uFileAccess::uWriteTimeout : public uFileAccess::uWriteFailure {
  public:
    uWriteTimeout( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
}; // uFileAccess::uWriteFailure


#pragma __U_USER_CODE__

#endif // __U_FILE_H__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
