/* ==================================================== ======== ======= *
 *
 *  uuima.cpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuima.cpp	ubit:03.06.04"
#include <list>
#include <iostream>
#include <string.h>
#include <udefs.hpp>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uprop.hpp>
#include <ucontext.hpp>
#include <ugraph.hpp>
#include <uctrl.hpp>
#include <uview.hpp>
#include <ucolor.hpp>
#include <ustr.hpp>
#include <uima.hpp>
#include <ugraph.hpp>
#include <unatima.hpp>
#include <uevent.hpp>
#include <uappli.hpp>
#include <ubox.hpp>
#include <update.hpp>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UIma::UIma(const char* file_name) : UElem() {
  name = null;
  show_unknown_ima = false;
  setImpl(file_name);
}

UIma::UIma(const UStr& _filename) : UElem() {
  name    = null;
  show_unknown_ima = false;
  setImpl(_filename.chars());
}

UIma::UIma(const char** xpm_data, u_modes m) : UElem(m) {
  name    = null;
  show_unknown_ima = false;
  setImpl(xpm_data);
}

UIma::~UIma() {
  if (name) free(name);
  cleanCache();
}

void UIma::cleanCache() {
  // NB: on ne detruit pas data (data n'est pas copie et 
  // pointe donc sur l'original)

  for (list<UNatIma*>::iterator p = natimas.begin(); p != natimas.end(); p++) {
    delete (*p);
  }
  natimas.clear();
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UIma::load() {
  if (!name && !data) return stat = UFilestat::CannotOpen;
  realize(*UAppli::getApp(), true);
  changed(true);  // *apres* le realize
  return stat;
}

int UIma::readFile(const UStr& fname) {
  setImpl(fname.chars());
  realize(*UAppli::getApp(), true);
  changed(true);  // *apres* le realize
  return stat;
}

int UIma::readFile(const char* fname) {
  setImpl(fname);
  realize(*UAppli::getApp(), true);
  changed(true); // *apres* le realize
  return stat;
}

int UIma::readData(const char** xpm_data) {
  setImpl(xpm_data);
  realize(*UAppli::getApp(), true);
  changed(true); // *apres* le realize
  return stat;
}

/* ==================================================== ======== ======= */

void UIma::set(const UStr& fname) {
  setImpl(fname.chars());
  changed(true);
}
void UIma::set(const char* fname) {
  setImpl(fname);
  changed(true);
}
void UIma::set(const char** xpm_data) {
  setImpl(xpm_data);
  changed(true);
}

/* ==================================================== ======== ======= */

void UIma::setImpl(const char* fname) {
  cleanCache();
  if (name) free(name);
  name      = fname ? CStr::strdup(fname) : null;
  data      = null;
  stat      = UFilestat::NotOpened;
  read_from_file = true;
  read_from_data = false;
}

void UIma::setImpl(const char** xpm_data) {
  cleanCache();
  if (name) free(name);
  name      = null;
  data      = xpm_data;  //!!att: NOT copied!
  stat      = UFilestat::NotOpened;
  read_from_file = false;
  read_from_data = true;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UIma::copy(const UIma& ima2) {
  if (&ima2 == this || ima2.natimas.empty()) return;
  setImpl((char*)null);

  name = ima2.name ? CStr::strdup(ima2.name) : null;
  stat = ima2.stat;
  natimas.push_back((*ima2.natimas.begin())->createScaledImage(1.,1.));
  changed(true);
}

void UIma::copyRescaled(const UIma& ima2, float xscale, float yscale) {
  if (&ima2 == this || ima2.natimas.empty()) return;
  setImpl((char*)null);

  name = ima2.name ? CStr::strdup(ima2.name) : null;
  stat = ima2.stat;
  natimas.push_back((*ima2.natimas.begin())->createScaledImage(xscale,yscale));
  changed(true);
}

void UIma::copyResized(const UIma& ima2, u_dim width, u_dim height) {
  if (&ima2 == this || ima2.natimas.empty()) return;
  setImpl((char*)null);

  name = ima2.name ? CStr::strdup(ima2.name) : null;
  stat = ima2.stat;

  list<UNatIma*>::iterator p = ima2.natimas.begin();
  float w2 = (*p)->getWidth();
  float h2 = (*p)->getHeight();

  if (w2 == 0 || h2 == 0) return;
  natimas.push_back((*p)->createScaledImage(width/w2, height/h2));
  changed(true);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UPix::UPix(const char* file_name) : UIma(file_name) {
  show_unknown_ima = true;
}

UPix::UPix(const UStr& file_name)  : UIma(file_name) {
  show_unknown_ima = true;
}

UPix::UPix(const char** xpm_data, u_modes m) : UIma(xpm_data, m) {
  show_unknown_ima = true;
}

UPix::~UPix() {
  //!att: appel necessaire car ~UIma appelara le cleanCache de UIma
  // (l'appel des fct virtuelles ne l'est dans dans les destructeurs!)
  cleanCache();
}

void UPix::cleanCache() {
  for (u_int k = 0; k < natpixs.size(); k++) {
    delete natpixs[k]; natpixs[k] = null;
  }
  natpixs.clear();
  UIma::cleanCache();
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UIma::update() 
{parents.updateParents(UUpdate::all);}

const char* UIma::getNameChars() const 
{return name;}

int UIma::getStatus() const
{return stat;}

bool UIma::isLoaded() const
{return stat >= UFilestat::Opened;}

u_dim UIma::getHeight() const 
{return (natimas.empty() ? -1 : (*natimas.begin())->getHeight());}

u_dim UIma::getWidth() const 
{return (natimas.empty() ? -1 : (*natimas.begin())->getWidth());}

bool UIma::isRealized() const 
{return (natimas.empty() ? false : (*natimas.begin())->isRealized());}

bool UIma::isShaped() const 
{return (natimas.empty() ? false : (*natimas.begin())->isShaped());}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UIma::getSize(UContext *props, u_dim *w, u_dim *h) const {
  UWinGraph &g = props->winview->wg();

  if (stat == UFilestat::NotOpened && natimas.empty())
    realize(*g.getDisp(), false);

  // !!ATTENTION: l'image peut etre partage par plusieurs objets,
  // d'ou la necessite d'initialiser TOUS les links correctement
  // meme si l'image en elle-meme n'est charge qu'une seule fois

  if (natimas.empty()) {
    if (show_unknown_ima) {            //empecher recursion infinie
      if (this != g.getDisp()->getConf().unknown_image) {
	g.getDisp()->getConf().unknown_image->getSize(props, w, h);
      }
      else {*w = 0; *h =0;}
    }
    else {*w = 0; *h =0;}
  }

  else {
    UNatIma* ni = findImaInCache(g.getNatDisp(), props->lscale);    
    if (!ni) ni = addImaInCache(g.getNatDisp(), props->lscale);
    if (!ni) {
      *w = *h = 0;
      return;
    }
    else {
      *w = ni->getWidth();
      *h = ni->getHeight();
    }
  }
}

/* ==================================================== ======== ======= */

void UPix::getSize(UContext *props, u_dim *w, u_dim *h) const {
  UIma::getSize(props, w, h);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UIma::paint(UWinGraph &g, UContext *props, const URegion &r) const {
  if (natimas.empty()) {
    if (show_unknown_ima) {                //empecher recursion infinie
      if (this != g.getDisp()->getConf().unknown_image) 
	g.getDisp()->getConf().unknown_image->paint(g, props, r);
    }
    //else nop;
  }
  else {
    // fautdrait peut etre un add ici en part pour DrawWallPaperLine !!
    UNatIma *ni = findImaInCache(g.getNatDisp(),props->lscale);
    //etre coherent avec getSize: ne rien faire sinon
    //cerr << "UIma::paint " << this << " natdisp " << g.getNatDisp()<< endl;
    if (ni) g.drawIma(ni, r.x, r.y);      
  }
}

/* ==================================================== ======== ======= */

void UPix::paint(UWinGraph &g, UContext *props, const URegion &r) const {
#ifdef WITH_GL
  UIma::paint(g, props, r);
#else

  unsigned int did = g.getDisp()->getID();

  if (natimas.empty()) {
    if (show_unknown_ima) {                //empecher recursion infinie
      if (this != g.getDisp()->getConf().unknown_image)
	g.getDisp()->getConf().unknown_image->paint(g, props, r);
    }
    //else nop;
  }

  // on n'utilise le pixmap tel quel QUE SI il a la meme taille 
  // et le bon natdisp
  else if (did < natpixs.size()
	   && natpixs[did] != null
	   && natpixs[did]->lscale == props->lscale) {
    g.drawIma(natpixs[did], r.x, r.y);
  }

  else {
    UNatIma* ni = findImaInCache(g.getNatDisp(), props->lscale);

    //etre coherent avec getSize: ne rien faire sinon
    if (ni) {
      // agrandir liste
      for (u_int k = natpixs.size(); k <= did; k++) 
	natpixs.push_back(null);

      if (natpixs[did] != null)	delete natpixs[did]; 

      // creates the new natpix
      natpixs[did] = new UNatPix(g.getNatDisp(), ni);

      if (natpixs[did]) {  	// cas normal: draw the natpix
	natpixs[did]->lscale = props->lscale;
	g.drawIma(natpixs[did], r.x, r.y);
      }

      //pbm eventuel de creation du natpix ==> draw ima
      else g.drawIma(ni, r.x, r.y);
    }
    //else cerr << "pixmap not found" << endl;
  }
#endif
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UNatIma* UIma::findImaInCache(UNatDisp *nd, int lscale) const { 

  int fsize = lscale + UFont::MEDIUM_LSIZE;
  // limite inferieure
  if (fsize < UFont::MIN_LSIZE) {
    lscale = UFont::MIN_LSIZE - UFont::MEDIUM_LSIZE;
  }

  //NB: lscale=0 => nominal size / first natima always at nominal size
  //printf("search at scale %d\n", lscale);

  // search in the cache if this image was already created with the same size
  // AND THE SAME DISPLAY !!

  for (list<UNatIma*>::iterator p = natimas.begin(); p != natimas.end(); p++) {
    if ((*p)->lscale == lscale && (*p)->natdisp == nd) 
      return *p;
  }

  return null; //not found
}

/* ==================================================== ======== ======= */

UNatIma* UIma::addImaInCache(UNatDisp *nd, int lscale) const {
  if (natimas.empty()) return null;

  // on detruit systematiquement toutes les copies du natdisp concerne
  // sauf l'original qui est toujours en premier
  // !!! on pourrait garder plusieurs copies a tailles diverses !!!

  list<UNatIma*>::iterator p = natimas.begin();   // hum, A REVOIR
  p++;                                            // skip 1st = the original
  while (p != natimas.end()) {
    if ((*p)->natdisp != nd) p++;
    else {
      list<UNatIma*>::iterator p2 = p; p2++;
      delete *p; 
      natimas.erase(p);
      p = p2;
    }
  }

  //derives xscale and yscale from lscale automatically from font sizes  
  float xyscale = UFontFamily::getXYScale(lscale);

  /* NB image depth must be <= to the NatDisp depth */

  UNatIma* ni = (*natimas.begin())->createScaledImage(nd, xyscale, xyscale);
  if (ni) {
    ni->lscale = lscale;
    natimas.push_back(ni);
  }

  return ni; 
}
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UIma::realize(class UDisp& disp, bool force_reload) const {
  // ne pas essayer de recharger une seconde fois quelque soit
  // le resultat de cet essai (pour ne pas boucler sur le load)
  
  if (force_reload || (stat == UFilestat::NotOpened && natimas.empty())) {

    if (read_from_data) {
      if (!data) 
	stat = UFilestat::InvalidData;
      else {
	UNatIma* ni = null;
	stat = UNatIma::readImage(disp.getNatDisp(), (const char*)data, 
				  UNatIma::XPM_DATA, ni);
	if (ni) natimas.push_back(ni);
      }
    }

    else if (read_from_file) {
      char *fpath = disp.getAppli().makeImaPath(name);
      UNatIma* ni = null;
      stat = UNatIma::readImage(disp.getNatDisp(), fpath, null, ni);
      if (ni) natimas.push_back(ni);
      free(fpath);
    }
  }

  return stat;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

// a wrapping for using some of the great FVWM pixmaps in the Ubit toolkit
// - these icons were originally designed by Ludvig A. Norin
// - other icons are in file uufvwmpix.cc
// - please refer to this file for details

const char *fvwm_folder_xpm[] = {
"16 16 6 1",
"       c None s None",
".      c #808080",
"X      c #c0c0c0",
"o      c #ffff00",
"O      c black",
"#      c white",
"                ",
"  .....         ",
" .XoXoX.        ",
".XoXoXoX......  ",
".############.O ",
".#oXoXoXoXoXo.O ",
".#XoXoXoXoXoX.O ",
".#oXoXoXoXoXo.O ",
".#XoXoXoXoXoX.O ",
".#oXoXoXoXoXo.O ",
".#XoXoXoXoXoX.O ",
".#oXoXoXoXoXo.O ",
"..............O ",
" OOOOOOOOOOOOOO ",
"                ",
"                "};

const char *fvwm_ofolder_xpm[] = {
" 16 16 6 1",
"  c None s None",
". c black",
"# c #808080",
"g c white",
"h c #c0c0c0",
"j c yellow",
"                ",
"   #####        ",
"  #ggggg#       ",
" #ghjhjhg###### ",
" #gjhjhjhggggg#.",
" #ghjhjhjhjhjh#.",
"#############j#.",
"#gggggggggg#.h#.",
"#ghjhjhjhjhj.##.",
" #ghjhjhjhjh#.#.",
" #gjhjhjhjhjh.#.",
"  #gjhjhjhjhj#..",
"  ############..",
"   .............",
"                ",
"                "};

const char *fvwm_doc_xpm[] = {
"16 16 4 1",
"       c None s None",
".      c black",
"X      c white",
"o      c #808080",
"                ",
"   .......      ",
"   .XXXXX..     ",
"   .XoooX.X.    ",
"   .XXXXX....   ",
"   .XooooXoo.o  ",
"   .XXXXXXXX.o  ",
"   .XooooooX.o  ",
"   .XXXXXXXX.o  ",
"   .XooooooX.o  ",
"   .XXXXXXXX.o  ",
"   .XooooooX.o  ",
"   .XXXXXXXX.o  ",
"   ..........o  ",
"    oooooooooo  ",
"                "};

const char *fvwm_diskette_xpm[] = {
"16 16 7 1 0 0",
"       c #000080",
".      s iconColor1    m black c black",
"X      s none  m none  c none",
"o      s iconColor6    m white c yellow",
"O      s iconGray1     m white c #e0e0e0",
"+      s iconColor2    m white c white",
"@      c #C0C0C0",
"  ...........  X",
"  ooooooooooo  .",
"  OOOOOOOOOOO  .",
"  +++++++++++  .",
"  OOOOOOOOOOO  .",
"  +++++++++++  .",
"  OOOOOOOOOOO  .",
"  +++++++++++  .",
"               .",
"    @@@@@@@    .",
"    @  @@@@    .",
"    @  @@@@    .",
"    @  @@@@    .",
"    @  @@@@    .",
"X   @@@@@@@    .",
"XX.............."};

/*
const char *fvwm_cross_xpm[] = {
"16 16 4 1",
"       c None s None",
".      c red",
"X      c #808080",
"o      c black",
"                ",
"                ",
"           .    ",
"   ..X    ...   ",
"    ..X  .....  ",
"     .......oo  ",
"      .....o    ",
"      ....o     ",
"     ......     ",
"     ..o ...    ",
"    ..o   ...   ",
"    .o     ..X  ",
"   .o       .o  ",
"   o         o  ",
"                ",
"                "};
*/
const char *fvwm_cross_xpm[] = {
" 16 16 23 1",
". c None s None",
"# c #ef4a52",
"a c #ef5252",
"b c #bd0000",
"c c #ff0000",
"d c #d6525a",
"e c #733939",
"f c #ce0000",
"g c #ce7373",
"h c #c60000",
"i c #732121",
"j c #e70000",
"k c #c68c8c",
"l c #730808",
"m c #f72129",
"n c #8c0000",
"o c #e73939",
"p c #9c0000",
"q c #947373",
"r c #ad0000",
"s c #845a5a",
"t c #7b4242",
"u c #733131",
"................",
"................",
"................",
"....##.....a#...",
"....bcd...dcb...",
"....efcg.gche...",
".....ijckcji....",
"......lcccl.....",
".......ccc......",
"......mcncm.....",
".....ocpqpco....",
"....acrs.srca...",
"....fbt...tbf...",
"....uu.....uu...",
"................",
"................"};

const char *fvwm_check_xpm[] = {
" 16 16 32 1",
". c None s None",
"# c #63ef73",
"a c #6bff7b",
"b c #08ff31",
"c c #009c08",
"d c #08b529",
"e c #31424a",
"f c #10d631",
"g c #183931",
"h c #c6bdc6",
"i c #10f731",
"j c #18ff39",
"k c #18f739",
"l c #083921",
"m c #bdadbd",
"n c #004210",
"o c #21ff42",
"p c #29ff42",
"q c #a5a5ad",
"r c #8c9494",
"s c #005208",
"t c #39ef52",
"u c #006321",
"v c #737b7b",
"w c #006b10",
"x c #002100",
"y c #5a636b",
"z c #008c18",
"A c #5a7b6b",
"B c #424a52",
"C c #00a521",
"D c #294242",
"................",
"................",
"................",
"............#a..",
"............bc..",
"...........bde..",
"..........bfgh..",
"...ij....jklm...",
"...nop..ponq....",
"...rsottour.....",
"....vwojoxm.....",
".....yzozA......",
"......BCB.......",
".......D........",
"................",
"................"};

const char *fvwm_select_xpm[] = {
" 16 16 3 1",
". c #000000",
"# c None s None",
"a c #ffffff",
"################",
"####.###########",
"####..##########",
"####.a.#########",
"####.aa.########",
"####.aaa.#######",
"####.aaaa.######",
"####.aaaaa.#####",
"####.aaaaaa.####",
"####.aaaa....###",
"####.a..a.######",
"####..#.aa.#####",
"####.###.a.#####",
"########.aa.####",
"#########.a.####",
"################"};

const char *fvwm_zoom_xpm[] = {
"16 16 3 1",
"  c None s None",
". c Black",
"X c White",
"                ",
"                ",
"    ....        ",
"   .    .       ",
"  .  X   .      ",
"  . X    .      ",
"  .      .      ",
"  .      .      ",
"   .    ..      ",
"    .......     ",
"         ...    ",
"          ...   ",
"           ...  ",
"            ..  ",
"                ",
"                "};

const char *fvwm_turn_xpm[] = {
"16 16 3 1",
"       c None s None",
".      c None s None",
"X      c navy",
"                ",
"     ..XXX.     ",
"    .XXXXXXX    ",
"   .XX.    .X   ",
"   XX.          ",
"  XXX       X   ",
"XXXXXXX    XXX  ",
" XXXXX    XXXXX ",
"  XXX    XXXXXXX",
"   X       XXX  ",
"          .XX   ",
"   X.    .XX.   ",
"    XXXXXXX.    ",
"     .XXX..     ",
"                ",
"                "};


static const char *right_xpm[] = {
"25 27 147 2",
" 	c None",
".  	c None",
//"+ 	c #000101",
"+ 	c None",
//"@ 	c #000202",
"@ 	c None",
"# 	c #010304",
"$ 	c #010506",
"% 	c #010607",
"& 	c #010405",
//"* 	c #000203",
"* 	c None",
"= 	c #020B0D",
"- 	c #031115",
"; 	c #041418",
"> 	c #020C0F",
", 	c #010709",
"' 	c #02090B",
") 	c #04161B",
"! 	c #194D5B",
"~ 	c #082E37",
"{ 	c #082B34",
"] 	c #06232A",
"^ 	c #04181D",
"/ 	c #030F12",
"( 	c #01080A",
"_ 	c #030E11",
": 	c #062229",
"< 	c #7CA9B5",
"[ 	c #BAD9E0",
"} 	c #609DAD",
"| 	c #0E4B5A",
"1 	c #0A3742",
"2 	c #072830",
"3 	c #051B21",
"4 	c #020A0C",
"5 	c #07252D",
"6 	c #81AFBA",
"7 	c #75ABB9",
"8 	c #CBE3EA",
"9 	c #E2F2F7",
"0 	c #64A4B4",
"a 	c #196173",
"b 	c #0B3B47",
"c 	c #051E24",
"d 	c #031317",
"e 	c #020C0E",
"f 	c #061F26",
"g 	c #3D7B8B",
"h 	c #BEDCE4",
"i 	c #E5F4F8",
"j 	c #7DB7C6",
"k 	c #B0D5DF",
"l 	c #E4F5F9",
"m 	c #A1CCD7",
"n 	c #1C6679",
"o 	c #0B3E4B",
"p 	c #061F25",
"q 	c #010708",
"r 	c #07262E",
"s 	c #0B3C49",
"t 	c #468C9E",
"u 	c #C1E0E7",
"v 	c #E7F6FA",
"w 	c #7DB9C8",
"x 	c #B2D7E1",
"y 	c #BFDEE6",
"z 	c #448A9B",
"A 	c #082C35",
"B 	c #051D23",
"C 	c #09303A",
"D 	c #0C4250",
"E 	c #468EA0",
"F 	c #C2E1E9",
"G 	c #B3D8E2",
"H 	c #7AB7C7",
"I 	c #E1F2F7",
"J 	c #BBDDE5",
"K 	c #5E99A9",
"L 	c #0B3946",
"M 	c #0A3743",
"N 	c #0E4D5D",
"O 	c #20778E",
"P 	c #A6D2DE",
"Q 	c #C8E5EC",
"R 	c #28859C",
"S 	c #4F9CAF",
"T 	c #C2DFE6",
"U 	c #E1F1F6",
"V 	c #082A33",
"W 	c #031216",
"X 	c #031013",
"Y 	c #082E38",
"Z 	c #0C4452",
"` 	c #115D70",
" .	c #1A788F",
"..	c #93C8D6",
"+.	c #23849D",
"@.	c #136A80",
"#.	c #29798E",
"$.	c #E4F2F7",
"%.	c #0B404D",
"&.	c #115A6C",
"*.	c #66AABC",
"=.	c #E4F3F8",
"-.	c #CBE5EC",
";.	c #58A3B6",
">.	c #A3CEDA",
",.	c #DDF0F6",
"'.	c #9FC4CE",
").	c #07272F",
"!.	c #031114",
"~.	c #04191E",
"{.	c #1B6578",
"].	c #A0CBD6",
"^.	c #E2F3F7",
"/.	c #62AABC",
"(.	c #CBE4EB",
"_.	c #E0F4F8",
":.	c #609EAE",
"<.	c #165666",
"[.	c #2B7385",
"}.	c #A2CCD7",
"|.	c #B2D7E0",
"1.	c #7DB8C7",
"2.	c #C2E0E7",
"3.	c #62A1B1",
"4.	c #0E4A59",
"5.	c #09323D",
"6.	c #06242B",
"7.	c #07242C",
"8.	c #7FAEBA",
"9.	c #96C6D2",
"0.	c #B3D7E1",
"a.	c #A2CDD8",
"b.	c #458C9E",
"c.	c #09323C",
"d.	c #04171C",
"e.	c #87B2BD",
"f.	c #C9E2E9",
"g.	c #A1CBD6",
"h.	c #062027",
"i.	c #04161A",
"j.	c #020D10",
"k.	c #75A6B2",
"l.	c #5D97A6",
"m.	c #0F4A59",
"n.	c #0A3844",
"o.	c #072329",
"p.	c #062128",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . + + . . . . . . . . . . . . . . . . . ",
". . . . @ # $ % & * + . . . . . . . . . . . . . . ",
". . . + $ = - ; - > , # @ . . . . . . . . . . . . ",
". . . * ' ) ! ~ { ] ^ / ( & @ . . . . . . . . . . ",
". . . # _ : < [ } | 1 2 3 - 4 $ * + . . . . . . . ",
". . . & / 5 6 7 8 9 0 a b { c d = % * + . . . . . ",
". . . * e f g h i j k l m n o ~ p ; e $ @ . . . . ",
". . . @ q d r s t u v w x i y z s A B / % @ . . . ",
". . . . * ( - f C D E F v G H I J K L B e # . . . ",
". . . . + * % > ) 5 M N O P Q R S T U V W $ + . . ",
". . . . . + # ( X B Y Z `  ...+.@.#.$.~ ; % + . . ",
". . . . + & ' W c ~ %.&.*.=.-.;.>.,.'.).!.& + . . ",
". . . + & e ~.V s {.].^.-./.(._.:.<.A ~.4 * . . . ",
". . . @ ' ~.C [.}.i |.1.i 2.3.4.5.6.) e & + . . . ",
". . . # _ 7.8.9 9.0.v a.b.D c.6.d._ , # + . . . . ",
". . . & X 2 e.f.9 g.n o Y h.i.j., # @ . . . . . . ",
". . . # j.h.k.l.m.n.V B d e q # + . . . . . . . . ",
". . . @ ( d o.7.p.~.!.4 $ * + . . . . . . . . . . ",
". . . + # , > _ e , & @ + . . . . . . . . . . . . ",
". . . . + @ # # * + . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . "};


static const char *up_xpm[] = {
"25 24 130 2",
"  	c None",
". 	c None",
//"+ 	c #000101",
"+ 	c None",
//"@ 	c #000202",
"@ 	c None",
//"# 	c #000203",
"# 	c None",
"$ 	c #010607",
"% 	c #02090B",
"& 	c #020B0D",
"* 	c #020A0C",
"= 	c #010708",
"- 	c #010304",
"; 	c #031216",
"> 	c #051A20",
", 	c #051E24",
"' 	c #051B21",
") 	c #041418",
"! 	c #010405",
"~ 	c #04161A",
"{ 	c #07272F",
"] 	c #246170",
"^ 	c #276676",
"/ 	c #246171",
"( 	c #082D36",
"_ 	c #04181D",
": 	c #010506",
"< 	c #031115",
"[ 	c #07262E",
"} 	c #3E7E8E",
"| 	c #D8EEF3",
"1 	c #82B5C2",
"2 	c #BADAE2",
"3 	c #5894A3",
"4 	c #072932",
"5 	c #031317",
"6 	c #051C22",
"7 	c #0A3945",
"8 	c #6CACBC",
"9 	c #13687E",
"0 	c #519AAD",
"a 	c #BADBE3",
"b 	c #0B3C49",
"c 	c #061F26",
"d 	c #020C0E",
"e 	c #082B34",
"f 	c #1C6476",
"g 	c #E2F2F7",
"h 	c #4196AC",
"i 	c #3593AB",
"j 	c #3F95AB",
"k 	c #E2F3F7",
"l 	c #458B9C",
"m 	c #082F39",
"n 	c #041519",
"o 	c #0B3C48",
"p 	c #A0CBD6",
"q 	c #B2D6E0",
"r 	c #E5F5F8",
"s 	c #8DC5D4",
"t 	c #7CB9C9",
"u 	c #C1DFE7",
"v 	c #0C414F",
"w 	c #062027",
"x 	c #1B6578",
"y 	c #E4F3F8",
"z 	c #7FBCCB",
"A 	c #C5E3EA",
"B 	c #146D84",
"C 	c #A7D3DE",
"D 	c #B3D8E2",
"E 	c #E5F4F8",
"F 	c #458D9E",
"G 	c #09303A",
"H 	c #A1CCD6",
"I 	c #B2D7E1",
"J 	c #4997AB",
"K 	c #10586A",
"L 	c #1F758B",
"M 	c #C1E0E7",
"N 	c #031013",
"O 	c #072830",
"P 	c #1B6375",
"Q 	c #7EBAC9",
"R 	c #C2E0E8",
"S 	c #0E4C5C",
"T 	c #0B3F4C",
"U 	c #0D4A5A",
"V 	c #C0E0E8",
"W 	c #7BB8C8",
"X 	c #E7F6FA",
"Y 	c #09313B",
"Z 	c #9FC8D3",
"` 	c #B2D5DE",
" .	c #E7F5F8",
"..	c #478FA2",
"+.	c #0A3742",
"@.	c #072931",
"#.	c #0A3540",
"$.	c #458C9E",
"%.	c #7AB5C4",
"&.	c #BFDDE4",
"*.	c #0A3641",
"=.	c #010709",
"-.	c #10424F",
";.	c #DCF0F6",
">.	c #79AFBD",
",.	c #DFF2F6",
"'.	c #0C414E",
").	c #07242C",
"!.	c #062229",
"~.	c #0B3D4A",
"{.	c #BDDCE4",
"].	c #77AEBC",
"^.	c #DDF0F6",
"/.	c #357384",
"(.	c #01080A",
"_.	c #3E7C8B",
":.	c #B6D8E0",
"<.	c #5994A3",
"[.	c #082A33",
"}.	c #020C0F",
"|.	c #3C7B8C",
"1.	c #B6D7DF",
"2.	c #407D8D",
"3.	c #062128",
"4.	c #07272E",
"5.	c #04161B",
"6.	c #082C35",
"7.	c #020D10",
"8.	c #031114",
"9.	c #030E11",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . + @ @ @ + + . . . . . . . . . ",
". . . . . . . . + # $ % & * = - + . . . . . . . . ",
". . . . . . . + - % ; > , ' ) & ! + . . . . . . . ",
". . . . . . . # % ~ { ] ^ / ( _ * # . . . . . . . ",
". . . . . . + : < [ } | 1 2 3 4 5 $ + . . . . . . ",
". . . . . . # * 6 7 2 8 9 0 a b c d - . . . . . . ",
". . . . . + : < e f g h i j k l m n $ + . . . . . ",
". . . . . @ * 6 o p q r s r t u v w d - . . . . . ",
". . . . + : < e x y z A B C D E F G n $ + . . . . ",
". . . . @ % 6 o H I E J K L E t M v w d # . . . . ",
". . . . ! N O P y Q R S T U V W X l ( ; : + . . . ",
". . . + $ ) Y Z `  ...+.@.#.$.E %.&.*._ =.@ . . . ",
". . . + $ n -.;.>.,.'.)._ !.~.{.].^./._ (.@ . . . ",
". . . + ! N )._.:.<.[.~ }.) { |.1.2.O ; $ + . . . ",
". . . . # (.) 3.@.4.5.* $ % ) 3.6.!.~ * - . . . . ",
". . . . + # =.7.8.9.% ! @ - =.7.8.9.(.- + . . . . ",
". . . . . + @ - ! ! # + . + @ - ! - @ + . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . "};

static const char *down_xpm[] = {
"    25    24      129            2",
".. c None",
//".# c #000101",
".# c None",
//".a c #000202",
".a c None",
//".b c #000203",
".b c None",
".c c #010607",
".d c #02090b",
".e c #020b0d",
".f c #020a0c",
".g c #010708",
".h c #010304",
".i c #031216",
".j c #051a20",
".k c #051e24",
".l c #051b21",
".m c #041418",
".n c #010405",
".o c #04161a",
".p c #07272f",
".q c #246170",
".r c #276676",
".s c #246171",
".t c #082d36",
".u c #04181d",
".v c #010506",
".w c #031115",
".x c #07262e",
".y c #3e7e8e",
".z c #d8eef3",
".A c #82b5c2",
".B c #badae2",
".C c #5894a3",
".D c #072932",
".E c #031317",
".F c #051c22",
".G c #0a3945",
".H c #6cacbc",
".I c #13687e",
".J c #519aad",
".K c #badbe3",
".L c #0b3c49",
".M c #061f26",
".N c #020c0e",
".O c #082b34",
".P c #1c6476",
".Q c #e2f2f7",
".R c #4196ac",
".S c #3593ab",
".T c #3f95ab",
".U c #e2f3f7",
".V c #458b9c",
".W c #082f39",
".X c #041519",
".Y c #0b3c48",
".Z c #a0cbd6",
".0 c #b2d6e0",
".1 c #e5f5f8",
".2 c #8dc5d4",
".3 c #7cb9c9",
".4 c #c1dfe7",
".5 c #0c414f",
".6 c #062027",
".7 c #1b6578",
".8 c #e4f3f8",
".9 c #7fbccb",
"#. c #c5e3ea",
"## c #146d84",
"#a c #a7d3de",
"#b c #b3d8e2",
"#c c #e5f4f8",
"#d c #458d9e",
"#e c #09303a",
"#f c #a1ccd6",
"#g c #b2d7e1",
"#h c #4997ab",
"#i c #10586a",
"#j c #1f758b",
"#k c #c1e0e7",
"#l c #031013",
"#m c #072830",
"#n c #1b6375",
"#o c #7ebac9",
"#p c #c2e0e8",
"#q c #0e4c5c",
"#r c #0b3f4c",
"#s c #0d4a5a",
"#t c #c0e0e8",
"#u c #7bb8c8",
"#v c #e7f6fa",
"#w c #09313b",
"#x c #9fc8d3",
"#y c #b2d5de",
"#z c #e7f5f8",
"#A c #478fa2",
"#B c #0a3742",
"#C c #072931",
"#D c #0a3540",
"#E c #458c9e",
"#F c #7ab5c4",
"#G c #bfdde4",
"#H c #0a3641",
"#I c #010709",
"#J c #10424f",
"#K c #dcf0f6",
"#L c #79afbd",
"#M c #dff2f6",
"#N c #0c414e",
"#O c #07242c",
"#P c #062229",
"#Q c #0b3d4a",
"#R c #bddce4",
"#S c #77aebc",
"#T c #ddf0f6",
"#U c #357384",
"#V c #01080a",
"#W c #3e7c8b",
"#X c #b6d8e0",
"#Y c #5994a3",
"#Z c #082a33",
"#0 c #020c0f",
"#1 c #3c7b8c",
"#2 c #b6d7df",
"#3 c #407d8d",
"#4 c #062128",
"#5 c #07272e",
"#6 c #04161b",
"#7 c #082c35",
"#8 c #020d10",
"#9 c #031114",
"a. c #030e11",
"..................................................",
"..................................................",
"..................................................",
"..................................................",
"...........#.a.h.n.h.a.#...#.b.n.n.h.a.#..........",
".........#.h#Va.#9#8#I.h.a.n.da.#9#8#I.b.#........",
".........h.f.o#P#7#4.m.d.c.f#6#5#C#4.m#V.b........",
".......#.c.i#m#3#2#1.p.m#0.o#Z#Y#X#W#O#l.n.#......",
".......a#V.u#U#T#S#R#Q#P.u#O#N#M#L#K#J.X.c.#......",
".......a#I.u#H#G#F#c#E#D#C#B#A#z#y#x#w.m.c.#......",
".......#.v.i.t.V#v#u#t#s#r#q#p#o.8#n#m#l.n........",
".........b.N.6.5#k.3#c#j#i#h#c#g#f.Y.F.d.a........",
".........#.c.X#e#d#c#b#a###..9.8.7.O.w.v.#........",
"...........h.N.6.5.4.3.1.2.1.0.Z.Y.F.f.a..........",
"...........#.c.X.W.V.U.T.S.R.Q.P.O.w.v.#..........",
".............h.N.M.L.K.J.I.H.B.G.F.f.b............",
".............#.c.E.D.C.B.A.z.y.x.w.v.#............",
"...............b.f.u.t.s.r.q.p.o.d.b..............",
"...............#.n.e.m.l.k.j.i.d.h.#..............",
".................#.h.g.f.e.d.c.b.#................",
"...................#.#.a.a.a.#....................",
"..................................................",
"..................................................",
".................................................."
};

static const char * left_xpm[] = {
"25 27 152 2",
"  	c None",
". 	c None",
//"+ 	c #000101",
"+ 	c None",
//"@ 	c #000203",
"@ 	c None",
"# 	c #010506",
"$ 	c #010708",
//"% 	c #000202",
"% 	c None",
"& 	c #010304",
"* 	c #020C0F",
"= 	c #031216",
"- 	c #04161B",
"; 	c #04161A",
"> 	c #030E11",
", 	c #010405",
"' 	c #01080A",
") 	c #04171C",
"! 	c #06232A",
"~ 	c #082E37",
"{ 	c #09333E",
"] 	c #38717F",
"^ 	c #051D23",
"/ 	c #02090B",
"( 	c #031114",
"_ 	c #051A1F",
": 	c #07262E",
"< 	c #0A3540",
"[ 	c #0E4A59",
"} 	c #478C9E",
"| 	c #C1DEE6",
"1 	c #E2F1F6",
"2 	c #082B34",
"3 	c #010607",
"4 	c #020B0D",
"5 	c #082A33",
"6 	c #0A3945",
"7 	c #0F5060",
"8 	c #63A3B4",
"9 	c #E1F2F7",
"0 	c #CBE4EA",
"a 	c #7BB1BF",
"b 	c #E5F3F7",
"c 	c #082E38",
"d 	c #041418",
"e 	c #020A0C",
"f 	c #051E24",
"g 	c #0B3C48",
"h 	c #1B6577",
"i 	c #A0CBD6",
"j 	c #E4F5F9",
"k 	c #CBE5EB",
"l 	c #62A8BA",
"m 	c #CBE4EB",
"n 	c #E3F4F9",
"o 	c #629AA8",
"p 	c #07272F",
"q 	c #031013",
"r 	c #072931",
"s 	c #428698",
"t 	c #A1CBD6",
"u 	c #E6F5FA",
"v 	c #B3D8E1",
"w 	c #7EB9C9",
"x 	c #E5F4F8",
"y 	c #C2E0E7",
"z 	c #62A1B1",
"A 	c #0E4858",
"B 	c #082C35",
"C 	c #04181D",
"D 	c #09333C",
"E 	c #418191",
"F 	c #B9DAE2",
"G 	c #DFF2F6",
"H 	c #77B4C4",
"I 	c #95C7D4",
"J 	c #C2E1E9",
"K 	c #478FA1",
"L 	c #0C4452",
"M 	c #09323D",
"N 	c #041519",
"O 	c #020D10",
"P 	c #062229",
"Q 	c #7AAAB6",
"R 	c #BDDCE3",
"S 	c #69AABB",
"T 	c #1A758D",
"U 	c #A9D4DF",
"V 	c #A8D4DF",
"W 	c #32899F",
"X 	c #0E5061",
"Y 	c #0A3A46",
"Z 	c #04191E",
"` 	c #010709",
" .	c #030F12",
"..	c #07252D",
"+.	c #81AEBA",
"@.	c #6EA8B6",
"#.	c #12677C",
"$.	c #17768E",
"%.	c #91C7D5",
"&.	c #28869E",
"*.	c #116175",
"=.	c #0D4958",
"-.	c #09323C",
";.	c #061F25",
">.	c #031115",
",.	c #3F7B8A",
"'.	c #D9F0F5",
").	c #C1DFE7",
"!.	c #58A2B5",
"~.	c #AFD6E0",
"{.	c #A4D0DB",
"].	c #1E6E83",
"^.	c #09313B",
"/.	c #062128",
"(.	c #031317",
"_.	c #0C414E",
":.	c #609CAC",
"<.	c #DEF2F6",
"[.	c #7EBACA",
"}.	c #B3D8E2",
"|.	c #A1CCD7",
"1.	c #2E7A8D",
"2.	c #062027",
"3.	c #09303A",
"4.	c #0C414F",
"5.	c #478E9F",
"6.	c #97C8D5",
"7.	c #97C7D4",
"8.	c #C2E0E9",
"9.	c #468A9C",
"0.	c #0A3742",
"a.	c #020C0E",
"b.	c #2E798C",
"c.	c #A2CDD8",
"d.	c #B2D7E0",
"e.	c #7DB8C7",
"f.	c #E7F5F8",
"g.	c #C4DEE5",
"h.	c #082D36",
"i.	c #0B3C49",
"j.	c #1A6376",
"k.	c #83B9C6",
"l.	c #E2F2F7",
"m.	c #CBE3E9",
"n.	c #E8F4F7",
"o.	c #051C22",
"p.	c #072932",
"q.	c #0A3844",
"r.	c #0F4D5C",
"s.	c #619CAA",
"t.	c #BDD8DF",
"u.	c #072830",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . + + + . . . . . ",
". . . . . . . . . . . . . . + @ # $ $ # % . . . . ",
". . . . . . . . . . . . + & $ * = - ; > $ % . . . ",
". . . . . . . . . . % , ' > ) ! ~ { ] ^ * & . . . ",
". . . . . . . + @ # / ( _ : < [ } | 1 2 = # + . . ",
". . . . . + @ 3 4 = ^ 5 6 7 8 9 0 a b c d 3 + . . ",
". . . . % , e = f 2 g h i j k l m n o p q , + . . ",
". . . + , * _ r 6 s t u v w x y z A B ) / @ . . . ",
". . . @ / C D E F G H I x J K L M ! N e & + . . . ",
". . . & O P Q R S T U V W X Y p Z > ` & + . . . . ",
". . . &  ...+.@.#.$.%.&.*.=.-.;.>./ , % . . . . . ",
". . . & * ;.,.'.).!.~.x {.].L ^./.N 4 # % . . . . ",
". . . % ` (.: _.:.<.x [.}.x |.1._.c ^ > # + . . . ",
". . . . @ ' (.2.3.4.5.J x 6.7.x 8.9.0.^ a.@ . . . ",
". . . . + @ $ * ; /.3._.b.c.u d.e.f.g.B = # + . . ",
". . . . . . % & $ a.d ;.h.i.j.k.l.m.n.3.d 3 + . . ",
". . . . . . . . + @ 3 4 = o.p.q.r.s.t.p ( # + . . ",
". . . . . . . . . . + @ # / q Z ! u.u.) e @ . . . ",
". . . . . . . . . . . . . % , ` * q > e , + . . . ",
". . . . . . . . . . . . . . . % @ , , @ + . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . ",
". . . . . . . . . . . . . . . . . . . . . . . . . "};

const char *fvwm_plus_xpm[] = {
" 13 13 2 1",
". c #ffffff",
"# c #000000",
".............",
".............",
"..#########..",
"..#.......#..",
"..#...#...#..",
"..#...#...#..",
"..#.#####.#..",
"..#...#...#..",
"..#...#...#..",
"..#.......#..",
"..#########..",
".............",
"............."};

const char *fvwm_minus_xpm[] = {
" 13 13 2 1",
". c #ffffff",
"# c #000000",
".............",
".............",
"..#########..",
"..#.......#..",
"..#.......#..",
"..#.......#..",
"..#.#####.#..",
"..#.......#..",
"..#.......#..",
"..#.......#..",
"..#########..",
".............",
"............."};

const char *fvwm_ellipsis_xpm[] = {
" 13 13 2 1",
". c #ffffff",
"# c #000000",
".............",
".............",
".............",
".............",
".............",
".............",
"...#.#.#.#...",
".............",
".............",
".............",
".............",
".............",
"............."};

const char *fvwm_rball_xpm[] = {
"10 10 4 1",
"       c None s None",
".      c red",
"X      c white",
"o      c gray50",
"          ",
"          ",
"   ...    ",
"  .XX..   ",
"  .XX..o  ",
"  .....o  ",
"   ...oo  ",
"    ooo   ",
"          ",
"          "};



// *** = predefined Pixmaps with transparent (ie. shaped) background

UPix UPix::folder(fvwm_folder_xpm, UMode::UCONST);
UPix UPix::ofolder(fvwm_ofolder_xpm, UMode::UCONST);
UPix UPix::doc(fvwm_doc_xpm, UMode::UCONST);
UPix UPix::diskette(fvwm_diskette_xpm, UMode::UCONST);
UPix UPix::plus(fvwm_plus_xpm, UMode::UCONST);
UPix UPix::minus(fvwm_minus_xpm, UMode::UCONST);
UPix UPix::ellipsis(fvwm_ellipsis_xpm, UMode::UCONST);

UPix UPix::rball(fvwm_rball_xpm, UMode::UCONST);
UPix UPix::cross(fvwm_cross_xpm, UMode::UCONST);
UPix UPix::check(fvwm_check_xpm, UMode::UCONST);
UPix UPix::select(fvwm_select_xpm, UMode::UCONST);
UPix UPix::zoom(fvwm_zoom_xpm, UMode::UCONST);
UPix UPix::turn(fvwm_turn_xpm, UMode::UCONST);

UPix UPix::left(left_xpm, UMode::UCONST);
UPix UPix::right(right_xpm, UMode::UCONST);
UPix UPix::up(up_xpm, UMode::UCONST);
UPix UPix::down(down_xpm, UMode::UCONST);

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:02] ======= */
