// -*- tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
// vi: set et ts=4 sw=2 sts=2:
/****************************************************************************/
/*                                                                          */
/* File:      quadrature.c                                                  */
/*                                                                          */
/* Purpose:   quadrature formulas                                           */
/*                                                                          */
/* Author:    Christian Wieners                                             */
/*            Institut fuer Computeranwendungen III                         */
/*            Universitaet Stuttgart                                        */
/*            Pfaffenwaldring 27                                            */
/*            70569 Stuttgart                                               */
/*            email: ug@ica3.uni-stuttgart.de                               */
/*                                                                          */
/* History:   Sep 25 95 begin                                               */
/*                                                                          */
/* Remarks:                                                                 */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/* include files                                                            */
/*            system include files                                          */
/*            application include files                                     */
/*                                                                          */
/****************************************************************************/

#include "config.h"

#include <stdlib.h>

#include "architecture.h"
#include "evm.h"
#include "shapes.h"
#include "quadrature.h"
#include "ugdevices.h"
#include "general.h"


USING_UG_NAMESPACES

/****************************************************************************/
/*                                                                          */
/* defines in the following order                                           */
/*                                                                          */
/*        compile time constants defining static data size (i.e. arrays)    */
/*        other constants                                                   */
/*        macros                                                            */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/* data structures used in this source file (exported data structures are   */
/*        in the corresponding include file!)                               */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/* definition of exported global variables                                  */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/* definition of variables global to this source file only (static!)        */
/*                                                                          */
/****************************************************************************/


/* RCS string */
static char RCS_ID("$Header$",UG_RCS_STRING);

/****************************************************************************/
/*                                                                          */
/* forward declarations of functions used before they are defined           */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/* Quadrature formulas                                                      */
/*                                                                          */
/****************************************************************************/

/*--------------- 1D -------------------------------------------------------*/
/*
 * Gaussian quadrature abscissas and weights (from Mathematica, 64 digit precision).
 *
 * The errors are for a function f:

        -Derivative(2)(f)/24
        -0.000231*Derivative(4)(f)
        -4.960317e-7*Derivative(6)(f)
        -5.623942e-10*Derivative(8)(f)
        -3.944965e-13*Derivative(10)(f)
        -1.880943e-16*Derivative(12)(f)
        -6.492408e-20*Derivative(14)(f)
        -1.697361e-23*Derivative(16)(f)
        -3.477566e-27*Derivative(18)(f)
        -5.734017e-31*Derivative(20)(f)

 * Quadrature1Dn means, that the Quadrature rule is exact for polynomials up
 * to degree n.
 */
#ifdef macintosh
#pragma mark Quadrature formulas for one dimension
#endif

/* Degree 1 ----------------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature1D1_l[] = {{0.5, 0.0}};
static const DOUBLE Quadrature1D1_w[] = {1.0};
static QUADRATURE Quadrature1D1 = {1, 1, Quadrature1D1_l, Quadrature1D1_w};

/* Degree 3 ----------------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature1D3_l[] = {
  { 0.2113248654051871177454256097490212721761991243649365619906988367580112, 0.0 },
  { 0.7886751345948128822545743902509787278238008756350634380093011632419888, 0.0 }
};
static const DOUBLE Quadrature1D3_w[] = {
  0.5,
  0.5
};
static QUADRATURE Quadrature1D3 = {2, 3, Quadrature1D3_l, Quadrature1D3_w};

/* Degree 5 ----------------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature1D5_l[] = {
  {0.112701665379258311482073460021760038916707829470840917341242623388652 , 0.0 },
  {0.5 , 0.0 },
  {0.887298334620741688517926539978239961083292170529159082658757376611348 , 0.0 }
};
static const DOUBLE Quadrature1D5_w[] = {
  0.277777777777777777777777777777777777777777777777777777777777777777778,
  0.444444444444444444444444444444444444444444444444444444444444444444444,
  0.277777777777777777777777777777777777777777777777777777777777777777778
};
static QUADRATURE Quadrature1D5 = {3, 5, Quadrature1D5_l, Quadrature1D5_w};

/* Degree 7 ----------------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature1D7_l[] = {
  {0.069431844202973712388026755553595247452137310185141181192139039546735,  0.0},
  {0.3300094782075718675986671204483776563997120651145428237035230115894900, 0.0},
  {0.6699905217924281324013328795516223436002879348854571762964769884105100, 0.0},
  {0.930568155797026287611973244446404752547862689814858818807860960453265,  0.0}
};
static const DOUBLE Quadrature1D7_w[] = {
  0.173927422568726928686531974610999703617674347916946770246264659759376,
  0.326072577431273071313468025389000296382325652083053229753735340240624,
  0.326072577431273071313468025389000296382325652083053229753735340240624,
  0.173927422568726928686531974610999703617674347916946770246264659759376
};
static QUADRATURE Quadrature1D7 = {4, 7, Quadrature1D7_l, Quadrature1D7_w};

/* Degree 9 ----------------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature1D9_l[] = {
  {0.046910077030668003601186560850303517437174044618734568563118856728115,  0.0},
  {0.2307653449471584544818427896498955975163566965472200218988841864702644, 0.0},
  {0.5, 0.0},
  {0.7692346550528415455181572103501044024836433034527799781011158135297356, 0.0},
  {0.953089922969331996398813439149696482562825955381265431436881143271885,  0.0}
};
static const DOUBLE Quadrature1D9_w[] = {
  0.118463442528094543757132020359958681321630001106207007791413944110859,
  0.239314335249683234020645757417819096456147776671570769986363833666919,
  0.284444444444444444444444444444444444444444444444444444444444444444444,
  0.239314335249683234020645757417819096456147776671570769986363833666919,
  0.118463442528094543757132020359958681321630001106207007791413944110859
};
static QUADRATURE Quadrature1D9 = {5, 9, Quadrature1D9_l, Quadrature1D9_w};

/* Degree 11 ----------------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature1D11_l[] = {
  {0.033765242898423986093849222753002695432617131143855087563725191736693,  0.0},
  {0.1693953067668677431693002024900473264967757178024149645927366470739083, 0.0},
  {0.3806904069584015456847491391596440322906946849299893249093024177128625, 0.0},
  {0.6193095930415984543152508608403559677093053150700106750906975822871375, 0.0},
  {0.8306046932331322568306997975099526735032242821975850354072633529260917, 0.0},
  {0.966234757101576013906150777246997304567382868856144912436274808263307,  0.0}
};
static const DOUBLE Quadrature1D11_w[] = {
  0.085662246189585172520148071086366446763411250742021991199317719899473,
  0.180380786524069303784916756918858055830760946373372741144869620118570,
  0.233956967286345523694935171994775497405827802884605267655812659981957,
  0.233956967286345523694935171994775497405827802884605267655812659981957,
  0.180380786524069303784916756918858055830760946373372741144869620118570,
  0.085662246189585172520148071086366446763411250742021991199317719899473
};
static QUADRATURE Quadrature1D11 = {6, 11, Quadrature1D11_l, Quadrature1D11_w};

/* Degree 13 ----------------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature1D13_l[] = {
  {0.025446043828620737736905157976074368799614531164691108225615448043468, 0.0},
  {0.129234407200302780068067613359605796462926176429304869940022324016285,  0.0},
  {0.2970774243113014165466967939615192683263089929503149368064783741026681, 0.0},
  {0.5, 0.0},
  {0.7029225756886985834533032060384807316736910070496850631935216258973319, 0.0},
  {0.870765592799697219931932386640394203537073823570695130059977675983715,  0.0},
  {0.974553956171379262263094842023925631200385468835308891774384551956532,  0.0}
};
static const DOUBLE Quadrature1D13_w[] = {
  0.064742483084434846635305716339541009164293701129973331988604319362328,
  0.139852695744638333950733885711889791243462532613299382268507016346809,
  0.190915025252559472475184887744487566939182541766931367375541725515353,
  0.208979591836734693877551020408163265306122448979591836734693877551020,
  0.190915025252559472475184887744487566939182541766931367375541725515353,
  0.139852695744638333950733885711889791243462532613299382268507016346809,
  0.064742483084434846635305716339541009164293701129973331988604319362328
};
static QUADRATURE Quadrature1D13 = {7, 13, Quadrature1D13_l, Quadrature1D13_w};

/* Degree 15 ----------------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature1D15_l[] = {
  {0.019855071751231884158219565715263504785882382849273980864180111313788,  0.0},
  {0.101666761293186630204223031762084781581414134192017583964914852480391,  0.0},
  {0.2372337950418355070911304754053768254790178784398035711245714503637726, 0.0},
  {0.4082826787521750975302619288199080096666210935435131088414057631503978, 0.0},
  {0.5917173212478249024697380711800919903333789064564868911585942368496022, 0.0},
  {0.7627662049581644929088695245946231745209821215601964288754285496362274, 0.0},
  {0.898333238706813369795776968237915218418585865807982416035085147519609,  0.0},
  {0.980144928248768115841780434284736495214117617150726019135819888686212,  0.0}
};
static const DOUBLE Quadrature1D15_w[] = {
  0.050614268145188129576265677154981095057697045525842478529501849032370,
  0.111190517226687235272177997213120442215065435025624782362954644646808,
  0.156853322938943643668981100993300656630164499501367468845131972537478,
  0.181341891689180991482575224638597806097073019947165270262411533783343,
  0.181341891689180991482575224638597806097073019947165270262411533783343,
  0.156853322938943643668981100993300656630164499501367468845131972537478,
  0.111190517226687235272177997213120442215065435025624782362954644646808,
  0.050614268145188129576265677154981095057697045525842478529501849032370
};
static QUADRATURE Quadrature1D15 = {8, 15, Quadrature1D15_l, Quadrature1D15_w};

/* Degree 17 ----------------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature1D17_l[] = {
  {0.015919880246186955082211898548163564975297599754037335224988344075460, 0.0},
  {0.081984446336682102850285105965132561727946640937662001947814010180272,  0.0},
  {0.1933142836497048013456489803292629076071396975297176535635935288593663, 0.0},
  {0.3378732882980955354807309926783316957140218696315134555864762615789067, 0.0},
  {0.5, 0.0},
  {0.6621267117019044645192690073216683042859781303684865444135237384210933, 0.0},
  {0.8066857163502951986543510196707370923928603024702823464364064711406337, 0.0},
  {0.918015553663317897149714894034867438272053359062337998052185989819728,  0.0},
  {0.984080119753813044917788101451836435024702400245962664775011655924540,  0.0}
};
static const DOUBLE Quadrature1D17_w[] = {
  0.040637194180787205985946079055261825337830860391205375355553838440343,
  0.090324080347428702029236015621456404757168910866020242249167953235679,
  0.130305348201467731159371434709316424885920102218649975969998501059805,
  0.156173538520001420034315203292221832799377430630952322777005582799572,
  0.165119677500629881582262534643487024439405391786344167296548248929201,
  0.156173538520001420034315203292221832799377430630952322777005582799572,
  0.130305348201467731159371434709316424885920102218649975969998501059805,
  0.090324080347428702029236015621456404757168910866020242249167953235679,
  0.040637194180787205985946079055261825337830860391205375355553838440343
};
static QUADRATURE Quadrature1D17 = {9, 17, Quadrature1D17_l, Quadrature1D17_w};

/* Degree 19 ----------------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature1D19_l[] = {
  {0.013046735741414139961017993957773973285865026653808940384393966651702,  0.0},
  {0.067468316655507744633951655788253475736228492517334773739020134077313,  0.0},
  {0.160295215850487796882836317442563212115352644082595266167591405523721,  0.0},
  {0.2833023029353764046003670284171079188999640811718767517486492434281165, 0.0},
  {0.4255628305091843945575869994351400076912175702896541521460053732420482, 0.0},
  {0.5744371694908156054424130005648599923087824297103458478539946267579518, 0.0},
  {0.7166976970646235953996329715828920811000359188281232482513507565718835, 0.0},
  {0.839704784149512203117163682557436787884647355917404733832408594476279,  0.0},
  {0.932531683344492255366048344211746524263771507482665226260979865922687,  0.0},
  {0.986953264258585860038982006042226026714134973346191059615606033348298,  0.0}
};
static const DOUBLE Quadrature1D19_w[] = {
  0.033335672154344068796784404946665896428932417160079072564347440806706,
  0.074725674575290296572888169828848666201278319834713683917738634376619,
  0.109543181257991021997767467114081596229385935261338544940478271817600,
  0.134633359654998177545613460784734676429879969230441897900281638121077,
  0.147762112357376435086946497325669164710523358513426800677154014877998,
  0.147762112357376435086946497325669164710523358513426800677154014877998,
  0.134633359654998177545613460784734676429879969230441897900281638121077,
  0.109543181257991021997767467114081596229385935261338544940478271817600,
  0.074725674575290296572888169828848666201278319834713683917738634376619,
  0.033335672154344068796784404946665896428932417160079072564347440806706
};
static QUADRATURE Quadrature1D19 = {10, 19, Quadrature1D19_l, Quadrature1D19_w};



/*--------------- 2D -------------------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature2D31_l[] = {{0.333333333333333, 0.333333333333333}};
static const DOUBLE Quadrature2D31_w[] = {1.0};
static QUADRATURE Quadrature2D31 = {1, 1, Quadrature2D31_l, Quadrature2D31_w};

static const DOUBLE_VECTOR_3D Quadrature2D32_l[] = {{0.66666666666666666, 0.16666666666666666},
                                                    {0.16666666666666666, 0.66666666666666666},
                                                    {0.16666666666666666, 0.16666666666666666}};
static const DOUBLE Quadrature2D32_w[] = {0.333333333333333,    0.333333333333333,      0.333333333333333};
static QUADRATURE Quadrature2D32 = {3, 2, Quadrature2D32_l, Quadrature2D32_w};

static const DOUBLE_VECTOR_3D Quadrature2D33_l[] = {{0.333333333333333, 0.3333333333333333},
                                                    {0.6,0.2},
                                                    {0.2,0.6},
                                                    {0.2,0.2}};
static const DOUBLE Quadrature2D33_w[] = {-0.5625, 0.520833333333333, 0.520833333333333, 0.520833333333333};
static QUADRATURE Quadrature2D33 = {4, 3, Quadrature2D33_l, Quadrature2D33_w};

static const DOUBLE_VECTOR_3D Quadrature2D34_l[] = {{0.816847572980459, 0.091576213509771},
                                                    {0.091576213509771, 0.816847572980459},
                                                    {0.091576213509771, 0.091576213509771},
                                                    {0.108103018168070, 0.445948490915965},
                                                    {0.445948490915965, 0.108103018168070},
                                                    {0.445948490915965, 0.445948490915965}};
static const DOUBLE Quadrature2D34_w[] = {0.109951743655322, 0.109951743655322, 0.109951743655322,
                                          0.223381589678011, 0.223381589678011, 0.223381589678011};
static QUADRATURE Quadrature2D34 = {6, 4, Quadrature2D34_l, Quadrature2D34_w};

static const DOUBLE_VECTOR_3D Quadrature2D35_l[] = {{0.333333333333333, 0.333333333333333},
                                                    {0.059715871789770, 0.470142064105115},
                                                    {0.470142064105115, 0.059715871789770},
                                                    {0.470142064105115, 0.470142064105115},
                                                    {0.797426985368435,   0.101286507323456},
                                                    {0.101286507323456,   0.797426985368435},
                                                    {0.101286507323456,   0.101286507323456}};
static const DOUBLE Quadrature2D35_w[] = {0.225,
                                          0.132394152788506,
                                          0.132394152788506,
                                          0.132394152788506,
                                          0.125939180544827,
                                          0.125939180544827,
                                          0.125939180544827};
static QUADRATURE Quadrature2D35 = {7, 5, Quadrature2D35_l, Quadrature2D35_w};





static const DOUBLE_VECTOR_3D Quadrature2D36_l[] = {{0.333333333333333, 0.333333333333333},
                                                    {0.059715871789770, 0.470142064105115},
                                                    {0.470142064105115, 0.059715871789770},
                                                    {0.470142064105115, 0.470142064105115},
                                                    {0.797426985368435,   0.101286507323456},
                                                    {0.101286507323456,   0.797426985368435},
                                                    {0.101286507323456,   0.101286507323456}};
static const DOUBLE Quadrature2D36_w[] = {      0.116786275726379,
                                                0.116786275726379,
                                                0.116786275726379,
                                                0.050844906370207,
                                                0.050844906370207,
                                                0.050844906370207,
                                                0.082851075618374,
                                                0.082851075618374,
                                                0.082851075618374};
static QUADRATURE Quadrature2D36 = {7, 5, Quadrature2D36_l, Quadrature2D36_w};

/******** For new GaussQuadratureRule function *****************************/
/*
 * Many of these  quadrature formulas come from the
 * "Encyclopaedia of Cubature Formulas" at
 * http://www.cs.kuleuven.ac.be/~nines/research/ecf/ecf.html
 * maintained by Ronald Cools.
 *
 */
#ifdef macintosh
#pragma mark Quadrature formulas for 2D: Triangles
#endif

/* ---- Polynomial degree 1 --------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P1_N1_GP[] = {
  {0.333333333333333333333333333333333,
   0.333333333333333333333333333333333}
};

static const DOUBLE Quadrature2D_Triangle_P1_N1_W[] = {1.0};

static QUADRATURE Quadrature2D_Triangle_P1_N1 = {
  /* Number of Gauss Points */
  1,
  /* Polynomial degree */
  1,
  /* Abscissae */
  Quadrature2D_Triangle_P1_N1_GP,
  /* Weight */
  Quadrature2D_Triangle_P1_N1_W
};


/* ---- Polynomial degree 2 --------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P2_N3a_GP[] = {
  {0.5, 0.5},
  {0.5, 0.0},
  {0.0, 0.5}
};

static const DOUBLE Quadrature2D_Triangle_P2_N3a_W[] =
{0.33333333333333333333333333333333333,
 0.33333333333333333333333333333333333,
 0.33333333333333333333333333333333333};

static QUADRATURE Quadrature2D_Triangle_P2_N3a = {
  /* Number of Gauss Points */
  3,
  /* Polynomial degree */
  2,
  /* Abscissae */
  Quadrature2D_Triangle_P2_N3a_GP,
  /* Weight */
  Quadrature2D_Triangle_P2_N3a_W
};

static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P2_N3b_GP[] =
{{0.16666666666666666666666666666667, 0.16666666666666666666666666666667},
 {0.16666666666666666666666666666667, 0.66666666666666666666666666666667},
 {0.66666666666666666666666666666667, 0.16666666666666666666666666666667}};

static const DOUBLE Quadrature2D_Triangle_P2_N3b_W[] =
{0.33333333333333333333333333333333333,
 0.33333333333333333333333333333333333,
 0.33333333333333333333333333333333333};


static QUADRATURE Quadrature2D_Triangle_P2_N3b = {
  /* Number of Gauss Points */
  3,
  /* Polynomial degree */
  2,
  /* Abscissae */
  Quadrature2D_Triangle_P2_N3b_GP,
  /* Weight */
  Quadrature2D_Triangle_P2_N3b_W
};



/* ---- Polynomial degree 3 --------------------------------------------*/

/* 4 Gauss points, 1 negative weight */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P3_N4_GP[] =
{{0.333333333333333333333333333333333, 0.333333333333333333333333333333333},
 {0.2, 0.2, 0.6},
 {0.2, 0.6, 0.2},
 {0.6, 0.2, 0.2}};

static const DOUBLE Quadrature2D_Triangle_P3_N4_W[] =
{-0.5625,
 0.52083333333333333333333333333333,
 0.52083333333333333333333333333333,
 0.52083333333333333333333333333333};

static QUADRATURE Quadrature2D_Triangle_P3_N4 = {
  /* Number of Gauss Points */
  4,
  /* Polynomial degree */
  3,
  /* Abscissae */
  Quadrature2D_Triangle_P3_N4_GP,
  /* Weight */
  Quadrature2D_Triangle_P3_N4_W
};


/* 6 Gauss points, 3 points on boundary, 3 negative weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P3_N6a_GP[] =
{{0.0, 0.0},
 {1.0, 0.0},
 {0.0, 1.0},
 {0.14724747683480533311373176020907, 0.14724747683480533311373176020907},
 {0.14724747683480533311373176020907, 0.70550504633038933377253647958187},
 {0.70550504633038933377253647958187, 0.14724747683480533311373176020907}};

static const DOUBLE Quadrature2D_Triangle_P3_N6a_W[] =
{-0.029854797457965333388233726614400,
 -0.029854797457965333388233726614400,
 -0.029854797457965333388233726614400,
 0.36318813079129866672156705994773,
 0.36318813079129866672156705994773,
 0.36318813079129866672156705994773};

static QUADRATURE Quadrature2D_Triangle_P3_N6a = {
  /* Number of Gauss Points */
  6,
  /* Polynomial degree */
  3,
  /* Abscissae */
  Quadrature2D_Triangle_P3_N6a_GP,
  /* Weight */
  Quadrature2D_Triangle_P3_N6a_W
};


/* 6 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P3_N6b_GP[] =
{{0.10903900907287721232483466756991, 0.23193336855303057249678456117469},
 {0.10903900907287721232483466756991, 0.65902762237409221517838077125540},
 {0.23193336855303057249678456117469, 0.10903900907287721232483466756991},
 {0.23193336855303057249678456117469, 0.65902762237409221517838077125540},
 {0.65902762237409221517838077125540, 0.10903900907287721232483466756991},
 {0.65902762237409221517838077125540, 0.23193336855303057249678456117469}};

static const DOUBLE Quadrature2D_Triangle_P3_N6b_W[] =
{0.16666666666666666666666666666667,
 0.16666666666666666666666666666667,
 0.16666666666666666666666666666667,
 0.16666666666666666666666666666667,
 0.16666666666666666666666666666667,
 0.16666666666666666666666666666667};

static QUADRATURE Quadrature2D_Triangle_P3_N6b = {
  /* Number of Gauss Points */
  6,
  /* Polynomial degree */
  3,
  /* Abscissae */
  Quadrature2D_Triangle_P3_N6b_GP,
  /* Weight */
  Quadrature2D_Triangle_P3_N6b_W
};

/* 3 inner Gauss points, 3 boundary Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P3_N6c_GP[] =
{{0.5, 0.5},
 {0.5, 0.0},
 {0.0, 0.5},
 {0.16666666666666666666666666666667, 0.16666666666666666666666666666667},
 {0.16666666666666666666666666666667, 0.66666666666666666666666666666667},
 {0.66666666666666666666666666666667, 0.16666666666666666666666666666667}};

static const DOUBLE Quadrature2D_Triangle_P3_N6c_W[] =
{0.033333333333333333333333333333333,
 0.033333333333333333333333333333333,
 0.033333333333333333333333333333333,
 0.3,
 0.3,
 0.3};

static QUADRATURE Quadrature2D_Triangle_P3_N6c = {
  /* Number of Gauss Points */
  6,
  /* Polynomial degree */
  3,
  /* Abscissae */
  Quadrature2D_Triangle_P3_N6c_GP,
  /* Weight */
  Quadrature2D_Triangle_P3_N6c_W
};


/* 7 Gauss Points, 6 on boundary, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P3_N7_GP[] =
{{0.0, 0.0},
 {1.0, 0.0},
 {0.0, 1.0},
 {0.5, 0.5},
 {0.5, 0.0},
 {0.0, 0.5},
 {0.333333333333333333333333333333333, 0.333333333333333333333333333333333}};

static const DOUBLE Quadrature2D_Triangle_P3_N7_W[] =
{0.05,
 0.05,
 0.05,
 0.13333333333333333333333333333333,
 0.13333333333333333333333333333333,
 0.13333333333333333333333333333333,
 0.45};

static QUADRATURE Quadrature2D_Triangle_P3_N7 = {
  /* Number of Gauss Points */
  7,
  /* Polynomial degree */
  3,
  /* Abscissae */
  Quadrature2D_Triangle_P3_N7_GP,
  /* Weight */
  Quadrature2D_Triangle_P3_N7_W
};


/* ---- Polynomial degree 4 --------------------------------------------*/

/* 6 inner points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P4_N6_GP[] =
{{0.091576213509770743459571463402202, 0.091576213509770743459571463402202},
 {0.091576213509770743459571463402202, 0.81684757298045851308085707319560},
 {0.81684757298045851308085707319560, 0.091576213509770743459571463402202},
 {0.44594849091596488631832925388305, 0.44594849091596488631832925388305},
 {0.44594849091596488631832925388305, 0.10810301816807022736334149223390},
 {0.10810301816807022736334149223390, 0.44594849091596488631832925388305}};

static const DOUBLE Quadrature2D_Triangle_P4_N6_W[] =
{0.10995174365532186763832632490021,
 0.10995174365532186763832632490021,
 0.10995174365532186763832632490021,
 0.22338158967801146569500700843312,
 0.22338158967801146569500700843312,
 0.22338158967801146569500700843312};

static QUADRATURE Quadrature2D_Triangle_P4_N6 = {
  /* Number of Gauss Points */
  6,
  /* Polynomial degree */
  4,
  /* Abscissae */
  Quadrature2D_Triangle_P4_N6_GP,
  /* Weight */
  Quadrature2D_Triangle_P4_N6_W
};

/* 7 inner points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P4_N7_GP[] =
{{0.33333333333333333333333333333333,  0.33333333333333333333333333333333},
 {0.025355134559131751049580723682707, 0.23793236647243354549156299492765},
 {0.025355134559131751049580723682707, 0.73671249896843470345885628138964},
 {0.23793236647243354549156299492765,  0.025355134559131751049580723682707},
 {0.23793236647243354549156299492765,  0.73671249896843470345885628138964},
 {0.73671249896843470345885628138964,  0.025355134559131751049580723682707},
 {0.73671249896843470345885628138964,  0.23793236647243354549156299492765}};

static const DOUBLE Quadrature2D_Triangle_P4_N7_W[] =
{0.375,
 0.10416666666666666666666666666667,
 0.10416666666666666666666666666667,
 0.10416666666666666666666666666667,
 0.10416666666666666666666666666667,
 0.10416666666666666666666666666667,
 0.10416666666666666666666666666667};

static QUADRATURE Quadrature2D_Triangle_P4_N7 = {
  /* Number of Gauss Points */
  7,
  /* Polynomial degree */
  4,
  /* Abscissae */
  Quadrature2D_Triangle_P4_N7_GP,
  /* Weight */
  Quadrature2D_Triangle_P4_N7_W
};

/* 9 Gauss points, 3 on boundary, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P4_N9_GP[] =
{{0.0, 0.0},
 {1.0, 0.0},
 {0.0, 1.0},
 {0.5, 0.5},
 {0.5, 0.0},
 {0.0, 0.5},
 {0.18858048469644503927115437402942, 0.18858048469644503927115437402942},
 {0.18858048469644503927115437402942, 0.62283903060710992145769125194117},
 {0.62283903060710992145769125194117, 0.18858048469644503927115437402942}};

static const DOUBLE Quadrature2D_Triangle_P4_N9_W[] =
{0.020540135345933363074668829812582,
 0.020540135345933363074668829812582,
 0.020540135345933363074668829812582,
 0.061975498868267142528034610334645,
 0.061975498868267142528034610334645,
 0.061975498868267142528034610334645,
 0.25081769911913282773062989318611,
 0.25081769911913282773062989318611,
 0.25081769911913282773062989318611};

static QUADRATURE Quadrature2D_Triangle_P4_N9 = {
  /* Number of Gauss Points */
  9,
  /* Polynomial degree */
  4,
  /* Abscissae */
  Quadrature2D_Triangle_P4_N9_GP,
  /* Weight */
  Quadrature2D_Triangle_P4_N9_W
};

/* ---- Polynomial degree 5 --------------------------------------------*/

/* 7 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P5_N7_GP[] =
{{0.333333333333333333333333333333333, 0.333333333333333333333333333333333},
 {0.10128650732345633880098736191512, 0.10128650732345633880098736191512},
 {0.10128650732345633880098736191512, 0.79742698535308732239802527616975},
 {0.79742698535308732239802527616975, 0.10128650732345633880098736191512},
 {0.47014206410511508977044120951345, 0.47014206410511508977044120951345},
 {0.47014206410511508977044120951345, 0.05971587178976982045911758097311},
 {0.05971587178976982045911758097311, 0.47014206410511508977044120951345}};

static const DOUBLE Quadrature2D_Triangle_P5_N7_W[] =
{0.225, 0.12593918054482715259568394550018,
 0.12593918054482715259568394550018,
 0.12593918054482715259568394550018,
 0.13239415278850618073764938783315,
 0.13239415278850618073764938783315,
 0.13239415278850618073764938783315};

static QUADRATURE Quadrature2D_Triangle_P5_N7 = {
  /* Number of Gauss Points */
  7,
  /* Polynomial degree */
  5,
  /* Abscissae */
  Quadrature2D_Triangle_P5_N7_GP,
  /* Weight */
  Quadrature2D_Triangle_P5_N7_W
};

/* 9 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P7_N9_GP[] =
{{0.43752524838338386732523898110054, 0.43752524838338386732523898110054},
 {0.43752524838338386732523898110054, 0.12494950323323226534952203779891},
 {0.12494950323323226534952203779891, 0.43752524838338386732523898110054},
 {0.037477420750087775658575783340592, 0.16540992738984142292404701391079},
 {0.037477420750087775658575783340592, 0.79711265186007080141737720274862},
 {0.16540992738984142292404701391079, 0.037477420750087775658575783340592},
 {0.16540992738984142292404701391079, 0.79711265186007080141737720274862},
 {0.79711265186007080141737720274862, 0.037477420750087775658575783340592},
 {0.79711265186007080141737720274862, 0.16540992738984142292404701391079}};

static const DOUBLE Quadrature2D_Triangle_P7_N9_W[] =
{0.20595050476088660301184485012830,
 0.20595050476088660301184485012830,
 0.20595050476088660301184485012830,
 0.063691414286223365160744241602518,
 0.063691414286223365160744241602518,
 0.063691414286223365160744241602518,
 0.063691414286223365160744241602518,
 0.063691414286223365160744241602518,
 0.063691414286223365160744241602518};

static QUADRATURE Quadrature2D_Triangle_P7_N9 = {
  /* Number of Gauss Points */
  9,
  /* Polynomial degree */
  5,
  /* Abscissae */
  Quadrature2D_Triangle_P7_N9_GP,
  /* Weight */
  Quadrature2D_Triangle_P7_N9_W
};

/* ---- Polynomial degree 6 --------------------------------------------*/

/* 12 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P6_N12a_GP[] =
{{0.063089014491502228340331602870819, 0.063089014491502228340331602870819},
 {0.063089014491502228340331602870819, 0.87382197101699554331933679425836},
 {0.87382197101699554331933679425836, 0.063089014491502228340331602870819},
 {0.24928674517091042129163855310702, 0.24928674517091042129163855310702},
 {0.24928674517091042129163855310702, 0.50142650965817915741672289378596},
 {0.50142650965817915741672289378596, 0.24928674517091042129163855310702},
 {0.053145049844816947353249671631398, 0.31035245103378440541660773395655},
 {0.053145049844816947353249671631398, 0.63650249912139864723014259441205},
 {0.31035245103378440541660773395655, 0.053145049844816947353249671631398},
 {0.31035245103378440541660773395655, 0.63650249912139864723014259441205},
 {0.63650249912139864723014259441205, 0.053145049844816947353249671631398},
 {0.63650249912139864723014259441205, 0.31035245103378440541660773395655}};

static const DOUBLE Quadrature2D_Triangle_P6_N12a_W[] =
{0.050844906370206816920936809106869,
 0.050844906370206816920936809106869,
 0.050844906370206816920936809106869,
 0.11678627572637936602528961138558,
 0.11678627572637936602528961138558,
 0.11678627572637936602528961138558,
 0.082851075618373575193553456420442,
 0.082851075618373575193553456420442,
 0.082851075618373575193553456420442,
 0.082851075618373575193553456420442,
 0.082851075618373575193553456420442,
 0.082851075618373575193553456420442};

static QUADRATURE Quadrature2D_Triangle_P6_N12a = {
  /* Number of Gauss Points */
  12,
  /* Polynomial degree */
  6,
  /* Abscissae */
  Quadrature2D_Triangle_P6_N12a_GP,
  /* Weight */
  Quadrature2D_Triangle_P6_N12a_W
};

/* 12 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P6_N12b_GP[] =
{{0.21942998254978296000012662922294, 0.21942998254978296000012662922294},
 {0.21942998254978296000012662922294, 0.56114003490043407999974674155413},
 {0.56114003490043407999974674155413, 0.21942998254978296000012662922294},
 {0.48013796411221504402894153229550, 0.48013796411221504402894153229550},
 {0.48013796411221504402894153229550, 0.03972407177556991194211693540900},
 {0.03972407177556991194211693540900, 0.48013796411221504402894153229550},
 {0.83900925971479105320836760570047, 0.14161901592396815841177387638623},
 {0.83900925971479105320836760570047, 0.01937172436124078837985851791330},
 {0.14161901592396815841177387638623, 0.83900925971479105320836760570047},
 {0.14161901592396815841177387638623, 0.01937172436124078837985851791330},
 {0.01937172436124078837985851791330, 0.83900925971479105320836760570047},
 {0.01937172436124078837985851791330, 0.14161901592396815841177387638623}};

static const DOUBLE Quadrature2D_Triangle_P6_N12b_W[] =
{0.17133312415298103012021658880582,
 0.17133312415298103012021658880582,
 0.17133312415298103012021658880582,
 0.080731089593030978309593700213708,
 0.080731089593030978309593700213708,
 0.080731089593030978309593700213708,
 0.040634559793660662451761522156903,
 0.040634559793660662451761522156903,
 0.040634559793660662451761522156903,
 0.040634559793660662451761522156903,
 0.040634559793660662451761522156903,
 0.040634559793660662451761522156903};

static QUADRATURE Quadrature2D_Triangle_P6_N12b = {
  /* Number of Gauss Points */
  12,
  /* Polynomial degree */
  6,
  /* Abscissae */
  Quadrature2D_Triangle_P6_N12b_GP,
  /* Weight */
  Quadrature2D_Triangle_P6_N12b_W
};

/* 13 Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P6_N13_GP[] =
{{0.865307354083457174835799758111522, 0.134692645916542825164200241888477},
 {0.865307354083457174835799758111522, 0.0},
 {0.134692645916542825164200241888477, 0.865307354083457174835799758111522},
 {0.134692645916542825164200241888477, 0.0},
 {0.0,                                 0.865307354083457174835799758111522},
 {0.0,                                 0.134692645916542825164200241888477},
 {0.333333333333333333333333333333333, 0.333333333333333333333333333333333},
 {0.47383081395365138455265792772597,  0.47383081395365138455265792772597},
 {0.47383081395365138455265792772597,  0.05233837209269723089468414454806},
 {0.05233837209269723089468414454806,  0.47383081395365138455265792772597},
 {0.17211766963081763562155759271492,  0.17211766963081763562155759271492},
 {0.17211766963081763562155759271492,  0.65576466073836472875688481457017},
 {0.65576466073836472875688481457017,  0.17211766963081763562155759271492}};

static const DOUBLE Quadrature2D_Triangle_P6_N13_W[] =
{0.027363023522182471676717151158859,
 0.027363023522182471676717151158859,
 0.027363023522182471676717151158859,
 0.027363023522182471676717151158859,
 0.027363023522182471676717151158859,
 0.027363023522182471676717151158859,
 0.15270896678835233561855889999549,
 0.098135868078891798318630971127531,
 0.098135868078891798318630971127531,
 0.098135868078891798318630971127531,
 0.12956842928062581312174842655625,
 0.12956842928062581312174842655625,
 0.12956842928062581312174842655625};

static QUADRATURE Quadrature2D_Triangle_P6_N13 = {
  /* Number of Gauss Points */
  13,
  /* Polynomial degree */
  6,
  /* Abscissae */
  Quadrature2D_Triangle_P6_N13_GP,
  /* Weight */
  Quadrature2D_Triangle_P6_N13_W
};


/* ---- Polynomial degree 7 --------------------------------------------*/

/* 12 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P7_N12_GP[] =
{{0.0623822650944021181736830009963499, 0.0675178670739160854425571310508685},
 {0.0675178670739160854425571310508685, 0.870099867831681796383759867952782},
 {0.870099867831681796383759867952782,  0.0623822650944021181736830009963499},
 {0.0552254566569266117374791902756449, 0.321502493851981822666307849199202},
 {0.321502493851981822666307849199202,  0.623272049491091565596212960525153},
 {0.623272049491091565596212960525153,  0.0552254566569266117374791902756449},
 {0.0343243029450971464696306424839376, 0.660949196186735657611980310197799},
 {0.660949196186735657611980310197799,  0.304726500868167195918389047318263},
 {0.304726500868167195918389047318263,  0.0343243029450971464696306424839376},
 {0.515842334353591779257463386826430,  0.277716166976391782569581871393723},
 {0.277716166976391782569581871393723,  0.20644149867001643817295474177985},
 {0.20644149867001643817295474177985,   0.515842334353591779257463386826430}};

static const DOUBLE Quadrature2D_Triangle_P7_N12_W[] =
{0.053034056314872502857508360921478,
 0.053034056314872502857508360921478,
 0.053034056314872502857508360921478,
 0.087762817428892110073539806278575,
 0.087762817428892110073539806278575,
 0.087762817428892110073539806278575,
 0.057550085569963171476890993800437,
 0.057550085569963171476890993800437,
 0.057550085569963171476890993800437,
 0.13498637401960554892539417233284,
 0.13498637401960554892539417233284,
 0.13498637401960554892539417233284};

static QUADRATURE Quadrature2D_Triangle_P7_N12 = {
  /* Number of Gauss Points */
  12,
  /* Polynomial degree */
  7,
  /* Abscissae */
  Quadrature2D_Triangle_P7_N12_GP,
  /* Weight */
  Quadrature2D_Triangle_P7_N12_W
};

/* 13 inner Gauss points, one negative weight */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P7_N13_GP[] =
{{0.333333333333333333333333333333333, 0.333333333333333333333333333333333},
 {0.26034596607903982692624246913924, 0.26034596607903982692624246913924},
 {0.26034596607903982692624246913924, 0.47930806784192034614751506172153},
 {0.47930806784192034614751506172153, 0.26034596607903982692624246913924},
 {0.065130102902215811538025906311975, 0.065130102902215811538025906311975},
 {0.065130102902215811538025906311975, 0.86973979419556837692394818737605},
 {0.86973979419556837692394818737605, 0.065130102902215811538025906311975},
 {0.048690315425316411793021558528413, 0.31286549600487386140664447676840},
 {0.048690315425316411793021558528413, 0.63844418856980972680033396470319},
 {0.31286549600487386140664447676840, 0.048690315425316411793021558528413},
 {0.31286549600487386140664447676840, 0.63844418856980972680033396470319},
 {0.63844418856980972680033396470319, 0.048690315425316411793021558528413},
 {0.63844418856980972680033396470319, 0.31286549600487386140664447676840}};

static const DOUBLE Quadrature2D_Triangle_P7_N13_W[] =
{-0.14957004446768175062971125547310,
 0.17561525743320781175351941115633,
 0.17561525743320781175351941115633,
 0.17561525743320781175351941115633,
 0.053347235608838491269987288899067,
 0.053347235608838491269987288899067,
 0.053347235608838491269987288899067,
 0.077113760890257140259865192551152,
 0.077113760890257140259865192551152,
 0.077113760890257140259865192551152,
 0.077113760890257140259865192551152,
 0.077113760890257140259865192551152,
 0.077113760890257140259865192551152};

static QUADRATURE Quadrature2D_Triangle_P7_N13 = {
  /* Number of Gauss Points */
  13,
  /* Polynomial degree */
  7,
  /* Abscissae */
  Quadrature2D_Triangle_P7_N13_GP,
  /* Weight */
  Quadrature2D_Triangle_P7_N13_W
};

static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P7_N15_GP[] =
{{0.064930513159164863078379776030397, 0.064930513159164863078379776030397},
 {0.064930513159164863078379776030397, 0.87013897368167027384324044793921},
 {0.87013897368167027384324044793921,  0.064930513159164863078379776030397},
 {0.19838447668150671917987659863333,  0.51703993906932294562270734401689},
 {0.19838447668150671917987659863333,  0.28457558424917033519741605734978},
 {0.51703993906932294562270734401689,  0.19838447668150671917987659863333},
 {0.51703993906932294562270734401689,  0.28457558424917033519741605734978},
 {0.28457558424917033519741605734978,  0.19838447668150671917987659863333},
 {0.28457558424917033519741605734978,  0.51703993906932294562270734401689},
 {0.64257734382269602053234940183007,  0.043863471792372471511798695971296},
 {0.64257734382269602053234940183007,  0.31355918438493150795585190219863},
 {0.043863471792372471511798695971296, 0.64257734382269602053234940183007},
 {0.043863471792372471511798695971296, 0.31355918438493150795585190219863},
 {0.31355918438493150795585190219863,  0.64257734382269602053234940183007},
 {0.31355918438493150795585190219863,  0.043863471792372471511798695971296}};

static const DOUBLE Quadrature2D_Triangle_P7_N15_W[] =
{0.053077801790232411671954974999695,
 0.053077801790232411671954974999695,
 0.053077801790232411671954974999695,
 0.070853083692133567318412583246404,
 0.070853083692133567318412583246404,
 0.070853083692133567318412583246404,
 0.070853083692133567318412583246404,
 0.070853083692133567318412583246404,
 0.070853083692133567318412583246404,
 0.069274682079416893512276595920415,
 0.069274682079416893512276595920415,
 0.069274682079416893512276595920415,
 0.069274682079416893512276595920415,
 0.069274682079416893512276595920415,
 0.069274682079416893512276595920415};

static QUADRATURE Quadrature2D_Triangle_P7_N15 = {
  /* Number of Gauss Points */
  15,
  /* Polynomial degree */
  7,
  /* Abscissae */
  Quadrature2D_Triangle_P7_N15_GP,
  /* Weights */
  Quadrature2D_Triangle_P7_N15_W
};


/* ---- Polynomial degree 8 --------------------------------------------*/

/* 16 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P8_N16_GP[] =
{{0.33333333333333333333333333333333,  0.33333333333333333333333333333333},
 {0.17056930775176020662229350149146,  0.17056930775176020662229350149146},
 {0.17056930775176020662229350149146,  0.65886138449647958675541299701707},
 {0.65886138449647958675541299701707,  0.17056930775176020662229350149146},
 {0.050547228317030975458423550596599, 0.050547228317030975458423550596599},
 {0.050547228317030975458423550596599, 0.89890554336593804908315289880680},
 {0.89890554336593804908315289880680,  0.050547228317030975458423550596599},
 {0.45929258829272315602881551449417,  0.45929258829272315602881551449417},
 {0.45929258829272315602881551449417,  0.08141482341455368794236897101166},
 {0.08141482341455368794236897101166,  0.45929258829272315602881551449417},
 {0.72849239295540428124100037917606,  0.26311282963463811342178578628464},
 {0.72849239295540428124100037917606,  0.00839477740995760533721383453930},
 {0.26311282963463811342178578628464,  0.72849239295540428124100037917606},
 {0.26311282963463811342178578628464,  0.00839477740995760533721383453930},
 {0.00839477740995760533721383453930,  0.72849239295540428124100037917606},
 {0.00839477740995760533721383453930,  0.26311282963463811342178578628464}};

static const DOUBLE Quadrature2D_Triangle_P8_N16_W[] =
{0.14431560767778716825109111048906,
 0.10321737053471825028179155029213,
 0.10321737053471825028179155029213,
 0.10321737053471825028179155029213,
 0.032458497623198080310925928341780,
 0.032458497623198080310925928341780,
 0.032458497623198080310925928341780,
 0.095091634267284624793896104388584,
 0.095091634267284624793896104388584,
 0.095091634267284624793896104388584,
 0.027230314174434994264844690073909,
 0.027230314174434994264844690073909,
 0.027230314174434994264844690073909,
 0.027230314174434994264844690073909,
 0.027230314174434994264844690073909,
 0.027230314174434994264844690073909};

static QUADRATURE Quadrature2D_Triangle_P8_N16 = {
  /* Number of Gauss Points */
  16,
  /* Polynomial degree */
  8,
  /* Abscissae */
  Quadrature2D_Triangle_P8_N16_GP,
  /* Weight */
  Quadrature2D_Triangle_P8_N16_W
};

/* ---- Polynomial degree 9 --------------------------------------------*/

/* 19 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P9_N19_GP[] =
{{0.333333333333333333333333333333333, 0.333333333333333333333333333333333},
 {0.48968251919873762778370692483619,  0.48968251919873762778370692483619},
 {0.48968251919873762778370692483619,  0.02063496160252474443258615032762},
 {0.02063496160252474443258615032762,  0.48968251919873762778370692483619},
 {0.43708959149293663726993036443535,  0.43708959149293663726993036443535},
 {0.43708959149293663726993036443535,  0.12582081701412672546013927112929},
 {0.12582081701412672546013927112929,  0.43708959149293663726993036443535},
 {0.18820353561903273024096128046733,  0.18820353561903273024096128046733},
 {0.18820353561903273024096128046733,  0.62359292876193453951807743906533},
 {0.62359292876193453951807743906533,  0.18820353561903273024096128046733},
 {0.044729513394452709865106589966276, 0.044729513394452709865106589966276},
 {0.044729513394452709865106589966276, 0.91054097321109458026978682006745},
 {0.91054097321109458026978682006745,  0.044729513394452709865106589966276},
 {0.74119859878449802069007987352342,  0.036838412054736283634817598783385},
 {0.74119859878449802069007987352342,  0.22196298916076569567510252769319},
 {0.036838412054736283634817598783385, 0.74119859878449802069007987352342},
 {0.036838412054736283634817598783385, 0.22196298916076569567510252769319},
 {0.22196298916076569567510252769319,  0.74119859878449802069007987352342},
 {0.22196298916076569567510252769319,  0.036838412054736283634817598783385}};

static const DOUBLE Quadrature2D_Triangle_P9_N19_W[] =
{0.097135796282798833819241982507289,
 0.031334700227139070536854831287209,
 0.031334700227139070536854831287209,
 0.031334700227139070536854831287209,
 0.077827541004774279316739356299404,
 0.077827541004774279316739356299404,
 0.077827541004774279316739356299404,
 0.079647738927210253032891774264045,
 0.079647738927210253032891774264045,
 0.079647738927210253032891774264045,
 0.025577675658698031261678798559000,
 0.025577675658698031261678798559000,
 0.025577675658698031261678798559000,
 0.043283539377289377289377289377289,
 0.043283539377289377289377289377289,
 0.043283539377289377289377289377289,
 0.043283539377289377289377289377289,
 0.043283539377289377289377289377289,
 0.043283539377289377289377289377289};

static QUADRATURE Quadrature2D_Triangle_P9_N19 = {
  /* Number of Gauss Points */
  19,
  /* Polynomial degree */
  9,
  /* Abscissae */
  Quadrature2D_Triangle_P9_N19_GP,
  /* Weight */
  Quadrature2D_Triangle_P9_N19_W
};

/* ---- Polynomial degree 10 --------------------------------------------*/

/* 25 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P10_N25a_GP[] =
{{0.333333333333333333333333333333333, 0.333333333333333333333333333333333},
 {0.42508621060209057296952951163804, 0.42508621060209057296952951163804},
 {0.42508621060209057296952951163804, 0.14982757879581885406094097672391},
 {0.14982757879581885406094097672391, 0.42508621060209057296952951163804},
 {0.023308867510000190714466386895980, 0.023308867510000190714466386895980},
 {0.023308867510000190714466386895980, 0.95338226497999961857106722620804},
 {0.95338226497999961857106722620804, 0.023308867510000190714466386895980},
 {0.62830740021349255642083766607883, 0.22376697357697300622568649026820},
 {0.62830740021349255642083766607883, 0.14792562620953443735347584365296},
 {0.22376697357697300622568649026820, 0.62830740021349255642083766607883},
 {0.22376697357697300622568649026820, 0.14792562620953443735347584365296},
 {0.14792562620953443735347584365296, 0.62830740021349255642083766607883},
 {0.14792562620953443735347584365296, 0.22376697357697300622568649026820},
 {0.61131382618139764891875500225390, 0.35874014186443146457815530072385},
 {0.61131382618139764891875500225390, 0.02994603195417088650308969702225},
 {0.35874014186443146457815530072385, 0.61131382618139764891875500225390},
 {0.35874014186443146457815530072385, 0.02994603195417088650308969702225},
 {0.02994603195417088650308969702225, 0.61131382618139764891875500225390},
 {0.02994603195417088650308969702225, 0.35874014186443146457815530072385},
 {0.82107206998562937337354441347218, 0.14329537042686714530585663061732},
 {0.82107206998562937337354441347218, 0.03563255958750348132059895591050},
 {0.14329537042686714530585663061732, 0.82107206998562937337354441347218},
 {0.14329537042686714530585663061732, 0.03563255958750348132059895591050},
 {0.03563255958750348132059895591050, 0.82107206998562937337354441347218},
 {0.03563255958750348132059895591050, 0.14329537042686714530585663061732}};

static const DOUBLE Quadrature2D_Triangle_P10_N25a_W[] =
{0.079894504741239707831247045213386,
 0.071123802232377334639291287398658,
 0.071123802232377334639291287398658,
 0.071123802232377334639291287398658,
 0.0082238186904641955186466203624719,
 0.0082238186904641955186466203624719,
 0.0082238186904641955186466203624719,
 0.045430592296170018007073629243933,
 0.045430592296170018007073629243933,
 0.045430592296170018007073629243933,
 0.045430592296170018007073629243933,
 0.045430592296170018007073629243933,
 0.045430592296170018007073629243933,
 0.037359856234305276826236499001975,
 0.037359856234305276826236499001975,
 0.037359856234305276826236499001975,
 0.037359856234305276826236499001975,
 0.037359856234305276826236499001975,
 0.037359856234305276826236499001975,
 0.030886656884563988782513077004629,
 0.030886656884563988782513077004629,
 0.030886656884563988782513077004629,
 0.030886656884563988782513077004629,
 0.030886656884563988782513077004629,
 0.030886656884563988782513077004629};

static QUADRATURE Quadrature2D_Triangle_P10_N25a = {
  /* Number of Gauss Points */
  25,
  /* Polynomial degree */
  10,
  /* Abscissae */
  Quadrature2D_Triangle_P10_N25a_GP,
  /* Weight */
  Quadrature2D_Triangle_P10_N25a_W
};

/* 25 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P10_N25b_GP[] =
{{0.333333333333333333333333333333333, 0.333333333333333333333333333333333},
 {0.14216110105656438509216210319096, 0.14216110105656438509216210319096},
 {0.14216110105656438509216210319096, 0.71567779788687122981567579361808},
 {0.71567779788687122981567579361808, 0.14216110105656438509216210319096},
 {0.032055373216943512930984589336490, 0.032055373216943512930984589336490},
 {0.032055373216943512930984589336490, 0.93588925356611297413803082132702},
 {0.93588925356611297413803082132702, 0.032055373216943512930984589336490},
 {0.53005411892734402827709567394569, 0.32181299528883542122509756098605},
 {0.53005411892734402827709567394569, 0.14813288578382055049780676506826},
 {0.32181299528883542122509756098605, 0.53005411892734402827709567394569},
 {0.32181299528883542122509756098605, 0.14813288578382055049780676506826},
 {0.14813288578382055049780676506826, 0.53005411892734402827709567394569},
 {0.14813288578382055049780676506826, 0.32181299528883542122509756098605},
 {0.60123332868345924545474289345869, 0.36914678182781098691142083711527},
 {0.60123332868345924545474289345869, 0.02961988948872976763383626942604},
 {0.36914678182781098691142083711527, 0.60123332868345924545474289345869},
 {0.36914678182781098691142083711527, 0.02961988948872976763383626942604},
 {0.02961988948872976763383626942604, 0.60123332868345924545474289345869},
 {0.02961988948872976763383626942604, 0.36914678182781098691142083711527},
 {0.80793060092287906507994990288174, 0.16370173373718249566961434154012},
 {0.80793060092287906507994990288174, 0.02836766533993843925043575557813},
 {0.16370173373718249566961434154012, 0.80793060092287906507994990288174},
 {0.16370173373718249566961434154012, 0.02836766533993843925043575557813},
 {0.02836766533993843925043575557813, 0.80793060092287906507994990288174},
 {0.02836766533993843925043575557813, 0.16370173373718249566961434154012}};

static const DOUBLE Quadrature2D_Triangle_P10_N25b_W[] =
{0.081743329146285966428118684984178,
 0.045957963604744728013787909632548,
 0.045957963604744728013787909632548,
 0.045957963604744728013787909632548,
 0.013352968813149566275572978399069,
 0.013352968813149566275572978399069,
 0.013352968813149566275572978399069,
 0.063904906396424045432898733762653,
 0.063904906396424045432898733762653,
 0.063904906396424045432898733762653,
 0.063904906396424045432898733762653,
 0.063904906396424045432898733762653,
 0.063904906396424045432898733762653,
 0.034184648162959428628691584041342,
 0.034184648162959428628691584041342,
 0.034184648162959428628691584041342,
 0.034184648162959428628691584041342,
 0.034184648162959428628691584041342,
 0.034184648162959428628691584041342,
 0.025297757707288384389042790682833,
 0.025297757707288384389042790682833,
 0.025297757707288384389042790682833,
 0.025297757707288384389042790682833,
 0.025297757707288384389042790682833,
 0.025297757707288384389042790682833};

static QUADRATURE Quadrature2D_Triangle_P10_N25b = {
  /* Number of Gauss Points */
  25,
  /* Polynomial degree */
  10,
  /* Abscissae */
  Quadrature2D_Triangle_P10_N25b_GP,
  /* Weight */
  Quadrature2D_Triangle_P10_N25b_W
};

/* ---- Polynomial degree 11 --------------------------------------------*/

/* 28 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P11_N28_GP[] =
{{0.858870281282636704039173938058347, 0.141129718717363295960826061941652},
 {0.858870281282636704039173938058347, 0.0},
 {0.141129718717363295960826061941652, 0.858870281282636704039173938058347},
 {0.141129718717363295960826061941652, 0.0},
 {0.0, 0.858870281282636704039173938058347},
 {0.0, 0.141129718717363295960826061941652},
 {0.333333333333333333333333333333333, 0.333333333333333333333333333333333},
 {0.025989140928287395260032485498841, 0.025989140928287395260032485498841},
 {0.025989140928287395260032485498841, 0.94802171814342520947993502900232},
 {0.94802171814342520947993502900232, 0.025989140928287395260032485498841},
 {0.094287502647922495630569776275405, 0.094287502647922495630569776275405},
 {0.094287502647922495630569776275405, 0.81142499470415500873886044744919},
 {0.81142499470415500873886044744919, 0.094287502647922495630569776275405},
 {0.49463677501721381374163260230644, 0.49463677501721381374163260230644},
 {0.49463677501721381374163260230644, 0.01072644996557237251673479538713},
 {0.01072644996557237251673479538713, 0.49463677501721381374163260230644},
 {0.20734338261451133345293402411297, 0.20734338261451133345293402411297},
 {0.20734338261451133345293402411297, 0.58531323477097733309413195177407},
 {0.58531323477097733309413195177407, 0.20734338261451133345293402411297},
 {0.43890780570049209506106538163613, 0.43890780570049209506106538163613},
 {0.43890780570049209506106538163613, 0.12218438859901580987786923672775},
 {0.12218438859901580987786923672775, 0.43890780570049209506106538163613},
 {0.67793765488259040154212614118875, 0.044841677589130443309052391468801},
 {0.67793765488259040154212614118875, 0.27722066752827915514882146734245},
 {0.044841677589130443309052391468801, 0.67793765488259040154212614118875},
 {0.044841677589130443309052391468801, 0.27722066752827915514882146734245},
 {0.27722066752827915514882146734245, 0.67793765488259040154212614118875},
 {0.27722066752827915514882146734245, 0.044841677589130443309052391468801}};

static const DOUBLE Quadrature2D_Triangle_P11_N28_W[] =
{0.0073623837833005542642588950473806,
 0.0073623837833005542642588950473806,
 0.0073623837833005542642588950473806,
 0.0073623837833005542642588950473806,
 0.0073623837833005542642588950473806,
 0.0073623837833005542642588950473806,
 0.087977301162232238798093169321456,
 0.0087443115537360230495164287998252,
 0.0087443115537360230495164287998252,
 0.0087443115537360230495164287998252,
 0.038081571993934937515024339435614,
 0.038081571993934937515024339435614,
 0.038081571993934937515024339435614,
 0.018855448056131292058476782591115,
 0.018855448056131292058476782591115,
 0.018855448056131292058476782591115,
 0.072159697544739526124029988586463,
 0.072159697544739526124029988586463,
 0.072159697544739526124029988586463,
 0.069329138705535899841765650903814,
 0.069329138705535899841765650903814,
 0.069329138705535899841765650903814,
 0.041056315429288566641652314907294,
 0.041056315429288566641652314907294,
 0.041056315429288566641652314907294,
 0.041056315429288566641652314907294,
 0.041056315429288566641652314907294,
 0.041056315429288566641652314907294};

static QUADRATURE Quadrature2D_Triangle_P11_N28 = {
  /* Number of Gauss Points */
  28,
  /* Polynomial degree */
  11,
  /* Abscissae */
  Quadrature2D_Triangle_P11_N28_GP,
  /* Weight */
  Quadrature2D_Triangle_P11_N28_W
};

/* ---- Polynomial degree 11 --------------------------------------------*/

/* 28 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature2D_Triangle_P12_N33_GP[] =
{{0.02356522045239,0.488217389773805},
 {0.488217389773805,0.02356522045239},
 {0.488217389773805,0.488217389773805},
 {0.43972439229446,0.43972439229446},
 {0.43972439229446,0.120551215411079},
 {0.120551215411079,0.43972439229446},
 {0.271210385012116,0.271210385012116},
 {0.271210385012116,0.457579229975768},
 {0.457579229975768,0.271210385012116},
 {0.127576145541586,0.127576145541586},
 {0.127576145541586,0.7448477089168279},
 {0.7448477089168279,0.127576145541586},
 {0.02131735045321,0.02131735045321},
 {0.02131735045321,0.9573652990935799},
 {0.9573652990935799,0.02131735045321},
 {0.115343494534698,0.275713269685514},
 {0.115343494534698,0.6089432357797879},
 {0.275713269685514,0.115343494534698},
 {0.275713269685514,0.6089432357797879},
 {0.6089432357797879,0.115343494534698},
 {0.6089432357797879,0.275713269685514},
 {0.022838332222257,0.28132558098994},
 {0.022838332222257,0.6958360867878031},
 {0.28132558098994,0.022838332222257},
 {0.28132558098994,0.6958360867878031},
 {0.6958360867878031,0.022838332222257},
 {0.6958360867878031,0.28132558098994},
 {0.02573405054833,0.116251915907597},
 {0.02573405054833,0.858014033544073},
 {0.116251915907597,0.02573405054833},
 {0.116251915907597,0.858014033544073},
 {0.858014033544073,0.02573405054833},
 {0.858014033544073,0.116251915907597}};

static const DOUBLE Quadrature2D_Triangle_P12_N33_W[] =
{0.025731066440455,
 0.025731066440455,
 0.025731066440455,
 0.043692544538038,
 0.043692544538038,
 0.043692544538038,
 0.062858224217885,
 0.062858224217885,
 0.062858224217885,
 0.034796112930709,
 0.034796112930709,
 0.034796112930709,
 0.006166261051559,
 0.006166261051559,
 0.006166261051559,
 0.040371557766381,
 0.040371557766381,
 0.040371557766381,
 0.040371557766381,
 0.040371557766381,
 0.040371557766381,
 0.022356773202303,
 0.022356773202303,
 0.022356773202303,
 0.022356773202303,
 0.022356773202303,
 0.022356773202303,
 0.017316231108659,
 0.017316231108659,
 0.017316231108659,
 0.017316231108659,
 0.017316231108659,
 0.017316231108659};

static QUADRATURE Quadrature2D_Triangle_P12_N33 = {
  /* Number of Gauss Points */
  33,
  /* Polynomial degree */
  12,
  /* Abscissae */
  Quadrature2D_Triangle_P12_N33_GP,
  /* Weight */
  Quadrature2D_Triangle_P12_N33_W
};

/******** Quadrilaterals ***************************************************/
#ifdef macintosh
#pragma mark Quadrature formulas for 2D: Quadrilaterals
#endif

static const DOUBLE_VECTOR_3D Quadrature2D40_l[] = {{0.0, 0.0},{1.0, 0.0},{1.0, 1.0},{0.0, 1.0}};
static const DOUBLE Quadrature2D40_w[] = {0.25,0.25,0.25,0.25};
static QUADRATURE Quadrature2D40 = {4, 0, Quadrature2D40_l, Quadrature2D40_w};

static const DOUBLE_VECTOR_3D Quadrature2D42_l[] = {{0.211324865405187, 0.211324865405187},
                                                    {0.788675134594813, 0.211324865405187},
                                                    {0.211324865405187, 0.788675134594813},
                                                    {0.788675134594813, 0.788675134594813}};
static const DOUBLE Quadrature2D42_w[] = {0.25,0.25,0.25,0.25};
static QUADRATURE Quadrature2D42 = {4, 2, Quadrature2D42_l, Quadrature2D42_w};

static const DOUBLE_VECTOR_3D Quadrature2D44_l[] = {
  {0.112701665379258, 0.112701665379258},
  {0.5, 0.112701665379258},
  {0.887298334620742, 0.112701665379258},
  {0.112701665379258, 0.5},
  {0.5, 0.5},
  {0.887298334620742, 0.5},
  {0.112701665379258, 0.887298334620742},
  {0.5, 0.887298334620742},
  {0.887298334620742, 0.887298334620742}
};
static const DOUBLE Quadrature2D44_w[] = {
  0.07716049382716,
  0.12345679012346,
  0.07716049382716,
  0.12345679012346,
  0.19753086419753,
  0.12345679012346,
  0.07716049382716,
  0.12345679012346,
  0.07716049382716
};
static QUADRATURE Quadrature2D44 = {9, 4, Quadrature2D44_l, Quadrature2D44_w};


/******** For new GaussQuadratureRule function *****************************/
/* ---- Polynomial degree 1 --------------------------------------------*/
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P1_N1_GP[] = {
  {0.5,0.5}
};

static const DOUBLE Quadrature2D_Quad_P1_N1_W[] =
{1.};

static QUADRATURE Quadrature2D_Quad_P1_N1 = {
  /* Number of Gauss Points */
  1,
  /* Polynomial degree */
  1,
  /* Abscissae */
  Quadrature2D_Quad_P1_N1_GP,
  /* Weight */
  Quadrature2D_Quad_P1_N1_W
};


/* ---- Polynomial degree 2 --------------------------------------------*/

static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P2_N3_GP[] =
{{0.211324865405187, 0.211324865405187},
 {0.788675134594813, 0.211324865405187},
 {0.211324865405187, 0.788675134594813},
 {0.788675134594813, 0.788675134594813}};

static const DOUBLE Quadrature2D_Quad_P2_N3_W[] =
{0.25,
 0.25,
 0.25,
 0.25};

static QUADRATURE Quadrature2D_Quad_P2_N4 = {
  /* Number of Gauss Points */
  4,
  /* Polynomial degree */
  2,
  /* Abscissae */
  Quadrature2D_Quad_P2_N3_GP,
  /* Weight */
  Quadrature2D_Quad_P2_N3_W
};


/* ---- Polynomial degree 3 --------------------------------------------*/

/* Gauss Points: 4 */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P3_N4_GPa[] =
{{0.091751709536136983633785987549018,0.5},
 {0.5,0.091751709536136983633785987549018},
 {0.5,0.90824829046386301636621401245098},
 {0.90824829046386301636621401245098,0.5}};

static const DOUBLE Quadrature2D_Quad_P3_N4_Wa[] =
{0.25,
 0.25,
 0.25,
 0.25};

static QUADRATURE Quadrature2D_Quad_P3_N4a = {
  /* Number of Gauss Points */
  4,
  /* Polynomial degree */
  3,
  /* Abscissae */
  Quadrature2D_Quad_P3_N4_GPa,
  /* Weight */
  Quadrature2D_Quad_P3_N4_Wa
};


/* Gauss Points: 4 */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P3_N4_GPb[] =
{{0.21132486540518711774542560974902, 0.21132486540518711774542560974902},
 {0.21132486540518711774542560974902, 0.78867513459481288225457439025098},
 {0.78867513459481288225457439025098, 0.21132486540518711774542560974902},
 {0.78867513459481288225457439025098, 0.78867513459481288225457439025098}};

static const DOUBLE Quadrature2D_Quad_P3_N4_Wb[] =
{0.25,
 0.25,
 0.25,
 0.25};

static QUADRATURE Quadrature2D_Quad_P3_N4b = {
  /* Number of Gauss Points */
  4,
  /* Polynomial degree */
  3,
  /* Abscissae */
  Quadrature2D_Quad_P3_N4_GPb,
  /* Weight */
  Quadrature2D_Quad_P3_N4_Wb
};


/* ---- Polynomial degree 4 --------------------------------------------*/

/* Gauss Points: 6 */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P4_N6_GP[] =
{{0.5,0.5},
 {0.98304589153964795245728880523899, 0.5},
 {0.72780186391809642112479237299488, 0.074042673347699754349082179816666},
 {0.72780186391809642112479237299488, 0.92595732665230024565091782018333},
 {0.13418502421343273531598225407969, 0.18454360551162298687829339850317},
 {0.13418502421343273531598225407969, 0.81545639448837701312170660149683}};

static const DOUBLE Quadrature2D_Quad_P4_N6_W[] =
{0.28571428571428571428571428571428,
 0.10989010989010989010989010989011,
 0.14151805175188302631601261486295,
 0.14151805175188302631601261486295,
 0.16067975044591917148618518733485,
 0.16067975044591917148618518733485};

static QUADRATURE Quadrature2D_Quad_P4_N6 = {
  /* Number of Gauss Points */
  6,
  /* Polynomial degree */
  4,
  /* Abscissae */
  Quadrature2D_Quad_P4_N6_GP,
  /* Weight */
  Quadrature2D_Quad_P4_N6_W
};


/* ---- Polynomial degree 5 --------------------------------------------*/

/* Gauss Points: 7 */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P5_N7_GP[] =
{{0.5,0.5},
 {0.016954108460352047542711194761008,0.5},
 {0.98304589153964795245728880523899,0.5},
 {0.21132486540518711774542560974902, 0.11270166537925831148207346002176},
 {0.21132486540518711774542560974902, 0.88729833462074168851792653997824},
 {0.78867513459481288225457439025098, 0.11270166537925831148207346002176},
 {0.78867513459481288225457439025098, 0.88729833462074168851792653997824}};

static const DOUBLE Quadrature2D_Quad_P5_N7_W[] =
{0.28571428571428571428571428571428,
 0.079365079365079365079365079365079,
 0.079365079365079365079365079365079,
 0.13888888888888888888888888888889,
 0.13888888888888888888888888888889,
 0.13888888888888888888888888888889,
 0.13888888888888888888888888888889};

static QUADRATURE Quadrature2D_Quad_P5_N7 = {
  /* Number of Gauss Points */
  7,
  /* Polynomial degree */
  5,
  /* Abscissae */
  Quadrature2D_Quad_P5_N7_GP,
  /* Weight */
  Quadrature2D_Quad_P5_N7_W
};


/* Gauss Points: 8 */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P5_N8_GP[] =
{{0.15843497446801338722596537731596,0.5},
 {0.5,0.15843497446801338722596537731596},
 {0.5,0.84156502553198661277403462268403},
 {0.84156502553198661277403462268403,0.5},
 {0.05904144815590156824973070772679, 0.05904144815590156824973070772679},
 {0.05904144815590156824973070772679, 0.94095855184409843175026929227321},
 {0.94095855184409843175026929227321, 0.05904144815590156824973070772679},
 {0.94095855184409843175026929227321, 0.94095855184409843175026929227321}};

static const DOUBLE Quadrature2D_Quad_P5_N8_W[] =
{0.20408163265306122448979591836735,
 0.20408163265306122448979591836735,
 0.20408163265306122448979591836735,
 0.20408163265306122448979591836735,
 0.045918367346938775510204081632653,
 0.045918367346938775510204081632653,
 0.045918367346938775510204081632653,
 0.045918367346938775510204081632653};

static QUADRATURE Quadrature2D_Quad_P5_N8 = {
  /* Number of Gauss Points */
  8,
  /* Polynomial degree */
  5,
  /* Abscissae */
  Quadrature2D_Quad_P5_N8_GP,
  /* Weight */
  Quadrature2D_Quad_P5_N8_W
};


/* ---- Polynomial degree 6 --------------------------------------------*/

/* Gauss Points: 10  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P6_N10_GPa[] =
{{0.91820281684881280226895072035422,0.5},
 {0.32126991730434640755691280777702,0.5},
 {0.93605076559656529501643767018721, 0.05561799267261773307341476419531},
 {0.93605076559656529501643767018721, 0.94438200732738226692658523580469},
 {0.65299258107771333074469136784128, 0.19757118026765748615073385233221},
 {0.65299258107771333074469136784128, 0.80242881973234251384926614766779},
 {0.29486455026667108063715992772201, 0.022276246679468126121235819612606},
 {0.29486455026667108063715992772201, 0.97772375332053187387876418038739},
 {0.063565344421560330374370395751836, 0.21727000328062297705281230543027},
 {0.063565344421560330374370395751836, 0.78272999671937702294718769456973}};

static const DOUBLE Quadrature2D_Quad_P6_N10_Wa[] =
{0.11383581142854335963838543032803,
 0.20684899330074137308917423457476,
 0.036000221149911346232150771640156,
 0.036000221149911346232150771640156,
 0.16706477606566628748245813610296,
 0.16706477606566628748245813610296,
 0.056368501222669839352763911963813,
 0.056368501222669839352763911963813,
 0.080224099197110160568847347841674,
 0.080224099197110160568847347841674};

static QUADRATURE Quadrature2D_Quad_P6_N10a = {
  /* Number of Gauss Points */
  10,
  /* Polynomial degree */
  6,
  /* Abscissae */
  Quadrature2D_Quad_P6_N10_GPa,
  /* Weight */
  Quadrature2D_Quad_P6_N10_Wa
};


/* Gauss Points: 10  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P6_N10_GPb[] =
{{0.93491668762502950427418478934001,0.5},
 {0.26029682419394437953793052024698,0.5},
 {0.90141875810382835311174464496491, 0.068128586826923055665232207935882},
 {0.90141875810382835311174464496491, 0.93187141317307694433476779206412},
 {0.63107183275402909278458720168118, 0.24065473930370882821379308002295},
 {0.63107183275402909278458720168118, 0.75934526069629117178620691997705},
 {0.31845170842596673057084116883887, 0.033013727513575246586382922369621},
 {0.31845170842596673057084116883887, 0.96698627248642475341361707763038},
 {0.051695683618773672697043573101627, 0.19551123199182184892371165084698},
 {0.051695683618773672697043573101627, 0.80448876800817815107628834915302}};

static const DOUBLE Quadrature2D_Quad_P6_N10_Wb[] =
{0.098187647741086985886023696708155,
 0.18869072031065263420233987781343,
 0.051541512647069756320993477612544,
 0.051541512647069756320993477612544,
 0.17249803462246593791122158677157,
 0.17249803462246593791122158677157,
 0.065129372183079243945303563930747,
 0.065129372183079243945303563930747,
 0.067391896521515251778299584424342,
 0.067391896521515251778299584424342};

static QUADRATURE Quadrature2D_Quad_P6_N10b = {
  /* Number of Gauss Points */
  10,
  /* Polynomial degree */
  6,
  /* Abscissae */
  Quadrature2D_Quad_P6_N10_GPb,
  /* Weight */
  Quadrature2D_Quad_P6_N10_Wb
};


/* ---- Polynomial degree 7 --------------------------------------------*/

/* Gauss Points: 12  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P7_N12_GP[] =
{{0.037089950113724269216716611708001,0.5},
 {0.5,0.037089950113724269216716611708001},
 {0.5,0.962910049886275730783283388292},
 {0.962910049886275730783283388292,0.5},
 {0.3097227833958421718104468204568,0.3097227833958421718104468204568},
 {0.3097227833958421718104468204568,0.6902772166041578281895531795432},
 {0.6902772166041578281895531795432,0.3097227833958421718104468204568},
 {0.6902772166041578281895531795432,0.6902772166041578281895531795432},
 {0.097010108540700628146071909324628, 0.097010108540700628146071909324628},
 {0.097010108540700628146071909324628, 0.90298989145929937185392809067537},
 {0.90298989145929937185392809067537, 0.097010108540700628146071909324628},
 {0.90298989145929937185392809067537, 0.90298989145929937185392809067537}};

static const DOUBLE Quadrature2D_Quad_P7_N12_W[] =
{0.060493827160493827160493827160494,
 0.060493827160493827160493827160494,
 0.060493827160493827160493827160494,
 0.060493827160493827160493827160494,
 0.13014822916684861428497985801168,
 0.13014822916684861428497985801168,
 0.13014822916684861428497985801168,
 0.13014822916684861428497985801168,
 0.059357943672657558554526314827823,
 0.059357943672657558554526314827823,
 0.059357943672657558554526314827823,
 0.059357943672657558554526314827823};

static QUADRATURE Quadrature2D_Quad_P7_N12 = {
  /* Number of Gauss Points */
  12,
  /* Polynomial degree */
  7,
  /* Abscissae */
  Quadrature2D_Quad_P7_N12_GP,
  /* Weight */
  Quadrature2D_Quad_P7_N12_W
};


/* ---- Polynomial degree 8 --------------------------------------------*/

/* Gauss Points: 16  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P8_N16_GP[] =
{{0.5,0.5},
 {0.87881458883025272376243091093078,0.5},
 {0.38156407887214905330677888913682,0.5},
 {0.005141035477736659342552582428439,0.5},
 {0.97526047782283344179383397817322, 0.18045434754981517329838236798414},
 {0.97526047782283344179383397817322, 0.81954565245018482670161763201586},
 {0.83194136844281658918106215312182, 0.031465461537504778123884459552616},
 {0.83194136844281658918106215312182, 0.96853453846249522187611554044738},
 {0.65210534086205225452112163378099, 0.23145823472925316105117594700364},
 {0.65210534086205225452112163378099, 0.76854176527074683894882405299636},
 {0.3817516407319402596713732685346, 0.056405746775187680490155875505254},
 {0.3817516407319402596713732685346,0.94359425322481231950984412449475},
 {0.15052326195671822003768421503399, 0.25265058966490183398785216925757},
 {0.15052326195671822003768421503399, 0.74734941033509816601214783074243},
 {0.049804612894210133218263327939714, 0.051252090860116231232972080068356},
 {0.049804612894210133218263327939714, 0.94874790913988376876702791993164}};

static const DOUBLE Quadrature2D_Quad_P8_N16_W[] =
{0.013841176405359943061584002713742,
 0.10109734218151885333357243373724,
 0.13338665123815861383831921172309,
 0.029263547196684801606179593070438,
 0.031403604403436701032964263059526,
 0.031403604403436701032964263059526,
 0.034136146183397116833286999035414,
 0.034136146183397116833286999035414,
 0.12085211980281423777756638059528,
 0.12085211980281423777756638059528,
 0.063132126607385923345085246169134,
 0.063132126607385923345085246169134,
 0.090315580970543142281516900544439,
 0.090315580970543142281516900544439,
 0.021366063521561772809752589973953,
 0.021366063521561772809752589973953};

static QUADRATURE Quadrature2D_Quad_P8_N16 = {
  /* Number of Gauss Points */
  16,
  /* Polynomial degree */
  8,
  /* Abscissae */
  Quadrature2D_Quad_P8_N16_GP,
  /* Weight */
  Quadrature2D_Quad_P8_N16_W
};


/* ---- Polynomial degree 9 --------------------------------------------*/

/* Gauss Points: 17  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P9_N17_GP[] =
{{0.5,0.5},
 {0.015575016819011139641654828510833, 0.18465994013416557291337985962548},
 {0.18465994013416557291337985962548, 0.98442498318098886035834517148917},
 {0.81534005986583442708662014037452, 0.015575016819011139641654828510833},
 {0.98442498318098886035834517148917, 0.81534005986583442708662014037452},
 {0.036019177020215166299623315308149, 0.87513854998945026676799199119468},
 {0.12486145001054973323200800880532, 0.036019177020215166299623315308149},
 {0.87513854998945026676799199119468, 0.96398082297978483370037668469185},
 {0.96398082297978483370037668469185, 0.12486145001054973323200800880532},
 {0.23813208989278533198085643668043, 0.27333008943217640461860617920896},
 {0.27333008943217640461860617920896, 0.76186791010721466801914356331957},
 {0.72666991056782359538139382079104, 0.23813208989278533198085643668043},
 {0.76186791010721466801914356331957, 0.72666991056782359538139382079104},
 {0.073692135333168846127392661740826, 0.53810416409630858659081343369466},
 {0.46189583590369141340918656630534, 0.073692135333168846127392661740826},
 {0.53810416409630858659081343369466, 0.92630786466683115387260733825917},
 {0.92630786466683115387260733825917, 0.46189583590369141340918656630534}};

static const DOUBLE Quadrature2D_Quad_P9_N17_W[] =
{0.13168724279835390946502057613169,
 0.022219844542549676741752721555521,
 0.022219844542549676741752721555521,
 0.022219844542549676741752721555521,
 0.022219844542549676741752721555521,
 0.028024900532399121318974731756361,
 0.028024900532399121318974731756361,
 0.028024900532399121318974731756361,
 0.028024900532399121318974731756361,
 0.099570609815517523819083929237449,
 0.099570609815517523819083929237449,
 0.099570609815517523819083929237449,
 0.099570609815517523819083929237449,
 0.067262834409945200753933473417747,
 0.067262834409945200753933473417747,
 0.067262834409945200753933473417747,
 0.067262834409945200753933473417747};

static QUADRATURE Quadrature2D_Quad_P9_N17 = {
  /* Number of Gauss Points */
  17,
  /* Polynomial degree */
  9,
  /* Abscissae */
  Quadrature2D_Quad_P9_N17_GP,
  /* Weight */
  Quadrature2D_Quad_P9_N17_W
};


/* Gauss Points: 18  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P9_N18_GPa[] =
{{0.2105858699403541472705858356571,0.5},
 {0.7894141300596458527294141643429,0.5},
 {0.5,0.011499549209978769703002359942252},
 {0.5,0.98850045079002123029699764005775},
 {0.5,0.30317971364075553243828103749997},
 {0.5,0.69682028635924446756171896250003},
 {0.060096393001235730522408916817767, 0.036010192453657355695967479052214},
 {0.060096393001235730522408916817767, 0.96398980754634264430403252094779},
 {0.93990360699876426947759108318223, 0.036010192453657355695967479052214},
 {0.93990360699876426947759108318223, 0.96398980754634264430403252094779},
 {0.24777044842260080772162094202283, 0.12326400448419247309776702896626},
 {0.24777044842260080772162094202283, 0.87673599551580752690223297103373},
 {0.75222955157739919227837905797717, 0.12326400448419247309776702896626},
 {0.75222955157739919227837905797717, 0.87673599551580752690223297103373},
 {0.042343822958863379086169378707506, 0.28850321452561743466838047639475},
 {0.042343822958863379086169378707506, 0.71149678547438256533161952360525},
 {0.95765617704113662091383062129249, 0.28850321452561743466838047639475},
 {0.95765617704113662091383062129249, 0.71149678547438256533161952360525}};

static const DOUBLE Quadrature2D_Quad_P9_N18_Wa[] =
{0.1023183988885828608218695685332,
 0.1023183988885828608218695685332,
 0.026620029453900579636098847217415,
 0.026620029453900579636098847217415,
 0.1133037202629274640944402276042,
 0.1133037202629274640944402276042,
 0.017104130615577326419783662965799,
 0.017104130615577326419783662965799,
 0.017104130615577326419783662965799,
 0.017104130615577326419783662965799,
 0.069758460524218253487912428458658,
 0.069758460524218253487912428458658,
 0.069758460524218253487912428458658,
 0.069758460524218253487912428458658,
 0.042016334557498967816099586898138,
 0.042016334557498967816099586898138,
 0.042016334557498967816099586898138,
 0.042016334557498967816099586898138};

static QUADRATURE Quadrature2D_Quad_P9_N18 = {
  /* Number of Gauss Points */
  18,
  /* Polynomial degree */
  9,
  /* Abscissae */
  Quadrature2D_Quad_P9_N18_GPa,
  /* Weight */
  Quadrature2D_Quad_P9_N18_Wa
};


/* Gauss Points: 18  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P9_N18_GPb[] =
{{0.25264106017420188295403271203719,0.5},
 {0.74735893982579811704596728796281,0.5},
 {0.5,0.009571514026679727888109745487752},
 {0.5,0.99042848597332027211189025451225},
 {0.5,0.2584426519013601717876076846495},
 {0.5,0.74155734809863982821239231535049},
 {0.031286666889666445427789217208991, 0.029274403500357845131854080913447},
 {0.031286666889666445427789217208991, 0.97072559649964215486814591908655},
 {0.96871333311033355457221078279101, 0.029274403500357845131854080913447},
 {0.96871333311033355457221078279101, 0.97072559649964215486814591908655},
 {0.21461499156571297792568919149899, 0.10392672741576376234563750393881},
 {0.21461499156571297792568919149899, 0.89607327258423623765436249606119},
 {0.78538500843428702207431080850101, 0.10392672741576376234563750393881},
 {0.78538500843428702207431080850101, 0.89607327258423623765436249606119},
 {0.051128879100757135152227374343407, 0.29999133051183153570184913808706},
 {0.051128879100757135152227374343407, 0.70000866948816846429815086191294},
 {0.94887112089924286484777262565659, 0.29999133051183153570184913808706},
 {0.94887112089924286484777262565659, 0.70000866948816846429815086191294}};

static const DOUBLE Quadrature2D_Quad_P9_N18_Wb[] =
{0.11303099532803713749186839867068,
 0.11303099532803713749186839867068,
 0.025608038177478739553241177555265,
 0.025608038177478739553241177555265,
 0.11400355588171750280622396099421,
 0.11400355588171750280622396099421,
 0.01071332931222427213401903778756,
 0.01071332931222427213401903778756,
 0.01071332931222427213401903778756,
 0.01071332931222427213401903778756,
 0.064471015901649110760197224785112,
 0.064471015901649110760197224785112,
 0.064471015901649110760197224785112,
 0.064471015901649110760197224785112,
 0.048494360092509927180116968817253,
 0.048494360092509927180116968817253,
 0.048494360092509927180116968817253,
 0.048494360092509927180116968817253};

static QUADRATURE Quadrature2D_Quad_P9_N18b = {
  /* Number of Gauss Points */
  18,
  /* Polynomial degree */
  9,
  /* Abscissae */
  Quadrature2D_Quad_P9_N18_GPb,
  /* Weight */
  Quadrature2D_Quad_P9_N18_Wb
};


/* Gauss Points: 20  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P9_N20_GPa[] =
{{0.007730094028873803783499699849507,0.5},
 {0.5,0.007730094028873803783499699849507},
 {0.5,0.99226990597112619621650030015049},
 {0.99226990597112619621650030015049,0.5},
 {0.25555682857881379188611568933666,0.5},
 {0.5,0.25555682857881379188611568933666},
 {0.5,0.74444317142118620811388431066334},
 {0.74444317142118620811388431066334,0.5},
 {0.030216356289239232932848461884167, 0.030216356289239232932848461884167},
 {0.030216356289239232932848461884167, 0.96978364371076076706715153811583},
 {0.96978364371076076706715153811583, 0.030216356289239232932848461884167},
 {0.96978364371076076706715153811583, 0.96978364371076076706715153811583},
 {0.081644837488005512952326854423903, 0.24631161316269349736125798275304},
 {0.081644837488005512952326854423903, 0.75368838683730650263874201724696},
 {0.24631161316269349736125798275304, 0.081644837488005512952326854423903},
 {0.24631161316269349736125798275304,0.9183551625119944870476731455761},
 {0.75368838683730650263874201724696, 0.081644837488005512952326854423903},
 {0.75368838683730650263874201724696,0.9183551625119944870476731455761},
 {0.9183551625119944870476731455761,0.24631161316269349736125798275304},
 {0.9183551625119944870476731455761,0.75368838683730650263874201724696}};

static const DOUBLE Quadrature2D_Quad_P9_N20_Wa[] =
{0.017903356177452741696183476992951,
 0.017903356177452741696183476992951,
 0.017903356177452741696183476992951,
 0.017903356177452741696183476992951,
 0.11352258813788630603303810087143,
 0.11352258813788630603303810087143,
 0.11352258813788630603303810087143,
 0.11352258813788630603303810087143,
 0.010696153866694512792292085003668,
 0.010696153866694512792292085003668,
 0.010696153866694512792292085003668,
 0.010696153866694512792292085003668,
 0.053938950908983219739243168565974,
 0.053938950908983219739243168565974,
 0.053938950908983219739243168565974,
 0.053938950908983219739243168565974,
 0.053938950908983219739243168565974,
 0.053938950908983219739243168565974,
 0.053938950908983219739243168565974,
 0.053938950908983219739243168565974};

static QUADRATURE Quadrature2D_Quad_P9_N20a = {
  /* Number of Gauss Points */
  20,
  /* Polynomial degree */
  9,
  /* Abscissae */
  Quadrature2D_Quad_P9_N20_GPa,
  /* Weight */
  Quadrature2D_Quad_P9_N20_Wa
};


/* Gauss Points: 20  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P9_N20_GPb[] =
{{0.25553657151281546521295346988535,0.5},
 {0.5,0.25553657151281546521295346988535},
 {0.5,0.74446342848718453478704653011465},
 {0.74446342848718453478704653011465,0.5},
 {0.1545597247568280635925558459308,0.1545597247568280635925558459308},
 {0.1545597247568280635925558459308,0.8454402752431719364074441540692},
 {0.8454402752431719364074441540692,0.1545597247568280635925558459308},
 {0.8454402752431719364074441540692,0.8454402752431719364074441540692},
 {0.030172370951581147071214124768416, 0.030172370951581147071214124768416},
 {0.030172370951581147071214124768416, 0.96982762904841885292878587523158},
 {0.96982762904841885292878587523158, 0.030172370951581147071214124768416},
 {0.96982762904841885292878587523158, 0.96982762904841885292878587523158},
 {0.040689779471638870172264910428611, 0.32756398731779821191438479093765},
 {0.040689779471638870172264910428611, 0.67243601268220178808561520906235},
 {0.32756398731779821191438479093765, 0.040689779471638870172264910428611},
 {0.32756398731779821191438479093765, 0.95931022052836112982773508957139},
 {0.67243601268220178808561520906235, 0.040689779471638870172264910428611},
 {0.67243601268220178808561520906235, 0.95931022052836112982773508957139},
 {0.95931022052836112982773508957139, 0.32756398731779821191438479093765},
 {0.95931022052836112982773508957139, 0.67243601268220178808561520906235}};

static const DOUBLE Quadrature2D_Quad_P9_N20_Wb[] =
{0.11354099017168725686044681140132,
 0.11354099017168725686044681140132,
 0.11354099017168725686044681140132,
 0.11354099017168725686044681140132,
 0.053550090231715407985606408576494,
 0.053550090231715407985606408576494,
 0.053550090231715407985606408576494,
 0.053550090231715407985606408576494,
 0.01068280796644394064503268038496,
 0.01068280796644394064503268038496,
 0.01068280796644394064503268038496,
 0.01068280796644394064503268038496,
 0.036113055815076697254457049818612,
 0.036113055815076697254457049818612,
 0.036113055815076697254457049818612,
 0.036113055815076697254457049818612,
 0.036113055815076697254457049818612,
 0.036113055815076697254457049818612,
 0.036113055815076697254457049818612,
 0.036113055815076697254457049818612};

static QUADRATURE Quadrature2D_Quad_P9_N20b = {
  /* Number of Gauss Points */
  20,
  /* Polynomial degree */
  9,
  /* Abscissae */
  Quadrature2D_Quad_P9_N20_GPb,
  /* Weight */
  Quadrature2D_Quad_P9_N20_Wb
};


/* ---- Polynomial degree 11 --------------------------------------------*/

/* Gauss Points: 24  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P11_N24_GP[] =
{{0.008680388229572263523754251497996, 0.15096194772521621761176509691252},
 {0.15096194772521621761176509691252,0.991319611770427736476245748502},
 {0.84903805227478378238823490308748, 0.008680388229572263523754251497996},
 {0.991319611770427736476245748502,0.84903805227478378238823490308748},
 {0.030256808591631546396783818915052, 0.91288791795148196865113729264497},
 {0.08711208204851803134886270735503, 0.030256808591631546396783818915052},
 {0.91288791795148196865113729264497, 0.96974319140836845360321618108495},
 {0.96974319140836845360321618108495, 0.08711208204851803134886270735503},
 {0.023230235899233992077497866588012, 0.59429306935932097730016228409137},
 {0.40570693064067902269983771590863, 0.023230235899233992077497866588012},
 {0.59429306935932097730016228409137, 0.97676976410076600792250213341199},
 {0.97676976410076600792250213341199, 0.40570693064067902269983771590863},
 {0.09373972584759344975530870904385, 0.34218828354237290200719514179895},
 {0.34218828354237290200719514179895, 0.90626027415240655024469129095615},
 {0.65781171645762709799280485820105, 0.09373972584759344975530870904385},
 {0.90626027415240655024469129095615, 0.65781171645762709799280485820105},
 {0.14399904346233184672546705243812, 0.76266012518227388117081594357001},
 {0.23733987481772611882918405642999, 0.14399904346233184672546705243812},
 {0.76266012518227388117081594357001, 0.85600095653766815327453294756188},
 {0.85600095653766815327453294756188, 0.23733987481772611882918405642999},
 {0.28757637557566537469228494424402, 0.52082903595601118413677340226885},
 {0.47917096404398881586322659773115, 0.28757637557566537469228494424402},
 {0.52082903595601118413677340226885, 0.71242362442433462530771505575598},
 {0.71242362442433462530771505575598, 0.47917096404398881586322659773115}};

static const DOUBLE Quadrature2D_Quad_P11_N24_W[] =
{0.012005190837680953640690793994395,
 0.012005190837680953640690793994395,
 0.012005190837680953640690793994395,
 0.012005190837680953640690793994395,
 0.016517832291137648918408770212387,
 0.016517832291137648918408770212387,
 0.016517832291137648918408770212387,
 0.016517832291137648918408770212387,
 0.024346694339667041049030109949887,
 0.024346694339667041049030109949887,
 0.024346694339667041049030109949887,
 0.024346694339667041049030109949887,
 0.052934087499737150125982915339065,
 0.052934087499737150125982915339065,
 0.052934087499737150125982915339065,
 0.052934087499737150125982915339065,
 0.056406515432215846850789504052122,
 0.056406515432215846850789504052122,
 0.056406515432215846850789504052122,
 0.056406515432215846850789504052122,
 0.087789679599561359415097906452143,
 0.087789679599561359415097906452143,
 0.087789679599561359415097906452143,
 0.087789679599561359415097906452143};

static QUADRATURE Quadrature2D_Quad_P11_N24 = {
  /* Number of Gauss Points */
  24,
  /* Polynomial degree */
  11,
  /* Abscissae */
  Quadrature2D_Quad_P11_N24_GP,
  /* Weight */
  Quadrature2D_Quad_P11_N24_W
};


/* ---- Polynomial degree 13 --------------------------------------------*/

/* Gauss Points: 33  */
static const DOUBLE_VECTOR_3D Quadrature2D_Quad_P13_N33_GP[] =
{{0.5,0.5},
 {0.008256658780063868107267967594656, 0.88940485577720971125947535810003},
 {0.11059514422279028874052464189997, 0.008256658780063868107267967594656},
 {0.88940485577720971125947535810003, 0.99174334121993613189273203240534},
 {0.99174334121993613189273203240534, 0.11059514422279028874052464189997},
 {0.021351150106846317171856187613716, 0.070221997179180535707420716167816},
 {0.070221997179180535707420716167816, 0.97864884989315368282814381238628},
 {0.92977800282081946429257928383218, 0.021351150106846317171856187613716},
 {0.97864884989315368282814381238628, 0.92977800282081946429257928383218},
 {0.020537414856232571230119334900391, 0.56909172993123267687360059018315},
 {0.43090827006876732312639940981685, 0.020537414856232571230119334900391},
 {0.56909172993123267687360059018315, 0.97946258514376742876988066509961},
 {0.97946258514376742876988066509961, 0.43090827006876732312639940981685},
 {0.029336387063537381525398683214232, 0.3046318919352694996615659951572},
 {0.3046318919352694996615659951572,0.97066361293646261847460131678577},
 {0.6953681080647305003384340048428, 0.029336387063537381525398683214232},
 {0.97066361293646261847460131678577,0.6953681080647305003384340048428},
 {0.074961663150125712015617206499451, 0.73790431260913795253379642170171},
 {0.26209568739086204746620357829829, 0.074961663150125712015617206499451},
 {0.73790431260913795253379642170171, 0.92503833684987428798438279350055},
 {0.92503833684987428798438279350055, 0.26209568739086204746620357829829},
 {0.12209732171395928186435768276542, 0.17608918140649463397820348570608},
 {0.17608918140649463397820348570608, 0.87790267828604071813564231723458},
 {0.82391081859350536602179651429392, 0.12209732171395928186435768276542},
 {0.87790267828604071813564231723458, 0.82391081859350536602179651429392},
 {0.15187496075412529302050214748279,0.4646292455017775318914703240629},
 {0.4646292455017775318914703240629,0.84812503924587470697949785251721},
 {0.5353707544982224681085296759371,0.15187496075412529302050214748279},
 {0.84812503924587470697949785251721,0.5353707544982224681085296759371},
 {0.29534771915298057834772392987155, 0.67135827802020339470656353671843},
 {0.32864172197979660529343646328157, 0.29534771915298057834772392987155},
 {0.67135827802020339470656353671843, 0.70465228084701942165227607012845},
 {0.70465228084701942165227607012845, 0.32864172197979660529343646328157}};

static const DOUBLE Quadrature2D_Quad_P13_N33_W[] =
{0.075095528857806340346498286032891,
 0.0074979597161247829164034139144791,
 0.0074979597161247829164034139144791,
 0.0074979597161247829164034139144791,
 0.0074979597161247829164034139144791,
 0.009543605329270917410092593737386,
 0.009543605329270917410092593737386,
 0.009543605329270917410092593737386,
 0.009543605329270917410092593737386,
 0.015106230954437495170216398298561,
 0.015106230954437495170216398298561,
 0.015106230954437495170216398298561,
 0.015106230954437495170216398298561,
 0.019373184633276334839456888339777,
 0.019373184633276334839456888339777,
 0.019373184633276334839456888339777,
 0.019373184633276334839456888339777,
 0.029711166825148900270449786544107,
 0.029711166825148900270449786544107,
 0.029711166825148900270449786544107,
 0.029711166825148900270449786544107,
 0.032440887592500677821385982654941,
 0.032440887592500677821385982654941,
 0.032440887592500677821385982654941,
 0.032440887592500677821385982654941,
 0.053335395364297347358251071045185,
 0.053335395364297347358251071045185,
 0.053335395364297347358251071045185,
 0.053335395364297347358251071045185,
 0.06421768737049195912711929395734,
 0.06421768737049195912711929395734,
 0.06421768737049195912711929395734,
 0.06421768737049195912711929395734};

static QUADRATURE Quadrature2D_Quad_P13_N33 = {
  /* Number of Gauss Points */
  33,
  /* Polynomial degree */
  13,
  /* Abscissae */
  Quadrature2D_Quad_P13_N33_GP,
  /* Weight */
  Quadrature2D_Quad_P13_N33_W
};



/******** End of GaussQuadratureRule functions *****************************/


static const DOUBLE_VECTOR_3D Quadrature3D40_l[] = {{0.25,0.25,0.25},
                                                    {0.1,0.1,0.4},
                                                    {0.1,0.4,0.1},
                                                    {0.4,0.1,0.1},
                                                    {0.1,0.4,0.4},
                                                    {0.4,0.1,0.4},
                                                    {0.4,0.4,0.1}};
static const DOUBLE Quadrature3D40_w[] = {0.142857142857143,0.142857142857143,0.142857142857143,0.142857142857143,
                                          0.142857142857143,0.142857142857143,0.142857142857143};
static QUADRATURE Quadrature3D40 = {7, 0, Quadrature3D40_l, Quadrature3D40_w};

static const DOUBLE_VECTOR_3D Quadrature3D41_l[] = {{0.25,0.25,0.25}};
static const DOUBLE Quadrature3D41_w[] = {1.0};
static QUADRATURE Quadrature3D41 = {1, 1, Quadrature3D41_l, Quadrature3D41_w};

static const DOUBLE_VECTOR_3D Quadrature3D42_l[] = {{0.58541020, 0.13819660, 0.13819660},
                                                    {0.13819660, 0.58541020, 0.13819660},
                                                    {0.13819660, 0.13819660, 0.58541020},
                                                    {0.13819660, 0.13819660, 0.13819660}};
static const DOUBLE Quadrature3D42_w[] = {0.25,0.25,0.25,0.25};
static QUADRATURE Quadrature3D42 = {4, 2, Quadrature3D42_l, Quadrature3D42_w};

static const DOUBLE_VECTOR_3D Quadrature3D43_l[] = {
  {0.0,0.0,0.0},
  {1.0,0.0,0.0},
  {0.0,1.0,0.0},
  {0.0,0.0,1.0},
  {0.333333333333,0.333333333333,0.0},
  {0.333333333333,0.0,0.333333333333},
  {0.0,0.333333333333,0.333333333333},
  {0.333333333333,0.333333333333,0.333333333333}
};
static const DOUBLE Quadrature3D43_w[] = {
  0.025,0.025,0.025,0.025,0.225,0.225,0.225,0.225
};
static QUADRATURE Quadrature3D43 = {8, 3, Quadrature3D43_l, Quadrature3D43_w};

/* Stroud 1971 */

#define s_1 0.091971078   /* (7 - sq(15) ) / 34 */
#define s_2 0.31979363    /* (7 + sq(15) ) / 34 */
#define t_1 0.72408677    /* (13 + 3*sq(15) ) / 34 */
#define t_2 0.040619117   /* (13 - 3*sq(15) ) / 34 */
#define u   0.056350833   /* (10 - 2*sq(15) ) / 40 */
#define v   0.44364917    /* (10 + 2*sq(15) ) / 40 */
#define A   0.11851852    /* 16 / 135 */
#define B_1 0.071937084   /* (2665 - 14*sq(15) ) / 37800 */
#define B_2 0.069068207   /* (2665 - 14*sq(15) ) / 37800 */
#define C   0.052910053   /* 20 / 378 */

static const DOUBLE_VECTOR_3D Quadrature3D45_l[] = {
  {0.25,0.25,0.25},
  {s_1,s_1,s_1},
  {t_1,s_1,s_1},
  {s_1,t_1,s_1},
  {s_1,s_1,t_1},
  {s_2,s_2,s_2},
  {t_2,s_2,s_2},
  {s_2,t_2,s_2},
  {s_2,s_2,t_2},
  {v,u,u},
  {u,v,u},
  {u,u,v},
  {v,v,u},
  {v,u,v},
  {u,v,v}
};
static const DOUBLE Quadrature3D45_w[] = {
  A,B_1,B_1,B_1,B_1,B_2,B_2,B_2,B_2,C,C,C,C,C,C
};
static QUADRATURE Quadrature3D45 = {15, 5, Quadrature3D45_l, Quadrature3D45_w};

#undef s_1
#undef s_2
#undef t_1
#undef t_2
#undef u
#undef v
#undef A
#undef B_1
#undef B_2
#undef C

static const DOUBLE_VECTOR_3D Quadrature3D52_l[] = {{0.58541020, 0.72819660, 0.13819660},
                                                    {0.13819660, 0.72819660, 0.13819660},
                                                    {0.13819660, 0.27630920, 0.58541020},
                                                    {0.13819660, 0.27630920, 0.13819660},
                                                    {0.72819660, 0.13819660, 0.13819660},
                                                    {0.72819660, 0.58541020, 0.13819660},
                                                    {0.27630920, 0.13819660, 0.58541020},
                                                    {0.27630920, 0.13819660, 0.13819660}};
static const DOUBLE Quadrature3D52_w[] = {0.125,0.125,0.125,0.125,0.125,0.125,0.125,0.125};
static QUADRATURE Quadrature3D52 = {8, 2, Quadrature3D52_l, Quadrature3D52_w};

static const DOUBLE_VECTOR_3D Quadrature3D60_l[] = {{0.0, 0.0, 0.0},
                                                    {1.0, 0.0, 0.0},
                                                    {0.0, 1.0, 0.0},
                                                    {0.0, 0.0, 1.0},
                                                    {1.0, 0.0, 1.0},
                                                    {0.0, 1.0, 1.0}};
static const DOUBLE Quadrature3D60_w[] = {0.16666666666666666, 0.16666666666666666,
                                          0.16666666666666666, 0.16666666666666666,
                                          0.16666666666666666, 0.16666666666666666};
static QUADRATURE Quadrature3D60 = {6, 0, Quadrature3D60_l, Quadrature3D60_w};

static const DOUBLE_VECTOR_3D Quadrature3D62_l[] = {{0.66666666666666666, 0.16666666666666666, 0.211324865405187},
                                                    {0.16666666666666666, 0.66666666666666666, 0.211324865405187},
                                                    {0.16666666666666666, 0.16666666666666666, 0.211324865405187},
                                                    {0.66666666666666666, 0.16666666666666666, 0.788675134594813},
                                                    {0.16666666666666666, 0.66666666666666666, 0.788675134594813},
                                                    {0.16666666666666666, 0.16666666666666666, 0.788675134594813}};
static const DOUBLE Quadrature3D62_w[] = {0.16666666666666666, 0.16666666666666666,
                                          0.16666666666666666, 0.16666666666666666,
                                          0.16666666666666666, 0.16666666666666666};
static QUADRATURE Quadrature3D62 = {6, 2, Quadrature3D62_l, Quadrature3D62_w};

static const DOUBLE_VECTOR_3D Quadrature3D63_l[] = {{0.333333333333333, 0.333333333333333, 0.211324865405187},
                                                    {0.6, 0.2, 0.211324865405187},
                                                    {0.2, 0.6, 0.211324865405187},
                                                    {0.2, 0.2, 0.211324865405187},
                                                    {0.333333333333333, 0.333333333333333, 0.788675134594813},
                                                    {0.6, 0.2, 0.788675134594813},
                                                    {0.2, 0.6, 0.788675134594813},
                                                    {0.2, 0.2, 0.788675134594813}};
static const DOUBLE Quadrature3D63_w[] = {-0.28125, 0.2604166666666666, 0.2604166666666666, 0.2604166666666666,
                                          -0.28125, 0.2604166666666666, 0.2604166666666666, 0.2604166666666666};
static QUADRATURE Quadrature3D63 = {8, 3, Quadrature3D63_l, Quadrature3D63_w};

static const DOUBLE_VECTOR_3D Quadrature3D80_l[] = {{0.0, 0.0, 0.0},
                                                    {0.0, 0.0, 1.0},
                                                    {0.0, 1.0, 0.0},
                                                    {0.0, 1.0, 1.0},
                                                    {1.0, 0.0, 0.0},
                                                    {1.0, 0.0, 1.0},
                                                    {1.0, 1.0, 0.0},
                                                    {1.0, 1.0, 1.0}};
static const DOUBLE Quadrature3D80_w[] = {0.125,0.125,0.125,0.125,0.125,0.125,0.125,0.125};
static QUADRATURE Quadrature3D80 = {8, 0, Quadrature3D80_l, Quadrature3D80_w};

static const DOUBLE_VECTOR_3D Quadrature3D82_l[] = {{0.211324865405187, 0.211324865405187, 0.211324865405187},
                                                    {0.788675134594813, 0.211324865405187, 0.211324865405187},
                                                    {0.211324865405187, 0.788675134594813, 0.211324865405187},
                                                    {0.788675134594813, 0.788675134594813, 0.211324865405187},
                                                    {0.211324865405187, 0.211324865405187, 0.788675134594813},
                                                    {0.788675134594813, 0.211324865405187, 0.788675134594813},
                                                    {0.211324865405187, 0.788675134594813, 0.788675134594813},
                                                    {0.788675134594813, 0.788675134594813, 0.788675134594813}};
static const DOUBLE Quadrature3D82_w[] = {0.125,0.125,0.125,0.125,0.125,0.125,0.125,0.125};
static QUADRATURE Quadrature3D82 = {8, 2, Quadrature3D82_l, Quadrature3D82_w};

#define A_1 0.112701665379258
#define A_2 0.887298334620742
#define W_1 0.277777777777777
#define W_2 0.444444444444444

static const DOUBLE_VECTOR_3D Quadrature3D84_l[] = {
  {A_1,A_1,A_1},
  {0.5,A_1,A_1},
  {A_2,A_1,A_1},
  {A_1,0.5,A_1},
  {0.5,0.5,A_1},
  {A_2,0.5,A_1},
  {A_1,A_2,A_1},
  {0.5,A_2,A_1},
  {A_2,A_2,A_1},
  {A_1,A_1,0.5},
  {0.5,A_1,0.5},
  {A_2,A_1,0.5},
  {A_1,0.5,0.5},
  {0.5,0.5,0.5},
  {A_2,0.5,0.5},
  {A_1,A_2,0.5},
  {0.5,A_2,0.5},
  {A_2,A_2,0.5},
  {A_1,A_1,A_2},
  {0.5,A_1,A_2},
  {A_2,A_1,A_2},
  {A_1,0.5,A_2},
  {0.5,0.5,A_2},
  {A_2,0.5,A_2},
  {A_1,A_2,A_2},
  {0.5,A_2,A_2},
  {A_2,A_2,A_2}
};
static const DOUBLE Quadrature3D84_w[] =
{W_1*W_1*W_1,W_2*W_1*W_1,W_1*W_1*W_1,
 W_1*W_2*W_1,W_2*W_2*W_1,W_1*W_2*W_1,
 W_1*W_1*W_1,W_2*W_1*W_1,W_1*W_1*W_1,
 W_1*W_1*W_2,W_2*W_1*W_2,W_1*W_1*W_2,
 W_1*W_2*W_2,W_2*W_2*W_2,W_1*W_2*W_2,
 W_1*W_1*W_2,W_2*W_1*W_2,W_1*W_1*W_2,
 W_1*W_1*W_1,W_2*W_1*W_1,W_1*W_1*W_1,
 W_1*W_2*W_1,W_2*W_2*W_1,W_1*W_2*W_1,
 W_1*W_1*W_1,W_2*W_1*W_1,W_1*W_1*W_1};
static QUADRATURE Quadrature3D84 = {27, 4, Quadrature3D84_l, Quadrature3D84_w};

#undef A_1
#undef A_2
#undef W_1
#undef W_2


/******** Quadrature rules for Hexahedron *******************************/
/* ---- Polynomial degree 1 --------------------------------------------*/

/* 1 inner Gauss points, positive weights */

static const DOUBLE_VECTOR_3D Quadrature3D_Hex_P1_N1_GP[] = {
  {0.5,0.5,0.5}
};
static const DOUBLE Quadrature3D_Hex_P1_N1_W[] =
{1.};
static QUADRATURE Quadrature2D_Hex_P1_N1 = {
  /* Number of Gauss Points */
  1,
  /* Polynomial degree */
  1,
  /* Integration points */
  Quadrature3D_Hex_P1_N1_GP,
  /* Weight */
  Quadrature3D_Hex_P1_N1_W
};

/* ---- Polynomial degree 3 --------------------------------------------*/

/* 6 inner Gauss points, positive weights */
static const DOUBLE_VECTOR_3D Quadrature3D_Hex_P3_N6_GP[] = {
  {0,0.5,0.5},{0.5,0,0.5},{0.5,0.5,0},
  {0.5,0.5,1.},{0.5,1.,0.5},{1.,0.5,0.5}
};

static const DOUBLE Quadrature3D_Hex_P3_N6_W[] = {
  0.16666666666666666666666666666667,0.16666666666666666666666666666667,
  0.16666666666666666666666666666667,0.16666666666666666666666666666667,
  0.16666666666666666666666666666667,0.16666666666666666666666666666667
};

static QUADRATURE Quadrature3D_Hex_P3_N6 = {
  /* Number of Gauss Points */
  6,
  /* Polynomial degree */
  3,
  /* Integration points */
  Quadrature3D_Hex_P3_N6_GP,
  /* Weight */
  Quadrature3D_Hex_P3_N6_W
};

/* ---- Polynomial degree 5 --------------------------------------------*/

/* 14 inner Gauss points, positive weights */

static const DOUBLE_VECTOR_3D Quadrature3D_Hex_P5_N14_GP[] = {
  {0.10208878712288926836772558976193,0.5,0.5},
  {0.5,0.10208878712288926836772558976193,0.5},
  {0.5,0.5,0.10208878712288926836772558976193},
  {0.5,0.5,0.89791121287711073163227441023807},
  {0.5,0.89791121287711073163227441023807,0.5},
  {0.89791121287711073163227441023807,0.5,0.5},
  {0.12060654468033592686548286094387,0.12060654468033592686548286094387,0.12060654468033592686548286094387},
  {0.12060654468033592686548286094387,0.12060654468033592686548286094387,0.87939345531966407313451713905613},
  {0.12060654468033592686548286094387,0.87939345531966407313451713905613,0.12060654468033592686548286094387},
  {0.12060654468033592686548286094387,0.87939345531966407313451713905613,0.87939345531966407313451713905613},
  {0.87939345531966407313451713905613,0.12060654468033592686548286094387,0.12060654468033592686548286094387},
  {0.87939345531966407313451713905613,0.12060654468033592686548286094387,0.87939345531966407313451713905613},
  {0.87939345531966407313451713905613,0.87939345531966407313451713905613,0.12060654468033592686548286094387},
  {0.87939345531966407313451713905613,0.87939345531966407313451713905613,0.87939345531966407313451713905613}
};

static const DOUBLE Quadrature3D_Hex_P5_N14_W[] = {
  0.11080332409972299168975069252078,0.11080332409972299168975069252078,
  0.11080332409972299168975069252078,0.11080332409972299168975069252078,
  0.11080332409972299168975069252078,0.11080332409972299168975069252078,
  0.041897506925207756232686980609418,0.041897506925207756232686980609418,
  0.041897506925207756232686980609418,0.041897506925207756232686980609418,
  0.041897506925207756232686980609418,0.041897506925207756232686980609418,
  0.041897506925207756232686980609418,0.041897506925207756232686980609418
};

static QUADRATURE Quadrature3D_Hex_P5_N14 = {
  /* Number of Gauss Points */
  14,
  /* Polynomial degree */
  5,
  /* Integration points */
  Quadrature3D_Hex_P5_N14_GP,
  /* Weight */
  Quadrature3D_Hex_P5_N14_W
};

/* ---- Polynomial degree 7 --------------------------------------------*/

/* 31 inner Gauss points, one negative weight */

static const DOUBLE_VECTOR_3D Quadrature3D_Hex_P7_N31_GP[] = {
  {0.5,0.5,0.5},{0.20722997811544004619368460912797,0.5,0.5},
  {0.5,0.20722997811544004619368460912797,0.5},
  {0.5,0.5,0.20722997811544004619368460912797},
  {0.5,0.5,0.79277002188455995380631539087203},
  {0.5,0.79277002188455995380631539087203,0.5},
  {0.79277002188455995380631539087203,0.5,0.5},
  {0.031419180715895980744476311167302,0.15276493200414761669898709805835,0.29217036619796743688019666371648},
  {0.031419180715895980744476311167302,0.29217036619796743688019666371648,0.84723506799585238330101290194165},
  {0.031419180715895980744476311167302,0.70782963380203256311980333628352,0.15276493200414761669898709805835},
  {0.031419180715895980744476311167302,0.84723506799585238330101290194165,0.70782963380203256311980333628352},
  {0.15276493200414761669898709805835,0.031419180715895980744476311167302,0.70782963380203256311980333628352},
  {0.15276493200414761669898709805835,0.29217036619796743688019666371648,0.031419180715895980744476311167302},
  {0.15276493200414761669898709805835,0.70782963380203256311980333628352,0.9685808192841040192555236888327},
  {0.15276493200414761669898709805835,0.9685808192841040192555236888327,0.29217036619796743688019666371648},
  {0.29217036619796743688019666371648,0.031419180715895980744476311167302,0.15276493200414761669898709805835},
  {0.29217036619796743688019666371648,0.15276493200414761669898709805835,0.9685808192841040192555236888327},
  {0.29217036619796743688019666371648,0.84723506799585238330101290194165,0.031419180715895980744476311167302},
  {0.29217036619796743688019666371648,0.9685808192841040192555236888327,0.84723506799585238330101290194165},
  {0.70782963380203256311980333628352,0.031419180715895980744476311167302,0.84723506799585238330101290194165},
  {0.70782963380203256311980333628352,0.15276493200414761669898709805835,0.031419180715895980744476311167302},
  {0.70782963380203256311980333628352,0.84723506799585238330101290194165,0.9685808192841040192555236888327},
  {0.70782963380203256311980333628352,0.9685808192841040192555236888327,0.15276493200414761669898709805835},
  {0.84723506799585238330101290194165,0.031419180715895980744476311167302,0.29217036619796743688019666371648},
  {0.84723506799585238330101290194165,0.29217036619796743688019666371648,0.9685808192841040192555236888327},
  {0.84723506799585238330101290194165,0.70782963380203256311980333628352,0.031419180715895980744476311167302},
  {0.84723506799585238330101290194165,0.9685808192841040192555236888327,0.70782963380203256311980333628352},
  {0.9685808192841040192555236888327,0.15276493200414761669898709805835,0.70782963380203256311980333628352},
  {0.9685808192841040192555236888327,0.29217036619796743688019666371648,0.15276493200414761669898709805835},
  {0.9685808192841040192555236888327,0.70782963380203256311980333628352,0.84723506799585238330101290194165},
  {0.9685808192841040192555236888327,0.84723506799585238330101290194165,0.29217036619796743688019666371648}
};

static const DOUBLE Quadrature3D_Hex_P7_N31_W[] = {
  -0.15942028985507246376811594202898,0.10888888888888888888888888888889,
  0.10888888888888888888888888888889,0.10888888888888888888888888888889,
  0.10888888888888888888888888888889,0.10888888888888888888888888888889,
  0.10888888888888888888888888888889,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652,0.021086956521739130434782608695652,
  0.021086956521739130434782608695652
};

static QUADRATURE Quadrature3D_Hex_P7_N31 = {
  /* Number of Gauss Points */
  31,
  /* Polynomial degree */
  7,
  /* Integration points */
  Quadrature3D_Hex_P7_N31_GP,
  /* Weight */
  Quadrature3D_Hex_P7_N31_W
};

/* ---- Polynomial degree 7 --------------------------------------------*/

/* 38 inner Gauss points, positive weights */

static const DOUBLE_VECTOR_3D Quadrature3D_Hex_P7_N38_GP[] = {
  {0.049156096089354355458594216857025,0.5,0.5},
  {0.5,0.049156096089354355458594216857025,0.5},
  {0.5,0.5,0.049156096089354355458594216857025},
  {0.5,0.5,0.95084390391064564454140578314297},
  {0.5,0.95084390391064564454140578314297,0.5},
  {0.95084390391064564454140578314297,0.5,0.5},
  {0.29581388925026266296520554999894,0.29581388925026266296520554999894,0.29581388925026266296520554999894},
  {0.29581388925026266296520554999894,0.29581388925026266296520554999894,0.70418611074973733703479445000106},
  {0.29581388925026266296520554999894,0.70418611074973733703479445000106,0.29581388925026266296520554999894},
  {0.29581388925026266296520554999894,0.70418611074973733703479445000106,0.70418611074973733703479445000106},
  {0.70418611074973733703479445000106,0.29581388925026266296520554999894,0.29581388925026266296520554999894},
  {0.70418611074973733703479445000106,0.29581388925026266296520554999894,0.70418611074973733703479445000106},
  {0.70418611074973733703479445000106,0.70418611074973733703479445000106,0.29581388925026266296520554999894},
  {0.70418611074973733703479445000106,0.70418611074973733703479445000106,0.70418611074973733703479445000106},
  {0.07023845489947290344176106210689,0.07023845489947290344176106210689,0.29263204313600613975014537762568},
  {0.07023845489947290344176106210689,0.07023845489947290344176106210689,0.70736795686399386024985462237432},
  {0.07023845489947290344176106210689,0.29263204313600613975014537762568,0.07023845489947290344176106210689},
  {0.07023845489947290344176106210689,0.29263204313600613975014537762568,0.92976154510052709655823893789311},
  {0.07023845489947290344176106210689,0.70736795686399386024985462237432,0.07023845489947290344176106210689},
  {0.07023845489947290344176106210689,0.70736795686399386024985462237432,0.92976154510052709655823893789311},
  {0.07023845489947290344176106210689,0.92976154510052709655823893789311,0.29263204313600613975014537762568},
  {0.07023845489947290344176106210689,0.92976154510052709655823893789311,0.70736795686399386024985462237432},
  {0.29263204313600613975014537762568,0.07023845489947290344176106210689,0.07023845489947290344176106210689},
  {0.29263204313600613975014537762568,0.07023845489947290344176106210689,0.92976154510052709655823893789311},
  {0.29263204313600613975014537762568,0.92976154510052709655823893789311,0.07023845489947290344176106210689},
  {0.29263204313600613975014537762568,0.92976154510052709655823893789311,0.92976154510052709655823893789311},
  {0.70736795686399386024985462237432,0.07023845489947290344176106210689,0.07023845489947290344176106210689},
  {0.70736795686399386024985462237432,0.07023845489947290344176106210689,0.92976154510052709655823893789311},
  {0.70736795686399386024985462237432,0.92976154510052709655823893789311,0.07023845489947290344176106210689},
  {0.70736795686399386024985462237432,0.92976154510052709655823893789311,0.92976154510052709655823893789311},
  {0.92976154510052709655823893789311,0.07023845489947290344176106210689,0.29263204313600613975014537762568},
  {0.92976154510052709655823893789311,0.07023845489947290344176106210689,0.70736795686399386024985462237432},
  {0.92976154510052709655823893789311,0.29263204313600613975014537762568,0.07023845489947290344176106210689},
  {0.92976154510052709655823893789311,0.29263204313600613975014537762568,0.92976154510052709655823893789311},
  {0.92976154510052709655823893789311,0.70736795686399386024985462237432,0.07023845489947290344176106210689},
  {0.92976154510052709655823893789311,0.70736795686399386024985462237432,0.92976154510052709655823893789311},
  {0.92976154510052709655823893789311,0.92976154510052709655823893789311,0.29263204313600613975014537762568},
  {0.92976154510052709655823893789311,0.92976154510052709655823893789311,0.70736795686399386024985462237432}
};


static const DOUBLE Quadrature3D_Hex_P7_N38_W[] = {
  0.036898717282827862897703887507847,0.036898717282827862897703887507847,
  0.036898717282827862897703887507847,0.036898717282827862897703887507847,
  0.036898717282827862897703887507847,0.036898717282827862897703887507847,
  0.050506927158275072803238047597141,0.050506927158275072803238047597141,
  0.050506927158275072803238047597141,0.050506927158275072803238047597141,
  0.050506927158275072803238047597141,0.050506927158275072803238047597141,
  0.050506927158275072803238047597141,0.050506927158275072803238047597141,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325,
  0.015606344959868010007828012257325,0.015606344959868010007828012257325
};


static QUADRATURE Quadrature3D_Hex_P7_N38 = {
  /* Number of Gauss Points */
  38,
  /* Polynomial degree */
  7,
  /* Integration points */
  Quadrature3D_Hex_P7_N38_GP,
  /* Weight */
  Quadrature3D_Hex_P7_N38_W
};


/* ---- Polynomial degree 8 --------------------------------------------*/

/* 47 inner Gauss points, positive weights */

static const DOUBLE_VECTOR_3D Quadrature3D_Hex_P8_N47_GP[] = {
  {0.5,0.5,0.5},{0.10876960178202420467359301228514,0.5,0.5},
  {0.5,0.10876960178202420467359301228514,0.5},
  {0.5,0.5,0.10876960178202420467359301228514},
  {0.5,0.5,0.89123039821797579532640698771486},
  {0.5,0.89123039821797579532640698771486,0.5},
  {0.89123039821797579532640698771486,0.5,0.5},
  {0.25595266514681675973663534926566,0.25595266514681675973663534926566,0.25595266514681675973663534926566},
  {0.25595266514681675973663534926566,0.25595266514681675973663534926566,0.74404733485318324026336465073434},
  {0.25595266514681675973663534926566,0.74404733485318324026336465073434,0.25595266514681675973663534926566},
  {0.25595266514681675973663534926566,0.74404733485318324026336465073434,0.74404733485318324026336465073434},
  {0.74404733485318324026336465073434,0.25595266514681675973663534926566,0.25595266514681675973663534926566},
  {0.74404733485318324026336465073434,0.25595266514681675973663534926566,0.74404733485318324026336465073434},
  {0.74404733485318324026336465073434,0.74404733485318324026336465073434,0.25595266514681675973663534926566},
  {0.74404733485318324026336465073434,0.74404733485318324026336465073434,0.74404733485318324026336465073434},
  {0.068890536169259405571788554444979,0.068890536169259405571788554444979,0.068890536169259405571788554444979},
  {0.068890536169259405571788554444979,0.068890536169259405571788554444979,0.93110946383074059442821144555502},
  {0.068890536169259405571788554444979,0.93110946383074059442821144555502,0.068890536169259405571788554444979},
  {0.068890536169259405571788554444979,0.93110946383074059442821144555502,0.93110946383074059442821144555502},
  {0.93110946383074059442821144555502,0.068890536169259405571788554444979,0.068890536169259405571788554444979},
  {0.93110946383074059442821144555502,0.068890536169259405571788554444979,0.93110946383074059442821144555502},
  {0.93110946383074059442821144555502,0.93110946383074059442821144555502,0.068890536169259405571788554444979},
  {0.93110946383074059442821144555502,0.93110946383074059442821144555502,0.93110946383074059442821144555502},
  {0.027901710853995902340656252613128,0.15121258314638150161030413502551,0.64055695470417092802904914092321},
  {0.027901710853995902340656252613128,0.35944304529582907197095085907679,0.15121258314638150161030413502551},
  {0.027901710853995902340656252613128,0.64055695470417092802904914092321,0.84878741685361849838969586497449},
  {0.027901710853995902340656252613128,0.84878741685361849838969586497449,0.35944304529582907197095085907679},
  {0.15121258314638150161030413502551,0.027901710853995902340656252613128,0.35944304529582907197095085907679},
  {0.15121258314638150161030413502551,0.35944304529582907197095085907679,0.97209828914600409765934374738687},
  {0.15121258314638150161030413502551,0.64055695470417092802904914092321,0.027901710853995902340656252613128},
  {0.15121258314638150161030413502551,0.97209828914600409765934374738687,0.64055695470417092802904914092321},
  {0.35944304529582907197095085907679,0.027901710853995902340656252613128,0.84878741685361849838969586497449},
  {0.35944304529582907197095085907679,0.15121258314638150161030413502551,0.027901710853995902340656252613128},
  {0.35944304529582907197095085907679,0.84878741685361849838969586497449,0.97209828914600409765934374738687},
  {0.35944304529582907197095085907679,0.97209828914600409765934374738687,0.15121258314638150161030413502551},
  {0.64055695470417092802904914092321,0.027901710853995902340656252613128,0.15121258314638150161030413502551},
  {0.64055695470417092802904914092321,0.15121258314638150161030413502551,0.97209828914600409765934374738687},
  {0.64055695470417092802904914092321,0.84878741685361849838969586497449,0.027901710853995902340656252613128},
  {0.64055695470417092802904914092321,0.97209828914600409765934374738687,0.84878741685361849838969586497449},
  {0.84878741685361849838969586497449,0.027901710853995902340656252613128,0.64055695470417092802904914092321},
  {0.84878741685361849838969586497449,0.35944304529582907197095085907679,0.027901710853995902340656252613128},
  {0.84878741685361849838969586497449,0.64055695470417092802904914092321,0.97209828914600409765934374738687},
  {0.84878741685361849838969586497449,0.97209828914600409765934374738687,0.35944304529582907197095085907679},
  {0.97209828914600409765934374738687,0.15121258314638150161030413502551,0.35944304529582907197095085907679},
  {0.97209828914600409765934374738687,0.35944304529582907197095085907679,0.84878741685361849838969586497449},
  {0.97209828914600409765934374738687,0.64055695470417092802904914092321,0.15121258314638150161030413502551},
  {0.97209828914600409765934374738687,0.84878741685361849838969586497449,0.64055695470417092802904914092321}
};

static const DOUBLE Quadrature3D_Hex_P8_N47_W[] = {
  0.056487964359399961311345477337444,0.037422397169042364962923202322021,
  0.037422397169042364962923202322021,0.037422397169042364962923202322021,
  0.037422397169042364962923202322021,0.037422397169042364962923202322021,
  0.037422397169042364962923202322021,0.037609519921405002492462336223895,
  0.037609519921405002492462336223895,0.037609519921405002492462336223895,
  0.037609519921405002492462336223895,0.037609519921405002492462336223895,
  0.037609519921405002492462336223895,0.037609519921405002492462336223895,
  0.037609519921405002492462336223895,0.0061855406984629765717271646964357,
  0.0061855406984629765717271646964357,0.0061855406984629765717271646964357,
  0.0061855406984629765717271646964357,0.0061855406984629765717271646964357,
  0.0061855406984629765717271646964357,0.0061855406984629765717271646964357,
  0.0061855406984629765717271646964357,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324,0.015359048652808417349899970890324,
  0.015359048652808417349899970890324
};

static QUADRATURE Quadrature3D_Hex_P8_N47 = {
  /* Number of Gauss Points */
  47,
  /* Polynomial degree */
  8,
  /* Integration points */
  Quadrature3D_Hex_P8_N47_GP,
  /* Weight */
  Quadrature3D_Hex_P8_N47_W
};


/* ---- Polynomial degree 9 --------------------------------------------*/

/* 58 inner Gauss points, positive weights */

static const DOUBLE_VECTOR_3D Quadrature3D_Hex_P9_N58_GP[] = {
  {0.19315926520414550308255755512972,0.5,0.5},
  {0.5,0.19315926520414550308255755512972,0.5},
  {0.5,0.5,0.19315926520414550308255755512972},
  {0.5,0.5,0.80684073479585449691744244487028},
  {0.5,0.80684073479585449691744244487028,0.5},
  {0.80684073479585449691744244487028,0.5,0.5},
  {0.061156438371160856756612120502834,0.061156438371160856756612120502834,0.5},
  {0.061156438371160856756612120502834,0.5,0.061156438371160856756612120502834},
  {0.061156438371160856756612120502834,0.5,0.93884356162883914324338787949717},
  {0.061156438371160856756612120502834,0.93884356162883914324338787949717,0.5},
  {0.5,0.061156438371160856756612120502834,0.061156438371160856756612120502834},
  {0.5,0.061156438371160856756612120502834,0.93884356162883914324338787949717},
  {0.5,0.93884356162883914324338787949717,0.061156438371160856756612120502834},
  {0.5,0.93884356162883914324338787949717,0.93884356162883914324338787949717},
  {0.93884356162883914324338787949717,0.061156438371160856756612120502834,0.5},
  {0.93884356162883914324338787949717,0.5,0.061156438371160856756612120502834},
  {0.93884356162883914324338787949717,0.5,0.93884356162883914324338787949717},
  {0.93884356162883914324338787949717,0.93884356162883914324338787949717,0.5},
  {0.21794459648998497286669050066846,0.21794459648998497286669050066846,0.21794459648998497286669050066846},
  {0.21794459648998497286669050066846,0.21794459648998497286669050066846,0.78205540351001502713330949933154},
  {0.21794459648998497286669050066846,0.78205540351001502713330949933154,0.21794459648998497286669050066846},
  {0.21794459648998497286669050066846,0.78205540351001502713330949933154,0.78205540351001502713330949933154},
  {0.78205540351001502713330949933154,0.21794459648998497286669050066846,0.21794459648998497286669050066846},
  {0.78205540351001502713330949933154,0.21794459648998497286669050066846,0.78205540351001502713330949933154},
  {0.78205540351001502713330949933154,0.78205540351001502713330949933154,0.21794459648998497286669050066846},
  {0.78205540351001502713330949933154,0.78205540351001502713330949933154,0.78205540351001502713330949933154},
  {0.064950107669012041192468095568038,0.064950107669012041192468095568038,0.064950107669012041192468095568038},
  {0.064950107669012041192468095568038,0.064950107669012041192468095568038,0.93504989233098795880753190443196},
  {0.064950107669012041192468095568038,0.93504989233098795880753190443196,0.064950107669012041192468095568038},
  {0.064950107669012041192468095568038,0.93504989233098795880753190443196,0.93504989233098795880753190443196},
  {0.93504989233098795880753190443196,0.064950107669012041192468095568038,0.064950107669012041192468095568038},
  {0.93504989233098795880753190443196,0.064950107669012041192468095568038,0.93504989233098795880753190443196},
  {0.93504989233098795880753190443196,0.93504989233098795880753190443196,0.064950107669012041192468095568038},
  {0.93504989233098795880753190443196,0.93504989233098795880753190443196,0.93504989233098795880753190443196},
  {0.030734789067664127335511565198461,0.28386604868456891779198756924153,0.28386604868456891779198756924153},
  {0.030734789067664127335511565198461,0.28386604868456891779198756924153,0.71613395131543108220801243075847},
  {0.030734789067664127335511565198461,0.71613395131543108220801243075847,0.28386604868456891779198756924153},
  {0.030734789067664127335511565198461,0.71613395131543108220801243075847,0.71613395131543108220801243075847},
  {0.28386604868456891779198756924153,0.030734789067664127335511565198461,0.28386604868456891779198756924153},
  {0.28386604868456891779198756924153,0.030734789067664127335511565198461,0.71613395131543108220801243075847},
  {0.28386604868456891779198756924153,0.28386604868456891779198756924153,0.030734789067664127335511565198461},
  {0.28386604868456891779198756924153,0.28386604868456891779198756924153,0.96926521093233587266448843480154},
  {0.28386604868456891779198756924153,0.71613395131543108220801243075847,0.030734789067664127335511565198461},
  {0.28386604868456891779198756924153,0.71613395131543108220801243075847,0.96926521093233587266448843480154},
  {0.28386604868456891779198756924153,0.96926521093233587266448843480154,0.28386604868456891779198756924153},
  {0.28386604868456891779198756924153,0.96926521093233587266448843480154,0.71613395131543108220801243075847},
  {0.71613395131543108220801243075847,0.030734789067664127335511565198461,0.28386604868456891779198756924153},
  {0.71613395131543108220801243075847,0.030734789067664127335511565198461,0.71613395131543108220801243075847},
  {0.71613395131543108220801243075847,0.28386604868456891779198756924153,0.030734789067664127335511565198461},
  {0.71613395131543108220801243075847,0.28386604868456891779198756924153,0.96926521093233587266448843480154},
  {0.71613395131543108220801243075847,0.71613395131543108220801243075847,0.030734789067664127335511565198461},
  {0.71613395131543108220801243075847,0.71613395131543108220801243075847,0.96926521093233587266448843480154},
  {0.71613395131543108220801243075847,0.96926521093233587266448843480154,0.28386604868456891779198756924153},
  {0.71613395131543108220801243075847,0.96926521093233587266448843480154,0.71613395131543108220801243075847},
  {0.96926521093233587266448843480154,0.28386604868456891779198756924153,0.28386604868456891779198756924153},
  {0.96926521093233587266448843480154,0.28386604868456891779198756924153,0.71613395131543108220801243075847},
  {0.96926521093233587266448843480154,0.71613395131543108220801243075847,0.28386604868456891779198756924153},
  {0.96926521093233587266448843480154,0.71613395131543108220801243075847,0.71613395131543108220801243075847}
};

static const DOUBLE Quadrature3D_Hex_P9_N58_W[] = {
  0.05415937446870681787622884914929,0.05415937446870681787622884914929,
  0.05415937446870681787622884914929,0.05415937446870681787622884914929,
  0.05415937446870681787622884914929,0.05415937446870681787622884914929,
  0.011473725767022205271405573614956,0.011473725767022205271405573614956,
  0.011473725767022205271405573614956,0.011473725767022205271405573614956,
  0.011473725767022205271405573614956,0.011473725767022205271405573614956,
  0.011473725767022205271405573614956,0.011473725767022205271405573614956,
  0.011473725767022205271405573614956,0.011473725767022205271405573614956,
  0.011473725767022205271405573614956,0.011473725767022205271405573614956,
  0.024857479768002937540108589823201,0.024857479768002937540108589823201,
  0.024857479768002937540108589823201,0.024857479768002937540108589823201,
  0.024857479768002937540108589823201,0.024857479768002937540108589823201,
  0.024857479768002937540108589823201,0.024857479768002937540108589823201,
  0.0062685994124186287334314359655827,0.0062685994124186287334314359655827,
  0.0062685994124186287334314359655827,0.0062685994124186287334314359655827,
  0.0062685994124186287334314359655827,0.0062685994124186287334314359655827,
  0.0062685994124186287334314359655827,0.0062685994124186287334314359655827,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938,
  0.012014600439171670804059992308938,0.012014600439171670804059992308938
};

static QUADRATURE Quadrature3D_Hex_P9_N58 = {
  /* Number of Gauss Points */
  58,
  /* Polynomial degree */
  9,
  /* Integration points */
  Quadrature3D_Hex_P9_N58_GP,
  /* Weight */
  Quadrature3D_Hex_P9_N58_W
};


/* ---- Polynomial degree 11 --------------------------------------------*/

/* 90 inner Gauss points, positive weights */

static const DOUBLE_VECTOR_3D Quadrature3D_Hex_P11_N90_GP[] = {
  {0.093692832950186751803812201310128,0.5,0.5},
  {0.5,0.093692832950186751803812201310128,0.5},
  {0.5,0.5,0.093692832950186751803812201310128},
  {0.5,0.5,0.90630716704981324819618779868987},
  {0.5,0.90630716704981324819618779868987,0.5},
  {0.90630716704981324819618779868987,0.5,0.5},
  {0.13266585650149599132680761506623,0.13266585650149599132680761506623,0.5},
  {0.13266585650149599132680761506623,0.5,0.13266585650149599132680761506623},
  {0.13266585650149599132680761506623,0.5,0.86733414349850400867319238493377},
  {0.13266585650149599132680761506623,0.86733414349850400867319238493377,0.5},
  {0.5,0.13266585650149599132680761506623,0.13266585650149599132680761506623},
  {0.5,0.13266585650149599132680761506623,0.86733414349850400867319238493377},
  {0.5,0.86733414349850400867319238493377,0.13266585650149599132680761506623},
  {0.5,0.86733414349850400867319238493377,0.86733414349850400867319238493377},
  {0.86733414349850400867319238493377,0.13266585650149599132680761506623,0.5},
  {0.86733414349850400867319238493377,0.5,0.13266585650149599132680761506623},
  {0.86733414349850400867319238493377,0.5,0.86733414349850400867319238493377},
  {0.86733414349850400867319238493377,0.86733414349850400867319238493377,0.5},
  {0.34330329774197263947711338472102,0.34330329774197263947711338472102,0.34330329774197263947711338472102},
  {0.34330329774197263947711338472102,0.34330329774197263947711338472102,0.65669670225802736052288661527898},
  {0.34330329774197263947711338472102,0.65669670225802736052288661527898,0.34330329774197263947711338472102},
  {0.34330329774197263947711338472102,0.65669670225802736052288661527898,0.65669670225802736052288661527898},
  {0.65669670225802736052288661527898,0.34330329774197263947711338472102,0.34330329774197263947711338472102},
  {0.65669670225802736052288661527898,0.34330329774197263947711338472102,0.65669670225802736052288661527898},
  {0.65669670225802736052288661527898,0.65669670225802736052288661527898,0.34330329774197263947711338472102},
  {0.65669670225802736052288661527898,0.65669670225802736052288661527898,0.65669670225802736052288661527898},
  {0.19916236790086864918279349710734,0.19916236790086864918279349710734,0.19916236790086864918279349710734},
  {0.19916236790086864918279349710734,0.19916236790086864918279349710734,0.80083763209913135081720650289266},
  {0.19916236790086864918279349710734,0.80083763209913135081720650289266,0.19916236790086864918279349710734},
  {0.19916236790086864918279349710734,0.80083763209913135081720650289266,0.80083763209913135081720650289266},
  {0.80083763209913135081720650289266,0.19916236790086864918279349710734,0.19916236790086864918279349710734},
  {0.80083763209913135081720650289266,0.19916236790086864918279349710734,0.80083763209913135081720650289266},
  {0.80083763209913135081720650289266,0.80083763209913135081720650289266,0.19916236790086864918279349710734},
  {0.80083763209913135081720650289266,0.80083763209913135081720650289266,0.80083763209913135081720650289266},
  {0.072272119491120007662454264654827,0.072272119491120007662454264654827,0.072272119491120007662454264654827},
  {0.072272119491120007662454264654827,0.072272119491120007662454264654827,0.92772788050887999233754573534517},
  {0.072272119491120007662454264654827,0.92772788050887999233754573534517,0.072272119491120007662454264654827},
  {0.072272119491120007662454264654827,0.92772788050887999233754573534517,0.92772788050887999233754573534517},
  {0.92772788050887999233754573534517,0.072272119491120007662454264654827,0.072272119491120007662454264654827},
  {0.92772788050887999233754573534517,0.072272119491120007662454264654827,0.92772788050887999233754573534517},
  {0.92772788050887999233754573534517,0.92772788050887999233754573534517,0.072272119491120007662454264654827},
  {0.92772788050887999233754573534517,0.92772788050887999233754573534517,0.92772788050887999233754573534517},
  {0.029377571394698368044424921184433,0.029377571394698368044424921184433,0.32304859270168493254484356459355},
  {0.029377571394698368044424921184433,0.029377571394698368044424921184433,0.67695140729831506745515643540645},
  {0.029377571394698368044424921184433,0.32304859270168493254484356459355,0.029377571394698368044424921184433},
  {0.029377571394698368044424921184433,0.32304859270168493254484356459355,0.97062242860530163195557507881557},
  {0.029377571394698368044424921184433,0.67695140729831506745515643540645,0.029377571394698368044424921184433},
  {0.029377571394698368044424921184433,0.67695140729831506745515643540645,0.97062242860530163195557507881557},
  {0.029377571394698368044424921184433,0.97062242860530163195557507881557,0.32304859270168493254484356459355},
  {0.029377571394698368044424921184433,0.97062242860530163195557507881557,0.67695140729831506745515643540645},
  {0.32304859270168493254484356459355,0.029377571394698368044424921184433,0.029377571394698368044424921184433},
  {0.32304859270168493254484356459355,0.029377571394698368044424921184433,0.97062242860530163195557507881557},
  {0.32304859270168493254484356459355,0.97062242860530163195557507881557,0.029377571394698368044424921184433},
  {0.32304859270168493254484356459355,0.97062242860530163195557507881557,0.97062242860530163195557507881557},
  {0.67695140729831506745515643540645,0.029377571394698368044424921184433,0.029377571394698368044424921184433},
  {0.67695140729831506745515643540645,0.029377571394698368044424921184433,0.97062242860530163195557507881557},
  {0.67695140729831506745515643540645,0.97062242860530163195557507881557,0.029377571394698368044424921184433},
  {0.67695140729831506745515643540645,0.97062242860530163195557507881557,0.97062242860530163195557507881557},
  {0.97062242860530163195557507881557,0.029377571394698368044424921184433,0.32304859270168493254484356459355},
  {0.97062242860530163195557507881557,0.029377571394698368044424921184433,0.67695140729831506745515643540645},
  {0.97062242860530163195557507881557,0.32304859270168493254484356459355,0.029377571394698368044424921184433},
  {0.97062242860530163195557507881557,0.32304859270168493254484356459355,0.97062242860530163195557507881557},
  {0.97062242860530163195557507881557,0.67695140729831506745515643540645,0.029377571394698368044424921184433},
  {0.97062242860530163195557507881557,0.67695140729831506745515643540645,0.97062242860530163195557507881557},
  {0.97062242860530163195557507881557,0.97062242860530163195557507881557,0.32304859270168493254484356459355},
  {0.97062242860530163195557507881557,0.97062242860530163195557507881557,0.67695140729831506745515643540645},
  {0.017450167243644868534859088437328,0.27460003244274528481105782713487,0.27460003244274528481105782713487},
  {0.017450167243644868534859088437328,0.27460003244274528481105782713487,0.72539996755725471518894217286513},
  {0.017450167243644868534859088437328,0.72539996755725471518894217286513,0.27460003244274528481105782713487},
  {0.017450167243644868534859088437328,0.72539996755725471518894217286513,0.72539996755725471518894217286513},
  {0.27460003244274528481105782713487,0.017450167243644868534859088437328,0.27460003244274528481105782713487},
  {0.27460003244274528481105782713487,0.017450167243644868534859088437328,0.72539996755725471518894217286513},
  {0.27460003244274528481105782713487,0.27460003244274528481105782713487,0.017450167243644868534859088437328},
  {0.27460003244274528481105782713487,0.27460003244274528481105782713487,0.98254983275635513146514091156267},
  {0.27460003244274528481105782713487,0.72539996755725471518894217286513,0.017450167243644868534859088437328},
  {0.27460003244274528481105782713487,0.72539996755725471518894217286513,0.98254983275635513146514091156267},
  {0.27460003244274528481105782713487,0.98254983275635513146514091156267,0.27460003244274528481105782713487},
  {0.27460003244274528481105782713487,0.98254983275635513146514091156267,0.72539996755725471518894217286513},
  {0.72539996755725471518894217286513,0.017450167243644868534859088437328,0.27460003244274528481105782713487},
  {0.72539996755725471518894217286513,0.017450167243644868534859088437328,0.72539996755725471518894217286513},
  {0.72539996755725471518894217286513,0.27460003244274528481105782713487,0.017450167243644868534859088437328},
  {0.72539996755725471518894217286513,0.27460003244274528481105782713487,0.98254983275635513146514091156267},
  {0.72539996755725471518894217286513,0.72539996755725471518894217286513,0.017450167243644868534859088437328},
  {0.72539996755725471518894217286513,0.72539996755725471518894217286513,0.98254983275635513146514091156267},
  {0.72539996755725471518894217286513,0.98254983275635513146514091156267,0.27460003244274528481105782713487},
  {0.72539996755725471518894217286513,0.98254983275635513146514091156267,0.72539996755725471518894217286513},
  {0.98254983275635513146514091156267,0.27460003244274528481105782713487,0.27460003244274528481105782713487},
  {0.98254983275635513146514091156267,0.27460003244274528481105782713487,0.72539996755725471518894217286513},
  {0.98254983275635513146514091156267,0.72539996755725471518894217286513,0.27460003244274528481105782713487},
  {0.98254983275635513146514091156267,0.72539996755725471518894217286513,0.72539996755725471518894217286513}
};

static const DOUBLE Quadrature3D_Hex_P11_N90_W[] = {
  0.025309634201600023823167141370877,0.025309634201600023823167141370877,
  0.025309634201600023823167141370877,0.025309634201600023823167141370877,
  0.025309634201600023823167141370877,0.025309634201600023823167141370877,
  0.018149918232514462286563225099282,0.018149918232514462286563225099282,
  0.018149918232514462286563225099282,0.018149918232514462286563225099282,
  0.018149918232514462286563225099282,0.018149918232514462286563225099282,
  0.018149918232514462286563225099282,0.018149918232514462286563225099282,
  0.018149918232514462286563225099282,0.018149918232514462286563225099282,
  0.018149918232514462286563225099282,0.018149918232514462286563225099282,
  0.026999005656871141164183333298055,0.026999005656871141164183333298055,
  0.026999005656871141164183333298055,0.026999005656871141164183333298055,
  0.026999005656871141164183333298055,0.026999005656871141164183333298055,
  0.026999005656871141164183333298055,0.026999005656871141164183333298055,
  0.014692293494557035048741475501335,0.014692293494557035048741475501335,
  0.014692293494557035048741475501335,0.014692293494557035048741475501335,
  0.014692293494557035048741475501335,0.014692293494557035048741475501335,
  0.014692293494557035048741475501335,0.014692293494557035048741475501335,
  0.0055804890098536552051251442852662,0.0055804890098536552051251442852662,
  0.0055804890098536552051251442852662,0.0055804890098536552051251442852662,
  0.0055804890098536552051251442852662,0.0055804890098536552051251442852662,
  0.0055804890098536552051251442852662,0.0055804890098536552051251442852662,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.002826787017352735527899528833623,0.002826787017352735527899528833623,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308,
  0.0076802492622294169003437555791308,0.0076802492622294169003437555791308
};

static QUADRATURE Quadrature3D_Hex_P11_N90 = {
  /* Number of Gauss Points */
  90,
  /* Polynomial degree */
  11,
  /* Integration points */
  Quadrature3D_Hex_P11_N90_GP,
  /* Weight */
  Quadrature3D_Hex_P11_N90_W
};


/****************************************************************************/
/*D
   GetQuadrature - providing a quadrature formula

   SYNOPSIS:
   QUADRATURE *GetQuadrature(INT dim, INT n, INT order);

   PARAMETERS:
   .  dim - dimension of the formular
   .  n - number of corners of the element
   .  order - order of approximation

   DESCRIPTION:
   This function returns a pointer to a quadrature formula.

   EXAMPLE:
   .vb
   QUADRATURE *quadrature;

   if ((quadrature = GetQuadrature(2,3,2)) == NULL)
          return(1);
   sum = 0.0;
   for (l=0; l<Q_NIP(quadrature); l++)
   {
    LOCAL_TO_GLOBAL(n,x,Q_LOCAL(quadrature,l),global);
    (*SourceFunc)(global,val);
    sum += val * Q_WEIGHT(quadrature,l);
   }
   sum = val * AreaOfTriangle;
   .ve

   RETURN VALUE:
   QUADRATURE *
   .n    pointer to quadrature
   .n    NULL if the quadrature formula cannot be found.
   D*/
/****************************************************************************/

QUADRATURE *NS_DIM_PREFIX GetQuadrature(INT dim, INT n, INT order)
{
  switch (dim)
  {
  case 1 :
    switch (order)
    {
    case 0 :
    case 1 : return(&Quadrature1D1);
    case 2 :
    case 3 : return(&Quadrature1D3);
    case 4 :
    case 5 : return(&Quadrature1D5);
    default : return(&Quadrature1D7);
    }
  case 2 :
    switch (n)
    {
    case 3 :
      switch (order)
      {
      case 1 : return(&Quadrature2D31);
      case 2 : return(&Quadrature2D32);
      case 3 : return(&Quadrature2D33);
      case 4 : return(&Quadrature2D34);
      default : return(&Quadrature2D35);
      }
    case 4 :
      switch (order)
      {
      case 0 : return(&Quadrature2D40);
      case 1 : return(&Quadrature2D42);
      case 2 : return(&Quadrature2D42);
      case 3 : return(&Quadrature2D44);
      case 4 : return(&Quadrature2D44);
      default : return(&Quadrature2D44);
      }
    }
  case 3 :
    switch (n)
    {
    case 4 :
      switch (order)
      {
      case 0 :  return(&Quadrature3D40);
      case 1 :  return(&Quadrature3D41);
      case 2 :  return(&Quadrature3D42);
      case 3 :  return(&Quadrature3D43);
      default : return(&Quadrature3D45);
      }
    case 5 : return(&Quadrature3D52);
    case 6 :
      switch (order)
      {
      case 0 :  return(&Quadrature3D60);
      default : return(&Quadrature3D62);
      }
    case 8 :
      switch (order)
      {
      case 0 : return(&Quadrature3D80);
      case 1 : return(&Quadrature3D82);
      case 2 : return(&Quadrature3D82);
      default : return(&Quadrature3D84);
      }
    }
  }
  return(NULL);
}


/****************************************************************************/
/*D
   GetQuadratureRule - providing a quadrature formula

   SYNOPSIS:
   QUADRATURE *GetQuadratureRule(INT dim, INT n, INT order);

   PARAMETERS:
   .  dim - dimension of the formular
   .  n - number of corners of the element
   .  order - order of approximation

   DESCRIPTION:
   This function returns a pointer to a quadrature formula.

   It is different from GetQuadrature in the case that the quadrature
   formula of order 'order' is not exactly available.  'GetQuadrature'
   just returns the highest order formula, GetQuadratureRule returns
   the formula of the smallest degree that integrates exactly until 'order'.

   EXAMPLE:
   .vb
   QUADRATURE *quadrature;

   if ((quadrature = GetQuadratureRule(2,3,2)) == NULL)
          return(1);
   sum = 0.0;
   for (l=0; l<Q_NIP(quadrature); l++)
   {
    LOCAL_TO_GLOBAL(n,x,Q_LOCAL(quadrature,l),global);
    (*SourceFunc)(global,val);
    sum += val * Q_WEIGHT(quadrature,l);
   }
   sum = val * AreaOfTriangle;
   .ve

   RETURN VALUE:
   QUADRATURE *
   .n    pointer to quadrature
   .n    NULL if the quadrature formula cannot be found.
   D*/
/****************************************************************************/

QUADRATURE* NS_DIM_PREFIX GetQuadratureRule(INT dim, INT n, INT order)
{
  switch (dim)
  {
  case 1 :
    switch (order)
    {
    case 0 :       return(&Quadrature1D1);
    case 1 :       return(&Quadrature1D1);
    case 2 :       return(&Quadrature1D3);
    case 3 :       return(&Quadrature1D3);
    case 4 :       return(&Quadrature1D5);
    case 5 :       return(&Quadrature1D5);
    case 6 :       return(&Quadrature1D7);
    case 7 :       return(&Quadrature1D7);
    case 8 :       return(&Quadrature1D9);
    case 9 :       return(&Quadrature1D9);
    case 10 :      return(&Quadrature1D11);
    case 11 :      return(&Quadrature1D11);
    case 12 :      return(&Quadrature1D13);
    case 13 :      return(&Quadrature1D13);
    case 14 :      return(&Quadrature1D15);
    case 15 :      return(&Quadrature1D15);
    case 16 :      return(&Quadrature1D17);
    case 17 :      return(&Quadrature1D17);
    case 18 :      return(&Quadrature1D19);
    case 19 :      return(&Quadrature1D19);
    default : return(&Quadrature1D19);
    }
  case 2 :
    switch (n)
    {
    case 3 :
      switch (order)
      {
      case 0 :       return(&Quadrature2D_Triangle_P1_N1);
      case 1 :       return(&Quadrature2D_Triangle_P1_N1);
      case 2 :       return(&Quadrature2D_Triangle_P2_N3b);
      case 3 :       return(&Quadrature2D_Triangle_P3_N4);
      case 4 :       return(&Quadrature2D_Triangle_P4_N6);
      case 5 :       return(&Quadrature2D_Triangle_P5_N7);
      case 6 :       return(&Quadrature2D_Triangle_P6_N12a);
      case 7 :       return(&Quadrature2D_Triangle_P7_N12);
      case 8 :       return(&Quadrature2D_Triangle_P8_N16);
      case 9 :       return(&Quadrature2D_Triangle_P9_N19);
      case 10 :      return(&Quadrature2D_Triangle_P10_N25a);
      case 11 :      return(&Quadrature2D_Triangle_P11_N28);
      case 12 :      return(&Quadrature2D_Triangle_P12_N33);
      default :      return(&Quadrature2D_Triangle_P12_N33);
      }
    case 4 :
      switch (order)
      {
      case 0 :
      case 1 :       return(&Quadrature2D_Quad_P1_N1);
      case 2 :       return(&Quadrature2D_Quad_P2_N4);
      case 3 :       return(&Quadrature2D_Quad_P3_N4a);
      case 4 :       return(&Quadrature2D_Quad_P4_N6);
      case 5 :       return(&Quadrature2D_Quad_P5_N7);
      case 6 :       return(&Quadrature2D_Quad_P6_N10a);
      case 7 :       return(&Quadrature2D_Quad_P7_N12);
      case 8 :       return(&Quadrature2D_Quad_P8_N16);
      case 9 :       return(&Quadrature2D_Quad_P9_N17);
      case 10 :      return(&Quadrature2D_Quad_P11_N24);
      case 11 :      return(&Quadrature2D_Quad_P11_N24);
      case 12 :      return(&Quadrature2D_Quad_P13_N33);
      case 13 :      return(&Quadrature2D_Quad_P13_N33);
      default :      return(&Quadrature2D_Quad_P13_N33);
      }
    }
  case 3 :
    switch (n)
    {
    case 4 :
      switch (order)
      {
      case 0 :  return(&Quadrature3D40);
      case 1 :  return(&Quadrature3D41);
      case 2 :  return(&Quadrature3D42);
      case 3 :  return(&Quadrature3D43);
      default : return(&Quadrature3D45);
      }
    case 5 : return(&Quadrature3D52);
    case 6 :
      switch (order)
      {
      case 0 :  return(&Quadrature3D60);
      default : return(&Quadrature3D62);
      }
    case 8 :
      switch (order)
      {
      case 0 : return(&Quadrature2D_Hex_P1_N1);
      case 1 : return(&Quadrature3D82);
      case 2 : return(&Quadrature3D82);
      case 3 : return(&Quadrature3D_Hex_P3_N6);
      case 4 : return(&Quadrature3D_Hex_P5_N14);
      case 5 : return(&Quadrature3D_Hex_P5_N14);
      case 6 : return(&Quadrature3D_Hex_P7_N31);
      case 7 : return(&Quadrature3D_Hex_P7_N31);
      case 8 : return(&Quadrature3D_Hex_P8_N47);
      case 9 : return(&Quadrature3D_Hex_P9_N58);
      case 10 : return(&Quadrature3D_Hex_P11_N90);
      case 11 : return(&Quadrature3D_Hex_P11_N90);
      default : return(&Quadrature3D_Hex_P11_N90);
      }
    }
  }
  return(NULL);
}

/****************************************************************************/
/*D
   GetSymmetricQuadratureRule - provide a quadrature formula with axial-symmetric integration points

   SYNOPSIS:
   QUADRATURE *GetSymmetricQuadratureRule(INT dim, INT n, INT order);

   PARAMETERS:
   .  dim - dimension of the formular
   .  n - number of corners of the element
   .  order - order of approximation

   DESCRIPTION:
   This function returns a pointer to a quadrature formula.  The
   quadrature rule uses symmetric integration points in 1D and 2D,
   i.e. the integration points are symmetric w.r.t. the point 0.5 in
   the 1D case and symmetric to the x- and y-axis in 2D.

   It is different from GetQuadrature in the case that the quadrature
   formula of order 'order' is not exactly available.  'GetQuadrature'
   just returns the highest order formula, GetQuadratureRule returns
   the formula of the smallest degree that integrates exactly until 'order'.

   EXAMPLE:
   .vb
   QUADRATURE *quadrature;

   if ((quadrature = GetSymmetricQuadratureRule(2,3,2)) == NULL)
          return(1);
   sum = 0.0;
   for (l=0; l<Q_NIP(quadrature); l++)
   {
    LOCAL_TO_GLOBAL(n,x,Q_LOCAL(quadrature,l),global);
    (*SourceFunc)(global,val);
    sum += val * Q_WEIGHT(quadrature,l);
   }
   sum = val * AreaOfTriangle;
   .ve

   RETURN VALUE:
   QUADRATURE *
   .n    pointer to quadrature
   .n    NULL if the quadrature formula cannot be found.
   D*/
/****************************************************************************/

QUADRATURE* NS_DIM_PREFIX GetSymmetricQuadratureRule(INT dim, INT n, INT order)
{
  switch (dim)
  {
  case 1 :
    switch (order)
    {
    case 0 :       return(&Quadrature1D1);
    case 1 :       return(&Quadrature1D1);
    case 2 :       return(&Quadrature1D3);
    case 3 :       return(&Quadrature1D3);
    case 4 :       return(&Quadrature1D5);
    case 5 :       return(&Quadrature1D5);
    case 6 :       return(&Quadrature1D7);
    case 7 :       return(&Quadrature1D7);
    case 8 :       return(&Quadrature1D9);
    case 9 :       return(&Quadrature1D9);
    case 10 :      return(&Quadrature1D11);
    case 11 :      return(&Quadrature1D11);
    case 12 :      return(&Quadrature1D13);
    case 13 :      return(&Quadrature1D13);
    case 14 :      return(&Quadrature1D15);
    case 15 :      return(&Quadrature1D15);
    case 16 :      return(&Quadrature1D17);
    case 17 :      return(&Quadrature1D17);
    case 18 :      return(&Quadrature1D19);
    case 19 :      return(&Quadrature1D19);
    default : return(&Quadrature1D19);
    }
  case 2 :
    switch (n)
    {
    case 3 :
      switch (order)
      {
      case 0 :       return(&Quadrature2D_Triangle_P1_N1);
      case 1 :       return(&Quadrature2D_Triangle_P1_N1);
      case 2 :       return(&Quadrature2D_Triangle_P2_N3b);
      case 3 :       return(&Quadrature2D_Triangle_P3_N4);
      case 4 :       return(&Quadrature2D_Triangle_P4_N6);
      case 5 :       return(&Quadrature2D_Triangle_P5_N7);
      case 6 :       return(&Quadrature2D_Triangle_P6_N12a);
      case 7 :       return(&Quadrature2D_Triangle_P7_N12);
      case 8 :       return(&Quadrature2D_Triangle_P8_N16);
      case 9 :       return(&Quadrature2D_Triangle_P9_N19);
      case 10 :      return(&Quadrature2D_Triangle_P10_N25a);
      case 11 :      return(&Quadrature2D_Triangle_P11_N28);
      case 12 :      return(&Quadrature2D_Triangle_P12_N33);
      default :      return(&Quadrature2D_Triangle_P12_N33);
      }
    case 4 :
      switch (order)
      {
      case 0 :
      case 1 :       return(&Quadrature2D_Quad_P1_N1);
      case 2 :       return(&Quadrature2D_Quad_P2_N4);
      case 3 :       return(&Quadrature2D_Quad_P3_N4a);
      case 4 :
      case 5 :       return(&Quadrature2D_Quad_P5_N7);
      case 6 :
      case 7 :       return(&Quadrature2D_Quad_P7_N12);
      case 8 :
      case 9 :       return(&Quadrature2D_Quad_P9_N20b);
      case 10 :      return(&Quadrature2D_Quad_P11_N24);
      default :      return(&Quadrature2D_Quad_P11_N24);
      }
    }
  case 3 :
    switch (n)
    {
    case 4 :
      switch (order)
      {
      case 0 :  return(&Quadrature3D40);
      case 1 :  return(&Quadrature3D41);
      case 2 :  return(&Quadrature3D42);
      case 3 :  return(&Quadrature3D43);
      default : return(&Quadrature3D45);
      }
    case 5 : return(&Quadrature3D52);
    case 6 :
      switch (order)
      {
      case 0 :  return(&Quadrature3D60);
      default : return(&Quadrature3D62);
      }
    case 8 :
      switch (order)
      {
      case 0 : return(&Quadrature2D_Hex_P1_N1);
      case 1 : return(&Quadrature3D82);
      case 2 : return(&Quadrature3D82);
      case 3 : return(&Quadrature3D_Hex_P3_N6);
      case 4 : return(&Quadrature3D_Hex_P5_N14);
      case 5 : return(&Quadrature3D_Hex_P5_N14);
      case 6 : return(&Quadrature3D_Hex_P7_N31);
      case 7 : return(&Quadrature3D_Hex_P7_N31);
      case 8 : return(&Quadrature3D_Hex_P8_N47);
      case 9 : return(&Quadrature3D_Hex_P9_N58);
      case 10 : return(&Quadrature3D_Hex_P11_N90);
      case 11 : return(&Quadrature3D_Hex_P11_N90);
      default : return(&Quadrature3D_Hex_P11_N90);
      }
    }
  }
  return(NULL);
}

static void TestQuadratureRules( void )
{
  INT i,n;
  DOUBLE w;
  QUADRATURE *q[32];

  printf("**** Testing quadrature rules ********\n");

#ifdef __TWODIM__
  printf("First Test: Sum of quadrature weights should 1.0.\n");
  q[0]=&Quadrature2D_Triangle_P1_N1;
  q[1]=&Quadrature2D_Triangle_P2_N3a;
  q[2]=&Quadrature2D_Triangle_P2_N3b;
  q[3]=&Quadrature2D_Triangle_P3_N4;
  q[4]=&Quadrature2D_Triangle_P3_N6a;
  q[5]=&Quadrature2D_Triangle_P3_N6b;
  q[6]=&Quadrature2D_Triangle_P3_N6c;
  q[7]=&Quadrature2D_Triangle_P3_N7;
  q[8]=&Quadrature2D_Triangle_P4_N6;
  q[9]=&Quadrature2D_Triangle_P4_N7;
  q[10]=&Quadrature2D_Triangle_P4_N9;
  q[11]=&Quadrature2D_Triangle_P5_N7;
  q[12]=&Quadrature2D_Triangle_P7_N9;
  q[13]=&Quadrature2D_Triangle_P6_N12a;
  q[14]=&Quadrature2D_Triangle_P6_N12b;
  q[15]=&Quadrature2D_Triangle_P6_N13;
  q[16]=&Quadrature2D_Triangle_P7_N12;
  q[17]=&Quadrature2D_Triangle_P7_N13;
  q[18]=&Quadrature2D_Triangle_P7_N15;
  q[19]=&Quadrature2D_Triangle_P8_N16;
  q[20]=&Quadrature2D_Triangle_P9_N19;
  q[21]=&Quadrature2D_Triangle_P10_N25a;
  q[22]=&Quadrature2D_Triangle_P10_N25b;
  q[23]=&Quadrature2D_Triangle_P11_N28;
  q[24]=&Quadrature2D_Triangle_P12_N33;

  for(n=0; n<25; n++) {
    w=0.0;
    for (i=0; i<q[n]->nip; i++)
      w+=q[n]->weight[i];
    printf("    Triangle rule %d, Degree %d, %d Gauss points.  Sum of weights=%g\n",
           n,q[n]->order,q[n]->nip,w);
  }

  q[0]=&Quadrature2D_Quad_P1_N1;
  q[1]=&Quadrature2D_Quad_P2_N4;
  q[2]=&Quadrature2D_Quad_P3_N4a;
  q[3]=&Quadrature2D_Quad_P3_N4b;
  q[4]=&Quadrature2D_Quad_P4_N6;
  q[5]=&Quadrature2D_Quad_P5_N8;
  q[6]=&Quadrature2D_Quad_P6_N10a;
  q[7]=&Quadrature2D_Quad_P6_N10b;
  q[8]=&Quadrature2D_Quad_P7_N12;
  q[9]=&Quadrature2D_Quad_P8_N16;
  q[10]=&Quadrature2D_Quad_P9_N17;
  q[11]=&Quadrature2D_Quad_P9_N18;
  q[12]=&Quadrature2D_Quad_P9_N18b;
  q[13]=&Quadrature2D_Quad_P9_N20a;
  q[14]=&Quadrature2D_Quad_P9_N20b;
  q[15]=&Quadrature2D_Quad_P11_N24;
  q[16]=&Quadrature2D_Quad_P13_N33;

  for(n=0; n<17; n++) {
    w=0.0;
    for (i=0; i<q[n]->nip; i++)
      w+=q[n]->weight[i];
    printf("    Quad rule %d, Degree %d, %d Gauss points.  Sum of weights=%g\n",
           n,q[n]->order,q[n]->nip,w);
  }
#else
  q[0]=&Quadrature2D_Hex_P1_N1;
  q[1]=&Quadrature3D_Hex_P3_N6;
  q[2]=&Quadrature3D_Hex_P5_N14;
  q[3]=&Quadrature3D_Hex_P7_N31;
  q[4]=&Quadrature3D_Hex_P7_N38;
  q[5]=&Quadrature3D_Hex_P8_N47;
  q[6]=&Quadrature3D_Hex_P9_N58;
  q[7]=&Quadrature3D_Hex_P11_N90;

  for(n=0; n<8; n++) {
    w=0.0;
    for (i=0; i<q[n]->nip; i++)
      w+=q[n]->weight[i];
    printf("    Hex rule %d, Degree %d, %d Gauss points.  Sum of weights=%g\n",
           n,q[n]->order,q[n]->nip,w);
  }
#endif

  q[0]=&Quadrature2D_Quad_P1_N1;
  q[1]=&Quadrature2D_Quad_P2_N4;
  q[2]=&Quadrature2D_Quad_P3_N4a;
  q[3]=&Quadrature2D_Quad_P3_N4b;
  q[4]=&Quadrature2D_Quad_P4_N6;
  q[5]=&Quadrature2D_Quad_P5_N8;
  q[6]=&Quadrature2D_Quad_P6_N10a;
  q[7]=&Quadrature2D_Quad_P6_N10b;
  q[8]=&Quadrature2D_Quad_P7_N12;
  q[9]=&Quadrature2D_Quad_P8_N16;
  q[10]=&Quadrature2D_Quad_P9_N17;
  q[11]=&Quadrature2D_Quad_P9_N18;
  q[12]=&Quadrature2D_Quad_P9_N18b;
  q[13]=&Quadrature2D_Quad_P9_N20a;
  q[14]=&Quadrature2D_Quad_P9_N20b;
  q[15]=&Quadrature2D_Quad_P11_N24;
  q[16]=&Quadrature2D_Quad_P13_N33;

  for(n=0; n<17; n++) {
    char buf[256];
    FILE *fp;
    sprintf(buf,"quad_rule_%d_%d_%d",n,q[n]->order,q[n]->nip);
    fp=fopen(buf,"w");
    for (i=0; i<q[n]->nip; i++)
      fprintf(fp,"%g %g\n",   q[n]->local[i][0],q[n]->local[i][1]);
    fclose(fp);
  }

}

INT NS_DIM_PREFIX GaussPoints(INT dim, INT n, INT order, DOUBLE_VECTOR *x, GAUSS_POINT *gp)
{
  DOUBLE Jdet,area;
  INT ip,nip;
  QUADRATURE *quadrature;

  AREA_OF_REF(n,area);

  if ((quadrature = GetQuadrature(dim,n,order)) == NULL)
    return(1);

  nip = Q_NIP(quadrature);
  for (ip=0; ip<nip; ip++, gp++)
  {
    V_DIM_COPY(Q_LOCAL(quadrature,ip),G_LOCAL(gp));
    LOCAL_TO_GLOBAL (n,x,G_LOCAL(gp),G_GLOBAL(gp));
    INVERSE_TRANSFORMATION(n,x,G_LOCAL(gp),G_JINV(gp),Jdet);
    G_WEIGHT(gp) = Q_WEIGHT(quadrature,ip) * area * ABS(Jdet);
  }

  return(nip);
}
