/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_TASK_H_
#define _GB2_TASK_H_

#include "core_api.h"

#include <QtCore/QDateTime>
#include <QtCore/QStringList>

namespace GB2 {

class TaskScheduler;

class GB2_COREAPI_EXPORT TaskStateInfo {
public:
	TaskStateInfo() : progress(-1), cancelFlag(false) {}
	/* Percent values in range 0..100, negative if unknown. */
	int     progress;
	bool    cancelFlag;
	QString stateDesc;
	QString error;

	bool hasErrors() const {return !error.isEmpty();}
};

class GB2_COREAPI_EXPORT TaskTimeInfo {
public:
	QDateTime creationTime;
	QDateTime startTime;
	QDateTime finishTime;
};

enum TaskFlag {
    TaskFlag_None = 0x0,

    TaskFlag_NoRun = 1<<1,
	
	TaskFlag_DeleteWhenFinished = 1<<2,
	
    TaskFlag_RunAfterAllSubtasksFinished = 1<<3,
	
	TaskFlag_SerialSubtasks	= 1<<4,
    
    TaskFlag_StopOnSubtaskError = 1<<5, // works only in bundle with TaskFlag_SerialSubtasks

    TaskFlag_FailOnSubtaskError = 1<<6, //subtask error is propagated automatically

    TaskFlag_FailOnSubtaskCancel = 1<<7 // error and cancel flag are set if subtask was canceled and parent is neither canceled nor have errors

};

#define TaskFlags_NR_DWF            (TaskFlags(TaskFlag_NoRun) | TaskFlag_DeleteWhenFinished)
#define TaskFlags_NR_DWF_SS         (TaskFlags_NR_DWF | TaskFlag_SerialSubtasks)
#define TaskFlags_NR_DWF_SSSOR      (TaskFlags_NR_DWF_SS | TaskFlag_StopOnSubtaskError)
#define TaskFlags_RASF_DWF          (TaskFlags(TaskFlag_RunAfterAllSubtasksFinished) | TaskFlag_DeleteWhenFinished)
#define TaskFlags_RASF_DWF_SS       (TaskFlags_RASF_DWF | TaskFlag_SerialSubtasks)
#define TaskFlags_RASF_DWF_SSSOR    (TaskFlags_RASF_DWF_SS| TaskFlag_StopOnSubtaskError)
#define TaskFlags_FAIL_OSCOE        (TaskFlags(TaskFlag_FailOnSubtaskError) | TaskFlag_FailOnSubtaskCancel)

typedef QFlags<TaskFlag> TaskFlags;

class GB2_COREAPI_EXPORT Task : public QObject {
	Q_OBJECT
	friend class TaskScheduler;
public:

	enum State {
		State_New,
		State_Prepared,
		State_Running,
		State_Finished
	};

	enum ProgressManagement {
		Progress_Manual,
		Progress_SubTasksBased
	};

	enum ReportResult {
		ReportResult_Finished,
		ReportResult_CallMeAgain
	};


	//Creates new task with State_New state
	Task(const QString& _name, TaskFlags f);

	//Prepares Task to run
	//Task must request/prepare all resources it needs, create subtasks and define progress management type
	//This method called after Task is added to Scheduler from the main thread
	//After calling this method task gets State_Prepared state
	virtual void prepare(){}

	// Called by Scheduler from the separate thread. No updates to Project/Document model can be done from this method
	// Task gets State_Running state when its first of its subtasks is run
	virtual void run() {};
	
	// Called from the main thread after run() is finished
	// Task must report all of it results if needed.
	// If task can't report right now (for example a model is state-locked)
	// task can return ReportResult_CallMeAgain.
	// If task locks some resources, it's a good place to release them
	// After task reporting succeeds, it gets State_Finished state
	virtual ReportResult report() {return ReportResult_Finished;}

	// Set's cancelFlag to true. Does not wait for task to be stopped
	void cancel();

    bool isCanceled() const {return stateInfo.cancelFlag;}

	// Returns subtasks of the task. Task must prepare it's subtask on prepare() call and can't change them latter.
	QList<Task*> getSubtasks() const {return subtasks;}
	
	QString getTaskName() const {return taskName;}
	
	State getState() const {return state;}

	const TaskStateInfo& getStateInfo() const { return stateInfo; }
	
	const TaskTimeInfo&  getTimeInfo() const { return timeInfo; }

	int getProgress() const {return stateInfo.progress;}

	ProgressManagement getProgressManagementType() const {return tpm;}

	TaskFlags getFlags() const {return flags;}

	bool hasFlags(TaskFlags f) const {return flags & f;}

	void addSubTask(Task* sub);

    bool hasErrors() const {return stateInfo.hasErrors();}

    const QString& getError() const {return stateInfo.error;}

    bool isFinished() const {return state == Task::State_Finished;}

    bool isRunning() const {return state == Task::State_Running;}

    bool isPrepared() const {return state == Task::State_Prepared;}

    bool isNew() const {return state == Task::State_New;}

    // when called for a finished task it must deallocate all resources it keeps
    virtual void cleanup();

    virtual bool hasSubtasksWithErrors() const  { return getSubtaskWithErrors() != NULL; }

    virtual bool propagateSubtaskError();
    
    virtual Task* getSubtaskWithErrors() const;

    virtual qint64 getTaskId() const {return taskId;}

    virtual bool isTopLevelTask() const {return getParentTask() == 0;}

    virtual Task* getParentTask() const {return parentTask;}

    virtual Task* getTopLevelParentTask() {return isTopLevelTask() ? this : parentTask->getTopLevelParentTask();}

    virtual bool isReportingSupported() const {return reportSupported;}
    
    virtual bool isReportingEnabled() const {return reportEnabled;}

    virtual void setReportingEnabled(bool v) {assert(isReportingSupported()); reportEnabled = v;}

    virtual QString generateReport() const {assert(0); return QString();}

    
    float getSubtaskProgressWeight() const {return progressWeightAsSubtask;}
    
    void setSubtaskProgressWeight(float v) {progressWeightAsSubtask = v;}

    bool useDescriptionFromSubtask() const {return useDescFromSubtask;}
    
    void setUseDescriptionFromSubtask(bool v) { useDescFromSubtask = v;}

    bool isVerboseLogMode() const {return verboseLog;}
    
    void setVerboseLogMode(bool v) {verboseLog = v;}

signals:
	void si_subtaskAdded(Task* sub);

	void si_progressChanged();
    void si_descriptionChanged();
    void si_stateChanged();

protected:
	/// Called by scheduler when subtask is finished.
	virtual QList<Task*> onSubTaskFinished(Task* subTask);

    void setTaskName(const QString& taskName);


	TaskStateInfo		stateInfo;
	TaskTimeInfo		timeInfo;
	TaskFlags			flags;
    ProgressManagement	tpm;

    //todo: move to TaskFlags ?
    bool                reportSupported;
    bool                reportEnabled;
    bool                useDescFromSubtask;
    bool                verboseLog; //for top level tasks verbose means that task start/stop info is dumped to info category
    float               progressWeightAsSubtask;

private:
	QString				taskName;
	State				state;
    Task*               parentTask;
	QList<Task*>		subtasks;
    qint64              taskId;
};


class GB2_COREAPI_EXPORT TaskScheduler : public QObject {
	Q_OBJECT
public:

	virtual void registerTopLevelTask(Task* t) = 0;
	
	virtual void unregisterTopLevelTask(Task* t) = 0;

	virtual const QList<Task*>& getTopLevelTasks() const = 0;

	virtual QDateTime estimatedFinishTime(Task*) const = 0;

	virtual void cancelAllTasks() = 0;

    virtual QString getStateName(Task* t) const = 0;

protected:
	TaskStateInfo&	getTaskStateInfo(Task* t) {return t->stateInfo;}

	TaskTimeInfo&	getTaskTimeInfo(Task* t) {return t->timeInfo;}

	void emit_taskProgressChanged(Task* t) {emit t->si_progressChanged();}
    
    void emit_taskDescriptionChanged(Task* t) {emit t->si_descriptionChanged();}

	QList<Task*> onSubTaskFinished(Task* parentTask, Task* subTask) {return parentTask->onSubTaskFinished(subTask);}

    void addSubTask(Task* t, Task* sub);
        
	void setTaskState(Task* t, Task::State newState);

signals:
	void si_topLevelTaskRegistered(Task*);

	void si_topLevelTaskUnregistered(Task*);

	void si_stateChanged(Task* task);
};

} //namespace

#endif
