/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AnnotationTableObject.h"
#include "GObjectTypes.h"

#include <QtGui/QTextDocument>

namespace GB2 {

Annotation::Annotation(SharedAnnotationData _d): obj(NULL), d(_d)
{
}

Annotation::~Annotation() {
	//todo: add state checks?
}

QString Annotation::getQualifiersTip(int maxRows) const {
	QString tip;
	if (d->qualifiers.size() != 0) {
		const int QUALIFIER_VALUE_CUT = 40;
		int rows = 0;
		tip += "<nobr>";
		bool first = true;
		foreach (Qualifier q, d->qualifiers) {
            if (++rows > maxRows) {
                break;
            }
			QString val = q.getQualifierValue();
			if(val.length() > QUALIFIER_VALUE_CUT) {
				val = val.left(QUALIFIER_VALUE_CUT) + " ...";
			}
            if (first) {
                first = false;
            } else {
                tip +=	"<br>";
            }
			tip += "<b>" + Qt::escape(q.getQualifierName()) + "</b> = " + Qt::escape(val);
		}
		tip += "</nobr>";
	}
	return tip;
}

const QString& Annotation::getAnnotationName() const {
	return d->name;
}

bool Annotation::isOnComplementStrand() const {
    return d->complement;
}

TriState Annotation::getAminoStrand() const {
    return d->aminoStrand;
}

const QList<LRegion>& Annotation::getLocation() const  {
	return d->location;
}

const QList<Qualifier>& Annotation::getQualifiers() const {
	return d->qualifiers;
}

void Annotation::setAnnotationName(QString& newName) {
	assert(!newName.isEmpty());
	QString oldName = d->name;
	d->name = newName;
	if (obj!=NULL) {
		AnnotationModification md(AnnotationModification_NameChanged, this);
		obj->emit_onAnnotationModified(md);
	}
}

void Annotation::setOnComplementStrand(bool v) {
    if (d->complement == v) {
        return;
    }
    d->complement = v;
	if (obj!=NULL) {
		AnnotationModification md(AnnotationModification_LocationChanged, this);
		obj->emit_onAnnotationModified(md);
	}
}

void Annotation::setAminoStrand(TriState v) {
    if (getAminoStrand() == v) {
        return;
    }
    d->aminoStrand = v;
    if (obj!=NULL) {
        AnnotationModification md(AnnotationModification_LocationChanged, this);
        obj->emit_onAnnotationModified(md);
    }
}

void Annotation::addLocationRegion(const LRegion& reg) {
#ifdef _DEBUG
	//check that annotation region does not cross other regions
    foreach(const LRegion& tmp, getLocation()) {
		assert(!tmp.intersects(reg));
	}
#endif
	d->location.push_back(reg);
	if (obj!=NULL) {
		AnnotationModification md(AnnotationModification_LocationChanged, this);
		obj->emit_onAnnotationModified(md);
	}
}

void Annotation::removeLocationRegion(const LRegion& reg) {
	d->location.removeOne(reg);
	if (obj!=NULL) {
		AnnotationModification md(AnnotationModification_LocationChanged, this);
		obj->emit_onAnnotationModified(md);
	}
}

void Annotation::removeQualifier(const Qualifier& q) {
	assert(d->qualifiers.contains(q));

	if (obj!=NULL) {
		QualifierModification md(AnnotationModification_QualifierRemoving, this, q);
		obj->emit_onAnnotationModified(md);
	}
	d->qualifiers.removeOne(q);
}

//////////////////////////////////////////////////////////////////////////
// Group

AnnotationGroup::AnnotationGroup(AnnotationTableObject* p, const QString& _name, AnnotationGroup* parentGrp) 
: name(_name), obj(p), parentGroup(parentGrp)
{
    assert(!name.isEmpty() && (!name.contains('/') || name == AnnotationGroup::ROOT_GROUP_NAME));
}

AnnotationGroup::~AnnotationGroup() {
    //annotations are not removed here -> contract with ~AnnotationTableObject
    foreach(AnnotationGroup* g, subgroups) {
        delete g;
    }
}
void AnnotationGroup::findAllAnnotationsInGroupSubTree(QSet<Annotation*>& set) const {
    set+=QSet<Annotation*>::fromList(annotations);
    foreach(AnnotationGroup* g, subgroups) {
        g->findAllAnnotationsInGroupSubTree(set);
    }
}

void AnnotationGroup::addAnnotation(Annotation* a) {
	assert(a->getGObject() == obj);
	assert(!annotations.contains(a) && ! a->groups.contains(this));

	annotations.append(a);
	a->groups.append(this);
	
	if (obj!=NULL) {
		obj->setModified(true);
		AnnotationGroupModification md(AnnotationModification_AddedToGroup, a, this);
		obj->emit_onAnnotationModified(md);
	}
}

void AnnotationGroup::removeAnnotations(const QList<Annotation*>& ans) {
    QList<Annotation*> toRemoveFromObj;
    foreach(Annotation* a, ans) {
        assert(annotations.contains(a) && a->groups.contains(this));
        if (a->groups.size() == 1) {
            toRemoveFromObj.append(a);
        } else {
            annotations.removeOne(a);
            a->groups.removeOne(this);
            if (obj!=NULL) {
                obj->setModified(true);
                AnnotationGroupModification md(AnnotationModification_RemovedFromGroup, a, this);
                obj->emit_onAnnotationModified(md);
            }
        }
    }
    if (!toRemoveFromObj.isEmpty()) {
        obj->removeAnnotations(toRemoveFromObj);
    }
}

void AnnotationGroup::removeAnnotation(Annotation* a) {
	assert(annotations.contains(a) && a->groups.contains(this));
    if (a->groups.size() == 1) {
        assert(a->groups.first() == this);
        obj->removeAnnotation(a);
    } else {
	    annotations.removeOne(a);
	    a->groups.removeOne(this);
        if (obj!=NULL) {
            obj->setModified(true);
            AnnotationGroupModification md(AnnotationModification_RemovedFromGroup, a, this);
            obj->emit_onAnnotationModified(md);
        }
    }
}

AnnotationGroup* AnnotationGroup::getSubgroup(const QString& path, bool create) {
	int idx = path.indexOf('/');
    QString name = idx < 0 ? path : (idx == 0 ? path.mid(idx+1) : path.left(idx));
	AnnotationGroup* group = NULL;
	foreach (AnnotationGroup *g, subgroups) {
		if (g->getGroupName() == name) {
			group = g;
			break;
		}
	}
	if (group == NULL  && create) {
		group = new AnnotationGroup(obj, name, this);
		subgroups.push_back(group);
		obj->emit_onGroupCreated(group);
	}
	if (idx <= 0 || group == NULL) {
		return group;
	}
	AnnotationGroup* result = group->getSubgroup(path.mid(idx+1), create);
	return result;
}

void AnnotationGroup::removeSubgroup(AnnotationGroup* g) {
	assert(g->getParentGroup() == this);
	if (g->getParentGroup()!=this) {
		return;
	}
	g->clear();
    subgroups.removeOne(g);
    g->obj = NULL;
	obj->emit_onGroupRemoved(this, g);
	delete g;
}

void AnnotationGroup::clear() {
	while (!subgroups.isEmpty()) {
		removeSubgroup(subgroups.first());
	}
    if (!annotations.isEmpty()) {
        foreach(Annotation* a, annotations) {
            removeAnnotation(a);
        }
    }
}

bool AnnotationGroup::isParentOf(AnnotationGroup* g) const {
    if (g->getGObject() != obj || g == this) {
        return false;
    }
    for (AnnotationGroup* pg = g->getParentGroup(); pg!=NULL; pg = pg->getParentGroup()) {
        if ( pg == this) {
            return true;
        }
    }
    return false;
}

void AnnotationGroup::setGroupName(const QString& newName) {
	if (name == newName) {
		return;
	}
	QString oldName = name;
	name = newName;
	obj->emit_onGroupRenamed(this, oldName);
}

QString AnnotationGroup::getGroupPath() const {
	if (parentGroup == NULL) {
		return QString("");
	}
	if (parentGroup->parentGroup == NULL) {
		return name;
	}
	return parentGroup->getGroupPath() + "/" + name;
}

//////////////////////////////////////////////////////////////////////////
/// Annotation table object

const QString AnnotationGroup::ROOT_GROUP_NAME("/");

AnnotationTableObject::AnnotationTableObject(const QString& objectName, const QVariantMap& hintsMap) 
: GObject(GObjectTypes::ANNOTATION_TABLE, objectName, hintsMap) 
{
	rootGroup = new AnnotationGroup(this, AnnotationGroup::ROOT_GROUP_NAME, NULL);
}

AnnotationTableObject::~AnnotationTableObject() {
	foreach(Annotation* a, annotations) {
		delete a;
	}
	delete rootGroup;
}

GObject* AnnotationTableObject::clone() const {
	AnnotationTableObject* cln = new AnnotationTableObject(getGObjectName(), getGHintsMap());
    foreach(Annotation* a, annotations) {
		Annotation* newA = new Annotation(a->d);
		const QList<AnnotationGroup*>& groups = a->getGroups();
		assert(!groups.isEmpty());
		for (int i=0, n = groups.size();i<n; i++) {
			const AnnotationGroup* oldGroup = groups[i];
			QString gPath = oldGroup->getGroupPath();
			if (i == 0) {
				cln->addAnnotation(newA, gPath);
			} else {
				AnnotationGroup* newGroup = cln->rootGroup->getSubgroup(gPath, true);
				newGroup->addAnnotation(newA);
			}
		}
	}
	cln->setModified(false);
	return cln;
}

void AnnotationTableObject::addAnnotation(Annotation* a, const QString& groupName) {
	assert(a->obj == NULL);
	const QString& aName = a->getAnnotationName();
	AnnotationGroup* defaultGroup = rootGroup->getSubgroup(groupName.isEmpty() ? aName : groupName, true); 
	a->obj = this;
	defaultGroup->addAnnotation(a);
	annotations.append(a);
	setModified(true);

	QList<Annotation*> v; v<<a;
	emit si_onAnnotationsAdded(v);
}

void AnnotationTableObject::addAnnotations(const QList<Annotation*>& list, const QString& groupName) {
    if (list.isEmpty()) {
        return;
    }
    foreach(Annotation* a, list) {
        assert(a->obj == NULL);
        const QString& aName = a->getAnnotationName();
        const QString& resultGroupName = groupName.isEmpty() ? aName : groupName;
        AnnotationGroup* defaultGroup = rootGroup->getSubgroup(resultGroupName, true); 
        a->obj = this;
        defaultGroup->addAnnotation(a);
        annotations.push_back(a);
    }
    setModified(true);

    emit si_onAnnotationsAdded(list);
}


void AnnotationTableObject::removeAnnotations(const QList<Annotation*>& annotations) {
    foreach(Annotation* a, annotations) {
        _removeAnnotation(a);
    }
    setModified(true);
    emit si_onAnnotationsRemoved(annotations);
    qDeleteAll(annotations);
}

void AnnotationTableObject::removeAnnotation(Annotation* a) {
    QList<Annotation*> v; v<<a;
    _removeAnnotation(a);
    setModified(true);
    emit si_onAnnotationsRemoved(v);
    delete a;
}

void AnnotationTableObject::_removeAnnotation(Annotation* a) {
	assert(a->getGObject() == this);
	a->obj = NULL;
	annotations.removeOne(a);
    foreach(AnnotationGroup* ag, a->getGroups()) {
        ag->annotations.removeOne(a);
    }
}

void AnnotationTableObject::selectAnnotationsByName(const QString& name, QList<Annotation*>& res) {
    foreach(Annotation* a, annotations) {
        if (a->getAnnotationName() == name) {
            res.append(a);
        }
    }
}


bool AnnotationTableObject::checkConstraints(const GObjectConstraints* c) const {
    const AnnotationTableObjectConstraints* ac = qobject_cast<const AnnotationTableObjectConstraints*>(c);
    assert(ac!=NULL);
    int fitSize = ac->sequenceSizeToFit;
    foreach(Annotation* a, annotations) {
        foreach(const LRegion& r, a->getLocation()) {
            if (r.endPos() > fitSize) {
                return false;
            }
        }
    }
    return true;
}

AnnotationTableObjectConstraints::AnnotationTableObjectConstraints(const AnnotationTableObjectConstraints& c, QObject* p) 
: GObjectConstraints(GObjectTypes::ANNOTATION_TABLE, p), sequenceSizeToFit(c.sequenceSizeToFit)
{

}

AnnotationTableObjectConstraints::AnnotationTableObjectConstraints(QObject* p) 
: GObjectConstraints(GObjectTypes::ANNOTATION_TABLE, p), sequenceSizeToFit(0) 
{
}


bool annotationLessThanByRegion(const Annotation* a1, const Annotation* a2) {
    assert(!a1->getLocation().isEmpty());
    assert(!a2->getLocation().isEmpty());

    const LRegion& r1 = a1->getLocation().first();
    const LRegion& r2 = a2->getLocation().first();
    
    return r1 < r2;
}

}//namespace

