/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MAlignmentObject.h"

namespace GB2 {

GObject* MAlignmentObject::clone() const {
	return new MAlignmentObject(msa, getGObjectName(), getGHintsMap());
}


void MAlignmentObject::insertGap(int seqNum, int pos, int nGaps) {
    assert(seqNum >=0 && seqNum <= msa.getNumSequences());
    assert(pos >=0 && pos <= msa.getLength());
    assert(nGaps > 0);
    assert(msa.isNormalized());
    assert(!isStateLocked());
    
    QByteArray gap(nGaps, MAlignment_GapChar);

    //insert nGaps to selected sequence
    //append nGaps to all other sequences
    for (int i=0, n = msa.getNumSequences(); i < n; i++) {
        MAlignmentItem& item = msa.alignedSeqs[i];
        if (i == seqNum) {
            item.sequence.insert(pos, gap);
        } else {
            item.sequence.append(gap);
        }
    }

    emit si_alignmentModified();

    setModified(true);
}

void MAlignmentObject::insertGap(int pos, int nGaps) {
    assert(pos >= 0 && pos <= msa.getLength());
    assert(nGaps > 0);
    assert(msa.isNormalized());
    assert(!isStateLocked());
    
    QByteArray gap(nGaps, MAlignment_GapChar);
    for (int i=0, n = msa.getNumSequences(); i < n; i++) {
        MAlignmentItem& item = msa.alignedSeqs[i];
        item.sequence.insert(pos, gap);
    }

    emit si_alignmentModified();
    setModified(true);
}

int MAlignmentObject::deleteGap(int seqNum, int pos, int maxGaps) {
    assert(seqNum >=0 && seqNum <= msa.getNumSequences());
    assert(maxGaps > 0 && pos + maxGaps <= msa.getLength());
    assert(pos >=0 && pos+maxGaps <= msa.getLength());
    assert(msa.isNormalized());
    assert(!isStateLocked());

    int nDeleted = 0;
    MAlignmentItem& delItem = msa.alignedSeqs[seqNum];
    for(int i = pos, n = pos + maxGaps; i<n; i++) {
        char c = delItem.sequence[i];
        if (c != MAlignment_GapChar) {
            break;
        }
        nDeleted++;
    }
    if (nDeleted == 0) {
        return 0;
    }
    delItem.sequence.remove(pos, nDeleted);
    delItem.sequence.append(QByteArray(nDeleted, MAlignment_GapChar));

    emit si_alignmentModified();
    setModified(true);

    return nDeleted;
}

int MAlignmentObject::deleteGap(int pos, int maxGaps) {
    assert(pos >=0 && pos <= msa.getLength());
    assert(msa.isNormalized());
    assert(!isStateLocked());
    
    //find min gaps in all sequences starting from pos
    int minGaps = maxGaps;
    foreach(const MAlignmentItem& item, msa.alignedSeqs) {
        int nGaps = 0;
        for (int i=pos, n = pos+maxGaps; i < n; i++, nGaps++) {
            if (item.sequence[i]!=MAlignment_GapChar) { 
                break;
            }
        }
        minGaps = qMin(minGaps, nGaps);
        if (minGaps == 0) {
            break;
        }
    }
    if (minGaps == 0) {
        return  0;
    }
    int nDeleted = minGaps;
    for (int i=0, n = msa.getNumSequences(); i < n; i++) {
        MAlignmentItem& item = msa.alignedSeqs[i];
        item.sequence.remove(pos, nDeleted);
    }


    emit si_alignmentModified();

    setModified(true);
    return nDeleted;
}

void MAlignmentObject::addSequence(QByteArray& seq, int seqNum, int gapPrefix) {
    Q_UNUSED(seq);
    Q_UNUSED(gapPrefix);
    assert(seqNum >=0 && seqNum <= msa.getNumSequences());
    assert(msa.isNormalized());
    assert(!isStateLocked());

    emit si_sequenceListModified();
    setModified(true);
}

void MAlignmentObject::removeSequence(int seqNum) {
    assert(seqNum >= 0 && seqNum <= msa.getNumSequences());
    assert(msa.isNormalized());
    assert(!isStateLocked());

    emit si_sequenceListModified();
    
    setModified(true);
}


void MAlignmentObject::setMAlignment(const MAlignment& newMa) {
    assert(!isStateLocked());
    assert(newMa.isNormalized());

    msa = newMa;

    emit si_sequenceListModified();
    
    setModified(true);
}

}//namespace


