/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ZlibAdapter.h"
#if defined(Q_OS_LINUX) || defined(Q_OS_UNIX)
#include <zlib.h>
#else
#include "zlib.h"
#endif
#include <assert.h>

namespace GB2 {

	class GzipUtil {
	public:
		GzipUtil(IOAdapter* io, bool doCompression);
		~GzipUtil(); 
		qint64 uncompress(char* outBuff, qint64 outSize);
		qint64 compress(const char* inBuff, qint64 inSize, bool finish = false);
		bool isCompressing() const {return doCompression;}
	private:
		static const int CHUNK = 16384;
		z_stream strm;
		char buf[CHUNK];
		IOAdapter* io;
		bool doCompression;
	};

	GzipUtil::GzipUtil(IOAdapter* io, bool doCompression) : io(io), doCompression(doCompression)
	{
		/* allocate inflate state */
		strm.zalloc = Z_NULL;
		strm.zfree = Z_NULL;
		strm.opaque = Z_NULL;
		strm.avail_in = 0;
		strm.next_in = Z_NULL;
		
		int ret = doCompression ? 
			/* write a simple gzip header and trailer around the compressed data */
			deflateInit2(&strm, Z_DEFAULT_COMPRESSION, Z_DEFLATED, 16 + 15, 8, Z_DEFAULT_STRATEGY) 
			/* enable zlib and gzip decoding with automatic header detection */
			: inflateInit2(&strm, 32 + 15);
		assert(ret == Z_OK);
	}

	GzipUtil::~GzipUtil()
	{
		if (doCompression) {
			int ret = compress(NULL, 0, true);
			assert(ret == 0);
			deflateEnd(&strm); 
		} else {
			inflateEnd(&strm);
		}
	}

	qint64 GzipUtil::uncompress(char* outBuff, qint64 outSize) 
	{
		/* Based on gun.c (example from zlib, copyrighted (C) 2003, 2005 Mark Adler) */
		strm.avail_out = outSize;
		strm.next_out = (Bytef*)outBuff;
		do {
			/* run inflate() on input until output buffer is full */
			if (strm.avail_in == 0) {
				// need more input
				strm.avail_in = io->readBlock(buf, CHUNK);
				strm.next_in = (Bytef*)buf;
			}
			if (strm.avail_in == quint32(-1)) {
				// TODO log error
				return -1;
			}
			if (strm.avail_in == 0)
				break;

			int ret = inflate(&strm, Z_SYNC_FLUSH);
			assert(ret != Z_STREAM_ERROR);  /* state not clobbered */
			switch (ret) {
			case Z_NEED_DICT:
			case Z_DATA_ERROR:
			case Z_MEM_ERROR:
				return -1;
			case Z_BUF_ERROR:
			case Z_STREAM_END:
				return outSize - strm.avail_out;
			}
			if (strm.avail_out != 0 && strm.avail_in != 0) {
				assert(0);
				break;
			}
		} while (strm.avail_out != 0);

		return outSize - strm.avail_out;
	}

	qint64 GzipUtil::compress(const char* inBuff, qint64 inSize, bool finish) 
	{
		int ret;
		/* Based on gun.c (example from zlib, copyrighted (C) 2003, 2005 Mark Adler) */
		strm.avail_in = inSize;
		strm.next_in = (Bytef*)inBuff;
		do {
			/* run deflate() on input until output buffer not full */
			strm.avail_out = CHUNK;
			strm.next_out = (Bytef*)buf;
			ret = deflate(&strm, finish ? Z_FINISH : Z_NO_FLUSH);
			assert(ret != Z_STREAM_ERROR);  /* state not clobbered */
			int have = CHUNK - strm.avail_out;
			qint64 l = io->writeBlock(buf, have);
			if (l != have) {
				// TODO log error
				return -1;
			}
		} while (strm.avail_out == 0);

		if (strm.avail_in != 0) {
			assert(0);     /* all input should be used */
			// TODO log error
			return -1;
		}

		assert(!finish || ret == Z_STREAM_END);        /* stream will be complete */

		return inSize;
	}

ZlibAdapter::ZlibAdapter(IOAdapter* io) 
: IOAdapter(io->getFactory()), io(io), z(NULL), buf(NULL), rewinded(0) {}		

ZlibAdapter::~ZlibAdapter() {
	close();
	delete io;
}

void ZlibAdapter::close() {
	delete z;
	z = NULL;
	if (buf) {
		delete buf->rawData();
		delete buf;
		buf = NULL;
	}
	if (io->isOpen()) io->close();
}

bool ZlibAdapter::open(const QString& url, IOAdapterMode m) {
	assert(!isOpen());
	close();
	bool res = io->open(url, m);
	if (res) {
		z = new GzipUtil(io, m == IOAdapterMode_Write);
		assert(z);
		if (m == IOAdapterMode_Read) {
			buf = new RingBuffer(new char[BUFLEN], BUFLEN);
			assert(buf);
		}
	}

	return res;
}

qint64 ZlibAdapter::readBlock(char* data, qint64 size) 
{
	if (!isOpen() || z->isCompressing()) {
		assert(0);// "not ready to read";
		return false;
	}
	// first use data put back to buffer if any
	int cached = 0;
	if (rewinded != 0) {
		assert(rewinded > 0 && rewinded <= buf->length());
		cached = buf->read(data, size, buf->length() - rewinded);
		if (cached == size) {
			rewinded -= size;
			return size;
		}
		assert(cached < size);
		rewinded = 0;
	}
	size = z->uncompress(data + cached, size - cached);
	if (size == -1) {
		return -1;
	}
	buf->append(data + cached, size);
	
	return size + cached;
}

qint64 ZlibAdapter::writeBlock(const char* data, qint64 size) {
	if (!isOpen() || !z->isCompressing()) {
		assert(0);// "not ready to write";
		return false;
	}
	qint64 l = z->compress(data, size);
	return l;
}

bool ZlibAdapter::skip(qint64 nBytes) {
	if (!isOpen() || z->isCompressing()) {
		assert(0);// "not ready to seek";
		return false;
	}
	assert(buf);
	nBytes -= rewinded;
	if (nBytes <= 0) {
		if (-nBytes <= buf->length()) {
			rewinded = -nBytes;
			return true;
		}
		return false;
	}
	rewinded = 0;
	char* tmp = new char[nBytes];
	qint64 skipped = readBlock(tmp, nBytes);
	delete tmp;
	return skipped == nBytes;
}

}//namespace
