/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GTest.h"

#include <assert.h>

#include <QtCore/QFile>

namespace GB2 {

/* TRANSLATOR GB2::GTest */

//////////////////////////////////////////////////////////////////////////
// GTest

GTest::GTest(const QString& taskName, GTest* cp, const GTestEnvironment* _env, 
             bool useRun, const QList<GTest*>& subtasks)
: Task(taskName, useRun ? TaskFlags_RASF_DWF_SSSOR : TaskFlags_NR_DWF_SSSOR), contextProvider(cp), env(_env)
{
    assert(env!=NULL);
    foreach(Task* t, subtasks) {
        addSubTask(t);
    }
}

QObject* GTest::getContext(const QString& name) const {
    assert(contextProvider != NULL);
    return contextProvider->subtestsContext.value(name);
}

void GTest::addContext(const QString& name, QObject* v) {
    assert(contextProvider != NULL);
    assert(!contextProvider->subtestsContext.contains(name));
    contextProvider->subtestsContext[name] = v;
}

void GTest::removeContext(const QString& name) {
    assert(contextProvider != NULL);
    assert(contextProvider->subtestsContext.contains(name));
    contextProvider->subtestsContext.remove(name);
}

//////////////////////////////////////////////////////////////////////////
// GTestSuite

GTestSuite::~GTestSuite() {
    foreach(GTestRef* ref, tests) {
        delete ref; 
    }
}


static QStringList findAllFiles(const QString& dirPath, const QString& ext, bool recursive, int rec) {
    QStringList res;
    if (rec > 100) { //symlink or other err
        //todo: report err?
        return res;
    }
    QDir dir(dirPath);
    
    //add files first
    QStringList files = ext.isEmpty() ? dir.entryList(QDir::Files) : dir.entryList(ext.split(":"), QDir::Files);
    foreach(const QString& file, files) {
        QFileInfo fi(dir.absolutePath() + "/" + file);
        res.append(fi.absoluteFilePath());
    }
    
    
    //process subdirs if needed
    if (recursive) {
        QStringList subDirs = dir.entryList(QDir::Dirs | QDir::NoDotAndDotDot);
        foreach(QString sub, subDirs) {
            QString subDirPath = dirPath + "/" + sub;
            QStringList subRes = findAllFiles(subDirPath, ext, true, rec++);
            res+=subRes;
        }
    }
    return res;
}

GTestSuite* GTestSuite::readTestSuite(const QString& url, QString& err) {
    QFile f(url);
    if (!f.open(QIODevice::ReadOnly)) {
        err = tr("cant_open_file");
        return NULL;
    }
    QByteArray  xmlData = f.readAll();
    f.close();

    QDomDocument suiteDoc;
    bool res = suiteDoc.setContent(xmlData);
    if (!res) {
        err = tr("not_an_xml_suite_file");
        return NULL;
    }
    
    if (suiteDoc.doctype().name()!="UGENE_TEST_FRAMEWORK_SUITE") {
        err = tr("not_a_test_suite_file");
        return NULL;
    }

    QDomElement suiteEl = suiteDoc.documentElement();
    if (suiteEl.tagName()!="suite") {
        err = tr("suite_elem_not_found");
        return NULL;
    }
    
    //Name 
    QString suiteName = suiteEl.attribute("name");
    if (suiteName.isEmpty()) {
        err = tr("suite_name_is_empty");
        return NULL;
    }

    //Env
    GTestEnvironment suiteEnv;
    QDomNodeList envNodes = suiteEl.elementsByTagName("env-var");
    for(int i=0;i<envNodes.size(); i++) {
        QDomNode n = envNodes.item(i);
        assert(n.isElement());
        if (!n.isElement()) {
            continue;
        }
        QDomElement varEl = n.toElement();
        QString varName = varEl.attribute("name");
        if (varName.isEmpty()) {
            err = tr("var_tag_without_name");
            return NULL;
        }
        QString varVal = varEl.attribute("value");
        suiteEnv.setVar(varName, varVal);
    }

    //Tests
    QFileInfo suiteUrl(url);
    QString suiteDir = suiteUrl.absoluteDir().absolutePath();
    QList<GTestRef*> suiteTests;
    QDomNodeList testDirEls = suiteEl.elementsByTagName("test-dir");
    for(int i=0;i<testDirEls.size(); i++) {
        QDomNode n = testDirEls.item(i);
        assert(n.isElement());
        if (!n.isElement()) {
            continue;
        }
        QDomElement testDirEl = n.toElement();
        QString dirPath = testDirEl.attribute("path");
        if (dirPath.isEmpty()) {
            err = tr("path_attribute_not_found");
            break;
        }
        QString fullTestDirPath = suiteDir + "/" + dirPath;
        QFileInfo testDir(fullTestDirPath);
        if (!testDir.exists()) {
            err = tr("test_dir_not_exists %1").arg(fullTestDirPath);
            break;
        }
        QString testFormatName = testDirEl.attribute("test-format");
        bool recursive = testDirEl.attribute("recursive") != "false";
        QString testExt = testDirEl.attribute("test-ext");
        QStringList testURLs = findAllFiles(fullTestDirPath, testExt, recursive, 0);
        foreach(const QString& tUrl, testURLs) {
            int shortNameLen = tUrl.length() - fullTestDirPath.length() - 1; // minus '/' char
            assert(shortNameLen > 0);
            QString tShortName = tUrl.right(shortNameLen);
            GTestRef* tref = new GTestRef(tUrl, tShortName, testFormatName);
            suiteTests.append(tref);
        }
    }
    if (!err.isEmpty()) {
        foreach(GTestRef* ref, suiteTests) {
            delete ref;
        }
        return NULL;
    }
    
    GTestSuite* suite = new GTestSuite();
    suite->url = suiteUrl.absoluteFilePath();
    suite->name = suiteName;
    suite->env  = suiteEnv;
    suite->tests = suiteTests;
    
    return suite;
}


//////////////////////////////////////////////////////////////////////////
// GTestState

void GTestState::clearState() {
    if (state == TriState_Unknown) {
        assert(errMessage.isEmpty());
        return;
    }
    errMessage.clear(); 
    state = TriState_Unknown;
    emit si_stateChanged(this);
}

void GTestState::setFailed(const QString& err) {
    if (state == TriState_No) {
        assert(!errMessage.isEmpty());
        return;
    }
    assert(!err.isEmpty());
    assert(errMessage.isEmpty());
    
    errMessage = err; 
    state = TriState_No;
    emit si_stateChanged(this);
}

void GTestState::setPassed() {
    if (state == TriState_Yes) {
        assert(errMessage.isEmpty());
        return;
    }
    errMessage.clear(); 
    state = TriState_Yes;
    emit si_stateChanged(this);
}

}//namespace
