/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AnnotationTableObjectTest.h"
#include "gobjects/MAlignmentObject.h"

#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/DocumentModel.h>
#include <core_api/GObject.h>

#include <util_tasks/LoadDocumentTask.h>
#include <gobjects/AnnotationTableObject.h>


namespace GB2 {

/* TRANSLATOR GB2::GTest */

#define VALUE_ATTR      "value"
#define DOC_ATTR        "doc"
#define OBJ_ATTR        "obj"
#define ANNOTATION_ATTR "annotation"
#define	NUMBER_ATTR     "number"
#define	COMPLEMENT_ATTR "complement"
#define	QUALIFIER_ATTR  "qualifier"
#define	LOCATION_ATTR   "location"

void GTest_CheckNumAnnotations::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

    QString v = el.attribute(VALUE_ATTR);
    if (v.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
    bool ok = false;
    num = v.toInt(&ok);
    if (!ok) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
    }
}

Task::ReportResult GTest_CheckNumAnnotations::report() {
    GObject *obj = getContext<GObject>(this, objContextName);
	if(obj==NULL) {
		stateInfo.error = GTest::tr("invalid object context");
		return ReportResult_Finished;
	}
    assert(obj!=NULL);
    AnnotationTableObject *anntbl = qobject_cast<AnnotationTableObject*>(obj);
    const QList<Annotation*>& annList = anntbl->getAnnotations();
    if (num!=annList.size()) {
        stateInfo.error = GTest::tr("annotations count not matched: %1, expected %2 ").arg(annList.size()).arg(num);
    }
    return ReportResult_Finished;
}

void GTest_FindAnnotationByNum::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    result = NULL;
    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }
    QString num_str = el.attribute(NUMBER_ATTR);
    if (num_str.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(NUMBER_ATTR);
        return;
    }
    bool ok = false;
    number = num_str.toInt(&ok);
    if(!ok || number < 0) {
        stateInfo.error = GTest::tr("invalid value: %1").arg(NUMBER_ATTR);
        return;
    }
	result = NULL;
    annotationContextName = el.attribute("index");
}

Task::ReportResult GTest_FindAnnotationByNum::report() {
    GObject *obj = getContext<GObject>(this, objContextName);
    if(obj==NULL) {
		stateInfo.error = QString("invalid GTest_FindGObjectByName context");
        return ReportResult_Finished;
	}

    AnnotationTableObject *anntbl = qobject_cast<AnnotationTableObject*>(obj);
	if(anntbl==NULL){
		stateInfo.error = QString("qobject_cast error: null-pointer annotation table");
        return ReportResult_Finished;
	}
    const QList<Annotation*>& annList = anntbl->getAnnotations();
    if(number >= annList.size()) {
        stateInfo.error = QString("annotation not found: number %1").arg(number);
        return ReportResult_Finished;
    }
    result = annList[number];
    assert(result!=NULL);
    if (!annotationContextName.isEmpty()) {
        addContext(annotationContextName, new GTestAnnotationDataItem(result->data(), this));
    }
    return ReportResult_Finished;
}

void GTest_FindAnnotationByNum::cleanup() {
    if (result!=NULL && !annotationContextName.isEmpty()) {
        removeContext(annotationContextName);
    }
}


void GTest_CheckAnnotationLocation::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    annCtxName = el.attribute(ANNOTATION_ATTR);
    if (annCtxName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }
    QString loc=el.attribute(LOCATION_ATTR);
    if (loc.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(LOCATION_ATTR);
        return;
    } 

    QString complement_str=el.attribute(COMPLEMENT_ATTR);
    if (complement_str.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(COMPLEMENT_ATTR);
        return;
    } 
    bool ok = false;
    complement = complement_str.toInt(&ok);
    if (!ok) {
        stateInfo.error = GTest::tr("value not set %1").arg(COMPLEMENT_ATTR);
    }

    QRegExp rx("(\\d+)(..)(\\d+)");
    QStringList list;
    int pos = 0;
    while ((pos = rx.indexIn(loc, pos)) != -1) {
        int start=rx.cap(1).toInt();
        int end=rx.cap(3).toInt();
        location.append(LRegion(start-1,end-start+1));
        pos += rx.matchedLength();
    }

}

Task::ReportResult GTest_CheckAnnotationLocation::report() {
    GTestAnnotationDataItem *annCtx = getContext<GTestAnnotationDataItem>(this,annCtxName);
    if(annCtx==NULL){
		stateInfo.error = GTest::tr("invalid annotation context");
        return ReportResult_Finished;
	}
    const SharedAnnotationData annotation = annCtx->getAnnotation();
    if (complement != annotation->complement) {
        stateInfo.error = GTest::tr("complement flags not matched");
        return ReportResult_Finished;
    }
    int n = location.size();
    const QList<LRegion>& alocation = annotation->location;
    if (n!=alocation.size()) {
        stateInfo.error = GTest::tr("number of regions not matched: %1, expected %2").arg(n).arg(alocation.size());
        return ReportResult_Finished;
    }
    for(int i=0; i<n; i++) {
        const LRegion& l = location[i];
        const LRegion& al = alocation[i];
        if (l!=al) {
			stateInfo.error = GTest::tr("location not matched, idx=%1, \"%2..%3\", expected \"%4..%5\"").arg(i).arg(al.startPos+1).arg(al.endPos()).arg(l.startPos+1).arg(l.endPos());
            return ReportResult_Finished;
        }
    }
    return ReportResult_Finished;
}


void GTest_CheckAnnotationQualifier::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    annCtxName = el.attribute(ANNOTATION_ATTR);
    if (annCtxName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

    qName=el.attribute(QUALIFIER_ATTR);
    if (qName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(QUALIFIER_ATTR);
        return;
    } 
    qValue =el.attribute(VALUE_ATTR);
}

Task::ReportResult GTest_CheckAnnotationQualifier::report() {
    GTestAnnotationDataItem *annCtx = getContext<GTestAnnotationDataItem>(this,annCtxName);
    if (annCtx == NULL){
		stateInfo.error = GTest::tr("invalid annotation context");
        return ReportResult_Finished;
	}
    const SharedAnnotationData a = annCtx->getAnnotation();
    QList<Qualifier> res;
    a->findQualifiers(qName, res);
    if (res.isEmpty()) {
        stateInfo.error = QString("Qualifier not found, name=%1").arg(qName);
        return ReportResult_Finished;
    }
    
    bool ok = false;
    QString value;
    foreach(const Qualifier& q, res) {
        if (q.getQualifierName() == qName) {
            value = q.getQualifierValue();
            if (value == qValue) {
                ok = true;
            }
            break;
        }
    }
    if (!ok) {
        stateInfo.error = QString("Qualifier value not matched, name=\"%1\" value=\"%2\", expected=\"%3\"").arg(qName).arg(value).arg(qValue);
    }
    return ReportResult_Finished;
}

//---------------------------------------------------------------
void GTest_CheckAnnotationsNumInTwoObjects::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC_ATTR);
        return;
    }

    secondDocContextName = el.attribute(VALUE_ATTR);
    if (secondDocContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
}

Task::ReportResult GTest_CheckAnnotationsNumInTwoObjects::report() {
    Document* doc = getContext<Document>(this, docContextName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(docContextName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, secondDocContextName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(secondDocContextName);
        return ReportResult_Finished;
    }

    const QList<GObject*>& objs = doc->getObjects();
    const QList<GObject*>& objs2 = doc2->getObjects();
    GObject*obj=NULL;
    GObject*obj2=NULL;
    AnnotationTableObject * myAnnotation;
    AnnotationTableObject * myAnnotation2;

    for(int i=0;(i!=objs.size())||(i!=objs2.size());i++){
        obj = objs.at(i);
        obj2 = objs2.at(i);
        
        if((obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE)&&(obj2->getGObjectType() == GObjectTypes::ANNOTATION_TABLE)){
            myAnnotation = qobject_cast<AnnotationTableObject*>(obj);
            if(myAnnotation == NULL){
                stateInfo.error = GTest::tr("can't cast to annotation from: %1 in position %2").arg(obj->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
            myAnnotation2 = qobject_cast<AnnotationTableObject*>(obj2);
            if(myAnnotation2 == NULL){
                stateInfo.error = GTest::tr("can't cast to annotation from: %1 in position %2").arg(obj2->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
//////////////////////////////////////////////////////////
            const QList<Annotation*>& annList = myAnnotation->getAnnotations();
            const QList<Annotation*>& annList2 = myAnnotation2->getAnnotations();
            
            if (annList2.size() != annList.size()) {
                stateInfo.error = GTest::tr("annotations count not matched: %1, expected %2 ").arg(annList2.size()).arg(annList.size());
            return ReportResult_Finished;
            }
         
//////////////////////////////////////////////////////////
        }

    }
    
    if(obj!=objs.last() ){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }
    if(obj2!=objs2.last()){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}

//---------------------------------------------------------------
void GTest_CheckAnnotationsLocationsInTwoObjects::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC_ATTR);
        return;
    }

    secondDocContextName = el.attribute(VALUE_ATTR);
    if (secondDocContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
}

Task::ReportResult GTest_CheckAnnotationsLocationsInTwoObjects::report() {
    Document* doc = getContext<Document>(this, docContextName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(docContextName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, secondDocContextName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(secondDocContextName);
        return ReportResult_Finished;
    }

    const QList<GObject*>& objs = doc->getObjects();
    const QList<GObject*>& objs2 = doc2->getObjects();
    GObject*obj=NULL;
    GObject*obj2=NULL;
    AnnotationTableObject * myAnnotation;
    AnnotationTableObject * myAnnotation2;

    for(int i=0;(i!=objs.size())||(i!=objs2.size());i++){
        obj = objs.at(i);
        obj2 = objs2.at(i);
        
        if((obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE)&&(obj2->getGObjectType() == GObjectTypes::ANNOTATION_TABLE)){
            myAnnotation = qobject_cast<AnnotationTableObject*>(obj);
            if(myAnnotation == NULL){
                stateInfo.error = GTest::tr("can't cast to annotation from: %1 in position %2").arg(obj->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
            myAnnotation2 = qobject_cast<AnnotationTableObject*>(obj2);
            if(myAnnotation2 == NULL){
                stateInfo.error = GTest::tr("can't cast to annotation from: %1 in position %2").arg(obj2->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
//////////////////////////////////////////////////////////
            const QList<Annotation*>& annList = myAnnotation->getAnnotations();
            const QList<Annotation*>& annList2 = myAnnotation2->getAnnotations();
            
            for(int n=0;(n != annList.size())&&(n != annList2.size());n++){
                if(annList.at(n)->getLocation() != annList2.at(n)->getLocation()){
                    stateInfo.error = GTest::tr("annotations locations  in position %1 not matched").arg(n);
                    return ReportResult_Finished;
                }
            }
//////////////////////////////////////////////////////////
        }

    }
    
    if(obj!=objs.last() ){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }
    if(obj2!=objs2.last()){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}

//---------------------------------------------------------------
void GTest_CheckAnnotationsQualifiersInTwoObjects::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC_ATTR);
        return;
    }

    secondDocContextName = el.attribute(VALUE_ATTR);
    if (secondDocContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
}

Task::ReportResult GTest_CheckAnnotationsQualifiersInTwoObjects::report() {
    Document* doc = getContext<Document>(this, docContextName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(docContextName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, secondDocContextName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(secondDocContextName);
        return ReportResult_Finished;
    }

    const QList<GObject*>& objs = doc->getObjects();
    const QList<GObject*>& objs2 = doc2->getObjects();
    GObject*obj=NULL;
    GObject*obj2=NULL;
    AnnotationTableObject * myAnnotation;
    AnnotationTableObject * myAnnotation2;

    for(int i=0;(i!=objs.size())||(i!=objs2.size());i++){
        obj = objs.at(i);
        obj2 = objs2.at(i);
        
        if((obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE)&&(obj2->getGObjectType() == GObjectTypes::ANNOTATION_TABLE)){
            myAnnotation = qobject_cast<AnnotationTableObject*>(obj);
            if(myAnnotation == NULL){
                stateInfo.error = GTest::tr("can't cast to annotation from: %1 in position %2").arg(obj->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
            myAnnotation2 = qobject_cast<AnnotationTableObject*>(obj2);
            if(myAnnotation2 == NULL){
                stateInfo.error = GTest::tr("can't cast to annotation from: %1 in position %2").arg(obj2->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
//////////////////////////////////////////////////////////
            const QList<Annotation*>& annList = myAnnotation->getAnnotations();
            const QList<Annotation*>& annList2 = myAnnotation2->getAnnotations();

            for(int n=0;(n != annList.size())&&(n != annList2.size());n++){
                if(annList.at(n)->getQualifiers() != annList2.at(n)->getQualifiers()){
                    stateInfo.error = GTest::tr("annotations qualifiers  in position %1 not matched").arg(n);
                    return ReportResult_Finished;
                }
            }

//////////////////////////////////////////////////////////
        }

    }
    
    if(obj!=objs.last() ){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }
    if(obj2!=objs2.last()){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}

//---------------------------------------------------------------
void GTest_CheckAnnotationsNamesInTwoObjects::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC_ATTR);
        return;
    }

    secondDocContextName = el.attribute(VALUE_ATTR);
    if (secondDocContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
}

Task::ReportResult GTest_CheckAnnotationsNamesInTwoObjects::report() {
    Document* doc = getContext<Document>(this, docContextName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(docContextName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, secondDocContextName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(secondDocContextName);
        return ReportResult_Finished;
    }

    const QList<GObject*>& objs = doc->getObjects();
    const QList<GObject*>& objs2 = doc2->getObjects();
    GObject*obj=NULL;
    GObject*obj2=NULL;
    AnnotationTableObject * myAnnotation;
    AnnotationTableObject * myAnnotation2;

    for(int i=0;(i!=objs.size())||(i!=objs2.size());i++){
        obj = objs.at(i);
        obj2 = objs2.at(i);
        
        if((obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE)&&(obj2->getGObjectType() == GObjectTypes::ANNOTATION_TABLE)){
            myAnnotation = qobject_cast<AnnotationTableObject*>(obj);
            if(myAnnotation == NULL){
                stateInfo.error = GTest::tr("can't cast to annotation from: %1 in position %2").arg(obj->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
            myAnnotation2 = qobject_cast<AnnotationTableObject*>(obj2);
            if(myAnnotation2 == NULL){
                stateInfo.error = GTest::tr("can't cast to annotation from: %1 in position %2").arg(obj2->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
//////////////////////////////////////////////////////////
            const QList<Annotation*>& annList = myAnnotation->getAnnotations();
            const QList<Annotation*>& annList2 = myAnnotation2->getAnnotations();
            
            for(int n=0;(n != annList.size())&&(n != annList2.size());n++){
                if(annList.at(n)->getAnnotationName() != annList2.at(n)->getAnnotationName()){
                    stateInfo.error = GTest::tr("annotations names  in position %1 not matched").arg(n);
                    return ReportResult_Finished;
                }
            }

//////////////////////////////////////////////////////////
        }

    }
    
    if(obj!=objs.last() ){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }
    if(obj2!=objs2.last()){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}

//---------------------------------------------------------------
QList<XMLTestFactory*> AnnotationTableObjectTest::createTestFactories() {
    QList<XMLTestFactory*> res;
    res.append(GTest_CheckNumAnnotations::createFactory());
    res.append(GTest_FindAnnotationByNum::createFactory());
    res.append(GTest_CheckAnnotationLocation::createFactory());
    res.append(GTest_CheckAnnotationQualifier::createFactory());
    res.append(GTest_CheckAnnotationsNumInTwoObjects::createFactory());
    res.append(GTest_CheckAnnotationsLocationsInTwoObjects::createFactory());
    res.append(GTest_CheckAnnotationsQualifiersInTwoObjects::createFactory());
    res.append(GTest_CheckAnnotationsNamesInTwoObjects::createFactory());
    return res;
}

}//namespace
